/* 
 * Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"
#include "wf_code_editor.h"

using namespace ScintillaNet;

using namespace MySQL;
using namespace MySQL::Forms;

//--------------------------------------------------------------------------------------------------

/**
 * Helper to set up the code editor with usually used settings.
 */
void setup_editor(Scintilla^ scintilla, bool use_tabs, int indentation, String^ language)
{
  // Setting the language currently loads most of the settings from an xml file.
  // This will later change when we switch to our own Scintilla editor impl.
  scintilla->ConfigurationManager->Language = language;
  scintilla->AcceptsReturn = true;
  scintilla->AcceptsTab = use_tabs;
  scintilla->UseFont = true;

  scintilla->Indentation->UseTabs = use_tabs;
  scintilla->Indentation->TabWidth = indentation;
  scintilla->Indentation->TabIndents = true;
  scintilla->Indentation->IndentWidth = indentation;
  scintilla->Indentation->BackspaceUnindents = true;
  scintilla->Indentation->SmartIndentType = SmartIndent::Simple;
  scintilla->Indentation->ShowGuides = true;

  scintilla->Margins->Margin0->Width = 35; // line numbers
  scintilla->Margins->FoldMarginColor = Color::FromArgb(230, 230, 230);
  scintilla->Styles[33]->BackColor = Color::FromArgb(220, 220, 220);
  scintilla->Margins->Margin1->Width = 16; // markers
  scintilla->Margins->Margin2->Width = 16; // indicators

  // Fixes improper ViewportSize from start.
  // Once row is stretched enough to make thumb appear this thumb becomes irreversible.
  scintilla->Scrolling->HorizontalWidth = 1;

  scintilla->EndOfLine->Mode = EndOfLineMode::LF;
}

//--------------------------------------------------------------------------------------------------

CodeEditorImpl::CodeEditorImpl(mforms::CodeEditor *self)
  : ViewImpl(self)
{
}

//--------------------------------------------------------------------------------------------------

bool CodeEditorImpl::create(mforms::CodeEditor *self)
{
  CodeEditorImpl^ editor= gcnew CodeEditorImpl(self);

  if (editor != nullptr)
  {
    Scintilla^ scintilla= ViewImpl::create<Scintilla>(self, editor);
    setup_editor(scintilla, true, 4, "python");

    scintilla->TextInserted += gcnew EventHandler<TextModifiedEventArgs^>(editor, &CodeEditorImpl::text_changed);
    scintilla->TextDeleted += gcnew EventHandler<TextModifiedEventArgs^>(editor, &CodeEditorImpl::text_changed);
    return true;
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

void CodeEditorImpl::text_changed(Object^ sender, TextModifiedEventArgs^ args)
{
  Scintilla^ scintilla= (Scintilla^) sender;
  if (scintilla->Tag != nullptr)
  {
    mforms::CodeEditor* editor= ViewImpl::get_backend_control<mforms::CodeEditor>(scintilla);
    if (editor != NULL)
      editor->callback();
  }
}

//--------------------------------------------------------------------------------------------------

void CodeEditorImpl::set_text(mforms::CodeEditor *self, const std::string &text)
{
  CodeEditorImpl^ editor= (CodeEditorImpl^) ObjectImpl::FromUnmanaged(self);

  if (editor != nullptr)
  {
    Scintilla^ scintilla= editor->get_control<Scintilla>();
    scintilla->Text = CppStringToNative(text);

    String^ tt = scintilla->Text;
    int i = 0;
  }
}

//--------------------------------------------------------------------------------------------------

const std::string CodeEditorImpl::get_text(mforms::CodeEditor* self, bool selection_only)
{
  CodeEditorImpl^ editor= (CodeEditorImpl^) ObjectImpl::FromUnmanaged(self);

  if (editor != nullptr)
  {
    Scintilla^ scintilla= editor->get_control<Scintilla>();
    if (selection_only)
      return NativeToCppString(scintilla->Selection->Text);
    else
      return NativeToCppString(scintilla->Text);
  }
  return "";
}

//--------------------------------------------------------------------------------------------------

bool CodeEditorImpl::has_selection(mforms::CodeEditor* self)
{
  CodeEditorImpl^ editor= (CodeEditorImpl^) ObjectImpl::FromUnmanaged(self);

  if (editor != nullptr)
  {
    Scintilla^ scintilla= editor->get_control<Scintilla>();
    return scintilla->Selection->Length > 0;
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

void CodeEditorImpl::clear_selection(mforms::CodeEditor* self)
{
  CodeEditorImpl^ editor= (CodeEditorImpl^) ObjectImpl::FromUnmanaged(self);

  if (editor != nullptr)
  {
    Scintilla^ scintilla= editor->get_control<Scintilla>();
    scintilla->Selection->Clear();
  }
}

//--------------------------------------------------------------------------------------------------

void CodeEditorImpl::set_language(mforms::CodeEditor* self, mforms::SyntaxHighlighterLanguage language)
{
  CodeEditorImpl^ editor= (CodeEditorImpl^) ObjectImpl::FromUnmanaged(self);

  if (editor != nullptr)
  {
    Scintilla^ scintilla= editor->get_control<Scintilla>();
    switch (language)
    {
      case mforms::LanguageCpp:
        setup_editor(scintilla, false, 2, "cpp"); // Currently string based. Will later use enum.
        break;

      case mforms::LanguageLua:
        setup_editor(scintilla, true, 4, "lua");
        break;

      case mforms::LanguagePython:
        setup_editor(scintilla, true, 4, "python");
        break;

      case mforms::LanguageMySQL:
        setup_editor(scintilla, false, 2, "mysql");
        break;

    }
  }
}

//--------------------------------------------------------------------------------------------------

void CodeEditorImpl::set_read_only(mforms::CodeEditor* self, bool flag)
{
  CodeEditorImpl^ editor= (CodeEditorImpl^) ObjectImpl::FromUnmanaged(self);

  if (editor != nullptr)
  {
    Scintilla^ scintilla= editor->get_control<Scintilla>();
    scintilla->IsReadOnly = flag;
  }
}

//--------------------------------------------------------------------------------------------------

