/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
 * Implementation of the listbox wrapper for the mforms library.
 */

#include "stdafx.h"
#include "wf_listbox.h"

using namespace MySQL;
using namespace MySQL::Forms;

//----------------- Static interface functions -----------------------------------------------------

bool ListBoxImpl::create(::mforms::ListBox *self, bool multi_select)
{
  ListBoxImpl ^listbox= gcnew ListBoxImpl(self);

  if (listbox != nullptr)
  {
    ListBox^ l= ViewImpl::create<ListBox>(self, listbox);
    l->Size= Size(100, 100);
    if (multi_select)
      l->SelectionMode= SelectionMode::MultiExtended;
    l->SelectedIndexChanged += gcnew System::EventHandler(&ListBoxImpl::selection_changed);
    return true;
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

void ListBoxImpl::selection_changed(System::Object ^sender, System::EventArgs ^e)
{
  Windows::Forms::ListBox^ list= (Windows::Forms::ListBox^)sender;

  if (list->Tag != nullptr)
  {
    ::mforms::ListBox* listbox= ViewImpl::get_backend_control<::mforms::ListBox>(list);
    if (listbox != 0)
      listbox->selection_changed();
  }
}

//--------------------------------------------------------------------------------------------------

void ListBoxImpl::clear(::mforms::ListBox *self)
{
  ListBoxImpl^ listbox= (ListBoxImpl^)ObjectImpl::FromUnmanaged(self);
  if (listbox != nullptr)
    listbox->clear();
}

//--------------------------------------------------------------------------------------------------

void ListBoxImpl::set_heading(::mforms::ListBox *self, const std::string &text)
{
  // TODO: what's the heading of a listbox?
}

//--------------------------------------------------------------------------------------------------

void ListBoxImpl::add_items(::mforms::ListBox *self, const list<string> &items)
{
  ListBoxImpl^ listbox= (ListBoxImpl^)ObjectImpl::FromUnmanaged(self);
  if (listbox != nullptr)
    listbox->add_items(items);
}

//--------------------------------------------------------------------------------------------------

int ListBoxImpl::add_item(::mforms::ListBox *self, const std::string &item)
{
  ListBoxImpl^ listbox= (ListBoxImpl^)ObjectImpl::FromUnmanaged(self);
  if (listbox != nullptr)
    return listbox->add_item(item);
  return -1;
}

//--------------------------------------------------------------------------------------------------

string ListBoxImpl::get_text(::mforms::ListBox *self)
{
  ListBoxImpl^ listbox= (ListBoxImpl^)ObjectImpl::FromUnmanaged(self);
  if (listbox != nullptr)
    return listbox->get_text();
  return "";
}

//--------------------------------------------------------------------------------------------------

void ListBoxImpl::set_index(::mforms::ListBox *self, int index)
{
  ListBoxImpl^ listbox= (ListBoxImpl^)ObjectImpl::FromUnmanaged(self);
  if (listbox != nullptr)
    listbox->set_index(index);
}

//--------------------------------------------------------------------------------------------------

int ListBoxImpl::get_index(::mforms::ListBox *self)
{
  ListBoxImpl^ listbox= (ListBoxImpl^)ObjectImpl::FromUnmanaged(self);
  if (listbox != nullptr)
    return listbox->get_index();
  return -1;
}

//--------------------------------------------------------------------------------------------------

std::vector<int> ListBoxImpl::get_selected_indices(::mforms::ListBox *self)
{
  ListBoxImpl^ listbox= (ListBoxImpl^)ObjectImpl::FromUnmanaged(self);
  if (listbox != nullptr)
    return listbox->get_selected_indices();
  return std::vector<int>();
}

//----------------- Actual implementation ----------------------------------------------------------

ListBoxImpl::ListBoxImpl(::mforms::ListBox *self)
  : ViewImpl(self)
{
}

//--------------------------------------------------------------------------------------------------

void ListBoxImpl::clear()
{
  get_control<ListBox>()->Items->Clear();
}

//--------------------------------------------------------------------------------------------------

void ListBoxImpl::set_heading(const std::string &text)
{
  // TODO: make composite with a box, a label and a listbox.
}

//--------------------------------------------------------------------------------------------------

void ListBoxImpl::add_items(const list<string> &items)
{
  ListBox^ listbox= get_control<ListBox>();
  listbox->BeginUpdate();
  try
  {
    for each(string entry in items)
      listbox->Items->Add(CppStringToNative(entry));
  }
  finally
  {
    listbox->EndUpdate();
  }
}

//--------------------------------------------------------------------------------------------------

int ListBoxImpl::add_item(const string &item)
{
  return get_control<ListBox>()->Items->Add(CppStringToNative(item));
}

//--------------------------------------------------------------------------------------------------

string ListBoxImpl::get_text()
{
  if (get_control<ListBox>()->SelectedIndex < 0)
    return "";

  return NativeToCppString(get_control<ListBox>()->SelectedItem->ToString());
}

//--------------------------------------------------------------------------------------------------

void ListBoxImpl::set_index(int index)
{
  get_control<ListBox>()->SelectedIndex= index;
}

//--------------------------------------------------------------------------------------------------

int ListBoxImpl::get_index()
{
  return get_control<ListBox>()->SelectedIndex;
}

//--------------------------------------------------------------------------------------------------

std::vector<int> ListBoxImpl::get_selected_indices()
{
  std::vector<int> result;
  ListBox^ listbox= get_control<ListBox>();
  for each (int index in listbox->SelectedIndices)
    result.push_back(index);
  return result;
}

//--------------------------------------------------------------------------------------------------

