/* 
 * © 2008 MySQL AB, 2008-2009 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "stdafx.h"

#include "grtdiff.h"
#include <assert.h>
#include <algorithm>
#include "diffchange.h"
#include "changefactory.h"
#include "grtlistdiff.h"

#include "grtpp_util.h"
#include "grts/structs.h"
#include "util_functions.h"

namespace grt
{

DiffChange *diff_make(const ValueRef &source, const ValueRef &target, const Omf *omf,const sigc::slot<bool, ValueRef , ValueRef ,std::string> sqlDefinitionCmp)
{
  return GrtDiff().diff(source, target, omf, sqlDefinitionCmp);
}

void diff_dump(const DiffChange &change)
{
  change.dump_log(0);
}


void diff_delete(DiffChange *change)
{
  delete change;
}


bool is_any(const ValueRef &v)
{
  return !v.valueptr() || v.type() == AnyType;
}

inline bool XOR(bool a, bool b)
{
  return a ^ b;
}

bool are_compatible(const ValueRef &source, const ValueRef &target, Type *cmptype)
{
  Type st= source.type();
  Type tt= target.type();

  if (cmptype) 
    *cmptype= (st == tt || tt == AnyType)? st: tt;

  return ((st == tt) && !is_any(source)) || XOR(is_any(source) , is_any(target));
}


bool are_compatible_lists(const BaseListRef &source, const BaseListRef &target, Type *cmptype)
{
  Type stl= is_any(source)? AnyType: source.content_type();
  Type ttl= is_any(target)? AnyType: target.content_type();

  Type type= (stl == ttl || ttl == AnyType)? stl: ttl;
  if (cmptype)
    *cmptype= type;

  return ((stl == ttl) && !is_any(source)) || XOR(is_any(source) , is_any(target)) 
      && (is_simple_type(type) || type == ObjectType);
}

DiffChange *GrtDiff::diff(const ValueRef &source, const ValueRef &target, const Omf* omf,const TSlotNormalizerSlot sqlDefinitionCmp)
{
  this->omf= omf? omf: NULL;
  return on_value(NULL, source, target, sqlDefinitionCmp);
}

DiffChange * GrtDiff::on_value( DiffChange *parent, const ValueRef &source, const ValueRef &target, const TSlotNormalizerSlot sqlDefinitionCmp )
{
  Type type;
  if (!are_compatible(source, target, &type))
    return on_uncompatible(parent, source, target);

  if (is_any(source))
    return ChangeFactory::create_value_added_change(parent, source, target);
  
  if (is_any(target))
    return ChangeFactory::create_value_removed_change(parent, source, target);

  // TODO use omf here to determine if we should replace value

  switch(type)
  {
  case IntegerType:
  case DoubleType:
  case StringType:
    return ChangeFactory::create_simple_value_change(parent, source, target);
    break;
  case ListType:
    return on_list(parent, BaseListRef::cast_from(source), BaseListRef::cast_from(target), sqlDefinitionCmp);
    break;
  case DictType:
    return on_dict(parent, DictRef::cast_from(source), DictRef::cast_from(target),sqlDefinitionCmp);
    break;
  case ObjectType:
    return on_object(parent, ObjectRef::cast_from(source), ObjectRef::cast_from(target),sqlDefinitionCmp);
    break;
  default:
    break;
  }
  assert(0);
  return NULL;
}

std::string GrtDiff::trim_zeros(const std::string& str)const
{
  if (str.empty())
    return str;
  size_t pos = str.find_first_not_of("0");
  if (pos == std::string::npos)//there is only zeroes so return "0"
    return std::string("0");
  if (pos == 0)//no leading zeroes return unaltered string
    return str;
  return str.substr(pos);
}

std::string GrtDiff::fixDefalutString(const std::string& str)const
{
  if (str.empty())
    return str;
  if (str == std::string("0000-00-00 00:00:00")) return std::string("");
  if (str == std::string("CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP")) return std::string("");
  if (str == std::string("NOW()")) return std::string("");
  if (str == std::string("CURRENT_TIMESTAMP()")) return std::string("");
  if (str == std::string("CURRENT_TIMESTAMP")) return std::string("");
  if (str == std::string("LOCALTIME()")) return std::string("");
  if (str == std::string("LOCALTIME")) return std::string("");
  if (str == std::string("LOCALTIMESTAMP")) return std::string("");
  if (str == std::string("LOCALTIMESTAMP()")) return std::string("");
  if (str == std::string("TRUE")) return std::string("1");
  if (str == std::string("FALSE")) return std::string("0");
  if (strcasecmp(str.c_str(),"NULL") == 0) return std::string("0");
  return trim_zeros(str);
};

class NameOnlyComparer
{
public:
  bool comparison(const ValueRef obj1, const ValueRef obj2, const std::string name)const
  {
    return name != "name";
  }
};

DiffChange *GrtDiff::on_object(DiffChange *parent, const ObjectRef &source, const ObjectRef &target,const TSlotNormalizerSlot sqlDefinitionCmp)
{
  ChangeSet changes;
  MetaClass *meta= source.get_metaclass();

  if (meta->has_member("isStub"))
  {
    ValueRef v1= source.get_member("isStub");
    ValueRef v2= target.get_member("isStub");
    if ((1 == IntegerRef::cast_from(v1)) || (1 == IntegerRef::cast_from(v2)))
      return NULL;
  }
  do
  {
    for (MetaClass::MemberList::const_iterator iter= meta->get_members_partial().begin(); iter != meta->get_members_partial().end(); ++iter)
    {
      if (iter->second.overrides) continue;

      std::string name1= iter->second.name;

      std::string attr;

      int dontfollow= !iter->second.owned_object && (name1 != "flags") && (name1 != "columns");//iter->second.type.type == grt::internal::List::static_type();

      attr= meta->get_member_attribute(name1, "dontdiff");
      int dontdiff= attr.size() && (atoi(attr.c_str()) & omf->dontdiff_mask);

      if (dontdiff || (name1 == "owner"))
        continue;

      ValueRef v1= source.get_member(name1);
      ValueRef v2= target.get_member(name1);

      //For IndexColumn names should be ignored
        if(name1 == "name" && source.is_instance("db.IndexColumn") && target.is_instance("db.IndexColumn") &&
          StringRef::can_wrap(v1) && StringRef::can_wrap(v2))
          continue;

        if (v1.valueptr() || v2.valueptr())
        {
          if(name1 == "defaultValue")
          {
            std::string s1= StringRef::cast_from(v1).c_str();
            std::string s2= StringRef::cast_from(v2).c_str();
            s1.erase(std::remove_if(s1.begin(),s1.end(),std::bind2nd(std::equal_to<std::string::value_type>(),'\'')),s1.end());
            s2.erase(std::remove_if(s2.begin(),s2.end(),std::bind2nd(std::equal_to<std::string::value_type>(),'\'')),s2.end());
            s1 = fixDefalutString(s1);
            s2 = fixDefalutString(s2);
            bool eq= (strcmp(s1.c_str(), s2.c_str()) == 0);
            if(eq)
              continue;
          }  

          if (sqlDefinitionCmp(source,target,name1))
            continue;

          //        if (name1 == "sqlDefinition") continue;
          //        if (name1 == "routines") continue;
          //        if (name1 == "triggers") continue;
          //        if (name1 == "returnDatatype") continue;
          //        if (name1 == "params") continue;
          /*
          if (name1 == "characterSetName") continue;
          if (name1 == "collationName") continue; 
          if (name1 == "flags") continue;
          */
          //v2 is our model
          DiffChange* change;
          if (name1 == "referencedColumns")
          {
            NameOnlyComparer comparer;
            change = ChangeFactory::create_object_attr_modified_change(
            parent, source, target, name1, on_value(NULL, v1, v2,sigc::mem_fun(comparer,&NameOnlyComparer::comparison)));
          }
          else
          change = ChangeFactory::create_object_attr_modified_change(
            parent, source, target, name1, dontfollow?
            ChangeFactory::create_simple_value_change(parent, v1, v2):
          on_value(NULL, v1, v2,sqlDefinitionCmp));

          changes.append(change);
#if 0
         if (name1 == "indices")
            continue;
          if (name1 == "formattedRawType")
          {
            //          std::cout<<"v1("<<GrtObjectRef::cast_from(target)->name().c_str()<<") v2("<<GrtObjectRef::cast_from(source)->name().c_str()<<")";
            //          std::cout<<name1<<" v1("<<StringRef::cast_from(v1).c_str()<<") v2("<<StringRef::cast_from(v2).c_str()<<")"<<std::endl;        
          }

          //        continue;

          //        if (name1 != "columns") continue;
          if (change)
          //if (name1 == "referencedColumns")
          {
            std::cout<<name1;
            if (dontfollow)
            {
              std::cout<<" Simple ";
              if (v1.is_valid() && v2.is_valid())
                if (v1.type() == internal::Object::static_type())
                  std::cout<<"v1("<<GrtObjectRef::cast_from(v1)->name().c_str()<<") v2("<<GrtObjectRef::cast_from(v2)->name().c_str()<<")";
                else if(v1.type() == internal::String::static_type())
                  std::cout<<"v1("<<StringRef::cast_from(v1).c_str()<<") v2("<<StringRef::cast_from(v2).c_str()<<")";
            }
            else
            {
              if (v1.type() == internal::String::static_type())
                std::cout<<" v1("<<StringRef::cast_from(v1).c_str()<<") v2("<<StringRef::cast_from(v2).c_str()<<")"<<std::endl;
//              ChangeFactory::create_object_attr_modified_change(
//                parent, source, target, name1, on_value(NULL, v1, v2,sqlDefinitionCmp));
            }
            std::cout<<std::endl;
          };
#endif
        }

    }

    meta= meta->parent();
  }
  while (meta != 0);
  return ChangeFactory::create_object_modified_change(parent, source, target, changes);
}
  
  
DiffChange *GrtDiff::on_list(DiffChange *parent, const BaseListRef &source, const BaseListRef &target, const TSlotNormalizerSlot sqlDefinitionCmp)
{
  Type type;

  if (!are_compatible_lists(source, target, &type))
    return on_uncompatible(parent, source, target);

  return GrtListDiff().diff(source, target, omf, sqlDefinitionCmp);
}


DiffChange *GrtDiff::on_dict(DiffChange *parent, const DictRef &source, const DictRef &target, const TSlotNormalizerSlot sqlDefinitionCmp)
{
  ChangeSet changes;

  for (internal::Dict::const_iterator iter= source.begin(); iter != source.end(); ++iter)
  {
    std::string key= iter->first;
    ValueRef source_item(iter->second);

    if (!target.has_key(key))
      changes.append(ChangeFactory::create_dict_item_removed_change(parent, source, target, key));
    else
      changes.append(ChangeFactory::create_dict_item_modified_change(parent, source, target, key, on_value(NULL, source_item, target.get(key), sqlDefinitionCmp)));
  }

  for (internal::Dict::const_iterator iter= target.begin(); iter != target.end(); ++iter)
  {
    std::string key= iter->first;
    ValueRef target_item(iter->second);

    if (!source.has_key(key))
      changes.append(ChangeFactory::create_dict_item_added_change(parent, source, target, key, target_item));
  }

  return ChangeFactory::create_dict_change(parent, source, target, changes);
}

DiffChange *GrtDiff::on_uncompatible(DiffChange *parent, const ValueRef &source, const ValueRef &target)
{
  return ChangeFactory::create_value_added_change(parent, source, target);
}

}
