# import the wb module
from wb import *
# import the grt module
import grt
# import the mforms module for GUI stuff
import mforms

from sql_reformatter import formatter_for_statement_ast


# define this Python module as a GRT module
ModuleInfo = DefineModule(name= "SQLIDEUtils", author= "Oracle Corp.", version="1.1")



@ModuleInfo.plugin("wb.sqlide.executeToTextOutput", caption= "Execute Query Into Text Output", input= [wbinputs.currentQueryBuffer()], pluginMenu= "SQL/Utilities")
@ModuleInfo.export(grt.INT, grt.classes.db_query_QueryBuffer)
def executeQueryAsText(qbuffer):
  editor= qbuffer.owner
  sql= qbuffer.selectedText or qbuffer.script
  resultsets= editor.executeScript(sql)
  editor.addToOutput("Query Output:\n", 1)
  for result in resultsets:
    editor.addToOutput("> %s\n\n" % result.sql, 0)
    line= []
    column_lengths=[]
    ncolumns= len(result.columns)
    for column in result.columns:
      line.append(column.name + " "*5)
      column_lengths.append(len(column.name)+5)

    separator = []
    for c in column_lengths:
        separator.append("-"*c)
    separator= " + ".join(separator)
    editor.addToOutput("+ "+separator+" +\n", 0)

    line= " | ".join(line)
    editor.addToOutput("| "+line+" |\n", 0)

    editor.addToOutput("+ "+separator+" +\n", 0)
 
    rows = []
    ok= result.goToFirstRow()
    while ok:
      line= []
      for i in range(ncolumns):
        value = result.stringFieldValue(i)
        if value is None:
          value = "NULL"
        line.append(value.ljust(column_lengths[i]))
      line= " | ".join(line)
      rows.append("| "+line+" |\n")
      ok= result.nextRow()
    # much faster to do it at once than add lines one by one
    editor.addToOutput("".join(rows), 0)

    editor.addToOutput("+ "+separator+" +\n", 0)
    editor.addToOutput("%i rows\n" % len(rows), 0)

  return 0


@ModuleInfo.plugin("wb.sqlide.capitalizeCell", caption= "Capitalize Cell", input= [wbinputs.currentEditableResultset(), wbinputs.clickedRow(), wbinputs.clickedColumn()], pluginMenu= "SQL/Resultset")
@ModuleInfo.export(grt.INT, grt.classes.db_query_Resultset, grt.INT, grt.INT)
def capitalizeCell(rs, row, column):
  rs.goToRow(row)
  s= rs.stringFieldValue(column)
  if s:
    s=" ".join([ss.capitalize() for ss in s.split()])
    rs.setStringFieldValue(column, s)

  return 0

@ModuleInfo.plugin("wb.sqlide.lowerCaseCell", caption= "Lowercase Cell", input= [wbinputs.currentEditableResultset(), wbinputs.clickedRow(), wbinputs.clickedColumn()], pluginMenu= "SQL/Resultset")
@ModuleInfo.export(grt.INT, grt.classes.db_query_Resultset, grt.INT, grt.INT)
def lowerCaseCell(rs, row, column):
  rs.goToRow(row)
  s= rs.stringFieldValue(column)
  if s:
    s= s.lower()
    rs.setStringFieldValue(column, s)

  return 0

@ModuleInfo.plugin("wb.sqlide.upperCaseCell", caption= "Uppercase Cell", input= [wbinputs.currentEditableResultset(), wbinputs.clickedRow(), wbinputs.clickedColumn()], pluginMenu= "SQL/Resultset")
@ModuleInfo.export(grt.INT, grt.classes.db_query_Resultset, grt.INT, grt.INT)
def upperCaseCell(rs, row, column):
  rs.goToRow(row)
  s= rs.stringFieldValue(column)
  if s:
    s= s.upper()
    rs.setStringFieldValue(column, s)
  return 0



#@ModuleInfo.plugin("wb.sqlide.selectFromTable", caption= "Select From Table", input= [wbinputs.currentQueryBuffer(), wbinputs.selectedLiveTable()], pluginMenu= "SQL/Catalog")
#@ModuleInfo.export(grt.INT, grt.classes.db_query_QueryBuffer, grt.classes.db_query_LiveDBObject)
#def selectTable(qbuffer, object):
#  sql= "SELECT * FROM %s"%object.name
#  qbuffer.replaceContents(sql)
#  return 0


  

@ModuleInfo.plugin("wb.sqlide.enbeautificate", caption = "Reformat SQL Query", input=[wbinputs.currentQueryBuffer()], pluginMenu="SQL/Utilities")
@ModuleInfo.export(grt.INT, grt.classes.db_query_QueryBuffer)
def enbeautificate(editor):
    from grt.modules import MysqlSqlFacade

    text = editor.selectedText
    selectionOnly = True
    if not text:
        selectionOnly = False
        text = editor.script

    def get_ast_range(ast):
        b = ast[4]
        e = ast[5]
        for c in ast[2]:
            b_, e_ = get_ast_range(c)
            b = b_ if b is None else min(b_, b)
            e = max(e_, e)
        return b, e
    
    curpos = 0
    new_text = ""
    ast_list = MysqlSqlFacade.parseAstFromSqlScript(text)
    for ast in ast_list:
        s, v, c, _line, _begin, _end = ast
        begin, end = get_ast_range(ast)
        new_text += text[curpos:begin]
        curpos = end

        if type(ast) is str:
            # error
            print ast
            return 1
        else:
            def trim_ast(node):
                s = node[0]
                v = node[1]
                c = node[2]
                l = []
                for i in c:
                    l.append(trim_ast(i))
                return (s, v, l)
        

            formatter = formatter_for_statement_ast(ast)
            if formatter:
                p = formatter(trim_ast(ast))
                fmted = p.run()
            else:
                fmted = text[begin:end-begin]
            new_text += fmted
    new_text += text[curpos:]

    if selectionOnly:
        editor.replaceSelection()
    else:
        editor.replaceContents(new_text)

    return 0


def apply_to_keywords(editor, callable):
    from grt.modules import MysqlSqlFacade
    non_keywords = ["ident", "TEXT_STRING", "text_string", "TEXT_STRING_filesystem", "TEXT_STRING_literal", "TEXT_STRING_sys"]

    text = editor.selectedText
    selectionOnly = True
    if not text:
        selectionOnly = False
        text = editor.script
    
    new_text = ""
    ast_list = MysqlSqlFacade.parseAstFromSqlScript(text)
    for ast in ast_list:
        if type(ast) is str:
            # error
            print ast
        else:
            def traverse(script, node):
                s, v, c, l, b, e = node
                if v and s not in non_keywords:
                    script = script[:b] + callable(script[b:e]) + script[e:]
                for i in c:
                    script = traverse(script, i)

                return script

            new_text = traverse(text, ast)

    if selectionOnly:
        editor.replaceSelection()
    else:
        editor.replaceContents(new_text)

    return 0


@ModuleInfo.plugin("wb.sqlide.upcaseKeywords", caption = "Make keywords in query uppercase", input=[wbinputs.currentQueryBuffer()], pluginMenu="SQL/Utilities")
@ModuleInfo.export(grt.INT, grt.classes.db_query_QueryBuffer)
def upcaseKeywords(editor):
    return apply_to_keywords(editor, lambda s: s.upper())


@ModuleInfo.plugin("wb.sqlide.lowercaseKeywords", caption = "Make keywords in query lowercase", input=[wbinputs.currentQueryBuffer()], pluginMenu="SQL/Utilities")
@ModuleInfo.export(grt.INT, grt.classes.db_query_QueryBuffer)
def lowercaseKeywords(editor):
    return apply_to_keywords(editor, lambda s: s.lower())



@ModuleInfo.plugin("wb.sqlide.indent", caption = "Indent Selected Lines", input=[wbinputs.currentQueryBuffer()], pluginMenu="SQL/Utilities")
@ModuleInfo.export(grt.INT, grt.classes.db_query_QueryBuffer)
def indent(editor):
    indentation = " "*4
    text = editor.selectedText
    if text:
        lines = text.split("\n")
        fullText = editor.script
        start = editor.selectionStart
        end = editor.selectionEnd
        if start > 0 and fullText[start-1] != "\n":
            line0 = lines[0]
            del lines[0]
            new_text = line0 + ("\n%s"%indentation).join(text.split("\n"))
            editor.replaceSelection(new_text)
        else:
            new_text = indentation + ("\n%s"%indentation).join(text.split("\n"))
            editor.replaceSelection(new_text)
    else:
        text = editor.script
        new_text = indentation + ("\n%s"%indentation).join(text.split("\n"))
        editor.replaceContents(new_text)
    return 0



@ModuleInfo.plugin("wb.sqlide.unindent", caption = "Unindent Selected Lines", input=[wbinputs.currentQueryBuffer()], pluginMenu="SQL/Utilities")
@ModuleInfo.export(grt.INT, grt.classes.db_query_QueryBuffer)
def unindent(editor):
    def _unindent(text):
        lines = text.split("\n")
        if any(not line.startswith(indentation) for line in lines if len(line.strip()) > 0):
            return None
        return "\n".join(line[len(indentation):] if line.strip() else line for line in lines)

    indentation = " "*4
    text = editor.selectedText
    if text:
        lines = text.split("\n")
        fullText = editor.script
        start = editor.selectionStart
        end = editor.selectionEnd
        if start > 0 and fullText[start-1] != "\n":
            line0 = lines[0]
            del lines[0]
            tmp = _unindent(text)
            if tmp:
                new_text = line0 + tmp
                editor.replaceSelection(new_text)
        else:
            tmp = _unindent(text)
            if tmp:
                new_text = tmp
                editor.replaceSelection(new_text)
    else:
        text = editor.script
        new_text = _unindent(text)
        if new_text:
            editor.replaceContents(new_text)
    return 0


