/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "recordset_export_form.h"
#include "sqlide/wb_sql_editor_form.h"
#include "sqlide/recordset_text_storage.h"
#include "sqlide/recordset_sql_storage.h"
#include "string_utilities.h"

using namespace mforms;
using namespace base;

RecordsetExportForm::RecordsetExportForm(Db_sql_editor *editor, Recordset::Ref rset)
: Form(0), _editor(editor), _rset(rset), _vbox(false), _path_box(true), _button_box(true)
{
  set_content(&_vbox);
  
  _vbox.set_padding(MF_WINDOW_PADDING);
  _vbox.set_spacing(MF_TABLE_ROW_SPACING);
  _vbox.add(&_table, true, true);
  _vbox.add(&_button_box, false, true);
  
  _table.set_row_spacing(MF_TABLE_ROW_SPACING);
  _table.set_column_spacing(MF_TABLE_COLUMN_SPACING);
  _table.set_row_count(4);
  _table.set_column_count(2);
  
  _button_box.set_spacing(MF_BUTTON_SPACING);
  
  if (!_rset)
  {
    _res_label.set_text_align(mforms::MiddleRight);
    _res_label.set_text(_("Recordset to Export:"));
    _table.add(&_res_label, 0, 1, 0, 1, mforms::HFillFlag);
    _table.add(&_res_sel, 1, 2, 0, 1, mforms::HFillFlag|mforms::HExpandFlag);
  
    _res_sel.clear();
    for (int i= 0; i < editor->recordset_count(); ++i)
      _res_sel.add_item(editor->recordset(i)->caption());
  }
  _type_label.set_text_align(mforms::MiddleRight);
  _type_label.set_text(_("File Format:"));
  _table.add(&_type_label, 0, 1, 1, 2, mforms::HFillFlag);
  _table.add(&_type_sel, 1, 2, 1, 2, mforms::HFillFlag|mforms::HExpandFlag);

  
  _type_sel.clear();
  _type_sel.signal_changed().connect(sigc::mem_fun(this, &RecordsetExportForm::change_type));
  if (editor->recordset_count() > 0)
  {
    std::vector<std::string> formats_descs;
    editor->recordset(0)->data_storages_for_export(_format_names, formats_descs);
    for (size_t i= 0; i < formats_descs.size(); i++)
    {
      _type_sel.add_item(formats_descs[i]);
      
      char *extension= g_utf8_strdown(_format_names[i].data(), _format_names[i].length());
      char *sep;
      sep = strchr(extension, '_');
      if (sep) *sep= 0;
      sep = strchr(extension, ' ');
      if (sep) *sep= 0;
      
      _format_exts.push_back(std::string(".").append(extension));
      g_free(extension);
    }
  }
  
  _path_label.set_text_align(mforms::MiddleRight);
  _path_label.set_text(_("File Path:"));
  _table.add(&_path_label, 0, 1, 2, 3, mforms::HFillFlag);
  _path_button.set_text("...");
  _path_button.signal_clicked().connect(sigc::mem_fun(this, &RecordsetExportForm::browse_path));
  _path_box.set_spacing(4);
  _path_box.add(&_path_text, true, true);
  _path_box.add(&_path_button, false, false);
  _table.add(&_path_box, 1, 2, 2, 3, mforms::HFillFlag|mforms::HExpandFlag);

  _name_label.set_text(_("Table Name:"));
  _table.add(&_name_label, 0, 1, 3, 4, mforms::HFillFlag);
  _table.add(&_name_text, 1, 2, 3, 4, mforms::HFillFlag|mforms::HExpandFlag);
  _name_text.set_tooltip(_("Name of table for the resultset, used in some export formats, such as SQL Inserts.\nThis must be specified manually since queries can involve any number of tables."));
  _name_text.set_value("table_name");
  
  _ok_button.set_text(_("Export"));
  _cancel_button.set_text(_("Cancel"));
  
  Utilities::add_end_ok_cancel_buttons(&_button_box, &_ok_button, &_cancel_button);
  
  set_title(_("Export Query Results to File"));
  set_size(420, 220);
  center();
  
  _name_label.set_enabled(false);
  _name_text.set_enabled(false);
}

std::string RecordsetExportForm::add_extension(const std::string &orig_path)
{
  std::string path = orig_path;
  if (!path.empty())
  {    
    // strip old extension
    for (std::vector<std::string>::const_iterator e = _format_exts.begin(); e != _format_exts.end(); ++e)
    {
      if (path.size() > e->size() && path.substr(path.size()-e->size()) == *e)
      {
        path = path.substr(0, path.size()-e->size());
        break;
      }
    }
    
    path.append(_format_exts[_type_sel.get_selected_index()]);
  }
  return path;
}

void RecordsetExportForm::browse_path()
{
  mforms::FileChooser chooser(mforms::SaveFile);
  
  chooser.set_title(_("Export to File"));
  chooser.set_directory(_path_text.get_string_value());
  if (chooser.run_modal())
  {
    std::string path = add_extension(chooser.get_path());
    _path_text.set_value(path);
  }
}

void RecordsetExportForm::change_type()
{
  std::string path = add_extension(_path_text.get_string_value());
  _path_text.set_value(path);
  
  std::string ext = _format_exts[_type_sel.get_selected_index()];
  if (ext == ".sql")  // XXX shouldn't be hardcoded
  {
    _name_label.set_enabled(true);
    _name_text.set_enabled(true);
  }
  else
  {
    _name_label.set_enabled(false);
    _name_text.set_enabled(false);
  }
}


void RecordsetExportForm::run()
{  
retry:
  if (run_modal(&_ok_button, &_cancel_button))
  {
    int i = _type_sel.get_selected_index();
    std::string format= _format_names[i < 0 ? 0 : i];
    Recordset::Ref rset;
    
    if (g_file_test(_path_text.get_string_value().c_str(), G_FILE_TEST_EXISTS))
    {
      if (mforms::Utilities::show_warning(_("Export Query Results"),
                                          base::strfmt(_("File %s already exists, do you want to replace it?"), 
                                                       _path_text.get_string_value().c_str()),
                                          _("Replace"), _("Cancel"), "") != mforms::ResultOk)
        goto retry;
    }
 
    _editor->grt_manager()->push_status_text(_("Exporting Query Results..."));

    if (_rset)
      rset = _rset;
    else
    {    
      int rset_index= _res_sel.get_selected_index();
      rset= _editor->recordset(rset_index);
    }
    Recordset_data_storage::Ref dataStorage= _editor->recordset(0)->data_storage_for_export(format);
    
    if (dynamic_cast<Recordset_text_storage*>(dataStorage.get()))
    {
      Recordset_text_storage *textStorage= dynamic_cast<Recordset_text_storage*> (dataStorage.get());
      textStorage->data_format(format);
      textStorage->file_path(_path_text.get_string_value());

      Recordset_sql_storage* storage = dynamic_cast<Recordset_sql_storage*>(rset->data_storage().get());
      textStorage->parameter_value("GENERATOR_QUERY", rset->generator_query());
      textStorage->parameter_value("GENERATE_DATE", bec::fmttime(time(NULL), DATETIME_FMT));

      std::string name = _name_text.get_string_value();
      textStorage->parameter_value("TABLE_NAME", (storage && !storage->table_name().empty())?storage->table_name():name.c_str());
    }
    
    try
    {
      dataStorage->serialize(rset);
      
      _editor->grt_manager()->replace_status_text(strfmt(_("Query Results Exported to %s"), _path_text.get_string_value().c_str()));
    }
    catch (std::exception &exc)
    {
      mforms::Utilities::show_error(_("Export Query Results"), exc.what(), _("OK"));
      
      _editor->grt_manager()->replace_status_text(_("Error in resultset export."));
    }
  }
}
