/*
 * Copyright (c) 2008, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
#include "stdafx.h"

#include <glib/gstdio.h>

#include "wb_context_ui.h"

#include "file_functions.h"
#include "string_utilities.h"

#include "server_instance_editor.h"
#include "grtui/grtdb_connection_editor.h"
#include "new_server_instance_wizard.h"
#include "new_connection_wizard.h"
#include "edit_sql_script_wizard.h"
#include "edit_table_data_wizard.h"


using namespace bec;
using namespace wb;
using namespace base;

/**
 * Helper method to construct a human-readable server description.
 */
std::string get_server_info(db_mgmt_ServerInstanceRef instance)
{
  std::string text;
  std::string system = instance->serverInfo().get_string("sys.system");

  if (instance->serverInfo().get_int("remoteAdmin"))
    text = strfmt("Host: %s  Type: %s", instance->loginInfo().get_string("ssh.hostName").c_str(),
      system.c_str());
  else
  {
    if (instance->serverInfo().get_int("windowsAdmin") || system == "Windows")
    {
      std::string host = instance->loginInfo().get_string("wmi.hostName");
      if (host == "localhost" || host.empty() || host == "127.0.0.1")
        text = "Local  Type: Windows";
      else
        text = "Host: " + host + "  Type: Windows";
    }
    else
    {
      std::string host = instance->connection().is_valid() ?
        instance->connection()->parameterValues().get_string("hostName") : "Invalid";

      if (host == "localhost" || host.empty() || host == "127.0.0.1")
        text = strfmt("Local  Type: %s", system.c_str());
      else
        text = strfmt("Host: %s  Type: DB Only", host.c_str());
    }
  }

  return text;
}

//--------------------------------------------------------------------------------------------------

static bool on_home_closing(WBContextUI* owner)
{
  owner->home_screen_closing();
  return true;
}

//--------------------------------------------------------------------------------------------------

/**
 * Creates the main home screen (Workbench Central, Workspace) if not yet done and docks it to 
 * the main application window.
 */
void WBContextUI::show_home_screen()
{
  if (_home_screen == NULL)
  {
    _home_screen= mforms::manage(new HomeScreen(_wb->get_grt()));
    _home_screen->set_on_close(sigc::bind(sigc::ptr_fun(on_home_closing), this));
    _home_screen->set_callback((home_screen_action_callback)&WBContextUI::home_action_callback, this);
    _home_screen->handle_context_menu = sigc::mem_fun(this, &WBContextUI::handle_home_context_menu);
    // setup context menus
    bec::MenuItemList items, pitems;
    bec::MenuItem item, sep;

    sep.type = bec::MenuSeparator;
    item.name= "open_connection";
    item.caption= "Query Database";
    items.push_back(item);
    {
      std::list<std::string> groups;
      bec::ArgumentPool argument_pool;
      _wb->update_plugin_arguments_pool(argument_pool);
      argument_pool.add_entries_for_object("selectedConnection", db_mgmt_ConnectionRef(_wb->get_grt()), "db.mgmt.Connection");
      groups.push_back("Menu/Home/Connections");
      pitems = _wb->get_grt_manager()->get_plugin_context_menu_items(groups, argument_pool);
      if (!pitems.empty())
      {
        items.push_back(sep);
        items.insert(items.end(), pitems.begin(), pitems.end());
      }
    }
    if (!items.empty())
      items.push_back(sep);
    item.name= "delete_connection";
    item.caption= "Delete Connection...";
    items.push_back(item);

    item.caption= "Delete All Connections...";
    item.name= "delete_connection_all";
    items.push_back(item);

    _home_screen->set_context_menu_items(HomeScreenSQLDevList, items);
    
    //
    items.clear();
    item.name= "open_model";
    item.caption= "Open Model File";
    items.push_back(item);    
    {
      std::list<std::string> groups;
      bec::ArgumentPool argument_pool;
      _wb->update_plugin_arguments_pool(argument_pool);
      argument_pool.add_simple_value("selectedModelFile", grt::ValueRef());
      groups.push_back("Menu/Home/ModelFiles");
      pitems = _wb->get_grt_manager()->get_plugin_context_menu_items(groups, argument_pool);
      if (!pitems.empty())
      {
        items.push_back(sep);
        items.insert(items.end(), pitems.begin(), pitems.end());
      }      
    }
    if (!items.empty())
      items.push_back(sep);    
    item.name= "remove_model";
    item.caption= "Remove Model File from List";
    items.push_back(item);
    
    item.caption= "Clear List";
    item.name= "remove_model_all";
    items.push_back(item);    
    _home_screen->set_context_menu_items(HomeScreenModellingList, items);
    
    //
    items.clear();
    item.name= "open_administrator";
    item.caption= "Manage Instance";
    items.push_back(item);        
    {
      std::list<std::string> groups;
      bec::ArgumentPool argument_pool;
      _wb->update_plugin_arguments_pool(argument_pool);
      argument_pool.add_entries_for_object("selectedInstance", db_mgmt_ServerInstanceRef(_wb->get_grt()), "db.mgmt.ServerInstance");
      groups.push_back("Menu/Home/Instance");
      pitems = _wb->get_grt_manager()->get_plugin_context_menu_items(groups, argument_pool);
      if (!pitems.empty())
      {
        items.push_back(sep);
        items.insert(items.end(), pitems.begin(), pitems.end());
      }      
    }
    if (!items.empty())
      items.push_back(sep);

    item.name= "delete_instance";
    item.caption= "Delete Server Instance...";
    items.push_back(item);
    
    item.caption= "Delete All Server Instances...";
    item.name= "delete_instance_all";
    items.push_back(item);
    
    _home_screen->set_context_menu_items(HomeScreenAdministrationList, items);
  }
  mforms::App::get()->dock_view(_home_screen, "maintab");
  mforms::App::get()->set_view_title(_home_screen, "Home");
  
  refresh_home_models();
  refresh_home_instances(0);
  refresh_home_connections();
  refresh_home_starters();
}

//--------------------------------------------------------------------------------------------------

/**
 * Called when the home screen is closed by the UI. We have to clear our reference then.
 */
void WBContextUI::home_screen_closing()
{
  _home_screen= NULL;
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::home_action_callback(HomeScreenAction action, const grt::ValueRef &object, WBContextUI *self)
{
  try
  {
    self->handle_home_action(action, object);
  }
  catch (const std::exception &exc)
  {
    mforms::Utilities::show_error("Internal Error",
                                  base::strfmt("Exception caught while processing action from home screen: %s", exc.what()), "OK");
  }
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::starter_action_callback(StarterAction action, const app_StarterRef &object,
                                          void *self)
{
  try
  {
    ((WBContextUI*) self)->handle_starter_action(action, object);
  }
  catch (const std::exception &exc)
  {
    mforms::Utilities::show_error("Internal Error",
                                  base::strfmt("Exception caught while processing plugin action: %s", exc.what()), "OK");
  }
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::handle_home_context_menu(HomeScreenListType list, const grt::ValueRef &object, const std::string &action)
{
  if (action == "open_connection")
  {
    handle_home_action(ActionOpenConnectionFromList, object);
  }
  else if (action == "delete_connection")
  {
    db_mgmt_ConnectionRef connection(db_mgmt_ConnectionRef::cast_from(object));
    
    std::string name= connection->name();
    std::string text= strfmt(_("Do you want to delete connection %s?"), name.c_str());
    int answer= Utilities::show_warning(_("Delete Connection"), text,  _("Delete"), _("Cancel"));
    if (answer == mforms::ResultOk)
    {
      grt::ListRef<db_mgmt_Connection> connections(_wb->get_root()->rdbmsMgmt()->storedConns());
      connections->remove(connection);
      refresh_home_connections();
    }
  }
  else if (action == "delete_connection_all")
  {
    std::string text= _("Do you really want to delete all defined connections?");
    int answer= Utilities::show_warning(_("Delete All Connections"), text,  _("Delete"), _("Cancel"));
    if (answer == mforms::ResultOk)
    {
      grt::ListRef<db_mgmt_Connection> connections(_wb->get_root()->rdbmsMgmt()->storedConns());
      while (connections->count() > 0)
        connections->remove(0);
      refresh_home_connections();
    }
  }
  else if (action == "open_model")
  {
    handle_home_action(ActionOpenEERModelFromList, object);
  }
  else if (action == "remove_model")
  {
    _wb->get_root()->options()->recentFiles()->remove(object);
    refresh_home_models();
  }
  else if (action == "remove_model_all")
  {
    std::string text= _("Do you want to remove all entries from the model list?");
    int answer= Utilities::show_warning(_("Clear Model Entry List"), text,  _("Delete"), _("Cancel"));
    if (answer == mforms::ResultOk)
    {
      grt::StringListRef file_names(_wb->get_root()->options()->recentFiles());
      for (int index= file_names->count() - 1; index >= 0; index--)
      {
        if (g_str_has_suffix(file_names[index].c_str(), ".mwb"))
          file_names->remove(index);
      }
      refresh_home_models();
    }
  }
  else if (action == "open_administrator")
  {
    handle_home_action(ActionManageInstanceFromList, object);
  }
  else if (action == "delete_instance")
  {
    db_mgmt_ServerInstanceRef instance(db_mgmt_ServerInstanceRef::cast_from(object));
    
    std::string name= instance->name();
    std::string text= strfmt(_("Do you want to delete server instance %s?"), name.c_str());
    int answer= Utilities::show_warning(_("Delete Server Instance"), text,  _("Delete"), _("Cancel"));
    if (answer == mforms::ResultOk)
    {
      grt::ListRef<db_mgmt_ServerInstance> instances(_wb->get_root()->rdbmsMgmt()->storedInstances());
      instances->remove(instance);
      refresh_home_instances();
    }
  }
  else if (action == "delete_instance_all")
  {
    std::string text= _("Do you really want to delete all defined server instances?");
    int answer= Utilities::show_warning(_("Delete All Server Instances"), text,  _("Delete"), _("Cancel"));
    if (answer == mforms::ResultOk)
    {
      grt::ListRef<db_mgmt_ServerInstance> instances(_wb->get_root()->rdbmsMgmt()->storedInstances());
      while (instances->count() > 0)
        instances->remove(0);
      refresh_home_instances();
    }
  }
  else 
  {
    bec::ArgumentPool argument_pool;
    _wb->update_plugin_arguments_pool(argument_pool);
    
    switch (list)
    {
      case HomeScreenSQLDevList:
        argument_pool.add_entries_for_object("selectedConnection", db_mgmt_ConnectionRef::cast_from(object));
        break;
      case HomeScreenModellingList:
        argument_pool.add_simple_value("selectedModelFile", grt::StringRef::cast_from(object));
        break;
      case HomeScreenAdministrationList:
        argument_pool.add_entries_for_object("selectedInstance", db_mgmt_ServerInstanceRef::cast_from(object));
        break;        
    }

    get_command_ui()->activate_command(action, argument_pool);
  }
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::start_plugin(const std::string& title, const std::string& command, bool force_external)
{
  try
  {
    std::string message_title= base::strfmt(_("Starting %s"), title.c_str());
    GUILock lock(_wb, message_title, _("Please stand by while the plugin is started ..."));
    if (base::starts_with(command, "plugin:"))
    {
      _wb->execute_plugin(command.substr(7, command.length()));
    }
    else
      if (base::starts_with(command, "browse:"))
        show_web_page(command.substr(7, command.length()), !force_external);
      else
        if (base::starts_with(command, "http://"))
          show_web_page(command, false);
  }
  catch (const std::exception &exc)
  {
    mforms::Utilities::show_error(base::strfmt("Could not open link or plugin"),
                                  exc.what(), "OK");
  }
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::handle_home_action(HomeScreenAction action, const grt::ValueRef &object)
{
  switch (action)
  {
    case ActionNone:
      break;
    case ActionWhatsNew:
    {
      GUILock lock(_wb, _("Opening 'What's New' web page"), _("The MySQL Workbench news web page is being opened "
                                                              "and should be available in a moment.\n\nPlease stand by..."));
      show_web_page("http://wb.mysql.com/?page_id=49", true);
    }
      break;
      
    case ActionStarter:
    {
      app_StarterRef starter= app_StarterRef::cast_from(object);
      start_plugin(starter->title(), starter->command());
    }
      break;
      
    case ActionStarterPopup:
    {
      StarterPopup popup(_wb);
      popup.set_callback(&WBContextUI::starter_action_callback, this);
      
      grt::DictRef values= grt::DictRef::cast_from(object);
      int x= values.get_int("left");
      int y= values.get_int("top");
      
      popup.show(x, y);
      break;
    }
      
    case ActionRemoveStarter:
    {
      int index= grt::IntegerRef::cast_from(object);
      _wb->get_root()->starters()->displayList()->remove(index);
      _wb->save_starters();
      refresh_home_starters();
    }
      break;
      
    case ActionOpenConnection:
    {
      _wb->show_status_text("Opening SQL Editor...");
      _wb->add_new_query_window(db_mgmt_ConnectionRef());
      break;
    }
      
    case ActionOpenConnectionFromList:
    {
      if (object.is_valid())
      {
        db_mgmt_ConnectionRef connection(db_mgmt_ConnectionRef::cast_from(object));
        GUILock lock(_wb, _("Opening SQL Editor"), strfmt(_("An SQL editor instance for '%s' is opening and should be available in a "
                                                            "moment.\n\nPlease stand by..."), connection->name().c_str()));
        
        _wb->show_status_text("Opening SQL Editor...");          
        _wb->add_new_query_window(connection);
      }
      break;
    }
      
    case ActionNewConnection:
    {
      NewConnectionWizard wizard(_wb->get_root()->rdbmsMgmt());
      wizard.set_title("Setup New Connection");
      wizard.run();
      _wb->save_app_options();
      refresh_home_connections();
      break;
    }
    case ActionAddSchemaObject:
      break;
    case ActionEditTable:
    {
      EditTableDataWizard wizard(_wb);
      _wb->show_status_text("Edit Table Data Wizard Opened.");
      wizard.run_modal();
      _wb->show_status_text("");
      break;
    }
    case ActionEditSQLScript:
    {
      EditSqlScriptWizard wizard(_wb);
      _wb->show_status_text("Edit SQL Script Wizard Opened.");
      wizard.run_modal();
      _wb->show_status_text("");
      break;
    }
      
    case ActionManageConnections:
    {
      grtui::DbConnectionEditor editor(_wb->get_root()->rdbmsMgmt());
      _wb->show_status_text("Connection Manager Opened.");
      editor.run();
      _wb->show_status_text("");
      // save connection list now
      _wb->save_app_options();
      refresh_home_connections();        
      break;
    }
      
    case ActionOpenEERModel:
    {
      // Note: wb->open_document has an own GUILock, so we must not set another one here.
      std::string filename = _wb->show_file_dialog("open", _("Open Workbench Model"), "mwb");
      if (!filename.empty())
        _wb->open_document(filename);
      else
        _wb->show_status_text("Cancelled");
      break;
    }
      
    case ActionOpenEERModelFromList:
    {
      // Note: wb->open_document has an own GUILock, so we must not set another one here.
      if (object.is_valid())
      {
        std::string path = *grt::StringRef::cast_from(object);
        _wb->show_status_text(strfmt("Opening %s...", path.c_str()));
        _wb->open_document(path);
      }
      break;
    }
    case ActionNewEERModel:
    {
      _wb->new_document();
    }
      break;
      
    case ActionNewModelFromDB:
    {
      _wb->new_document();          
      if (_wb->get_document().is_valid())
      {
        ArgumentPool args;
        
        // delete the default schema
        if (_wb->get_document()->physicalModels()[0]->catalog()->schemata().count() > 0)
          _wb->get_document()->physicalModels()[0]->catalog()->schemata().remove(0);
        
        _wb->update_plugin_arguments_pool(args);
        args.add_entries_for_object("activeCatalog", _wb->get_document()->physicalModels()[0]->catalog(),
                                    "db.Catalog");
        
        _wb->execute_plugin("db.plugin.database.rev_eng", args);
      }
      else
        _wb->show_status_text("Error creating document");
      break;
    }
      
    case ActionNewModelFromScript:
    {
      _wb->new_document();
      if (_wb->get_document().is_valid())
      {
        ArgumentPool args;
        
        // delete the default schema
        if (_wb->get_document()->physicalModels()[0]->catalog()->schemata().count() > 0)
          _wb->get_document()->physicalModels()[0]->catalog()->schemata().remove(0);
        
        _wb->update_plugin_arguments_pool(args);
        args.add_entries_for_object("activeCatalog", _wb->get_document()->physicalModels()[0]->catalog(),
                                    "db.Catalog");
        _wb->execute_plugin("db.mysql.plugin.import.sql", args);
      }
      else
        _wb->show_status_text("Error creating document");
      break;
    }
      
    case ActionServerAdministration:
    {          
      app_PluginRef plugin(_wb->get_plugin_manager()->get_plugin("wb.admin.placeholder"));
      
      if (plugin.is_valid())
        try
      {
        grt::BaseListRef args(_wb->get_grt_manager()->get_grt(), grt::AnyType);
        _wb->get_plugin_manager()->open_plugin(plugin, args);
      }
      catch(grt::module_error& err)
      {
        std::string msg = strfmt("Error in wb.admin module: %s",err.what());
        _wb->show_status_text(msg);
      }
      break;
    }      
      
    case ActionManageInstanceFromList:
    {
      if (object.is_valid())
      {
        db_mgmt_ServerInstanceRef instance(db_mgmt_ServerInstanceRef::cast_from(object));
        if (!instance->connection().is_valid())
          mforms::Utilities::show_error("Cannot Open Administrator",
                                        "The selected server instance does not have a MySQL server connection assigned and cannot be opened.",
                                        "OK");
        else
        {
          GUILock lock(_wb, _("Opening MySQL Workbench Administrator"), strfmt(_("The MySQL Workbench Administrator for %s "
                                                                                 "is being opened and should be available in a moment.\n\nPlease stand by..."), instance->name().c_str()));
          _wb->add_new_admin_window(instance);
        }
      }
      break;
    }
      
    case ActionNewServerInstance:
    {
      NewServerInstanceWizard wizard(_wb);
      
      _wb->show_status_text(_("Creating new server instance"));
      if (wizard.run_modal())
      {
        _wb->save_app_options();
        refresh_home_connections();
        refresh_home_instances((int)_wb->get_root()->rdbmsMgmt()->storedInstances().count()-1);
        _wb->show_status_text(_("New server instance created"));
      }
      else
        _wb->show_status_text(_("New server instance creation cancelled"));
    }
      break;
      
    case ActionDumpRestore:
    {
      grt::BaseListRef args(_wb->get_grt_manager()->get_grt(), grt::AnyType);
      
      app_PluginRef plugin(_wb->get_plugin_manager()->get_plugin("wb.admin.dumpManager"));
      
      if (plugin.is_valid())
        try
      {
        _wb->get_plugin_manager()->open_plugin(plugin, args);
      }
      catch(grt::module_error& err)
      {
        std::string msg = strfmt("Error in dump module: %s",err.what());
        _wb->show_status_text(msg);
      }
      break;
    }
      
    case ActionManageSecurity:
    {
      grt::BaseListRef args(_wb->get_grt_manager()->get_grt(), grt::AnyType);
      
      app_PluginRef plugin(_wb->get_plugin_manager()->get_plugin("wb.admin.securityManager"));
      
      if (plugin.is_valid())
        try
      {
        _wb->get_plugin_manager()->open_plugin(plugin, args);
      }
      catch(grt::module_error& err)
      {
        std::string msg = strfmt("Error in securityManager module: %s",err.what());
        _wb->show_status_text(msg);
      }
      break;
    }
    case ActionManageServerInstances:
    {
      ServerInstanceEditor editor(_wb->get_grt_manager(), _wb->get_root()->rdbmsMgmt());
      _wb->show_status_text("Server Instance Manager Opened.");
      db_mgmt_ServerInstanceRef instance(editor.run());
      _wb->show_status_text("");
      // save instance list now
      _wb->save_app_options();
      refresh_home_connections();
      refresh_home_instances((int)_wb->get_root()->rdbmsMgmt()->storedInstances().get_index(instance));        
      break;
    }
  }
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::handle_starter_action(StarterAction action, const app_StarterRef &starter)
{
  int index= 0;
  switch (action)
  {
    case StarterPluginCentral:
    {
      GUILock lock(_wb, _("Opening Plugin Central"), _("The MySQL Workbench Plugin Central web page "
                                                       "is being opened and should be available in a moment.\n\nPlease stand by..."));
      show_web_page("http://wb.mysql.com/wb-plugins/", true);
    }
      break;
      
    case StarterStartPlugin:
    {
      std::string title= starter->title();
      std::string command= starter->command();
      start_plugin(title, command);
    }
      break;
      
    case StarterStartPluginExternal:
    {
      std::string title= starter->title();
      std::string command= starter->command();
      start_plugin(title, command, true);
    }
      break;
      
    case StarterFindPlugin:
    {
      GUILock lock(_wb, _("Starting Plugin Search"), _("The MySQL Workbench Plugin Search web page "
                                                       "is being opened and should be available in a moment.\n\nPlease stand by..."));
      show_web_page("http://wb.mysql.com/wb-plugins/", true);
    }
      break;
      
    case StarterAuthorHome:
      show_web_page(starter->authorHome(), true);
      break;
      
    case StarterSetAt6:
      index++; // fall through
    case StarterSetAt5:
      index++; // fall through
    case StarterSetAt4:
      index++; // fall through
    case StarterSetAt3:
      index++; // fall through
    case StarterSetAt2:
      index++; // fall through
    case StarterSetAt1:
      if ((int) _wb->get_root()->starters()->displayList().count() > index)
        _wb->get_root()->starters()->displayList().set(index, starter);
      else
        _wb->get_root()->starters()->displayList().insert(starter);
      _wb->save_starters();
      refresh_home_starters();
      break;
      
    case StarterInsertAt6:
      index++; // fall through
    case StarterInsertAt5:
      index++; // fall through
    case StarterInsertAt4:
      index++; // fall through
    case StarterInsertAt3:
      index++; // fall through
    case StarterInsertAt2:
      index++; // fall through
    case StarterInsertAt1:
      if ((int) _wb->get_root()->starters()->displayList().count() > index)
        _wb->get_root()->starters()->displayList().insert(starter, index);
      else
        _wb->get_root()->starters()->displayList().insert(starter);
      _wb->save_starters();
      refresh_home_starters();
      break;
      
  }
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::refresh_home_instances(int selected)
{
  if (!_home_screen) return;
  grt::ListRef<db_mgmt_ServerInstance> instances(_wb->get_root()->rdbmsMgmt()->storedInstances());
  
  _home_screen->clear_list(HomeScreenAdministrationList);

  // If there are no connections defined yet then create entries for all currently installed
  // local servers.
  if (instances->count() == 0)
  {
    grt::GRT* grt = _wb->get_grt();
    grt::Module* module = grt->get_module("Workbench");
    if (module == NULL)
      throw std::logic_error("Internal error: can't find Workbench module.");

    grt::StringListRef arguments(grt);
    module->call_function("createInstancesFromLocalServers", arguments);
  }

  for (grt::ListRef<db_mgmt_ServerInstance>::const_iterator end = instances.end(),
       inst = instances.begin(); inst != end; ++inst)
  {
    std::string text= get_server_info(*inst);
    
    _home_screen->add_list_entry(HomeScreenAdministrationList, (*inst)->name(), text, *inst);
  }
  
  _home_screen->select_list_entry(HomeScreenAdministrationList, selected);
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::refresh_home_connections(int selected)
{
  if (!_home_screen) return;
  grt::ListRef<db_mgmt_Connection> connections(_wb->get_root()->rdbmsMgmt()->storedConns());
  
  _home_screen->clear_list(HomeScreenSQLDevList);

  // If there are no connections defined yet then create entries for all currently installed
  // local servers.
  if (connections->count() == 0)
  {
    grt::GRT* grt = _wb->get_grt();
    grt::Module* module = grt->get_module("Workbench");
    if (module == NULL)
      throw std::logic_error("Internal error: can't find Workbench module.");
    
    grt::StringListRef arguments(grt);
    module->call_function("createConnectionsFromLocalServers", arguments);
  }

  for (grt::ListRef<db_mgmt_Connection>::const_iterator end = connections.end(),
       inst = connections.begin(); inst != end; ++inst)
  {
    grt::DictRef dict((*inst)->parameterValues());
    std::string tunnel;
    
    std::string host_entry= strfmt("%s:%i", dict.get_string("hostName").c_str(), (int) dict.get_int("port"));
    if ((*inst)->driver().is_valid() && (*inst)->driver()->name() == "MysqlNativeSSH")
      tunnel = "(ssh tunnel)";
    else
      if ((*inst)->driver().is_valid() && (*inst)->driver()->name() == "MysqlNativeSocket")
      {
        std::string socket= dict.get_string("socket"); // socket or pipe
        if (socket.length() == 0)
          socket= "MySQL"; // default pipe name
        // TODO: what about the default for sockets? Need to distinct the OS here for decision.
        host_entry= "Localhost via pipe " + socket;
      }
    
    std::string schema = dict.get_string("schema");
    if (!schema.empty())
      schema = "Schema: "+schema;
    
    std::string text = strfmt("User: %s  Host: %s %s  %s", dict.get_string("userName").c_str(), 
                              host_entry.c_str(), tunnel.c_str(), schema.c_str());
    
    _home_screen->add_list_entry(HomeScreenSQLDevList, (*inst)->name(), text, *inst);
  }
  
  _home_screen->select_list_entry(HomeScreenSQLDevList, selected);
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::refresh_home_models(int selected)
{
  if (!_home_screen)
    return;
  _home_screen->clear_list(HomeScreenModellingList);
  
  for (grt::StringListRef::const_iterator end = _wb->get_root()->options()->recentFiles().end(),
       f = _wb->get_root()->options()->recentFiles().begin(); f != end; ++f)
  {
    if (!g_str_has_suffix((*f).c_str(), ".mwb"))
      continue;
    
#ifdef _WIN32
    struct _stat stbuf;
    
    if (base_stat((*f).c_str(), &stbuf) < 0)
      _home_screen->add_list_entry(HomeScreenModellingList, *f, "File not accessible", grt::ValueRef());
    else
    {
      struct tm t;
      localtime_s(&t, &stbuf.st_mtime);
      
      std::string text = strfmt("Last modified %s", asctime(&t));
      if (text[text.size()-1] == '\n')
        text = text.substr(0, text.size()-1);
      
      char *file = g_path_get_basename((*f).c_str());
      if (g_str_has_suffix(file, ".mwb"))
        file[strlen(file)-4]= 0;
      _home_screen->add_list_entry(HomeScreenModellingList, file, text, *f);
      g_free(file);
    }
#else
    struct stat stbuf;
    
    if (base_stat((*f).c_str(), &stbuf) < 0)
      _home_screen->add_list_entry(HomeScreenModellingList, *f, "File not accessible", *f);
    else
    {
      char *buf = asctime(localtime(&stbuf.st_mtime));
      std::string text = strfmt("Last modified %s", buf?buf:"???");
      
      char *file = g_path_get_basename((*f).c_str());
      if (g_str_has_suffix(file, ".mwb"))
        file[strlen(file)-4]= 0;
      _home_screen->add_list_entry(HomeScreenModellingList, file, text, *f);
      g_free(file);
    }
#endif
  }
  _home_screen->select_list_entry(HomeScreenModellingList, selected);
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::refresh_home_starters()
{
  if (!_home_screen)
    return;
  _home_screen->clear_list(HomeScreenStarters);
  
  grt::ListRef<app_Starter> starters= _wb->get_root()->starters()->displayList();
  for (grt::ListRef<app_Starter>::const_iterator iterator= starters.begin(); iterator != starters.end(); 
       iterator++)
  {
    _home_screen->add_list_entry(HomeScreenStarters, "", "", *iterator, (*iterator)->smallIcon());
  }
  _home_screen->set_needs_repaint();
}

