/* 
 * Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#ifndef __WB_H__
#define __WB_H__

#include "wb_config.h"
#include "workbench/wb_context.h"
#include "workbench/wb_context_ui.h"
#include "model/wb_model_diagram_form.h"
#include "workbench/wb_history_tree.h"
#include "model/wb_diagram_options.h"
#include "model/wb_user_datatypes.h"
#include "model/wb_overview_physical.h"

#include "common/preferences_form.h"
#include "common/document_properties_form.h"

#include "GrtTemplates.h"
#include "SigCSlot.h"
#include "FindDialogBE.h"
#include "Overview.h"
#include "ModelViewForm.h"

#include "wf_appview.h"
#include "wf_mforms.h"

#if WBContext_VERSION > 5
# error outdated wrapper Wb.h
#endif

#ifdef _MSC_VER
#include  <vcclr.h> // .net interop helpers
using namespace MySQL::Grt;
using namespace MySQL::GUI::Mdc;
using namespace System;
using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;
#endif

namespace MySQL {
namespace Workbench {

public enum class RefreshType 
{
  RefreshNeeded=    ::wb::RefreshNeeded,

  RefreshNothing=    ::wb::RefreshNothing,
  RefreshSchemaList= ::wb::RefreshSchemaList,
  RefreshSchema=     ::wb::RefreshSchema,
  RefreshNewDiagram= ::wb::RefreshNewDiagram,
  RefreshSelection=  ::wb::RefreshSelection,
  RefreshToolbar=    ::wb::RefreshToolbar,
  RefreshCloseEditor= ::wb::RefreshCloseEditor,
  RefreshFindOutput= ::wb::RefreshFindOutput,
  RefreshNewModel=   ::wb::RefreshNewModel,
  RefreshOverviewNodeInfo=::wb::RefreshOverviewNodeInfo,
  RefreshOverviewNodeChildren=::wb::RefreshOverviewNodeChildren,
  RefreshViewName=   ::wb::RefreshViewName,
  RefreshDocument=   ::wb::RefreshDocument,
  RefreshCloseDocument=::wb::RefreshCloseDocument,
  RefreshZoom=       ::wb::RefreshZoom,
  RefreshMenubar=    ::wb::RefreshMenubar,
  RefreshTimer=      ::wb::RefreshTimer,
  RefreshFinishEdits=::wb::RefreshFinishEdits,
};



public enum class GUIPluginFlags
{
  ForceNewWindowFlag=   ::bec::ForceNewWindowFlag,
  StandaloneWindowFlag= ::bec::StandaloneWindowFlag
};

public enum class AppCommand
{
  AppDockView,
  AppSelectView,
  AppUndockView,
  AppSetViewTitle,
  AppQuit,
  AppGetResourcePath,
  AppSetStatusText,
  AppGetBounds,
};

public enum class Msg_type
{
  MT_warning=  grt::WarningMsg,
  MT_error=    grt::ErrorMsg,
  MT_info=     grt::InfoMsg,
  MT_progress= grt::ProgressMsg,
};

//--------------------------------------------------------------------------------------------------

public ref class PageSettings
{
public:
  PageSettings()
    : paper_type(nullptr), margin_top(0), margin_bottom(0), margin_left(0), margin_right(0), 
      orientation(nullptr)
  {
  }

  PageSettings(const app_PageSettingsRef &settings)
    : paper_type(CppStringToNative(settings->paperType().is_valid() ? settings->paperType()->name() : "")),
    margin_top(settings->marginTop()), margin_bottom(settings->marginBottom()),
    margin_left(settings->marginLeft()), margin_right(settings->marginRight()),
    orientation(CppStringToNative(settings->orientation()))
  {
  }

  void update_object(grt::ListRef<app_PaperType> paperTypes, app_PageSettingsRef settings)
  {
    settings->paperType(grt::find_named_object_in_list(paperTypes, NativeToCppString(paper_type)));

    settings->marginTop(grt::DoubleRef(margin_top));
    settings->marginBottom(grt::DoubleRef(margin_bottom));
    settings->marginLeft(grt::DoubleRef(margin_left));
    settings->marginRight(grt::DoubleRef(margin_right));

    settings->orientation(NativeToCppString(orientation));
  }

  String^ paper_type;
  double margin_top;
  double margin_bottom;
  double margin_left;
  double margin_right;
  String^ orientation;
};


public ref class PaperSize
{
public:
  PaperSize(const wb::WBPaperSize &paperSize)
    : name(CppStringToNative(paperSize.name)), 
    caption(CppStringToNative(paperSize.caption)), 
    margins_set(paperSize.margins_set),
    margin_top(paperSize.margin_top), margin_bottom(paperSize.margin_bottom), 
    margin_left(paperSize.margin_left), margin_right(paperSize.margin_right), 
    width(paperSize.width), height(paperSize.height),
    description(CppStringToNative(paperSize.description))
  {
  }

  String^ name;
  String^ caption;
  double width;
  double height;
  bool margins_set;
  double margin_top;
  double margin_bottom;
  double margin_left;
  double margin_right;
  String^ description;
};


// ----------------------------------------------------------------------------


public ref class WbFrontendCallbacks
{
public:
  // ----------------------------------------------------------------------------
  // The delegates used by C#
  delegate void VoidDelegate();
  delegate void VoidBoolDelegate(bool f);
  delegate void VoidStrDelegate(String^ str1);
  delegate void VoidStrStrDelegate(String^ str1, String^ str2);
  delegate String^ StrStrStrStrDelegate(String^ str1, String^ str2, String^ str3);
  delegate bool BoolStrStrStrDelegate(String^ str1, String^ str2, String^ str3);
  delegate bool BoolStrStrStrStrDelegate(String^ str0, String^ str1, String^ str2, String^ str3);
  delegate bool BoolStrStrFloatDelegate(String^ str1, String^ str2, float f3);
  delegate bool BoolStrIntStrPtrDelegate(String^ str, int i, [Out] String^ %result);
  delegate BaseWindowsCanvasView^ CanvasViewStringStringIntPtrDelegate(String^ str1, String^ str2, IntPtr payload);
  delegate void VoidCanvasViewDelegate(BaseWindowsCanvasView^ canvasView);
  delegate void VoidStrUIFormDelegate(String^ str1, UIForm^ form);
  delegate IntPtr IntPtrGRTManagerModuleStrStrGrtListFlagsDelegate(GrtManager^ grtManager, GrtModule^ module, String^ str1, String^ str2, GrtValue^ grtlist, GUIPluginFlags flags);
  delegate void VoidIntPtrDelegate(IntPtr p2);
  delegate void VoidRefreshTypeStringIntPtrDelegate(RefreshType refresh, String^ str, IntPtr ptr);
  delegate void VoidMenuItemListIntIntDelegate(List<MySQL::Grt::MenuItem^>^ plugins, int x, int y);
  delegate String^ StringAppCommandAppViewStringDelegate(AppCommand, MySQL::Forms::AppViewImpl^ view, String^ str);

private:

  ::wb::WBFrontendCallbacks *inner;
  GCHandle _cb_handle; // Keeps an unmanaged reference to this class for mforms::App callbacks.

  // ----------------------------------------------------------------------------
  // Wrapper delegates used by the C++.net wrapper

  // void ()
  delegate void VoidWrapperDelegate();
  typedef void (*WbFrontendCallbacks::VOID_CB)();
  // void (bool)
  delegate void VoidBoolWrapperDelegate(bool b);
  typedef void (*WbFrontendCallbacks::VOID_BOOL_CB)(bool b);
  // void (string)
  delegate void VoidStrWrapperDelegate(const std::string& str1);
  typedef void (*WbFrontendCallbacks::VOID_STR_CB)(const std::string& str1);
  // void (string, string)
  delegate void VoidStrStrWrapperDelegate(const std::string& str1, const std::string& str2);
  typedef void (*WbFrontendCallbacks::VOID_STR_STR_CB)(const std::string& str1, const std::string& str2);
  // string (string, string, string)
  delegate std::string StrStrStrStrWrapperDelegate(const std::string& str1, const std::string& str2, const std::string& str3);
  typedef std::string (*WbFrontendCallbacks::STR_STR_STR_STR_CB)(const std::string& str1, const std::string& str2, const std::string& str3);
  // bool (string, string, string)
  delegate bool BoolStrStrStrWrapperDelegate(const std::string& str1, const std::string& str2, const std::string& str3);
  typedef bool (*WbFrontendCallbacks::BOOL_STR_STR_STR_CB)(const std::string& str1, const std::string& str2, const std::string& str3);
  // int (string, string, string, string, string)
  delegate int IntStrStrStrStrStrWrapperDelegate(const std::string& title, const std::string& str1, const std::string& str2, const std::string& str3, const std::string& str4);
  typedef int (*WbFrontendCallbacks::INT_STR_STR_STR_STR_STR_CB)(const std::string& title, const std::string& str1, const std::string& str2, const std::string& str3, const std::string& str4);
  // bool (string, string, float)
  delegate bool BoolStrStrFloatWrapperDelegate(const std::string& str1, const std::string& str2, float f3);
  typedef bool (*WbFrontendCallbacks::BOOL_STR_STR_FLOAT_CB)(const std::string& str1, const std::string& str2, float f3);
  // bool (string, int, string&)
  delegate bool BoolStrIntStrPtrWrapperDelegate(const std::string &str, int, std::string &res);
  typedef bool (*WbFrontendCallbacks::BOOL_STR_INT_STRPTR_CB)(const std::string &str, int, std::string &res);
  // CanvasView * (model_DiagramRef&)
  delegate ::mdc::CanvasView* CanvasViewDiagramWrapperDelegate(const model_DiagramRef& diagram);
  typedef ::mdc::CanvasView* (*WbFrontendCallbacks::CANVASVIEW_DIAGRAM_CB)(const model_DiagramRef& diagram);
  // void (CanvasView *)
  delegate void VoidCanvasViewWrapperDelegate(::mdc::CanvasView *canvas_view);
  typedef void (*WbFrontendCallbacks::VOID_CANVASVIEW_CB)(::mdc::CanvasView *canvas_view);
  // _int64 (GrtManager *, Module*, string, string, GrtList, GUIPluginFlags)
  delegate __int64 IntGRTManagerModuleStrStrGrtListFlagsWrapperDelegate(::bec::GRTManager* grt_manager, grt::Module *, const std::string& str2, const std::string& str3, const grt::BaseListRef &grt_list, bec::GUIPluginFlags flags);
  typedef __int64 (*WbFrontendCallbacks::INT_GRTMANAGER_MODULE_STR_STR_GRTLIST_FLAGS_CB)(::bec::GRTManager* grt_manager, grt::Module *module, const std::string& str2, const std::string& str3, const grt::BaseListRef &grt_list, bec::GUIPluginFlags flags);
  // _int64 (GrtManager *, string, GrtList)
  delegate void VoidStrUIFormWrapperDelegate(const std::string& str1, boost::shared_ptr<bec::UIForm> form);
  typedef void (*WbFrontendCallbacks::VOID_STR_UIFORM_CB)(const std::string& str1, boost::shared_ptr<bec::UIForm> form);
  // void (__int64)
  delegate void VoidIntPtrWrapperDelegate(__int64 native_handle);
  typedef void (*WbFrontendCallbacks::VOID_INT64_CB)(__int64 native_handle);
  // void (RefreshType, string, __int64)
  delegate void VoidRefreshTypeStringIntPtrWrapperDelegate(::wb::RefreshType refresh, const std::string &str, __int64 ptr);
  typedef void (*WbFrontendCallbacks::VOID_REFRESHTYPE_STRING_INTPTR_CB)(::wb::RefreshType refresh, const std::string &str, __int64 ptr);
  // void (bec::MenuItemList, int, int)
  delegate void VoidMenuItemListIntIntWrapperDelegate(::bec::MenuItemList *plugins, int x, int y);
  typedef void (*WbFrontendCallbacks::VOID_MenuItemLIST_INT_INT_CB)(::bec::MenuItemList *plugins, int x, int y);

  // ----------------------------------------------------------------------------
  // Variables used to store the C# delegates and the C++.Net wrapper delegates
  // Intermediate callback that is called from C++ and calls the C# delegate

  BoolStrIntStrPtrDelegate^ request_input_delegate;
  BoolStrIntStrPtrWrapperDelegate^ request_input_wrapper_delegate;
  bool request_input_wrapper(const std::string &title, int flags, std::string& res)
  {
    String^ result= nullptr;
    bool flag;

    flag= request_input_delegate(CppStringToNative(title), flags, result);
    if (result != nullptr)
      res= NativeToCppString(result);

    return flag;
  }

  StrStrStrStrDelegate^ show_file_dialog_delegate;
  StrStrStrStrWrapperDelegate^ show_file_dialog_wrapper_delegate;
  std::string show_file_dialog_wrapper(const std::string& str1, const std::string& str2, const std::string& str3)
  { return NativeToCppString(show_file_dialog_delegate(CppStringToNative(str1), CppStringToNative(str2), CppStringToNative(str3))); }

  VoidStrDelegate^ show_status_text_delegate;
  VoidStrWrapperDelegate^ show_status_text_wrapper_delegate;
  void show_status_text_wrapper(const std::string& str1)
  { show_status_text_delegate(CppStringToNative(str1)); }

  BoolStrStrFloatDelegate^ show_progress_delegate;
  BoolStrStrFloatWrapperDelegate^ show_progress_wrapper_delegate;
  bool show_progress_wrapper(const std::string& str1, const std::string& str2, float f3)
  { return show_progress_delegate(CppStringToNative(str1), CppStringToNative(str2), f3); }

  VoidStrDelegate^ shell_output_delegate;
  VoidStrWrapperDelegate^ shell_output_wrapper_delegate;
  void shell_output_wrapper(const std::string& str1)
  { shell_output_delegate(CppStringToNative(str1)); }

  // CanvasView
  CanvasViewStringStringIntPtrDelegate^ create_diagram_delegate;
  CanvasViewDiagramWrapperDelegate^ create_diagram_wrapper_delegate;
  ::mdc::CanvasView *create_diagram_wrapper(const model_DiagramRef& model)
  {
    BaseWindowsCanvasView ^cv = create_diagram_delegate(CppStringToNative(model->id()),
      CppStringToNative(model->name()), IntPtr(&model.content()));
    return cv->get_unmanaged_object();
  }

  VoidCanvasViewDelegate^ destroy_view_delegate;
  VoidCanvasViewWrapperDelegate^ destroy_view_wrapper_delegate;
  void destroy_view_wrapper(::mdc::CanvasView *canvas_view)
  {
    BaseWindowsCanvasView^ wcv = 
      BaseWindowsCanvasView::GetFromFixedId((IntPtr)canvas_view->get_user_data());
    destroy_view_delegate(wcv);
  }

  VoidCanvasViewDelegate^ switched_view_delegate;
  VoidCanvasViewWrapperDelegate^ switched_view_wrapper_delegate;
  void switched_view_wrapper(::mdc::CanvasView *canvas_view)
  {
    BaseWindowsCanvasView^ wcv = 
      BaseWindowsCanvasView::GetFromFixedId((IntPtr)canvas_view->get_user_data());
    switched_view_delegate(wcv);
  }

  VoidCanvasViewDelegate^ tool_changed_delegate;
  VoidCanvasViewWrapperDelegate^ tool_changed_wrapper_delegate;
  void tool_changed_wrapper(::mdc::CanvasView *canvas_view)
  {
    BaseWindowsCanvasView^ wcv = 
      BaseWindowsCanvasView::GetFromFixedId((IntPtr)canvas_view->get_user_data());
    tool_changed_delegate(wcv);
  }

  VoidStrUIFormDelegate^ create_main_form_view_delegate;
  VoidStrUIFormWrapperDelegate^ create_main_form_view_wrapper_delegate;
  void create_main_form_view_wrapper(const std::string& view_name, boost::shared_ptr<bec::UIForm> form_be);


  // Editors
  IntPtrGRTManagerModuleStrStrGrtListFlagsDelegate^ open_editor_delegate;
  IntGRTManagerModuleStrStrGrtListFlagsWrapperDelegate^ open_editor_wrapper_delegate;
  __int64 open_editor_wrapper(::bec::GRTManager* grt_manager, grt::Module *module, const std::string& str2, const std::string& str3, const grt::BaseListRef &grt_list, bec::GUIPluginFlags flags)
  {
    return open_editor_delegate(gcnew GrtManager(grt_manager), gcnew GrtModule(module), 
      CppStringToNative(str2), CppStringToNative(str3),
      gcnew GrtValue(grt_list), (GUIPluginFlags)flags).ToInt64();
  }

  VoidIntPtrDelegate^ show_editor_delegate;
  VoidIntPtrWrapperDelegate^ show_editor_wrapper_delegate;
  void show_editor_wrapper(__int64 native_handle)
  {
    show_editor_delegate(IntPtr(native_handle));
  }

  VoidIntPtrDelegate^ hide_editor_delegate;
  VoidIntPtrWrapperDelegate^ hide_editor_wrapper_delegate;
  void hide_editor_wrapper(__int64 native_handle)
  {
    hide_editor_delegate(IntPtr(native_handle));
  }


  // Etc

  VoidRefreshTypeStringIntPtrDelegate ^refresh_gui_delegate;
  VoidRefreshTypeStringIntPtrWrapperDelegate ^refresh_gui_wrapper_delegate;
  void refresh_gui_wrapper(::wb::RefreshType refresh, const std::string &str, __int64 ptr)
  { refresh_gui_delegate((RefreshType)refresh, CppStringToNative(str), IntPtr(ptr)); }


  VoidMenuItemListIntIntDelegate ^popup_menu_delegate;
  VoidMenuItemListIntIntWrapperDelegate ^popup_menu_wrapper_delegate;
  void popup_menu_wrapper(::bec::MenuItemList *items, int x, int y)
  {
    popup_menu_delegate(CppVectorToObjectList<bec::MenuItem, MySQL::Grt::MenuItem>(*items), x, y);
  }


  VoidStrDelegate ^perform_command_delegate;
  VoidStrWrapperDelegate ^perform_command_wrapper_delegate;
  void perform_command_wrapper(const std::string &command)
  { perform_command_delegate(CppStringToNative(command)); }


  VoidBoolDelegate^ lock_gui_delegate;
  VoidBoolWrapperDelegate^ lock_gui_wrapper_delegate;
  void lock_gui_wrapper(bool flag)
  {
    lock_gui_delegate(flag);
  }

  VoidDelegate^ quit_application_delegate;
  VoidWrapperDelegate^ quit_application_wrapper_delegate;
  void quit_application_wrapper()
  {
    quit_application_delegate();
  }

  StringAppCommandAppViewStringDelegate^ app_command_delegate;
  std::string app_command_wrapper(AppCommand command, mforms::AppView* view, const std::string &str)
  {
    MySQL::Forms::AppViewImpl^ native_view= nullptr;
    if (view != NULL)
      native_view= (MySQL::Forms::AppViewImpl^) MySQL::Forms::ObjectImpl::FromUnmanaged(view);
    String^ result= app_command_delegate(command, native_view, CppStringToNative(str));
    return NativeToCppString(result);
  }

public:

  WbFrontendCallbacks(MySQL::Forms::Manager^ mgr,
                      StrStrStrStrDelegate^ show_file_dialog,
                      VoidStrDelegate^ show_status_text,
                      BoolStrStrFloatDelegate^ show_progress,
                      BoolStrIntStrPtrDelegate^ request_input,
                      CanvasViewStringStringIntPtrDelegate^ create_diagram,
                      VoidCanvasViewDelegate^ destroy_view,
                      VoidCanvasViewDelegate^ switched_view,
                      VoidCanvasViewDelegate^ tool_changed,
                      VoidStrUIFormDelegate^ create_main_form_view,
                      IntPtrGRTManagerModuleStrStrGrtListFlagsDelegate^ open_editor,
                      VoidIntPtrDelegate^ show_editor,
                      VoidIntPtrDelegate^ hide_editor,
                      VoidRefreshTypeStringIntPtrDelegate^ refresh_gui,
                      VoidBoolDelegate^ lock_gui,
                      VoidMenuItemListIntIntDelegate^ popup_menu,
                      VoidStrDelegate^ perform_command,
                      VoidDelegate^ quit_application,
                      StringAppCommandAppViewStringDelegate^ app_command)
                      : inner(new ::wb::WBFrontendCallbacks)
  {
    set_show_file_dialog(show_file_dialog);
    set_show_status_text(show_status_text);
    set_show_progress(show_progress);
    set_request_input(request_input);

    set_create_diagram(create_diagram);
    set_destroy_view(destroy_view);
    set_switched_view(switched_view);
    set_tool_changed(tool_changed);

    set_create_main_form_view(create_main_form_view);
    set_open_editor(open_editor);
    set_show_editor(show_editor);
    set_hide_editor(hide_editor);

    set_refresh_gui(refresh_gui);
    set_popup_menu(popup_menu);
    set_perform_command(perform_command);

    set_lock_gui(lock_gui);

    set_quit_application(quit_application);
    set_app_command(mgr, app_command);
  }

  ~WbFrontendCallbacks()
  {
    _cb_handle.Free();
    //delete inner;
  }

  ::wb::WBFrontendCallbacks *get_unmanaged_object()
  {
    return inner;
  }

  // ----------------------------------------------------------------------------
  // Setting the variables to the correct delegates and callbacks

  void set_show_file_dialog(StrStrStrStrDelegate^ dt)
  {
    show_file_dialog_delegate= dt;
    show_file_dialog_wrapper_delegate= 
      gcnew StrStrStrStrWrapperDelegate(this, &WbFrontendCallbacks::show_file_dialog_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(show_file_dialog_wrapper_delegate);
    STR_STR_STR_STR_CB cb = static_cast<STR_STR_STR_STR_CB>(ip.ToPointer());
    inner->show_file_dialog= sigc::ptr_fun(cb);
  }

  void set_show_status_text(VoidStrDelegate^ dt)
  {
    show_status_text_delegate= dt;
    show_status_text_wrapper_delegate= 
      gcnew VoidStrWrapperDelegate(this, &WbFrontendCallbacks::show_status_text_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(show_status_text_wrapper_delegate);
    VOID_STR_CB cb = static_cast<VOID_STR_CB>(ip.ToPointer());
    inner->show_status_text= sigc::ptr_fun(cb);
  }

  void set_show_progress(BoolStrStrFloatDelegate^ dt)
  {
    show_progress_delegate= dt;
    show_progress_wrapper_delegate= 
      gcnew BoolStrStrFloatWrapperDelegate(this, &WbFrontendCallbacks::show_progress_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(show_progress_wrapper_delegate);
    BOOL_STR_STR_FLOAT_CB cb = static_cast<BOOL_STR_STR_FLOAT_CB>(ip.ToPointer());
    inner->show_progress= sigc::ptr_fun(cb);
  }

  void set_request_input(BoolStrIntStrPtrDelegate^ dt)
  {
    request_input_delegate= dt;
    request_input_wrapper_delegate= 
      gcnew BoolStrIntStrPtrWrapperDelegate(this, &WbFrontendCallbacks::request_input_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(request_input_wrapper_delegate);
    BOOL_STR_INT_STRPTR_CB cb = static_cast<BOOL_STR_INT_STRPTR_CB>(ip.ToPointer());
    inner->request_input= sigc::ptr_fun(cb);
  }

  void set_create_diagram(CanvasViewStringStringIntPtrDelegate^ dt)
  {
    create_diagram_delegate= dt;
    create_diagram_wrapper_delegate=
      gcnew CanvasViewDiagramWrapperDelegate(this, &WbFrontendCallbacks::create_diagram_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(create_diagram_wrapper_delegate);
    CANVASVIEW_DIAGRAM_CB cb = static_cast<CANVASVIEW_DIAGRAM_CB>(ip.ToPointer());
    inner->create_diagram= sigc::ptr_fun(cb);
  }

  void set_destroy_view(VoidCanvasViewDelegate^ dt)
  {
    destroy_view_delegate= dt;
    destroy_view_wrapper_delegate=
      gcnew VoidCanvasViewWrapperDelegate(this, &WbFrontendCallbacks::destroy_view_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(destroy_view_wrapper_delegate);
    VOID_CANVASVIEW_CB cb = static_cast<VOID_CANVASVIEW_CB>(ip.ToPointer());
    inner->destroy_view= sigc::ptr_fun(cb);
  }

  void set_switched_view(VoidCanvasViewDelegate^ dt)
  {
    switched_view_delegate= dt;
    switched_view_wrapper_delegate=
      gcnew VoidCanvasViewWrapperDelegate(this, &WbFrontendCallbacks::switched_view_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(switched_view_wrapper_delegate);
    VOID_CANVASVIEW_CB cb = static_cast<VOID_CANVASVIEW_CB>(ip.ToPointer());
    inner->switched_view= sigc::ptr_fun(cb);
  }

  void set_tool_changed(VoidCanvasViewDelegate^ dt)
  {
    tool_changed_delegate= dt;
    tool_changed_wrapper_delegate=
      gcnew VoidCanvasViewWrapperDelegate(this, &WbFrontendCallbacks::tool_changed_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(tool_changed_wrapper_delegate);
    VOID_CANVASVIEW_CB cb = static_cast<VOID_CANVASVIEW_CB>(ip.ToPointer());
    inner->tool_changed= sigc::ptr_fun(cb);
  }





  void set_create_main_form_view(VoidStrUIFormDelegate^ dt)
  {
    create_main_form_view_delegate= dt;
    create_main_form_view_wrapper_delegate=
      gcnew VoidStrUIFormWrapperDelegate(this, &WbFrontendCallbacks::create_main_form_view_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(create_main_form_view_wrapper_delegate);
    VOID_STR_UIFORM_CB cb = static_cast<VOID_STR_UIFORM_CB>(ip.ToPointer());
    inner->create_main_form_view= sigc::ptr_fun(cb);
  }

  void set_open_editor(IntPtrGRTManagerModuleStrStrGrtListFlagsDelegate^ dt)
  {
    open_editor_delegate= dt;
    open_editor_wrapper_delegate=
      gcnew IntGRTManagerModuleStrStrGrtListFlagsWrapperDelegate(this, &WbFrontendCallbacks::open_editor_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(open_editor_wrapper_delegate);
    INT_GRTMANAGER_MODULE_STR_STR_GRTLIST_FLAGS_CB cb = static_cast<INT_GRTMANAGER_MODULE_STR_STR_GRTLIST_FLAGS_CB>(ip.ToPointer());
    inner->open_editor= sigc::ptr_fun(cb);
  }

  void set_show_editor(VoidIntPtrDelegate^ dt)
  {
    show_editor_delegate= dt;
    show_editor_wrapper_delegate= 
      gcnew VoidIntPtrWrapperDelegate(this, &WbFrontendCallbacks::show_editor_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(show_editor_wrapper_delegate);
    VOID_INT64_CB cb = static_cast<VOID_INT64_CB>(ip.ToPointer());
    inner->show_editor= sigc::ptr_fun(cb);
  }

  void set_hide_editor(VoidIntPtrDelegate^ dt)
  {
    hide_editor_delegate= dt;
    hide_editor_wrapper_delegate= 
      gcnew VoidIntPtrWrapperDelegate(this, &WbFrontendCallbacks::hide_editor_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(hide_editor_wrapper_delegate);
    VOID_INT64_CB cb = static_cast<VOID_INT64_CB>(ip.ToPointer());
    inner->hide_editor= sigc::ptr_fun(cb);
  }


  void set_refresh_gui(VoidRefreshTypeStringIntPtrDelegate ^ dt)
  {
    refresh_gui_delegate= dt;
    refresh_gui_wrapper_delegate= 
      gcnew VoidRefreshTypeStringIntPtrWrapperDelegate(this, &WbFrontendCallbacks::refresh_gui_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(refresh_gui_wrapper_delegate);
    VOID_REFRESHTYPE_STRING_INTPTR_CB cb = static_cast<VOID_REFRESHTYPE_STRING_INTPTR_CB>(ip.ToPointer());
    inner->refresh_gui= sigc::ptr_fun(cb);
  }

  void set_popup_menu(VoidMenuItemListIntIntDelegate ^ dt)
  {
    popup_menu_delegate= dt;
    popup_menu_wrapper_delegate= 
      gcnew VoidMenuItemListIntIntWrapperDelegate(this, &WbFrontendCallbacks::popup_menu_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(popup_menu_wrapper_delegate);
    VOID_MenuItemLIST_INT_INT_CB cb = static_cast<VOID_MenuItemLIST_INT_INT_CB>(ip.ToPointer());
    inner->popup_menu = sigc::ptr_fun(cb);
  }

  void set_perform_command(VoidStrDelegate ^ dt)
  {
    perform_command_delegate= dt;
    perform_command_wrapper_delegate= 
      gcnew VoidStrWrapperDelegate(this, &WbFrontendCallbacks::perform_command_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(perform_command_wrapper_delegate);
    VOID_STR_CB cb = static_cast<VOID_STR_CB>(ip.ToPointer());
    inner->perform_command = sigc::ptr_fun(cb);
  }

  void set_lock_gui(VoidBoolDelegate^ dt)
  {
    lock_gui_delegate= dt;
    lock_gui_wrapper_delegate=
      gcnew VoidBoolWrapperDelegate(this, &WbFrontendCallbacks::lock_gui_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(lock_gui_wrapper_delegate);
    VOID_BOOL_CB cb = static_cast<VOID_BOOL_CB>(ip.ToPointer());
    inner->lock_gui= sigc::ptr_fun(cb);
  }

  void set_quit_application(VoidDelegate^ dt)
  {
    quit_application_delegate= dt;
    quit_application_wrapper_delegate=
      gcnew VoidWrapperDelegate(this, &WbFrontendCallbacks::quit_application_wrapper);
    IntPtr ip = Marshal::GetFunctionPointerForDelegate(quit_application_wrapper_delegate);
    VOID_CB cb = static_cast<VOID_CB>(ip.ToPointer());
    inner->quit_application= sigc::ptr_fun(cb);
  }

  // Application command callback handling.
  static Object^ GetFromFixedId(IntPtr ip);
  static void dock_view(mforms::App *app, mforms::AppView *view, const std::string &position);
  static bool select_view(mforms::App *app, const std::string &ident);
  static void undock_view(mforms::App *app, mforms::AppView *view);
  static void set_view_title(mforms::App *app, mforms::AppView *view, const std::string &title);
  static std::string get_resource_path(mforms::App *app, const std::string &file);
  static void set_status_text(mforms::App *app, const std::string &text);
  static std::string get_bounds(mforms::App *app);
  static int enter_event_loop(mforms::App *app, float max_wait_time);
  static void exit_event_loop(mforms::App *app, int ret_code);

  void set_app_command(MySQL::Forms::Manager^ mgr, StringAppCommandAppViewStringDelegate^ dt);
};


public ref class WbOptions
{
  String^ basedir;
  String^ plugin_search_path;
  String^ struct_search_path;
  String^ module_search_path;
  String^ library_search_path;
  String^ user_data_dir;
  String^ open_at_startup;
  String^ open_at_startup_type;
  String^ run_at_startup;
  String^ run_at_startup_lang;
  bool force_sw_rendering;
  bool force_opengl_rendering;
  bool use_log_file;
  bool quit_when_done;

public:
/* unused
  WbOptions(const ::wb::WBOptions& opt)
    : basedir(CppStringToNative(opt.basedir)),
    plugin_search_path(CppStringToNative(opt.plugin_search_path)),
    struct_search_path(CppStringToNative(opt.struct_search_path)),
    module_search_path(CppStringToNative(opt.module_search_path)),
    library_search_path(CppStringToNative(opt.library_search_path)),
    user_data_dir(CppStringToNative(opt.user_data_dir)),
    open_at_startup(CppStringToNative(opt.open_at_startup)),
    open_at_startup_type(CppStringToNative(opt.open_at_startup_type)),
    run_at_startup(CppStringToNative(opt.run_at_startup)),
    force_sw_rendering(opt.force_sw_rendering),
    force_opengl_rendering(opt.force_opengl_rendering),
    use_log_file(opt.use_log_file),
    quit_when_done(opt.quit_when_done)
  {}*/

  WbOptions(String^ basedir, String^ plugin_search_path, String^ struct_search_path,
    String^ module_search_path, String^ library_search_path, String^ user_data_dir, String^ open_at_startup, 
    String^ open_at_startup_type, String^ run_at_startup, String^ run_at_startup_lang,
    bool force_sw_rendering, bool force_opengl_rendering, bool use_log_file, bool quit_when_done) 
    : basedir(basedir),
    plugin_search_path(plugin_search_path),
    struct_search_path(struct_search_path),
    module_search_path(module_search_path),
    library_search_path(library_search_path),
    user_data_dir(user_data_dir),
    open_at_startup(open_at_startup),
    open_at_startup_type(open_at_startup_type),
    run_at_startup(run_at_startup),
    run_at_startup_lang(run_at_startup_lang),
    force_sw_rendering(force_sw_rendering),
    force_opengl_rendering(force_opengl_rendering),
    use_log_file(use_log_file),
    quit_when_done(quit_when_done)
  {
  }

  ::wb::WBOptions *get_unmanaged_object()
  {
    ::wb::WBOptions *opt= new ::wb::WBOptions;
    opt->basedir= NativeToCppString(basedir);
    opt->plugin_search_path= NativeToCppString(plugin_search_path);
    opt->struct_search_path= NativeToCppString(struct_search_path);
    opt->module_search_path= NativeToCppString(module_search_path);
    opt->library_search_path= NativeToCppString(library_search_path);
    opt->user_data_dir= NativeToCppString(user_data_dir);
    opt->open_at_startup= NativeToCppString(open_at_startup);
    opt->open_at_startup_type= NativeToCppString(open_at_startup_type);
    opt->run_at_startup= NativeToCppString(run_at_startup);
    opt->run_language= NativeToCppString(run_at_startup_lang);
    opt->force_sw_rendering= force_sw_rendering;
    opt->force_opengl_rendering= force_opengl_rendering;
    opt->use_log_file= use_log_file;
    opt->quit_when_done= quit_when_done;
    return opt;
  }
/* unused 
  String^ get_basedir() { return basedir; }
  String^ get_plugin_search_path() { return plugin_search_path; }
  String^ get_struct_search_path() { return struct_search_path; }
  String^ get_module_search_path() { return module_search_path; }
  String^ get_user_data_dir() { return user_data_dir; }
  String^ get_open_at_startup() { return open_at_startup; }
  bool get_force_sw_rendering() { return force_sw_rendering; }
  bool get_force_opengl_rendering() { return force_opengl_rendering; }
  bool get_use_log_file() { return use_log_file; }
  */
};


public ref class WbContext
{
  ::wb::WBContextUI *inner;

  ::MySQL::Grt::GrtManager^ manager;
  ::MySQL::Workbench::Overview^ physical_overview;
  ::MySQL::Grt::GrtValueTree^ catalog_tree;
  ::MySQL::Grt::TreeModel^ history_tree;
  ::MySQL::Grt::ListModel^ usertypes_tree;

  ::System::Collections::ArrayList open_editor_slot_wrappers;

  DelegateSlot0<void,void> ^undo_delegate;
  DelegateSlot0<bool,bool> ^can_undo_delegate;
  DelegateSlot0<void,void> ^redo_delegate;
  DelegateSlot0<bool,bool> ^can_redo_delegate;
  DelegateSlot0<void,void> ^copy_delegate;
  DelegateSlot0<bool,bool> ^can_copy_delegate;
  DelegateSlot0<void,void> ^cut_delegate;
  DelegateSlot0<bool,bool> ^can_cut_delegate;
  DelegateSlot0<void,void> ^paste_delegate;
  DelegateSlot0<bool,bool> ^can_paste_delegate;
  DelegateSlot0<void,void> ^select_all_delegate;
  DelegateSlot0<bool,bool> ^can_select_all_delegate;
  DelegateSlot0<void,void> ^delete_delegate;
  DelegateSlot0<bool,bool> ^can_delete_delegate;
  DelegateSlot0<void,void> ^search_delegate;
  DelegateSlot0<void,void> ^find_delegate;
  DelegateSlot0<bool,bool> ^can_find_delegate;
  DelegateSlot0<void,void> ^find_replace_delegate;
  DelegateSlot0<bool,bool> ^can_find_replace_delegate;

public:
  delegate NativeHandle CreateMainFormViewSlotDelegate(::MySQL::Grt::GrtManager^ manager, ::MySQL::Grt::GrtValue^ args);
  ref class CreateMainFormViewSlotWrapper
  {

    typedef NativeHandle (*CreateMainFormViewSlotWrapper::CREATE_MAIN_FORM_VIEW)(bec::GRTManager* mgr, grt::BaseListRef args);
    delegate NativeHandle CreateMainFormViewSlotWrapperDelegate(bec::GRTManager*, grt::BaseListRef);

    IntPtr native_slot_ptr;
    CreateMainFormViewSlotDelegate^ slot;
    CreateMainFormViewSlotWrapperDelegate^ slot_wrapper;

    NativeHandle wrapper(bec::GRTManager* mgr, grt::BaseListRef args)
    {
      return slot(gcnew ::MySQL::Grt::GrtManager(mgr), gcnew ::MySQL::Grt::GrtValue(args));
    }
  public:

    CreateMainFormViewSlotWrapper(CreateMainFormViewSlotDelegate^ s) :
        slot(s)
    {
      slot_wrapper= gcnew CreateMainFormViewSlotWrapperDelegate(this, &CreateMainFormViewSlotWrapper::wrapper);
      native_slot_ptr = Marshal::GetFunctionPointerForDelegate(slot_wrapper);
    }

    //::bec::EditorManagerImpl::CreateMainFormViewSlot get_native_slot()
    //{
    //  CreateMainFormViewSlotDelegate^ sw_delegate= slot;
    //  CreateMainFormViewSlotWrapper::CreateMainFormViewSlotWrapperDelegate^ sw_wrapper_delegate= 
    //    gcnew CreateMainFormViewSlotWrapper::CreateMainFormViewSlotWrapperDelegate(this, &CreateMainFormViewSlotWrapper::wrapper);
    //  IntPtr ip = Marshal::GetFunctionPointerForDelegate(sw_wrapper_delegate);
    //  CREATE_MAIN_FORM_VIEW cb = static_cast<CREATE_MAIN_FORM_VIEW>(ip.ToPointer());
    //  return sigc::ptr_fun(cb);
    //}
  };

  delegate NativeHandle OpenEditorSlotDelegate(::MySQL::Grt::GrtManager^ manager, ::MySQL::Grt::GrtValue^ args);

  ref class OpenEditorSlotWrapper
  {

    typedef NativeHandle (*OpenEditorSlotWrapper::OPEN_EDITOR_CB)(bec::GRTManager* mgr, grt::BaseListRef args);
    delegate NativeHandle OpenEditorSlotWrapperDelegate(bec::GRTManager*, grt::BaseListRef);

    IntPtr native_slot_ptr;
    OpenEditorSlotDelegate^ slot;
    OpenEditorSlotWrapperDelegate^ slot_wrapper;

    NativeHandle wrapper(bec::GRTManager* mgr, grt::BaseListRef args)
    {
      return slot(gcnew ::MySQL::Grt::GrtManager(mgr), gcnew ::MySQL::Grt::GrtValue(args));
    }
  public:

    OpenEditorSlotWrapper(OpenEditorSlotDelegate^ s) :
        slot(s)
    {
      slot_wrapper= gcnew OpenEditorSlotWrapperDelegate(this, &OpenEditorSlotWrapper::wrapper);
      native_slot_ptr = Marshal::GetFunctionPointerForDelegate(slot_wrapper);
    }

    //::bec::EditorManagerImpl::OpenEditorSlot get_native_slot()
    //{
    //  OpenEditorSlotDelegate^ sw_delegate= slot;
    //  OpenEditorSlotWrapper::OpenEditorSlotWrapperDelegate^ sw_wrapper_delegate= 
    //    gcnew OpenEditorSlotWrapper::OpenEditorSlotWrapperDelegate(this, &OpenEditorSlotWrapper::wrapper);
    //  IntPtr ip = Marshal::GetFunctionPointerForDelegate(sw_wrapper_delegate);
    //  OPEN_EDITOR_CB cb = static_cast<OPEN_EDITOR_CB>(ip.ToPointer());
    //  return sigc::ptr_fun(cb);
    //}
  };

  //WbContext(::wb::WBContext *inn) : inner(inn), manager(nullptr)
  //{ }

  WbContext(bool verbose) : inner(new ::wb::WBContextUI(verbose)), manager(nullptr)
  { }

  ~WbContext()
  {
  }

  ::wb::WBContextUI *get_unmanaged_object() 
  { return inner; }

  bool init(WbFrontendCallbacks^ callbacks, WbOptions^ options)
  {
    return inner->init(
      callbacks->get_unmanaged_object(), 
      options->get_unmanaged_object());
  }

  bool use_opengl()
  {
    return inner->get_wb()->use_opengl();
  }

  bool is_busy()
  {
    return inner->get_wb()->get_grt_manager()->get_dispatcher()->get_busy();
  }

  bool request_quit()
  {
    return inner->request_quit();
  }

  void perform_quit()
  {
    inner->perform_quit();
  }

  void finalize()
  {
    inner->finalize();
  }

  GrtManager^ get_grt_manager()
  {
    if (manager == nullptr)
      manager= gcnew GrtManager(inner->get_wb()->get_grt_manager());

    return manager; 
  }

  String^ get_title()
  { return CppStringToNative(inner->get_title()); }

  bool has_unsaved_changes()
  { return inner->get_wb()->has_unsaved_changes(); }

  void open_document(String^ file)
  {
    inner->get_wb()->open_document(NativeToCppString(file));
  }

  bool save_changes()
  { return inner->get_wb()->save_changes(); }

  void flush_idle_tasks()
  {
    try 
    {
      inner->get_wb()->flush_idle_tasks(); 
    }
    catch(std::exception *ex)
    {
      throw gcnew BackendException(ex);
    }
    catch(std::exception &ex)
    {
      throw gcnew BackendException(&ex);
    }
    catch(...)
    {
      throw; // Don't wrap the exception into something completely useless. The original exception
             // often is converted automatically into a proper managed exception.
    }
  }

  double delay_for_next_timer()
  {
    return inner->get_wb()->get_grt_manager()->delay_for_next_timeout();
  }

  void flush_timers()
  {
    return inner->get_wb()->get_grt_manager()->flush_timers();
  }


  // edit menu handling
  void edit_undo()
  {
    if (inner->get_active_main_form())
      inner->get_active_main_form()->undo();
  }

  bool edit_can_undo()
  {
    if (inner->get_active_main_form() && inner->get_active_main_form()->can_undo())
      return true;
    return false;
  }

  void edit_redo()
  {
    if (inner->get_active_main_form())
      inner->get_active_main_form()->redo();
  }

  bool edit_can_redo()
  {
    if (inner->get_active_main_form() && inner->get_active_main_form()->can_redo())
      return true;
    return false;
  }

  void edit_copy()
  {
    if (inner->get_active_form())
      inner->get_active_form()->copy();
  }

  bool edit_can_copy()
  {
    if (inner->get_active_form() && inner->get_active_form()->can_copy())
      return true;
    return false;
  }

  void edit_cut()
  {
    if (inner->get_active_form())
      inner->get_active_form()->cut();
  }

  bool edit_can_cut()
  {
    if (inner->get_active_form() && inner->get_active_form()->can_cut())
      return true;
    return false;
  }

  void edit_paste()
  {
    if (inner->get_active_form())
      inner->get_active_form()->paste();
  }

  bool edit_can_paste()
  {
    if (inner->get_active_form() && inner->get_active_form()->can_paste())
      return true;
    return false;
  }

  void edit_select_all()
  {
    if (inner->get_active_form())
      inner->get_active_form()->select_all();
  }

  bool edit_can_select_all()
  {
    if (inner->get_active_form() && inner->get_active_form()->can_select_all())
      return true;
    return false;
  }

  void edit_delete()
  {
    if (inner->get_active_form())
      inner->get_active_form()->delete_selection();
  }

  bool edit_can_delete()
  {
    if (inner->get_active_form() && inner->get_active_form()->can_delete())
      return true;
    return false;
  }


  bool try_searching_diagram(String ^text)
  {
    bec::UIForm *form= inner->get_active_main_form();
    if (form && dynamic_cast<wb::ModelDiagramForm*>(form))
    {
      dynamic_cast<wb::ModelDiagramForm*>(form)->search_and_focus_object(NativeToCppString(text));
      return true;
    }
    return false;
  }

  
  typedef DelegateSlot0<void, void> CommandActionDelegate;
  typedef DelegateSlot0<bool, bool> CommandValidateDelegate;
  

  void set_edit_menu_delegates(CommandActionDelegate::ManagedDelegate ^undo_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_undo_delegate,
                               CommandActionDelegate::ManagedDelegate ^redo_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_redo_delegate,
                               CommandActionDelegate::ManagedDelegate ^copy_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_copy_delegate,
                               CommandActionDelegate::ManagedDelegate ^cut_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_cut_delegate,
                               CommandActionDelegate::ManagedDelegate ^paste_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_paste_delegate,
                               CommandActionDelegate::ManagedDelegate ^select_all_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_select_all_delegate,
                               CommandActionDelegate::ManagedDelegate ^delete_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_delete_delegate,
                               CommandActionDelegate::ManagedDelegate ^search_delegate,
                               CommandActionDelegate::ManagedDelegate ^find_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_find_delegate,
                               CommandActionDelegate::ManagedDelegate ^find_replace_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_find_replace_delegate)
  {
    this->undo_delegate= gcnew CommandActionDelegate(undo_delegate);
    this->can_undo_delegate= gcnew CommandValidateDelegate(can_undo_delegate);
    this->redo_delegate= gcnew CommandActionDelegate(redo_delegate);
    this->can_redo_delegate= gcnew CommandValidateDelegate(can_redo_delegate);
    this->copy_delegate= gcnew CommandActionDelegate(copy_delegate);
    this->can_copy_delegate= gcnew CommandValidateDelegate(can_copy_delegate);
    this->cut_delegate= gcnew CommandActionDelegate(cut_delegate);
    this->can_cut_delegate= gcnew CommandValidateDelegate(can_cut_delegate);
    this->paste_delegate= gcnew CommandActionDelegate(paste_delegate);
    this->can_paste_delegate= gcnew CommandValidateDelegate(can_paste_delegate);
    this->select_all_delegate= gcnew CommandActionDelegate(select_all_delegate);
    this->can_select_all_delegate= gcnew CommandValidateDelegate(can_select_all_delegate);
    this->delete_delegate= gcnew CommandActionDelegate(delete_delegate);
    this->can_delete_delegate= gcnew CommandValidateDelegate(can_delete_delegate);
    this->search_delegate= gcnew CommandActionDelegate(search_delegate);
    this->find_delegate= gcnew CommandActionDelegate(find_delegate);
    this->can_find_delegate= gcnew CommandValidateDelegate(can_find_delegate);
    this->find_replace_delegate= gcnew CommandActionDelegate(find_replace_delegate);
    this->can_find_replace_delegate= gcnew CommandValidateDelegate(can_find_replace_delegate);

    inner->get_command_ui()->add_builtin_command("undo", 
        this->undo_delegate->get_slot(),
        this->can_undo_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("redo", 
        this->redo_delegate->get_slot(),
        this->can_redo_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("copy", 
        this->copy_delegate->get_slot(),
        this->can_copy_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("cut", 
        this->cut_delegate->get_slot(),
        this->can_cut_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("paste", 
        this->paste_delegate->get_slot(),
        this->can_paste_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("selectAll", 
        this->select_all_delegate->get_slot(),
        this->can_select_all_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("delete", 
        this->delete_delegate->get_slot(),
        this->can_delete_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("find", 
        this->find_delegate->get_slot(),
        this->can_find_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("search_replace", 
        this->find_replace_delegate->get_slot(),
        this->can_find_replace_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("searchbox", 
        this->search_delegate->get_slot());
  }
  


  // plugins
  void add_frontend_commands(List<String^>^ commands)
  {
    inner->get_command_ui()->add_frontend_commands(NativeToCppStringList2(commands));
  }

  List<MySQL::Grt::MenuItem^>^ get_main_menus();
  List<MySQL::Grt::MenuItem^>^ get_items_for_menu(String^ menu);

  List<ToolbarItem^>^ get_items_for_toolbar(String^ toolbar)
  { return CppVectorToObjectList<::bec::ToolbarItem, ToolbarItem>(inner->get_command_ui()->get_toolbar_items(NativeToCppString(toolbar))); }


  void activate_command(String^ name)
  {
    std::string command= NativeToCppString(name);
    inner->get_command_ui()->activate_command(command);
  }

  List<String^>^ get_dropdown_items(String ^name, String^ option, [Out] String^ %selected)
  {
    std::string c_selected;
    List<String^>^ list;
    list= CppStringListToNative(inner->get_command_ui()->get_dropdown_items(NativeToCppString(name), NativeToCppString(option), c_selected));
    selected= CppStringToNative(c_selected);
    return list;
  }


  void select_dropdown_item(String^ name, String^ option, String^ item)
  { inner->get_command_ui()->select_dropdown_item(NativeToCppString(name), NativeToCppString(option), NativeToCppString(item)); }


  void toggle_checkbox_item(String^ name, String^ option, bool state)
  { inner->get_command_ui()->toggle_checkbox_item(NativeToCppString(name), NativeToCppString(option), state); }

  virtual FindDialogBE^ get_find_dialog()
  {
    return gcnew FindDialogBE(inner->get_find_dialog());
  }

  // Overview

  Overview^ get_physical_overview()
  {
    if (physical_overview == nullptr)
      physical_overview= gcnew Overview(inner->get_physical_overview());
    return physical_overview;
  }

  ::MySQL::Grt::GrtValueTree^ get_catalog_tree()
  {
    if (catalog_tree == nullptr)
      catalog_tree= gcnew ::MySQL::Grt::GrtValueTree(inner->get_catalog_tree());

    return catalog_tree;
  }

  ::MySQL::Grt::TreeModel^ get_history_tree()
  {
    if (history_tree == nullptr)
      history_tree= gcnew ::MySQL::Grt::TreeModel(inner->get_history_tree());
    
    return history_tree;
  }

  ::MySQL::Grt::ListModel^ get_usertypes_tree()
  {
    if (usertypes_tree == nullptr)
      usertypes_tree= gcnew ::MySQL::Grt::ListModel(inner->get_usertypes_tree());
    
    return usertypes_tree;
  } 

  ::MySQL::Grt::GrtValueInspector^ get_inspector_for_selection(UIForm^ form, [Out] List<String^>^ %items)
  {
    std::vector<std::string> vitems;
    bec::ValueInspectorBE *insp= 
      inner->create_inspector_for_selection(form->get_unmanaged_object(), vitems);
    if (!insp)
      insp= inner->create_inspector_for_selection(vitems);
    if (insp)
    {
      items= CppStringListToNative(vitems);
      return gcnew ::MySQL::Grt::GrtValueInspector(insp, true);
    }
    return nullptr;
  }

  bool are_lists_equal(GrtValue^ v1, GrtValue^ v2)
  {
    if ((nullptr == v1) || (nullptr == v2))
      return (v1 == v2);
    ::grt::ValueRef v1_(v1->get_unmanaged_object());
    ::grt::ValueRef v2_(v2->get_unmanaged_object());
    ::grt::ObjectListRef l1_= ::grt::ObjectListRef::cast_from(v1_);
    ::grt::ObjectListRef l2_= ::grt::ObjectListRef::cast_from(v2_);

    return ::grt::compare_list_contents(l1_, l2_);
  }

  String^ get_description_for_selection(UIForm^ form, [Out] GrtValue^ %activeObjList, [Out] List<String^>^ %items)
  {
    std::vector<std::string> vitems;
    ::grt::ListRef<GrtObject> activeObjList_;
    std::string val= 
      inner->get_description_for_selection(form->get_unmanaged_object(), activeObjList_, vitems);
    activeObjList= gcnew GrtValue(activeObjList_);
    items= CppStringListToNative(vitems);
    return CppStringToNative(val);
  }

  String^ get_description_for_selection([Out] GrtValue^ %activeObjList, [Out] List<String^>^ %items)
  {
    std::vector<std::string> vitems;
    ::grt::ListRef<GrtObject> activeObjList_;
    std::string val= inner->get_description_for_selection(activeObjList_, vitems);
    activeObjList= gcnew GrtValue(activeObjList_);
    items= CppStringListToNative(vitems);
    return CppStringToNative(val);
  }

  void set_description_for_selection(GrtValue^ activeObjList, String^ val)
  {
    std::string val_= NativeToCppString(val);
    ::grt::ListRef<GrtObject> activeObjList_(::grt::ListRef<GrtObject>::cast_from(activeObjList->get_unmanaged_object()));
    inner->set_description_for_selection(activeObjList_, val_);
  }

  ::MySQL::Workbench::ModelViewForm^ get_diagram_form_for_diagram(String^ id)
  {
    return gcnew ModelViewForm(inner->get_wb()->get_model_context()->get_diagram_form_for_diagram_id(NativeToCppString(id)));
  }


  void set_active_form(UIForm^ uiform)
  {
    if (uiform == nullptr)
      inner->set_active_form(0);
    else
      inner->set_active_form(uiform->get_unmanaged_object());
  }

  void set_active_form_from_appview(MySQL::Forms::AppViewDockContent^ form)
  {
    if (form == nullptr)
      inner->set_active_form(0);
    else
      inner->set_active_form(form->get_unmanaged_object());
  }

  String ^get_active_context()
  { return CppStringToNative(inner->get_active_context()); }


  void show_output()
  {
    inner->show_output();
  }

  void close_gui_plugin(IntPtr handle)
  {
    inner->get_wb()->close_gui_plugin((__int64)handle);
  }
  
  void execute_plugin(String^ name)
  {
    inner->get_wb()->execute_plugin(NativeToCppStringRaw(name));
  }

  // State support.
  String^ read_state(String^ name, String^ domain, String^ default_value);
  int read_state(String^ name, String^ domain, const int default_value);
  double read_state(String^ name, String^ domain, const double default_value);
  bool read_state(String^ name, String^ domain, const bool default_value);

  void save_state(String^ name, String^ domain, String^ value);
  void save_state(String^ name, String^ domain, const int value);
  void save_state(String^ name, String^ domain, const double value);
  void save_state(String^ name, String^ domain, const bool value);

  // Preferences support.
  String^ read_option_value(String^ model, String^ key, String^ default_value);

  // Paper.
  List<PaperSize^>^ get_paper_sizes()
  {
    return MySQL::Grt::CppListToObjectList<::wb::WBPaperSize,PaperSize>(inner->get_paper_sizes(false));
  }


  PageSettings^ get_page_settings()
  {
    app_PageSettingsRef settings(inner->get_page_settings());
    if (settings.is_valid())
      return gcnew PageSettings(settings);
    return nullptr;
  }


  void set_page_settings(PageSettings^ settings)
  {
    settings->update_object(inner->get_wb()->get_root()->options()->paperTypes(), inner->get_page_settings());
    inner->get_wb()->get_model_context()->update_page_settings();
  }

  /**
   * To be called by the front end once the main form is ready.
   */
  void finished_loading(WbOptions^ options)
  {
    inner->init_finish(options->get_unmanaged_object());
  }

  void close_document_finish()
  {
    inner->get_wb()->close_document_finish();

    // release overview object
    physical_overview = nullptr;
  }

  void new_model_finish()
  {
    inner->get_wb()->new_model_finish();
  }

  String^ get_filename()
  {
    return CppStringToNative(inner->get_wb()->get_filename());
  }  

};



public ref class DiagramOptionsBE
{
  wb::DiagramOptionsBE *inner;

  MySQL::Grt::DelegateSlot0<void, void> ^_delegate;

public:
  DiagramOptionsBE(WindowsGDICanvasView ^view, WbContext ^wbContext, MySQL::Grt::DelegateSlot0<void, void>::ManagedDelegate ^deleg)
  {
    _delegate= gcnew MySQL::Grt::DelegateSlot0<void, void>(deleg);

    inner= new wb::DiagramOptionsBE(view->get_unmanaged_object(), 
      wbContext->get_unmanaged_object()->get_wb()->get_model_context()->get_active_model_diagram(true), wbContext->get_unmanaged_object()->get_wb());

    inner->signal_changed().connect(_delegate->get_slot());
  }

  ~DiagramOptionsBE()
  {
    delete _delegate;
    delete inner;
  }

  String ^get_name()
  {
    return CppStringToNative(inner->get_name());
  }

  void set_name(String ^name)
  {
    inner->set_name(NativeToCppString(name));
  }

  int get_xpages() 
  {
    return inner->get_xpages();
  }

  int get_ypages()
  {
    return inner->get_ypages();
  }

  void set_xpages(int c)
  {
    inner->set_xpages(c);
  }

  void set_ypages(int c)
  {
    inner->set_ypages(c);
  }

  void commit()
  {
    inner->commit();
  }

  void update_size()
  {
    inner->update_size();
  }
};


} // namespace Workbench
} // namespace MySQL

#endif // __WB_H__
