/* 
 * Copyright (c) 2008, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#ifndef _MFORMS_VIEW_H_
#define _MFORMS_VIEW_H_

#include <mforms/base.h>
#include <sigc++/sigc++.h>

#include "geometry.h"

namespace mforms {

  class View;

  // Describes in which direction a control might be resized by the layouters.
  enum AutoResizeMode {
    ResizeNone,
    ResizeVertical,
    ResizeHorizontal,
    ResizeBoth
  };

  // Layout definition for background images.
  enum ImageLayout {
    LayoutNone,    //!< The image is left-aligned at the top across the control's client rectangle. 
    LayoutTile,    //!< The image is tiled across the control's client rectangle. 
    LayoutCenter,  //!< The image is centered within the control's client rectangle. 
    LayoutStretch, //!< The image is stretched across the control's client rectangle. 
    LayoutZoom     //!< The image is enlarged within the control's client rectangle. 
  };

  // Mouse button indicators for mouse handling routines.
  enum MouseButton {
    MouseButtonLeft = 0,
    MouseButtonRight = 1,
    MouseButtonMiddle = 2,
  };

#ifndef DOXYGEN_SHOULD_SKIP_THIS
#ifndef SWIG
  struct ViewImplPtrs
  {
    void (__stdcall *destroy)(View *self);

    int (__stdcall *get_width)(View *self);
    int (__stdcall *get_height)(View *self);
    int (__stdcall *get_preferred_width)(View *self);
    int (__stdcall *get_preferred_height)(View *self);
    void (__stdcall *set_size)(View *self, int, int);
    void (__stdcall *set_padding)(View *self, const MySQL::Geometry::Padding&);

    int (__stdcall *get_x)(View *self);
    int (__stdcall *get_y)(View *self);
    void (__stdcall *set_position)(View *self, int, int);
    void (__stdcall *client_to_screen)(View *self, MySQL::Geometry::Point&);

    void (__stdcall *show)(View *self, bool);
    bool (__stdcall *is_shown)(View *self);
    
    void (__stdcall *set_tooltip)(View *self, const std::string&);
    void (__stdcall *set_name)(View *self, const std::string&);
    void (__stdcall *set_font)(View *self, const std::string&);
    
    void (__stdcall *set_enabled)(View *self, bool);
    void (__stdcall *relayout)(View *self);
    void (__stdcall *set_needs_repaint)(View *self);

    void (__stdcall *suspend_layout)(View *self, bool);
    void (__stdcall *set_front_color)(View *self, const std::string&);
    void (__stdcall *set_back_color)(View *self, const std::string&);
    void (__stdcall *set_back_image)(View *self, const std::string&, ImageLayout layout);

    void (__stdcall *flush_events)(View *self);
  };
#endif
#endif
  
  class Form;

  class MFORMS_EXPORT View : public Object
  {
    friend class ControlFactory;

    ViewImplPtrs *_view_impl;
  private:
    std::string _name;
    AutoResizeMode _resize_mode;
    std::list<std::pair<View*, bool> > _subviews;
    bool _layout_dirty;
    
  protected:
    View();

    View *_parent;

    sigc::signal<void> _signal_resized;

    void add_subview(View *sv);
    void remove_subview(View *sv);
    int get_subview_index(View *sv);
    View *get_subview_at_index(int index);
    
  public:
    virtual ~View();
    
    virtual void set_managed();
    
    View *find_subview(const std::string &name);
    bool contains_subview(View* subview);
    
    AutoResizeMode get_resize_mode();
    void set_resize_mode(AutoResizeMode mode);

    sigc::signal<void> signal_resized() { return _signal_resized; }
    
    void set_name(const std::string &name);
    void set_tooltip(const std::string &text);
    void set_font(const std::string &fontDescription); // e.g. "Trebuchet MS bold 9"
    std::string get_name();
    void set_parent(View *parent);
    View *get_parent() const;
    Form *get_parent_form() const;
    virtual int get_width();
    virtual int get_height();
    virtual int get_preferred_width();
    virtual int get_preferred_height();
    virtual int get_x();
    virtual int get_y();
    virtual void set_position(int x, int y);
    virtual void set_size(int width, int height);
    virtual void set_padding(const MySQL::Geometry::Padding& padding);
    void client_to_screen(MySQL::Geometry::Point& position);


    void show(bool flag= true);
    bool is_shown();
    void set_enabled(bool flag);
    void set_needs_repaint();

    virtual void needs_resize();
    virtual void relayout();
    virtual void set_layout_dirty(bool value);
    virtual bool is_layout_dirty();
    virtual void on_resize();
    
    void suspend_layout();
    void resume_layout();
    void set_front_color(const std::string& color);
    void set_back_color(const std::string& color);
    void set_back_image(const std::string& path, ImageLayout layout);
    
    void show_retain_counts(int depth=0);

    virtual std::string get_string_value() { return ""; }
    virtual int get_int_value() { return 0; }
    virtual bool get_bool_value() { return false; }

    virtual void flush_events();
  };
};

#endif
