/* 
 * Copyright (c) 2009, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "wb_live_schema_tree.h"
#include "grtdb/charset_utils.h"
#include "base/string_utilities.h"

using namespace wb;
using namespace bec;
using namespace base;
using namespace grt;

static bool compare_schema_node(const LiveSchemaTree::SchemaNode &a, const LiveSchemaTree::SchemaNode &b)
{
  return a.name < b.name;
}


static std::string opt_quote_ident(const std::string &ident)
{
  bool needs_quotation= false;
  for (std::string::const_iterator i= ident.begin(); i != ident.end(); ++i)
  {
    if ((*i >= 'a' && *i <= 'z') || (*i >= 'A' && *i <= 'Z') || (*i >= '0' && *i <= '9')
      || (*i == '_')|| (*i == '\\')|| (*i == '"'))
      continue;
    needs_quotation= true;
    break;
  }
  if (needs_quotation)
    return "`"+ident+"`";
  else
    return ident;
}


LiveSchemaTree::LiveSchemaTree(grt::GRT* grt)
: _grt(grt), _schemata_vector_serial(0), _is_schema_contents_enabled(true)
{
}


std::vector<LiveSchemaTree::ObjectNode> * LiveSchemaTree::get_object_nodes_by_index(const bec::NodeId &index_node, ObjectType *object_type)
{
  SchemaNode &snode(_schemata[index_node[0]]);
  std::vector<ObjectNode> *obj_nodes= NULL;
  ObjectType type;
  
  switch (index_node[1])
  {
    case 0:
      type= Table;
      obj_nodes= &snode.tables;
      break;
    case 1:
      type= View;
      obj_nodes= &snode.views;
      break;
    case 2:
      type= Routine;
      obj_nodes= &snode.routines;
      break;
    default:
      return NULL;
  }

  if (obj_nodes && object_type)
    *object_type= type;

  return obj_nodes;
}


LiveSchemaTree::ObjectNode * LiveSchemaTree::get_object_node_by_index(const bec::NodeId &index_node, ObjectType *object_type)
{
  std::vector<ObjectNode> *obj_nodes= get_object_nodes_by_index(index_node, object_type);

  if (obj_nodes && (index_node[2] < (int)obj_nodes->size()))
  {
    ObjectNode *obj_node= &(*obj_nodes)[index_node[2]];
    return obj_node;
  }
  else
  {
    return NULL;
  }
}


std::vector<LiveSchemaTree::ObjectNode> * LiveSchemaTree::get_object_nodes_by_type(SchemaNode *schema_node, ObjectType obj_type)
{
  std::vector<ObjectNode> * obj_nodes;
  switch (obj_type)
  {
  case Table: obj_nodes= &schema_node->tables; break;
  case View: obj_nodes= &schema_node->views; break;
  case Routine: obj_nodes= &schema_node->routines; break;
  default: obj_nodes= NULL; break;
  }
  return obj_nodes;
}


void LiveSchemaTree::refresh()
{
  refresh_ex(false);
}


void LiveSchemaTree::refresh_ex(bool hard_refresh)
{
  std::list<std::string> schema_list= list_fetch_slot();
  refresh_ex(schema_list, hard_refresh);
}


void LiveSchemaTree::refresh_ex(const std::list<std::string> &schema_list, bool hard_refresh)
{
  std::map<std::string, SchemaNode> old_schemata;
  
  pre_refresh_signal.emit();
  
  if (!hard_refresh)
  {
    // save old data in a map so that we can reuse the contents
    for (std::vector<SchemaNode>::const_iterator iter= _schemata.begin();
         iter != _schemata.end(); ++iter)
    {
      old_schemata[iter->name]= *iter;
    }
  }

  _schemata.clear();
  _schemata.reserve(schema_list.size());
  for (std::list<std::string>::const_iterator iter= schema_list.begin();
       iter != schema_list.end(); ++iter)
  {
    if (old_schemata.find(*iter) == old_schemata.end())
    {
      SchemaNode node;
      node.name= *iter;
      node.fetched= false;
      node.fetching= false;
      _schemata.push_back(node);
    }
    else
      _schemata.push_back(old_schemata[*iter]);
  }
  
  std::sort(_schemata.begin(), _schemata.end(), compare_schema_node);
  
  _schemata_vector_serial++;  

  if (hard_refresh)
    tree_changed();
}


int LiveSchemaTree::count_children(const bec::NodeId &node)
{
  if (node.depth() == 0)
    return _schemata.size();
  else if (!_is_schema_contents_enabled)
    return 0;
  else if (node.depth() == 1)
  {
    return 3;
  }
  else if (node.depth() == 2)
  {
    SchemaNode &snode(_schemata[node[0]]);
    switch (node[1])
    {
      case 0: return snode.tables.size();
      case 1: return snode.views.size();
      case 2: return snode.routines.size();
    }
    return 0;
  }
  else if (node.depth() == 3)
  {
    if (ObjectNode *obj_node= get_object_node_by_index(node))
       return obj_node->columns.size();
  }
  return 0;
}


bool LiveSchemaTree::get_field(const bec::NodeId &node, int column, std::string &value)
{
  if (_schemata.empty())
  {
    return false;
  }
  if ((node.depth() > 0) && ((int)_schemata.size() <= node[0]))
    return false;

  if (node.depth() == 1)
  {
    value= _schemata[node[0]].name;
    if (column == 4242) // hack to return formatted column text in linux
    {
      if (value == _active_schema)
        value = "<b>"+value+"</b>";
    }
#if defined(_WIN32)
    SchemaNode &snode(_schemata[node[0]]);
    if (snode.fetching)
      value+= " - fetching...";
#endif
    return true;
  }
  else if (node.depth() == 2)
  {
    SchemaNode &snode(_schemata[node[0]]);
    if (snode.fetching)
    {
      value= _("fetching...");
      return true;
    }
    else
    {
      switch (node[1])
      {
        case 0: value= _("Tables"); return true;
        case 1: value= _("Views"); return true;
        case 2: value= _("Routines"); return true;
      }
    }
  }
  else if (node.depth() == 3)
  {
    if (ObjectNode *obj_node= get_object_node_by_index(node))
    {
      value= obj_node->name;
      return true;
    }
  }
  else if (node.depth() == 4)
  {
    if (ObjectNode *obj_node= get_object_node_by_index(node))
    {
      value= obj_node->columns[node[3]].name;
      return true;
    }
    return true;
  }
  return false;
}


std::string LiveSchemaTree::get_field_description(const bec::NodeId &node, int column)
{
  if (node.depth() == 3)
  {
    ObjectNode *obj_node= load_object_details(node);
    if (obj_node)
      return obj_node->details;
  }
  return "";
}


bec::IconId LiveSchemaTree::get_field_icon(const bec::NodeId &node, int column, bec::IconSize size)
{
  std::string suffix;
#ifdef _WIN32
  suffix = "win.$.png";
#else
  suffix = "$.png";
#endif

  if (node.depth() == 1)
  {
    return bec::IconManager::get_instance()->get_icon_id("db.Schema.side." + suffix, bec::Icon16);
  }
  else if (node.depth() == 2)
  {
    switch (node[1])
    {
      case 0: return bec::IconManager::get_instance()->get_icon_id("db.Table.many.side." + suffix, bec::Icon16);
      case 1: return bec::IconManager::get_instance()->get_icon_id("db.View.many.side." + suffix, bec::Icon16);
      case 2: return bec::IconManager::get_instance()->get_icon_id("db.Routine.many.side." + suffix, bec::Icon16);
    }
    return 0;
  }
  else if (node.depth() == 3)
  {
    switch (node[1])
    {
      case 0: return bec::IconManager::get_instance()->get_icon_id("db.Table.side." + suffix, bec::Icon16);
      case 1: return bec::IconManager::get_instance()->get_icon_id("db.View.side." + suffix, bec::Icon16);
      case 2: return bec::IconManager::get_instance()->get_icon_id("db.Routine.side." + suffix, bec::Icon16);
    }
  }
  else if (node.depth() == 4)
  {
    ObjectNode *obj_node= get_object_node_by_index(node);
    if (obj_node)
    {
      if (obj_node->columns[node[3]].is_pk)
        return bec::IconManager::get_instance()->get_icon_id("db.Column.pk.side." + suffix, bec::Icon16);
      //else if (obj_node->columns[node[3]].is_fk)
      //  return bec::IconManager::get_instance()->get_icon_id("db.Column.fk.side." + suffix, bec::Icon16);
      else
        return bec::IconManager::get_instance()->get_icon_id("db.Column.side." + suffix, bec::Icon16);
    }
  }
  return 0;
}

void LiveSchemaTree::set_active_schema(const std::string &schema)
{
  _active_schema = schema;
}

bool LiveSchemaTree::get_field(const bec::NodeId &node, int column, int &value)
{
  value = 0;
  if (node.depth() == 1)
  {
    if (_active_schema == _schemata[node[0]].name)
      value = 1;
  }
  return true;
}

bec::NodeId LiveSchemaTree::get_child(const bec::NodeId &node, int index)
{
  return bec::NodeId(node).append(index);
}


void LiveSchemaTree::update_live_object_state(ObjectType type, const std::string &schema_name, const std::string &old_obj_name, const std::string &new_obj_name)
{
  bool is_obj_created= old_obj_name.empty();
  bool is_obj_dropped= new_obj_name.empty();
  if (old_obj_name == new_obj_name)
  {
    is_obj_created= false;
    is_obj_dropped= false;
  }

  SchemaNode *snode= NULL;
  if (!schema_name.empty())
  {
    NodeId schema_node_id= get_node_for_schema(schema_name);
    snode= &_schemata[schema_node_id[0]];
  }
  ObjectNode obj_node;
  obj_node.name= new_obj_name;
  std::vector<ObjectNode> *obj_nodes= NULL;

  switch (type)
  {
    case Schema:
      {
        bool is_new_obj_found= false;
        for (int n= _schemata.size()-1; n >= 0; --n)
        {
          if (_schemata[n].name == new_obj_name)
            is_new_obj_found= true;
          if (is_obj_dropped && (_schemata[n].name == old_obj_name))
            _schemata.erase(_schemata.begin()+n);
        }
        if (is_obj_created && !is_new_obj_found)
        {
          SchemaNode new_schema_node;
          new_schema_node.name= new_obj_name;
          new_schema_node.fetched= false;
          new_schema_node.fetching= false;
          _schemata.push_back(new_schema_node);
          std::sort(_schemata.begin(), _schemata.end(), compare_schema_node);
        }
      }
      break;
    case Table:
      obj_nodes= &snode->tables;
      break;
    case View:
      obj_nodes= &snode->views;
      break;
    case Routine:
      obj_nodes= &snode->routines;
      obj_node.details= " "; // means no details
      break;
  }

  if (!obj_nodes)
    return;

  bool is_new_obj_found= false;
  for (int n= obj_nodes->size()-1; n >= 0; --n)
  {
    if ((*obj_nodes)[n].name == new_obj_name)
      is_new_obj_found= true;
    if (is_obj_dropped && ((*obj_nodes)[n].name == old_obj_name))
      obj_nodes->erase(obj_nodes->begin()+n);
    if (!is_obj_dropped && !is_obj_created)
      (*obj_nodes)[n].details= "";
  }
  if (is_obj_created && !is_new_obj_found)
  {
    obj_nodes->push_back(obj_node);
    std::sort(obj_nodes->begin(), obj_nodes->end());
  }
}


void LiveSchemaTree::schema_contents_arrived(const SchemaNode &node, int index, int serial)
{
  if (serial != _schemata_vector_serial)
  {
    for (std::vector<SchemaNode>::iterator iter= _schemata.begin(); iter != _schemata.end(); ++iter)
    {
      if (iter->name == node.name)
      {
        iter->fetched= true;
        iter->fetching= false;
        iter->tables= node.tables;
        iter->views= node.views;
        iter->routines= node.routines;
        break;
      }
    }
    // unknown schema, just ignore it
  }
  else
  {
    SchemaNode &snode(_schemata[index]);

    snode.fetched= true;
    snode.fetching= false;
    snode.tables= node.tables;
    snode.views= node.views;
    snode.routines= node.routines;
  }
}


void LiveSchemaTree::object_details_arrived(const SchemaNode &schema_node, const ObjectNode &obj_node, ObjectType obj_type, int schema_index, int obj_index, int serial)
{
  if (serial != _schemata_vector_serial)
  {
    for (std::vector<SchemaNode>::iterator iter= _schemata.begin(); iter != _schemata.end(); ++iter)
    {
      if (iter->name == schema_node.name)
      {
        std::vector<ObjectNode> * obj_nodes= get_object_nodes_by_type(&(*iter), obj_type);
        for (std::vector<ObjectNode>::iterator iter= obj_nodes->begin(); iter != obj_nodes->end(); ++iter)
        {
          if (iter->name == obj_node.name)
          {
            iter->details= obj_node.details;
            iter->columns= obj_node.columns;
            iter->fetched= true;
            iter->fetching= false;
          }
        }
        break;
      }
    }
  }
  else
  {
    SchemaNode *snode(&_schemata[schema_index]);
    std::vector<ObjectNode> * obj_nodes= get_object_nodes_by_type(snode, obj_type);
    ObjectNode *onode= &(*obj_nodes)[obj_index];
    onode->details= obj_node.details;
    onode->columns= obj_node.columns;
    onode->fetched= true;
    onode->fetching= false;
  }
}


bool LiveSchemaTree::is_expandable(const bec::NodeId &node)
{
  if (!_is_schema_contents_enabled)
  {
    return false;
  }
  else if (node.depth() < 3)
  {
    return true;
  }
  else if (node.depth() == 3)
  {
    return (node[1] < 2);
  }
  return false;
}


bool LiveSchemaTree::expand_node(const bec::NodeId &node)
{
  if (!_is_schema_contents_enabled && (node.depth() > 0))
  {
    return true;
  }
  else if (node.depth() == 0)
  {
    return true;
  }
  else if (node.depth() == 1)
  {
    SchemaNode &snode(_schemata[node[0]]);
    if (!snode.fetched && !snode.fetching)
    {
      snode.fetching= true;
      fetch_slot(node, snode.name,
        sigc::bind(sigc::mem_fun(this, &LiveSchemaTree::schema_contents_arrived),
          node[0], _schemata_vector_serial));
    }
    return true;
  }
  else if (node.depth() == 2)
  {
    return true;
  }
  else if (node.depth() == 3)
  {
    load_object_details(node);
    return true;
  }
  return false;
}


LiveSchemaTree::ObjectNode * LiveSchemaTree::load_object_details(const NodeId &index_node)
{
  ObjectType obj_type;
  ObjectNode *obj_node= get_object_node_by_index(index_node, &obj_type);
  if (!obj_node || obj_node->fetching || !obj_node->details.empty())
    return obj_node;
  obj_node->fetching= true;
  details_fetch_slot(obj_type, _schemata[index_node[0]].name, obj_node->name,
    sigc::bind(sigc::mem_fun(this, &LiveSchemaTree::object_details_arrived),
      index_node[0], index_node[2], _schemata_vector_serial));
  if (obj_node->details.empty())
    obj_node->details= " "; // means object's details are fetched
  return obj_node;
}


bec::NodeId LiveSchemaTree::get_node_for_schema(const std::string &name)
{
  int index= 0;
  for (std::vector<SchemaNode>::iterator iter= _schemata.begin(); iter != _schemata.end(); ++iter, ++index)
  {
    if (iter->name == name)
      return bec::NodeId(index);
  }
  return bec::NodeId();
}


bool LiveSchemaTree::activate_node(const bec::NodeId &node)
{
  switch (node.depth())
  {
  case 1:
    {
      activate_object_signal.emit("activate", Schema, _schemata[node[0]].name, "");      
      return true;
    }
    break;
  case 3:
    {
      ObjectType type;
      ObjectNode *obj_node= get_object_node_by_index(node, &type);
      if (!obj_node)
        return false;
      SchemaNode *snode(&_schemata[node[0]]);
      std::string schema_name= opt_quote_ident(snode->name);
      std::string obj_name= opt_quote_ident(obj_node->name);
      std::string text= strfmt("%s.%s", schema_name.c_str(), obj_name.c_str());
      sql_editor_text_insert_signal(text);
      //!activate_object_signal.emit("activate", type, snode->name, obj_node->name);
      return true;
    }
    break;
  case 4:
    {
      if (ObjectNode *obj_node= get_object_node_by_index(node))
      {
        std::string column_name= obj_node->columns[node[3]].name;
        column_name= opt_quote_ident(column_name);
        sql_editor_text_insert_signal(column_name);
        return true;
      }
    }
  }
  return false;
}


bec::MenuItemList LiveSchemaTree::get_popup_items_for_nodes(const std::vector<bec::NodeId> &nodes)
{
  bec::MenuItemList items;

  {
    bec::MenuItem item;
    item.type = MenuAction;

    bec::MenuItem edit_item;
    edit_item.caption= _("Edit Table Data");
    edit_item.name= "edit_data";
    edit_item.enabled= false;
        
    bec::MenuItem view_item;
    {
      std::string caption= _("Select Rows");
      {
        DictRef options= DictRef::cast_from(_grt->get("/wb/options/options"));
        bool limit_rows= (0 != options.get_int("SqlEditor:LimitRows"));
        int limit_rows_count= options.get_int("SqlEditor:LimitRowsCount");
        if (limit_rows && (limit_rows_count <= 0))
          limit_rows= false;
        if (limit_rows)
          caption+= strfmt(_(" - Limit %i"), limit_rows_count);
      }
      view_item.caption= caption;
    }
    view_item.name= "select_data";
    view_item.enabled= false;

    bec::MenuItem active_schema_item;
    active_schema_item.caption= _("Set as Default Schema");
    active_schema_item.name= "set_active_schema";
    active_schema_item.enabled= false;

    bec::MenuItem clipboard_copy_item;
    clipboard_copy_item.caption= _("Copy to Clipboard");
    clipboard_copy_item.name= "clipboard_copy";
    clipboard_copy_item.type= MenuCascade;
    clipboard_copy_item.enabled= false;

    bec::MenuItem editor_insert_item;
    editor_insert_item.caption= _("Send to SQL Editor");
    editor_insert_item.name= "editor_insert";
    editor_insert_item.type= MenuCascade;
    editor_insert_item.enabled= false;

    bec::MenuItem object_name_short_item;
    object_name_short_item.caption= _("Name (short)");
    object_name_short_item.name= "object_name_short";
    object_name_short_item.enabled= false;

    bec::MenuItem object_name_long_item;
    object_name_long_item.caption= _("Name (long)");
    object_name_long_item.name= "object_name_long";
    object_name_long_item.enabled= false;

    bec::MenuItem table_select_statement_item;
    table_select_statement_item.caption= _("Select All Statement");
    table_select_statement_item.name= "table_select_statement";
    table_select_statement_item.enabled= false;

    bec::MenuItem table_insert_statement_item;
    table_insert_statement_item.caption= _("Insert Statement");
    table_insert_statement_item.name= "table_insert_statement";
    table_insert_statement_item.enabled= false;

    bec::MenuItem table_update_statement_item;
    table_update_statement_item.caption= _("Update Statement");
    table_update_statement_item.name= "table_update_statement";
    table_update_statement_item.enabled= false;

    bec::MenuItem table_delete_statement_item;
    table_delete_statement_item.caption= _("Delete Statement");
    table_delete_statement_item.name= "table_delete_statement";
    table_delete_statement_item.enabled= false;

    bec::MenuItem table_create_statement_item;
    table_create_statement_item.caption= _("Create Statement");
    table_create_statement_item.name= "table_create_statement";
    table_create_statement_item.enabled= false;

    bec::MenuItem table_column_names_item;
    table_column_names_item.caption= _("Column Names");
    table_column_names_item.name= "table_column_names";
    table_column_names_item.enabled= false;

    bec::MenuItem create_item;
    create_item.caption= _("Create...");
    create_item.name= "create";
    create_item.enabled= false;

    bec::MenuItem alter_item;
    alter_item.caption= _("Alter...");
    alter_item.name= "alter";
    alter_item.enabled= false;

    bec::MenuItem drop_item;
    drop_item.caption= _("Drop...");
    drop_item.name= "drop";
    drop_item.enabled= false;

    if (nodes.size() != 1)
    {
      clipboard_copy_item.enabled= false;
      editor_insert_item.enabled= false;
      alter_item.enabled= false;
      drop_item.enabled= false;
      create_item.enabled= (nodes.size() == 0);
      if (create_item.enabled)
        create_item.caption= _("Create Schema...");
    }
    else
    {
      bec::NodeId node= nodes.empty() ? bec::NodeId() : nodes[0];
      switch (node.depth())
      {
        case 1:
          object_name_long_item.caption= _("Name");
          object_name_long_item.enabled= true;
          active_schema_item.enabled= true;
          clipboard_copy_item.enabled= true;
          editor_insert_item.enabled= true;
          create_item.enabled= true;
          create_item.caption= _("Create Schema...");
          alter_item.caption= _("Alter Schema...");
          drop_item.caption= _("Drop Schema...");
          alter_item.enabled= true;
          drop_item.enabled= true;
          table_create_statement_item.enabled= true;
          break;
        case 3:
          {
            switch (node[1])
            {
            case 0: // Table
              alter_item.caption= _("Alter Table...");
              drop_item.caption= _("Drop Table...");
              edit_item.enabled= true;
              view_item.enabled= true;
              table_column_names_item.enabled= true;
              table_select_statement_item.enabled= true;
              table_insert_statement_item.enabled= true;
              table_update_statement_item.enabled= true;
              table_delete_statement_item.enabled= true;
              table_create_statement_item.enabled= true;
              break;
            case 1: // View
              alter_item.caption= _("Alter View...");
              drop_item.caption= _("Drop View...");
              view_item.enabled= true;
              table_column_names_item.enabled= true;
              table_select_statement_item.enabled= true;
              table_insert_statement_item.enabled= true;
              table_update_statement_item.enabled= true;
              table_delete_statement_item.enabled= true;
              table_create_statement_item.enabled= true;
              break;
            case 2: // SP
              alter_item.caption= _("Alter Routine...");
              drop_item.caption= _("Drop Routine...");
              break;                  
            }
            clipboard_copy_item.enabled= true;
            editor_insert_item.enabled= true;
            object_name_short_item.enabled= true;
            object_name_long_item.enabled= true;
            alter_item.enabled= true;
            drop_item.enabled= true;
          }
          //break;
        case 2:
          {
            switch (node[1])
            {
            case 0: // Table
              create_item.caption= _("Create Table...");
              break;
            case 1: // View
              create_item.caption= _("Create View...");
              break;
            case 2: // SP
              create_item.caption= _("Create Routine...");
              break;                  
            }
            create_item.enabled= true;
          }
          break;
        default:
          break;
      }
    }
    
    if (view_item.enabled)
      items.push_back(view_item);
    if (edit_item.enabled)
      items.push_back(edit_item);
    if (active_schema_item.enabled)
    {
      items.push_back(active_schema_item);
      bec::MenuItem item;
      item.type= MenuSeparator;
      items.push_back(item);
    }
    if (clipboard_copy_item.enabled)
    {
      MenuItemList subitems;
      subitems.push_back(object_name_short_item);
      subitems.push_back(object_name_long_item);
      subitems.push_back(table_column_names_item);
      subitems.push_back(table_select_statement_item);
      subitems.push_back(table_insert_statement_item);
      subitems.push_back(table_update_statement_item);
      subitems.push_back(table_delete_statement_item);
      subitems.push_back(table_create_statement_item);
      for (MenuItemList::iterator i= subitems.begin(), end= subitems.end(); i != end; ++i)
      {
        if (!i->enabled)
          continue;
        i->name= "clipboard_" + i->name;
        clipboard_copy_item.subitems.push_back(*i);
      }
      items.push_back(clipboard_copy_item);
    }
    if (editor_insert_item.enabled)
    {
      MenuItemList subitems;
      subitems.push_back(object_name_short_item);
      subitems.push_back(object_name_long_item);
      subitems.push_back(table_column_names_item);
      subitems.push_back(table_select_statement_item);
      subitems.push_back(table_insert_statement_item);
      subitems.push_back(table_update_statement_item);
      subitems.push_back(table_delete_statement_item);
      subitems.push_back(table_create_statement_item);
      for (MenuItemList::iterator i= subitems.begin(), end= subitems.end(); i != end; ++i)
      {
        if (!i->enabled)
          continue;
        i->name= "editor_" + i->name;
        editor_insert_item.subitems.push_back(*i);
      }
      items.push_back(editor_insert_item);
    }
    if (!items.empty() && (alter_item.enabled || create_item.enabled || drop_item.enabled))
    {
      item.enabled= false;
      item.type = MenuSeparator;
      items.push_back(item);
    }
    if (alter_item.enabled)
      items.push_back(alter_item);
    if (create_item.enabled)
      items.push_back(create_item);
    if (drop_item.enabled)
      items.push_back(drop_item);
  }

  try
  {
    bec::MenuItem item;

    std::string schema;
    std::string object;
    std::string type;
    bec::NodeId node= nodes.empty() ? bec::NodeId() : nodes[0];
    switch (node.depth())
    {
      case 1:
        schema= _schemata[node[0]].name;
        object= "";
        type= "schema";
        break;
      case 3:
      {
        schema= _schemata[node[0]].name;
        switch (node[1])
        {
          case 0: // Table
            type= "table";
            object= _schemata[node[0]].tables[node[2]].name;
            break;
          case 1: // View
            type= "view";
            object= _schemata[node[0]].views[node[2]].name;
            break;
          case 2: // SP
            type= "routine";
            object= _schemata[node[0]].routines[node[2]].name;
            break;
        }
      }
    }
    
    if (!type.empty())
    {
      // plugin items
      bec::MenuItemList plugin_items(object_plugin_items_slot(schema, object, type));

      if (!plugin_items.empty())
      {
        bec::MenuItem item;
        item.type = MenuSeparator;
        items.push_back(item);    
        items.insert(items.end(), plugin_items.begin(), plugin_items.end());
      }
    }
  }
  catch (const std::exception &exc)
  {
    g_message("exception fetching context menu items for live tree: %s", exc.what());
  }
  
  {
    bec::MenuItem item;

    if (!items.empty())
    {
      item.type = MenuSeparator;
      items.push_back(item);    
    }
    item.type = MenuAction;
    item.caption= _("Refresh All");
    item.name= "refresh";
    items.push_back(item);
  }

  return items;
}


bool LiveSchemaTree::activate_popup_item_for_nodes(const std::string &name, const std::vector<bec::NodeId> &unsorted_nodes)
{
  if (name == "refresh")
  {
    refresh_signal.emit();
    return true;
  }
  else if ((name == "alter") || (name == "drop"))
  {
    LiveObjectSignal &live_object_signal= (name == "drop") ? drop_live_object_signal : alter_live_object_signal;

    bec::NodeId node= unsorted_nodes[0];
    switch (node.depth())
    {
      case 1:
        live_object_signal.emit(Schema, "", _schemata[node[0]].name);
        break;

      case 3:
        {
          ObjectType type;
          ObjectNode *obj_node= get_object_node_by_index(node, &type);
          if (!obj_node)
            return false;
          SchemaNode *snode(&_schemata[node[0]]);
          live_object_signal.emit(type, snode->name, obj_node->name);
        }
        break;
    }
    return true;
  }
  else if (name == "select_data" || name == "edit_data" || name == "copy_column_names" ||
    (0 == name.find("clipboard_")) || (0 == name.find("editor_")))
  {
    bec::NodeId node= unsorted_nodes[0];
    switch (node.depth())
    {
    case 1:
      activate_object_signal.emit(name, Schema, "", _schemata[node[0]].name);
      break;
          
    case 3:
      {
        ObjectType type;
        ObjectNode *obj_node= get_object_node_by_index(node, &type);
        if (!obj_node)
          return false;
        SchemaNode *snode(&_schemata[node[0]]);
        activate_object_signal.emit(name, type, snode->name, obj_node->name);
      }
      break;
    }
    return true;    
  }
  else if (name == "create")
  {
    if (unsorted_nodes.size() == 0)
    {
      create_live_object_signal.emit(Schema, "", "");
    }
    else
    {
      bec::NodeId node= unsorted_nodes[0];
      switch (node.depth())
      {
        case 1:
          create_live_object_signal.emit(Schema, "", "");
          break;

        case 2:
        case 3:
        {
          SchemaNode &snode(_schemata[node[0]]);
          switch (node[1])
          {
            case 0: create_live_object_signal.emit(Table, snode.name, ""); break;
            case 1: create_live_object_signal.emit(View, snode.name, ""); break;
            case 2: create_live_object_signal.emit(Routine, snode.name, ""); break;
          }
        }
        break;
      }
    }
    return true;
  }
  else if (name == "set_active_schema")
  {
    bec::NodeId node= unsorted_nodes[0];
    active_schema_change_signal.emit(_schemata[node[0]].name);
    return true;
  }

  try
  {
    bec::MenuItem item;
    
    std::string schema;
    std::string object;
    std::string type;
    bec::NodeId node= unsorted_nodes.empty() ? bec::NodeId() : unsorted_nodes[0];
    switch (node.depth())
    {
      case 1:
        schema= _schemata[node[0]].name;
        object= "";
        type= "schema";
        break;
      case 3:
      {
        schema= _schemata[node[0]].name;
        switch (node[1])
        {
          case 0: // Table
            type= "table";
            object= _schemata[node[0]].tables[node[2]].name;
            break;
          case 1: // View
            type= "view";
            object= _schemata[node[0]].views[node[2]].name;
            break;
          case 2: // SP
            type= "routine";
            object= _schemata[node[0]].routines[node[2]].name;
            break;
        }
      }
    }

    if (!type.empty())
      return call_object_plugin_items_slot(name, schema, object, type);
  }
  catch (const std::exception &exc)
  {
    g_message("exception fetching context menu items for live tree: %s", exc.what());
  }
  
  return false;
}


int LiveSchemaTree::get_index_of_schema(const std::string &schema_name) const
{
  for (int n= 0, count= _schemata.size(); n < count; ++n)
    if (schema_name == _schemata[n].name)
      return n;
  std::string upper_schema_name= toupper(schema_name);
  for (int n= 0, count= _schemata.size(); n < count; ++n)
    if (upper_schema_name == toupper(_schemata[n].name))
      return n;
  return -1;
}


bool LiveSchemaTree::is_schema_contents_enabled() const
{
  return _is_schema_contents_enabled;
}


void LiveSchemaTree::is_schema_contents_enabled(bool value)
{
  _is_schema_contents_enabled= value;
}

//--------------------------------------------------------------------------------------------------

/**
 * Returns true if the given node corresponds to the currently active schema.
 */
bool LiveSchemaTree::is_highlighted(const NodeId& node)
{
  if (node.depth() > 1)
    return false;

  if (node[0] > -1 && node[0] < (int) _schemata.size())
    if (_active_schema == _schemata[node[0]].name)
      return true;

  return false;
}

//--------------------------------------------------------------------------------------------------
