/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
#ifndef _BE_COMMON_H_
#define _BE_COMMON_H_

#include <cstring>
#include <vector>
#include <string>
#include <glib.h>
#include <sigc++/sigc++.h>

#ifndef _WIN32
#include <stdexcept>
#include <stdarg.h>
#endif

#include <string.h>

#include "wbpublic_public_interface.h"

#ifdef __GNUC__
#define UNUSED __attribute__((unused))
#else
#define UNUSED
#endif

namespace bec {

#define DATETIME_FMT "%Y-%m-%d %H:%M"
  std::string WBPUBLICBACKEND_PUBLIC_FUNC fmttime(time_t t= 0, const char *fmt= "%b %d, %Y");

  std::string WBPUBLICBACKEND_PUBLIC_FUNC replace_string(const std::string &s,
                            const std::string &from,
                            const std::string &to);
  void WBPUBLICBACKEND_PUBLIC_FUNC replace_string_inplace(std::string &text,
                            const std::string &from,
                            const std::string &to);
  std::string WBPUBLICBACKEND_PUBLIC_FUNC rtrim(const std::string &value);

  
  // replaces a variable from a string in format %variable%
  // a filter can be passed to the variable as in %variable|filter%
  // supported filters are upper, lower and capitalize
  std::string WBPUBLICBACKEND_PUBLIC_FUNC replace_variable(const std::string &format, const std::string &variable, const std::string &value);
  
  std::string WBPUBLICBACKEND_PUBLIC_FUNC append_extension_if_needed(const std::string &path,
                                                              const std::string &ext);

  std::string WBPUBLICBACKEND_PUBLIC_FUNC normalize_path(const std::string path);
  
  std::string WBPUBLICBACKEND_PUBLIC_FUNC expand_tilde(const std::string &path);

  inline bool has_prefix(const std::string &str, const std::string &pref)
  {
    if (strncmp(str.c_str(), pref.c_str(), pref.length())==0)
      return true;
    return false;
  }


  inline bool has_suffix(const std::string &str, const std::string &suf)
  {
    if (suf.length() < str.length() && strncmp(str.c_str()+str.length()-suf.length(), suf.c_str(), suf.length())==0)
      return true;
    return false;
  }


  std::string WBPUBLICBACKEND_PUBLIC_FUNC make_path(const std::string &prefix, const std::string &file);

  inline std::string pathlist_append(const std::string &l, const std::string &s)
  {
    if (l.empty())
      return s;
    return l+G_SEARCHPATH_SEPARATOR+s;
  }

  inline std::string pathlist_prepend(const std::string &l, const std::string &s)
  {
    if (l.empty())
      return s;
    return s+G_SEARCHPATH_SEPARATOR+l;
  }

  //! \brief RAII style GMutex locker/unlocker  
  class WBPUBLICBACKEND_PUBLIC_FUNC GMutexLock
  {
    GMutex *mutex;
  public:
    GMutexLock(GMutex *mtx) : mutex(mtx)
    {
      if (mutex)
        g_mutex_lock(mutex);
    }
    GMutexLock(const GMutexLock &lock)
    {
      GMutexLock *lock_= const_cast<GMutexLock*>(&lock);
      operator=(*lock_);
    }
    ~GMutexLock()
    {
      reset();
    }
    GMutex * release()
    {
      GMutex *res= mutex;
      mutex= NULL;
      return res;
    }
    void reset()
    {
      if (mutex)
      {
        g_mutex_unlock(mutex);
        mutex= NULL;
      }
    }
    GMutexLock & operator=(GMutexLock& lock)
    {
      mutex= lock.release();
      return *this;
    }
  };

  //! \brief RAII style GStaticMutex locker/unlocker  
  class WBPUBLICBACKEND_PUBLIC_FUNC GStaticMutexLock // Note: must compile with strict-aliasing disabled (-fno-strict-aliasing in gcc)
  {
      GStaticMutex    &mutex;
    public:
    GStaticMutexLock(GStaticMutex& mtx);
    ~GStaticMutexLock();
  };

  class WBPUBLICBACKEND_PUBLIC_FUNC GStaticRecMutexLock // Note: must compile with strict-aliasing disabled (-fno-strict-aliasing in gcc)
  {
    GStaticRecMutex    &mutex;
  public:
    GStaticRecMutexLock(GStaticRecMutex& mtx);
    ~GStaticRecMutexLock();
  };

  class WBPUBLICBACKEND_PUBLIC_FUNC TimerActionThread
  {
  public:
    typedef sigc::slot<void> Action;
    static TimerActionThread * create(const Action &action, gulong milliseconds);
    ~TimerActionThread();
    void stop(bool clear_exit_signal);
    sigc::signal<void> on_exit;
  private:
    GMutex *_action_mutex;
    sigc::slot<void> _action;
    gulong _microseconds;
    GThread *_thread;
    TimerActionThread(const Action &action, gulong milliseconds);
    static gpointer start(gpointer data);
    void main_loop();
  };

  class WBPUBLICBACKEND_PUBLIC_FUNC ScopeExitTrigger
  {
  public:
    typedef sigc::slot<void> Slot;
    ScopeExitTrigger() {}
    ScopeExitTrigger(const Slot &cb) : slot(cb) {}
    ~ScopeExitTrigger() { slot(); }
    ScopeExitTrigger & operator=(const Slot &cb) { slot= cb; return *this; }
    Slot slot;
  };
  
};

#endif /* _BE_COMMON_H_ */
