/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "grt_modules_tree.h"
#include <grtpp_util.h>


using namespace grt;
using namespace bec;


ModulesTreeBE::ModulesTreeBE(grt::GRT *grt)
  : _grt(grt)
{
  refresh();
}


void ModulesTreeBE::refresh()
{
  _module_count= _grt->get_modules().size();
}


int ModulesTreeBE::count_children(const NodeId &parent)
{
  if (parent == get_root())
    return _module_count;

  if (parent[0] < _module_count && get_node_depth(parent) == 1)
  {
    Module *module= _grt->get_modules()[parent[0]];
    if (module)
      return module->get_functions().size();
  }
  return 0;
}


NodeId ModulesTreeBE::get_child(const NodeId &parent, int index)
{
  if (parent == get_root() && index >= 0 && index < _module_count)
    return NodeId(index);

  if (get_node_depth(parent) == 1)
  {
    Module *module= _grt->get_modules()[parent[0]];
    if (module)
    {
      if (index >= 0 && index < (int)module->get_functions().size())
        return NodeId(parent).append(index);
    }
  }
  return NodeId();
}


bool ModulesTreeBE::get_field(const NodeId &node, int column, std::string &value)
{
  if (get_node_depth(node) == 1)
  {
    switch (column)
    {
    case Name:
      {
        Module *module= _grt->get_modules()[node[0]];
        if (module)
        {
          value= module->name();
          return true;
        }
      }
      break;
    }
  }
  else if (get_node_depth(node) == 2)
  {
    switch (column)
    {
    case Name:
      {
        Module *module= _grt->get_modules()[node[0]];
        if (module)
        {
          const Module::Function &func= module->get_functions()[node[1]];
          
          value= func.name;
        }
        return true;
      }
      break;
    }
  }
  return false;
}


std::string ModulesTreeBE::get_field_description(const NodeId &node, int column)
{
  std::string value;

  if (get_node_depth(node) == 1)
  {
    Module *module= _grt->get_modules()[node[0]];
    if (module)
    {
      std::string descr;
      
      descr.append("Module: "+module->name()+"\n");
      descr.append("Path: "+module->path()+"\n");
      descr.append("Language: "+module->get_loader()->get_loader_name()+"\n");
      descr.append("Extends: "+module->extends()+"\n");
      descr.append("Implements: ");
      for (std::vector<std::string>::const_iterator iter= module->get_interfaces().begin();
           iter != module->get_interfaces().end(); ++iter)
      {
        descr.append(*iter).append("\n");
      }
      value= descr;
    }
  }
  else if (get_node_depth(node) == 2)
  {
    Module *module= _grt->get_modules()[node[0]];
    if (module)
    {
      const Module::Function &func= module->get_functions()[node[1]];

      value= "Function: "+func.name+"\n";
      value.append("Arguments:\n");
      value.append(fmt_arg_spec_list(func.arg_types)).append("\n");

      value.append("Return: ").append(fmt_type_spec(func.ret_type));
    }
  }
  return value;
}


IconId ModulesTreeBE::get_field_icon(const NodeId &node, int column, IconSize size)
{
  if ((ModulesTreeColumns)column == Name)
  {
    if (get_node_depth(node) == 1)
      return IconManager::get_instance()->get_icon_id("grt_module.png");
    else if (get_node_depth(node) == 2)
      return IconManager::get_instance()->get_icon_id("grt_function.png");
  }
  return IconManager::get_instance()->get_icon_id("");
}


Type ModulesTreeBE::get_field_type(const NodeId &node, int column)
{
  return StringType;
}
