/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */


#include "stdafx.h"

#include "sqlide_generics.h"
#include <sqlite/execute.hpp>
#ifndef _WIN32
#include <sys/time.h>
#endif
#include <locale>


namespace sqlide
{


class IsVarTypeEqTo : public boost::static_visitor<bool>
{
public:
  template <typename T> result_type operator()(const T &v1, const T &v2) const { return true; }
  template <typename T1, typename T2> result_type operator()(const T1 &v1, const T2 &v2) const { return false; }
};
static const IsVarTypeEqTo is_var_type_eq_to;

bool is_var_null(const sqlite::Variant &value)
{
  static const sqlite::Variant null_value= sqlite::Null();
  return boost::apply_visitor(is_var_type_eq_to, value, null_value);
}

bool is_var_unknown(const sqlite::Variant &value)
{
  static const sqlite::Variant unknown_value= sqlite::Unknown();
  return boost::apply_visitor(is_var_type_eq_to, value, unknown_value);
}

bool is_var_blob(const sqlite::Variant &value)
{
  static const sqlite::Variant blob_value= sqlite::BlobRef();
  return boost::apply_visitor(is_var_type_eq_to, value, blob_value);
}


void optimize_sqlite_connection_for_speed(sqlite::connection *conn)
{
  //!sqlite::execute(*conn, "pragma locking_mode = exclusive", true);
  sqlite::execute(*conn, "pragma fsync = 0", true);
  sqlite::execute(*conn, "pragma synchronous = off", true);
  sqlite::execute(*conn, "pragma encoding = \"UTF-8\"", true);
  sqlite::execute(*conn, "pragma temp_store = memory", true);
  sqlite::execute(*conn, "pragma auto_vacuum = 0", true);
}


Sqlite_transaction_guarder::Sqlite_transaction_guarder(sqlite::connection *conn)
:
_conn(conn),
_in_trans(false)
{
  if (_conn)
  {
    sqlite::execute(*conn, "begin immediate", true);
    _in_trans= true;
  }
}


Sqlite_transaction_guarder::~Sqlite_transaction_guarder()
{
  if (!_in_trans)
    return;
  const char *action= std::uncaught_exception() ? "rollback" : "commit";
  sqlite::execute(*_conn, action, true);
}


void Sqlite_transaction_guarder::commit()
{
  sqlite::execute(*_conn, "commit", true);
  _in_trans= false;
}


void Sqlite_transaction_guarder::commit_and_start_new_transaction()
{
  commit();
  sqlite::execute(*_conn, "begin", true);
  _in_trans= true;
}


} // namespace sqlide


time_t timestamp()
{
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  return GetTickCount();
#else
  struct timeval tv;
  gettimeofday(&tv, NULL);
  return tv.tv_sec * 1000 + tv.tv_usec / 1000;
#endif
}


std::tm local_timestamp()
{
  std::time_t ltime;
  time(&ltime);
  std::tm t;
#ifdef _WIN32
  localtime_s(&t, &ltime);
#else
  localtime_r(&ltime, &t);
#endif
  return t;
}


std::string format_time(const std::tm &t, const char* format)
{
  const size_t BUFFER_SIZE= 256;
  char buf[BUFFER_SIZE];
  strftime(buf, BUFFER_SIZE, format, &t);
  return std::string(buf);
}


std::string current_time(const char* format)
{
  return format_time(local_timestamp(), format);
}
