using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;

using MySQL.Controls;
using MySQL.Grt;
using MySQL.Grt.Db;
using MySQL.Forms;

namespace MySQL.GUI.Workbench.Plugins
{
  /// <summary>
  /// Generic GRT Object Editor
  /// </summary>
  public partial class ObjectEditorPlugin : DockablePlugin
  {
    #region Member Variables

    /// <summary>
    /// Specifies if the controls on the form are currently getting initialized
    /// </summary>
    private bool initializingControls = false;

    private int refreshBlocked = 0;

    /// <summary>
    /// Used to implement the IWorkbenchDocument interface and support closing notification
    /// </summary>
    protected WorkbenchDocumentClosing onWorkbenchDocumentClosing = null;

    protected BaseEditor editorBE;

    private SplitContainer mainSplitContainer = null;
    private bool isEdictingLiveObjectCancelled = false;
    //private ProgressBar applyingChangesProgressBar = null;
    Button closeLiveEditorButton;
    Button revertLiveObjectButton;
    Button applyLiveObjectButton;
    private TextBox applyingChangesLogBox = null;
    public ObjectEditorView ContainerForm;

    #endregion

    #region SqlEditorHelper

    protected class SqlEditorHelper
    {
      private BaseEditor editorBE;
      private DBObjectEditorBE dbObjEditorBE;
      public Control parent;
      public MySQL.Grt.Db.Sql.SqlEditor sqlEditor;
      public delegate void ProcessParserLogDelegate(List<String> messages);
      private ProcessParserLogDelegate processParserLog;
      public delegate void SetSqlDelegate(string sql, bool sync);
      private SetSqlDelegate setSql;

      class Token
      {
        public int line;
        public int errTokLinePos;
        public int errTokLen;
        public string msg;
      }
      List<Token> errTokens;

      public SqlEditorHelper(GrtManager grtManager, Control parent, ProcessParserLogDelegate processParserLog)
      {
        errTokens = new List<Token>();

        sqlEditor = new MySQL.Grt.Db.Sql.SqlEditor(grtManager);
        sqlEditor.BackgroundAction = Parse;

        this.parent = parent;
        parent.Controls.Add(sqlEditor);
        sqlEditor.Dock = DockStyle.Fill;
        sqlEditor.Scrolling.ScrollBars = ScrollBars.Both;
        sqlEditor.Scrolling.HorizontalWidth = 20000;

        this.processParserLog = processParserLog;
      }

      public void SetEditorBackend(BaseEditor editorBE, SetSqlDelegate setSql)
      {
        if (this.editorBE != editorBE)
        {
          if (this.editorBE != null)
          {
            this.editorBE.Dispose();
            this.editorBE = null;
          }
          this.editorBE = editorBE;
          dbObjEditorBE = editorBE as DBObjectEditorBE;
          if (sqlEditor.BE != null)
          {
            MySQL.Grt.Db.Sql.Sql_editor sqlEditorBE = sqlEditor.BE;
            sqlEditor.BE = null;
            sqlEditorBE.Dispose();
          }
          sqlEditor.BE = editorBE.get_sql_editor();
          if (null != dbObjEditorBE)
          {
            //!dbObjEditorBE.set_sql_parser_log_cb(new DBObjectEditorBE.VoidStringListDelegate(processParserLog));
            dbObjEditorBE.set_sql_parser_log_cb(ProcessParserLog);
            dbObjEditorBE.set_sql_parser_err_cb(ProcessSyntaxError);
          }
        }

        this.setSql = setSql;
      }

      public void UnregisterCallbacks()
      {
        if (null != dbObjEditorBE)
        {
          dbObjEditorBE.set_sql_parser_log_cb(null);
          dbObjEditorBE.set_sql_parser_err_cb(null);
        }
      }

      public BaseEditor EditorBE
      {
        get { return editorBE; }
      }

      public void Parse(bool sync)
      {
        errTokens.Clear();
        if (null != sqlEditor)
        {
          sqlEditor.ResetSqlCheckState();
          if (null != setSql)
            setSql(sqlEditor.Text, sync);
        }
      }

      public int ProcessSyntaxError(int line, int errTokLinePos, int errTokLen, String msg)
      {
        Token tok = new Token();
        tok.line = line;
        tok.errTokLinePos = errTokLinePos;
        tok.errTokLen = errTokLen;
        tok.msg = msg;

        errTokens.Add(tok);

        return 0;
      }

      private void ProcessParserLog(List<String> messages)
      {
        if (null != sqlEditor)
          foreach (Token tok in errTokens)
            sqlEditor.ProcessSqlError(tok.line, tok.errTokLinePos, tok.errTokLen, tok.msg);

        if (null != processParserLog)
          processParserLog(messages);
      }
    };

    protected SqlEditorHelper sqlEd;

    #endregion

    #region Constructors

    /// <summary>
    /// Standard constructor
    /// </summary>
    protected ObjectEditorPlugin() : base()
    {
    }

    /// <summary>
    /// Overloaded constructor taking the GRT Manager and GRT object to edit
    /// </summary>
    /// <param name="GrtManager">The GRT Manager</param>
    /// <param name="GrtObject">The object to edit</param>
    public ObjectEditorPlugin(GrtManager GrtManager, GrtValue GrtList)
      : base(GrtManager, GrtList)
    {
      CreateHandle();
    }

    public ObjectEditorPlugin(BaseEditor editor)
      : base(editor.get_grt_manager(), editor.get_object())
    {
      CreateHandle();
      editorBE = editor;
    }

    #endregion

    #region Properties

    protected bool InitializingControls
    {
      get { return initializingControls; }
      set { initializingControls = value; }
    }

    public bool InsideInitializationOrRefresh()
    {
      return InitializingControls || insideRefreshFormData;
    }

    public new Control ActiveControl
    {
      get
      {
        if (null != ContainerForm)
          return ContainerForm.ActiveControl;
        else
          return ActiveControl;
      }
      set
      {
        if (null != ContainerForm)
          ContainerForm.ActiveControl = value;
        else
          ActiveControl = value;
      }
    }

    #endregion

    #region IWorkbenchDocument Interface

    public override WorkbenchDocumentClosing OnWorkbenchDocumentClosing
    {
      get { return onWorkbenchDocumentClosing; }
      set { onWorkbenchDocumentClosing = value; }
    }

    public override UIForm BackendClass 
    {
      get { return editorBE; } 
    }

    public override void RefreshGUI(RefreshType refresh, String str, IntPtr ptr)
    {
    }

    public override void PerformCommand(String command)
    {
    }

    public override DockablePlugin FindPluginOfType(Type type)
    {
      if (GetType() == type)
        return this;
      return null;
    }

    public override bool ClosePluginOfType(Type type)
    {
      if (GetType() == type)
      {
        Close();
        return true;
      }
      return false;
    }

    #endregion

    #region Form Implementation

    public bool IsEditingLiveObject
    {
      get { return (null == editorBE) ? false : editorBE.is_editing_live_object(); }
    }

    public void ApplyChangesToLiveObject()
    {
      editorBE.set_live_object_change_error_cb(ProcessLiveObjectChangeError);
      editorBE.set_live_object_change_progress_cb(ProcessLiveObjectChangeProgress);
      editorBE.set_live_object_change_statistics_cb(ProcessLiveObjectChangeStatistics);

      editorBE.apply_changes_to_live_object();
    }

    private void AdjustEditModeControls(Control mainTabControl)
    {
      if (null == mainTabControl)
        return;

      bool isEditingLiveObject = IsEditingLiveObject;
      bool isEditingLiveObjectUI = ((null != mainSplitContainer) && (mainTabControl.Parent == mainSplitContainer.Panel1));
      if (isEditingLiveObject == isEditingLiveObjectUI)
        return;

      Control mainContainer = (isEditingLiveObjectUI) ? mainSplitContainer.Parent : mainTabControl.Parent;
      mainContainer.SuspendLayout();

      try
      {
        if (isEditingLiveObject)
        {
          if (null == mainSplitContainer)
          {
            mainSplitContainer = new SplitContainer();
            mainSplitContainer.Dock = DockStyle.Fill;
            mainSplitContainer.Orientation = Orientation.Horizontal;
            mainSplitContainer.SplitterWidth = 2;
            mainSplitContainer.FixedPanel = FixedPanel.Panel2;

            Panel liveObjectControlsPanel = new Panel();
            liveObjectControlsPanel.Parent = mainSplitContainer.Panel2;
            liveObjectControlsPanel.Dock = DockStyle.Fill;
            {
              closeLiveEditorButton = new Button();
              closeLiveEditorButton.UseVisualStyleBackColor = true;
              closeLiveEditorButton.Parent = liveObjectControlsPanel;
              closeLiveEditorButton.Text = "Close";
              closeLiveEditorButton.Location = new Point(
                mainSplitContainer.Panel2.ClientSize.Width - closeLiveEditorButton.Width - closeLiveEditorButton.Margin.Right,
                closeLiveEditorButton.Margin.Top);
              closeLiveEditorButton.Click += new EventHandler(closeLiveObjectEditorButton_Click);
            }
            {
              revertLiveObjectButton = new Button();
              revertLiveObjectButton.UseVisualStyleBackColor = true;
              revertLiveObjectButton.Parent = liveObjectControlsPanel;
              revertLiveObjectButton.Text = "Revert";
              revertLiveObjectButton.Location = new Point(
                closeLiveEditorButton.Location.X - closeLiveEditorButton.Margin.Left - revertLiveObjectButton.Width - revertLiveObjectButton.Margin.Right,
                closeLiveEditorButton.Location.Y);
              revertLiveObjectButton.Click += new EventHandler(revertChangesToLiveObjectButton_Click);
            }
            {
              applyLiveObjectButton = new Button();
              applyLiveObjectButton.UseVisualStyleBackColor = true;
              applyLiveObjectButton.Parent = liveObjectControlsPanel;
              applyLiveObjectButton.Text = "Apply";
              applyLiveObjectButton.Location = new Point(
                revertLiveObjectButton.Location.X - revertLiveObjectButton.Margin.Left - applyLiveObjectButton.Width - applyLiveObjectButton.Margin.Right,
                revertLiveObjectButton.Location.Y);
              applyLiveObjectButton.Click += new EventHandler(applyChangesToLiveObjectButton_Click);
            }
            
            /*
            applyingChangesProgressBar = new ProgressBar();
            {
              applyingChangesProgressBar.Parent = liveObjectControlsPanel;
              applyingChangesProgressBar.Location = new POINT(
                revertLiveObjectButton.Location.X - revertLiveObjectButton.Margin.Left - applyingChangesProgressBar.Width - applyingChangesProgressBar.Margin.Right,
                closeLiveEditorButton.Location.Y);
            }
            */
            mainSplitContainer.Panel2MinSize = closeLiveEditorButton.Height + closeLiveEditorButton.Margin.Vertical;
            mainSplitContainer.SplitterDistance = mainContainer.ClientSize.Height - mainSplitContainer.Panel2MinSize - mainSplitContainer.SplitterWidth;

            closeLiveEditorButton.Anchor = revertLiveObjectButton.Anchor = applyLiveObjectButton.Anchor = /*applyingChangesProgressBar.Anchor =*/
              AnchorStyles.Right | AnchorStyles.Bottom;

            applyingChangesLogBox = new TextBox();
            {
              applyingChangesLogBox.Parent = liveObjectControlsPanel;
              applyingChangesLogBox.ScrollBars = ScrollBars.Vertical;
              applyingChangesLogBox.Multiline = true;
              applyingChangesLogBox.Location = new Point(
                applyingChangesLogBox.Margin.Left,
                closeLiveEditorButton.Location.Y);
              applyingChangesLogBox.Text = "DBMS feedback messages will go here upon applying changes.";
            }
          }

          mainSplitContainer.Parent = mainTabControl.Parent;
          mainTabControl.Parent = mainSplitContainer.Panel1;

          //applyingChangesLogBox.Width = applyingChangesProgressBar.Location.X - applyingChangesProgressBar.Margin.Left - applyingChangesLogBox.Margin.Horizontal;
          applyingChangesLogBox.Width = applyLiveObjectButton.Location.X - applyLiveObjectButton.Margin.Left - applyLiveObjectButton.Margin.Horizontal;
          applyingChangesLogBox.Height = mainSplitContainer.Panel2.Height - applyingChangesLogBox.Margin.Vertical;
          applyingChangesLogBox.Anchor = AnchorStyles.Bottom | AnchorStyles.Left | AnchorStyles.Right | AnchorStyles.Top;
        }
        else
        {
          mainSplitContainer.Parent = null;
          mainTabControl.Parent = mainContainer;
        }
      }
      finally
      {
        mainContainer.ResumeLayout();
      }
    }

    private void applyChangesToLiveObjectButton_Click(object sender, EventArgs e)
    {
      if (null != sqlEd)
        sqlEd.sqlEditor.RunBackgroundAction(true);

      applyingChangesLogBox.ForeColor = SystemColors.WindowText;
      applyingChangesLogBox.Text = "";
      //applyingChangesProgressBar.Value = applyingChangesProgressBar.Minimum;

      ApplyChangesToLiveObject();
    }

    private void revertChangesToLiveObjectButton_Click(object sender, EventArgs e)
    {
      if (null != sqlEd)
        sqlEd.sqlEditor.IsDirty = false;

      editorBE.refresh_live_object();
    }

    private void closeLiveObjectEditorButton_Click(object sender, EventArgs e)
    {
      if (sqlEd != null)
        sqlEd.sqlEditor.RunBackgroundAction(true);
      Close(false);
    }

    private int ProcessLiveObjectChangeError(Int64 errCode, String errMsg, String sql)
    {
      String errCodeMsg = (errCode == -1) ? "" : String.Format("SQL Error {0}: ", errCode);
      String msg = String.Format("{0}{1}\r\nSQL: {2}\r\n\r\n", errCodeMsg, errMsg, sql.Trim());
      applyingChangesLogBox.ForeColor = Color.Red;
      applyingChangesLogBox.Text += msg;
      return 0;
    }

    private int ProcessLiveObjectChangeProgress(float progress)
    {
      //applyingChangesProgressBar.Value =
        //applyingChangesProgressBar.Minimum + (int)(progress * (applyingChangesProgressBar.Maximum - applyingChangesProgressBar.Minimum));
      return 0;
    }

    private int ProcessLiveObjectChangeStatistics(int successCount, int failCount)
    {
      //applyingChangesProgressBar.Value = applyingChangesProgressBar.Maximum;
      if (0 == failCount)
        applyingChangesLogBox.Text = "Changes have been applied OK";
      return 0;
    }

    public override string TabText
    {
      set
      {
        base.TabText = value;
        if (null != ContainerForm)
          ContainerForm.TabText = value;
      }
    }

    public bool ShouldCloseOnObjectDelete(String oid)
    {
      return editorBE.should_close_on_delete_of(oid);
    }

    public virtual bool ChangeGrtList(GrtManager GrtManager, GrtValue GrtList)
    {
      return false;
    }

    public virtual void Close(bool force)
    {
      if (null != ContainerForm)
        ContainerForm.Close(force);
      else
        Close();
    }

    protected override void OnFormClosing(FormClosingEventArgs e)
    {
      base.OnFormClosing(e);

      if (e.Cancel)
        return;

      if (null != sqlEd)
        sqlEd.sqlEditor.RunBackgroundAction(true);

      if (IsEditingLiveObject && !isEdictingLiveObjectCancelled)
      {
        bool can_close = editorBE.can_close();
        if (!can_close)
        {
          e.Cancel = true;
          return;
        }
      }

      if (onWorkbenchDocumentClosing != null)
        onWorkbenchDocumentClosing(this, e);

      if (editorBE != null)
        editorBE.disable_auto_refresh();
    }

    protected void BlockUIRefreshes()
    {
      refreshBlocked++;
    }

    protected void UnblockUIRefreshes()
    {
      refreshBlocked--;
      if (refreshBlocked == 0)
        CallRefreshFormData();
      else if (refreshBlocked < 0)
        throw new Exception("too many UnblockUIRefreshes");
    }

    protected bool insideRefreshFormData= false;
    protected delegate void RefreshFormDataCallback();
    protected virtual void RefreshFormData() {}

    protected bool insideRefreshPartialFormData= false;
    protected delegate void RefreshPartialFormDataCallback(int where);
    protected virtual void RefreshPartialFormData(int where) { }

    protected virtual void CallRefreshFormData() 
    {
      if (insideRefreshFormData || refreshBlocked > 0)
        return;

      insideRefreshFormData = true;
      try
      {
        Control[] controls = Controls.Find("mainTabControl", true);
        if (controls.Length > 0)
        {
          FlatTabControl mainTabControl = controls[0] as FlatTabControl;
          AdjustEditModeControls(mainTabControl);
        }

        RefreshFormData();
      }
      finally 
      {
        insideRefreshFormData = false;
      }
    }

    protected virtual void CallRefreshPartialFormData(int where) 
    {
      if (insideRefreshPartialFormData || refreshBlocked > 0)
        return;

      insideRefreshPartialFormData = true;
      try
      {
        RefreshPartialFormData(where);
      }
      finally 
      {
        insideRefreshPartialFormData = false;
      }
    }

    // this method might be called from non-UI thread
    protected void RefreshFormDataInvoke() 
    {
      if (InvokeRequired)
        this.Invoke(new RefreshFormDataCallback(CallRefreshFormData));
      else
        CallRefreshFormData();
    }

    protected void RefreshPartialFormDataInvoke(int arg)
    {
      if (InvokeRequired)
        this.Invoke(new RefreshPartialFormDataCallback(CallRefreshPartialFormData), new Object[] { arg });
      else
        CallRefreshPartialFormData(arg);
    }
    #endregion
  }
}