/* 
 * Copyright (c) 2008, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
#ifndef __MFORMS_SIMPLEGRIDVIEW_H__
#define __MFORMS_SIMPLEGRIDVIEW_H__

#include <mforms/view.h>
#include <sigc++/sigc++.h>
#include <string>
#include <vector>
#include <map>

namespace mforms
{
enum CellsAccess {CellsReadable = 0, CellsEditable = 1};

// Actual GridPath::Impl will be defined in appropriate wf_gridview.h, lf_gridview.h, MFGridView.h
class SimpleGridPath
{
  struct Impl;
};

struct SimpleGridImplPtrs;

//==============================================================================
//
//==============================================================================
class SimpleGrid : public mforms::View
{
  public:
    SimpleGrid();

    typedef SimpleGridPath  Path;
    typedef std::string     Value;

    // signals to notify customers about data change
    typedef sigc::signal<void, const Path&, const int>     SignalCellValueChanged;// (rowid, column)
    typedef sigc::slot<bool, SimpleGrid*, const Path&>     ProcessRowSlot;

    typedef sigc::slot<bool, SimpleGrid*, const Value&>              ValidationSlot;
    typedef sigc::slot<bool, SimpleGrid*, const Value&>              FindSlot;
    typedef sigc::slot<bool, SimpleGrid*, const Path&, const int>    ValidationHintSlot; // (self*, rowid, col_id)

    // The add_* functions hints implementation if any column optimization can be
    // used. add_column adds column which cells can store 'any' value.
    // Specialized functions add_text_column and add_check_column hint that the column
    // will only contain specified types, text and booleans respectively
    // CellsAccess == {CellsReadable = 0, CellsEditable = 1};
    int       add_column(const std::string& name, mforms::CellsAccess access);
    int  add_text_column(const std::string& name, mforms::CellsAccess access);
    int add_check_column(const std::string& name, mforms::CellsAccess access);

    // Returns Id of the added group
    bool add_rows_group(const std::string& group_name);
    // Attempts to insert empty row at the given position @pos in the group @gid
    // Indexing starts from 0. Special value of @pos == -1 means append
    Path insert_row(const std::string& group_name, const int pos);
    std::string get_group_for_row(const Path& pid);

    // get_old_value should return the value cell had before edit
    Value get_old_value(const Path& rid, const int col_id);
    Value     get_value(const Path& rid, const int col_id);
    bool      set_value(const Path& rid, const int col_id, const Value& cv);

    // set_enum_def passes list of items to display in combobox when edit is started
    // on cell
    bool set_enum_def(const Path& rid, const int col_id, const std::vector<std::string>& list);

    // set_validator sets callable slot, which verifies cell's value and returns true of false
    bool set_validator(const Path& rid, const int col_id, ValidationSlot vs = ValidationSlot());
    // set_validator_hinter sets callable slot, which notifies user about correct value or value range
    // exact way of notifying is tbd
    bool set_validator_hinter(const Path& rid, const int col_id, ValidationHintSlot vs = ValidationHintSlot());

    // filter_rows filters rows according to criteria defined in callable slot. If slot returns true
    // the row must be visible. Order of rows within groups must be preserved, the group header
    // also must stay.
    void filter_rows(ProcessRowSlot slot);
    void reset_filter();

    // Function to do lookups within a column and all-columns.
    Path find_in_column(const Path& start, const int col_id, const Value& v);
    Path find(const Path& start, FindSlot fslot);
    // Highlighting and unhighlighting rows(col_id == -1) or single cells
    void highlight(const Path& path, const int col_id = -1);
    void unhighlight(const Path& path, const int col_id = -1);

    SignalCellValueChanged& signal_cell_value_changed()
    {
      return _signal_cell_changed;
    }
  private:
    SignalCellValueChanged    _signal_cell_changed;
    SimpleGridImplPtrs  *_impl;
};

//==============================================================================
//
//==============================================================================
struct MFORMS_EXPORT SimpleGridImplPtrs
{
  bool (__stdcall *create)(SimpleGrid*);
  int (__stdcall *add_column)(SimpleGrid*, const std::string&, mforms::CellsAccess);
  int (__stdcall *add_text_column)(SimpleGrid*, const std::string&, mforms::CellsAccess);
  int (__stdcall *add_check_column)(SimpleGrid*, const std::string&, mforms::CellsAccess);
  bool (__stdcall *add_rows_group)(SimpleGrid*, const std::string& group_name);
  SimpleGrid::Path (__stdcall *insert_row)(SimpleGrid*, const std::string&, const int);
  std::string (__stdcall *get_group_for_row)(SimpleGrid*, const SimpleGrid::Path&);
  SimpleGrid::Value (__stdcall *get_old_value)(SimpleGrid*, const SimpleGrid::Path&, const int);
  SimpleGrid::Value (__stdcall *get_value)(SimpleGrid*, const SimpleGrid::Path&, const int);
  bool (__stdcall *set_value)(SimpleGrid*, const SimpleGrid::Path&, const int, const SimpleGrid::Value&);
  bool (__stdcall *set_enum_def)(SimpleGrid*, const SimpleGrid::Path&, const int, const std::vector<std::string>&);
  bool (__stdcall *set_validator)(SimpleGrid*, const SimpleGrid::Path&, const int, SimpleGrid::ValidationSlot);
  bool (__stdcall *set_validator_hinter)(SimpleGrid*, const SimpleGrid::Path&, const int, SimpleGrid::ValidationHintSlot);
  void (__stdcall *filter_rows)(SimpleGrid*, SimpleGrid::ProcessRowSlot);
  void (__stdcall *reset_filter)(SimpleGrid*);
  SimpleGrid::Path (__stdcall *find_in_column)(SimpleGrid*, const SimpleGrid::Path&, const int, const SimpleGrid::Value&);
  SimpleGrid::Path (__stdcall *find)(SimpleGrid*, const SimpleGrid::Path&, SimpleGrid::FindSlot);
  void (__stdcall *highlight)(SimpleGrid*, const SimpleGrid::Path&, const int);
  void (__stdcall *unhighlight)(SimpleGrid*, const SimpleGrid::Path&, const int);
};

}

#endif
