/* 
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "wf_toolbar.h"
#include "wf_view.h"

using namespace System::Windows::Forms;
using namespace System::IO;
using namespace System::Drawing;

using namespace MySQL::Forms;
using namespace MySQL::Controls;

//----------------- ToolBarItemImpl ----------------------------------------------------------------

ToolBarItemImpl::ToolBarItemImpl(mforms::ToolBarItem* item)
  : ObjectImpl(item)
{
}

//--------------------------------------------------------------------------------------------------

void ToolBarItemImpl::set_normal_icon(const std::string& path)
{
  _normal_icon = CppStringToNative(path);
  if (File::Exists(_normal_icon))
  {
    ToolStripItem^ item = get_control<ToolStripItem>();
    if (item != nullptr)
      item->Image = Image::FromFile(_normal_icon);
  }
}

//--------------------------------------------------------------------------------------------------

void ToolBarItemImpl::set_active_icon(const std::string& path)
{
  _active_icon = CppStringToNative(path);
}

//--------------------------------------------------------------------------------------------------

void ToolBarItemImpl::set_item_checked(bool state)
{
  ToolStripButton^ button = get_control<ToolStripButton>();
  if (button != nullptr && button->Checked != state)
  {
    button->Checked = state;
    if (state && File::Exists(_active_icon))
      button->Image = Image::FromFile(_active_icon);
    else
      if (File::Exists(_normal_icon))
        button->Image = Image::FromFile(_normal_icon);
  }
}

//--------------------------------------------------------------------------------------------------

void ToolBarItemImpl::OnItemActivation(Object^ sender, System::EventArgs^ e)
{
  ToolStripItem^ native_item = (ToolStripItem^) sender;

  if (native_item->Tag != nullptr)
  {
    mforms::ToolBarItem* item = ViewImpl::get_backend_control<mforms::ToolBarItem>(native_item);
    if (item != NULL)
      item->callback();
  }
}

//--------------------------------------------------------------------------------------------------

void ToolBarItemImpl::OnColorItemActivation(Object^ sender, System::EventArgs^ e)
{
  ToolStripButton^ button = dynamic_cast<ToolStripButton^>(sender);
  ToolStripDropDownButton^ dropdown = dynamic_cast<ToolStripDropDownButton^>(button->Tag);
  dropdown->Image = button->Image; // Show what was selected in the button too.
  dropdown->Text = button->Text;

  if (dropdown->Tag != nullptr)
  {
    mforms::ToolBarItem* item = ViewImpl::get_backend_control<mforms::ToolBarItem>(dropdown);
    if (item != NULL)
      item->callback();
  }
}

//----------------- ToolBarImpl --------------------------------------------------------------------

ToolBarImpl::ToolBarImpl(mforms::ToolBar* toolbar)
  : ViewImpl(toolbar)
{

}

//--------------------------------------------------------------------------------------------------

ToolBarImpl::~ToolBarImpl()
{

}

//--------------------------------------------------------------------------------------------------

bool ToolBarImpl::create_tool_bar(mforms::ToolBar* item, mforms::ToolBarType type)
{
  ToolBarImpl^ toolbar = gcnew ToolBarImpl(item);

  ToolStrip^ native_toolbar = nullptr;
  switch (type)
  {
    case mforms::MainToolBar:
      native_toolbar = ViewImpl::create<ToolStrip>(item, toolbar);
      native_toolbar->Name = "ToolBar";
      native_toolbar->Renderer = gcnew FlatMainToolStripRenderer();
      native_toolbar->GripMargin = Padding(4, 6, 4, 6);
      native_toolbar->Margin = Padding(4, 8, 2, 8);
      break;

    case mforms::OptionsToolBar:
      native_toolbar = ViewImpl::create<ToolStrip>(item, toolbar);
      native_toolbar->Name = "ToolBar";
      native_toolbar->GripMargin = Padding(4, 6, 4, 6);
      native_toolbar->Margin = Padding(4, 8, 2, 8);
      native_toolbar->RenderMode = ToolStripRenderMode::ManagerRenderMode;
      break;

    case mforms::ToolPickerToolBar:
      native_toolbar = ViewImpl::create<ToolStrip>(item, toolbar);
      native_toolbar->Name = "ToolBar";
      native_toolbar->BackColor = SystemColors::ButtonFace;
      native_toolbar->GripStyle = ToolStripGripStyle::Hidden;
      native_toolbar->ImageScalingSize = Drawing::Size(24, 22);
      native_toolbar->LayoutStyle = ToolStripLayoutStyle::VerticalStackWithOverflow;
      native_toolbar->Padding = Padding(2, 0, 0, 0);
      native_toolbar->RenderMode = ToolStripRenderMode::ManagerRenderMode;
      break;
  }

  if (native_toolbar != nullptr)
    native_toolbar->AutoSize = true;
  return (native_toolbar != nullptr);
}

//--------------------------------------------------------------------------------------------------

void ToolBarImpl::insert_item(mforms::ToolBar* toolbar, int index, mforms::ToolBarItem* item)
{
  ToolBarImpl^ toolbar_impl = (ToolBarImpl^) ViewImpl::FromUnmanaged(toolbar);
  ToolBarItemImpl^ toolbar_item = (ToolBarItemImpl^) ViewImpl::FromUnmanaged(item);

  if (toolbar_impl != nullptr && toolbar_item != nullptr)
  {
    ToolStrip^ native_toolbar = toolbar_impl->get_control<ToolStrip>();
    ToolStripItem^ native_item = toolbar_item->get_control<ToolStripItem>();
    if (index < 0 || index >= native_toolbar->Items->Count)
    {
      index = native_toolbar->Items->Count;
      native_toolbar->Items->Add(native_item);
    }
    else
      native_toolbar->Items->Insert(index, native_item);

    // To simulate expander items we align all entries after such an entry on the right hand side.
    bool hasExpander = false;
    for (int i = index - 1; i >= 0; i--)
      if (native_toolbar->Items[i]->GetType() == ExpandableToolStripSeparator::typeid)
      {
        hasExpander = true;
        break;
      }

    // If there is already an expander left to the new entry then we don't need to handle the case
    // that the new entry is itself an expander.
    if (hasExpander)
      native_item->Alignment = ToolStripItemAlignment::Right;
    if (!hasExpander && item->get_type() == mforms::ExpanderItem)
    {
      // This is the first expander so right-align all following entries up to the next expander.
      for (int i = index + 1; i < native_toolbar->Items->Count; i++)
      {
        native_toolbar->Items[i]->Alignment = ToolStripItemAlignment::Right;
        if (native_toolbar->Items[i]->GetType() == ExpandableToolStripSeparator::typeid)
          break;
      }
    }
  }
}

//--------------------------------------------------------------------------------------------------

void ToolBarImpl::remove_item(mforms::ToolBar* toolbar, mforms::ToolBarItem* item)
{
  ToolBarImpl^ toolbar_impl = (ToolBarImpl^) ViewImpl::FromUnmanaged(toolbar);
  ToolBarItemImpl^ toolbar_item = (ToolBarItemImpl^) ViewImpl::FromUnmanaged(item);

  if (toolbar_impl != nullptr && toolbar_item != nullptr)
  {
    ToolStrip^ native_toolbar = toolbar_impl->get_control<ToolStrip>();
    ToolStripItem^ native_item = toolbar_item->get_control<ToolStripItem>();
    int index = native_toolbar->Items->IndexOf(native_item);

    // Early out if the given item is not in this toolbar.
    if (index < 0)
      return;

    // Check if we have to reset the right alignment of certain controls.
    // Is there an expander item left to what we remove?
    bool hasExpander = false;
    for (int i = index - 1; i >= 0; i--)
      if (native_toolbar->Items[i]->GetType() == ExpandableToolStripSeparator::typeid)
      {
        hasExpander = true;
        break;
      }

      // If there is already an expander left to the new entry then we don't need to handle the case
      // that the entry to remove is itself an expander.
      if (!hasExpander && item->get_type() == mforms::ExpanderItem)
      {
        // This is the first expander so set all following items to left align until the next expander.
        for (int i = index + 1; i < native_toolbar->Items->Count; i++)
        {
          native_toolbar->Items[i]->Alignment = ToolStripItemAlignment::Left;
          if (native_toolbar->Items[i]->GetType() == ExpandableToolStripSeparator::typeid)
            break;
        }
      }

    native_toolbar->Items->Remove(native_item);
  }
}

//--------------------------------------------------------------------------------------------------

bool ToolBarImpl::create_tool_item(mforms::ToolBarItem* item, mforms::ToolBarItemType type)
{
  ToolBarItemImpl^ toolbar_item = gcnew ToolBarItemImpl(item);

  ToolStripItem^ native_item;
  switch (type)
  {
  case mforms::LabelItem:
    native_item = ViewImpl::create<ToolStripLabel>(item, toolbar_item, native_item);
    native_item->ForeColor = Color::Black;
    break;

  case mforms::ToggleItem:
    native_item = ViewImpl::create<ToolStripButton>(item, toolbar_item, native_item);
    native_item->DisplayStyle = ToolStripItemDisplayStyle::Image;
    native_item->Click += gcnew System::EventHandler(toolbar_item, &ToolBarItemImpl::OnItemActivation); 
    break;

  case mforms::SearchFieldItem:
    {
      ToolStripTextBox^ box = ViewImpl::create<ToolStripTextBox>(item, toolbar_item, native_item);
      box->Enabled = true;
      box->AutoSize = false;
      box->Size = Size(180, 20);
      box->TextBox->MaximumSize = Size(180, 20);
      //box->KeyPress += new KeyPressEventHandler(tb_KeyPress);

      native_item = box;
      break;
    }

  case mforms::SelectorItem:
    {
      ToolStripComboBox^ selector = ViewImpl::create<ToolStripComboBox>(item, toolbar_item, native_item);
      selector->DropDownStyle = ComboBoxStyle::DropDownList;
      selector->SelectedIndexChanged += gcnew EventHandler(toolbar_item, &ToolBarItemImpl::OnItemActivation);
      native_item = selector;
      break;
    }

  case mforms::ColorSelectorItem:
    {
      ToolStripDropDownButton^ selector = ViewImpl::create<ToolStripDropDownButton>(item, toolbar_item, native_item);
      ToolStripDropDown^ dropdown = gcnew ToolStripDropDown();
      selector->DropDown = dropdown;
      selector->DisplayStyle = ToolStripItemDisplayStyle::Image;
      selector->ShowDropDownArrow = true;
      selector->DropDownDirection = ToolStripDropDownDirection::BelowRight;
      selector->AutoSize = false;
      selector->Size = Size(75, 21);
      native_item = selector;

      break;
    }

  case mforms::SeparatorItem:
    native_item = ViewImpl::create<ToolStripSeparator>(item, toolbar_item, native_item);
    break;

  case mforms::ExpanderItem:
    native_item = ViewImpl::create<ExpandableToolStripSeparator>(item, toolbar_item, native_item);
    break;

  default: // mforms::ActionItem
    native_item = ViewImpl::create<ToolStripButton>(item, toolbar_item, native_item);
    native_item->Click += gcnew System::EventHandler(toolbar_item, &ToolBarItemImpl::OnItemActivation); 
  }

  native_item->ImageScaling = ToolStripItemImageScaling::None;

  return true;
}

//--------------------------------------------------------------------------------------------------

void ToolBarImpl::set_item_icon(mforms::ToolBarItem* item, const std::string& path)
{
  ToolBarItemImpl^ toolbar_item = (ToolBarItemImpl^) ViewImpl::FromUnmanaged(item);
  if (toolbar_item != nullptr)
    toolbar_item->set_normal_icon(path);
  // TODO: set the icon depending on the item state.
}

//--------------------------------------------------------------------------------------------------

void ToolBarImpl::set_item_alt_icon(mforms::ToolBarItem* item, const std::string& path)
{
  ToolBarItemImpl^ toolbar_item = (ToolBarItemImpl^) ViewImpl::FromUnmanaged(item);
  if (toolbar_item != nullptr)
    toolbar_item->set_active_icon(path);
  // TODO: set the icon depending on the item state.
}

//--------------------------------------------------------------------------------------------------

void ToolBarImpl::set_item_text(mforms::ToolBarItem* item, const std::string& text)
{
  ToolBarItemImpl^ toolbar_item = (ToolBarItemImpl^) ViewImpl::FromUnmanaged(item);
  if (toolbar_item != nullptr)
  {
    ToolStripItem^ native_item = toolbar_item->get_control<ToolStripItem>();
    native_item->Text = CppStringToNative(text);
    if (item->get_type() == mforms::ColorSelectorItem)
    {
      // If this item is a color selector then find the entry with the corresponding color
      // and use its image for the selector.
      ToolStripDropDownButton^ selector = dynamic_cast<ToolStripDropDownButton^>(native_item);
      bool found = false;
      for each (ToolStripButton^ button in selector->DropDownItems)
      {
        if (button->Text == native_item->Text)
        {
          native_item->Image = button->Image;
          found = true;
          break;
        }
      }

      if (!found)
        native_item->Image = create_color_image(native_item->Text);
    }
  }
}

//--------------------------------------------------------------------------------------------------

std::string ToolBarImpl::get_item_text(mforms::ToolBarItem* item)
{
  ToolBarItemImpl^ toolbar_item = (ToolBarItemImpl^) ViewImpl::FromUnmanaged(item);
  if (toolbar_item != nullptr)
    return NativeToCppString(toolbar_item->get_control<ToolStripItem>()->Text);

  return "";
}

//--------------------------------------------------------------------------------------------------

void ToolBarImpl::set_item_enabled(mforms::ToolBarItem* item, bool state)
{
  ToolBarItemImpl^ toolbar_item = (ToolBarItemImpl^) ViewImpl::FromUnmanaged(item);
  if (toolbar_item != nullptr)
    toolbar_item->get_control<ToolStripItem>()->Enabled = state;
}

//--------------------------------------------------------------------------------------------------

bool ToolBarImpl::get_item_enabled(mforms::ToolBarItem* item)
{
  ToolBarItemImpl^ toolbar_item = (ToolBarItemImpl^) ViewImpl::FromUnmanaged(item);
  if (toolbar_item != nullptr)
    return toolbar_item->get_control<ToolStripItem>()->Enabled;

  return false;
}

//--------------------------------------------------------------------------------------------------

void ToolBarImpl::set_item_checked(mforms::ToolBarItem* item, bool state)
{
  ToolBarItemImpl^ toolbar_item = (ToolBarItemImpl^) ViewImpl::FromUnmanaged(item);
  if (toolbar_item != nullptr)
    toolbar_item->set_item_checked(state);
}

//--------------------------------------------------------------------------------------------------

bool ToolBarImpl::get_item_checked(mforms::ToolBarItem* item)
{
  ToolBarItemImpl^ toolbar_item = (ToolBarItemImpl^) ViewImpl::FromUnmanaged(item);
  if (toolbar_item != nullptr)
  {
    ToolStripButton^ button = toolbar_item->get_control<ToolStripButton>();
    if (button != nullptr)
      return button->Checked;
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

void ToolBarImpl::set_item_tooltip(mforms::ToolBarItem* item, const std::string& text)
{
  ToolBarItemImpl^ toolbar_item = (ToolBarItemImpl^) ViewImpl::FromUnmanaged(item);
  if (toolbar_item != nullptr)
    toolbar_item->get_control<ToolStripItem>()->ToolTipText = CppStringToNative(text);
}

//--------------------------------------------------------------------------------------------------

void ToolBarImpl::set_selector_items(mforms::ToolBarItem* item, const std::vector<std::string>& values)
{
  ToolBarItemImpl^ toolbar_item = (ToolBarItemImpl^) ViewImpl::FromUnmanaged(item);
  if (toolbar_item != nullptr)
  {
    switch (item->get_type())
    {
    case mforms::SelectorItem:
      {
        ToolStripComboBox^ combobox = toolbar_item->get_control<ToolStripComboBox>();
        if (combobox != nullptr)
        {
          // Normal combobox.
          List<String^> list = CppStringListToNative(values);
          combobox->Items->AddRange(list.ToArray());
        }
        break;
      }
    case mforms::ColorSelectorItem:
      {
        ToolStripDropDownButton^ selector = toolbar_item->get_control<ToolStripDropDownButton>();
        if (selector != nullptr)
        {
          // Color selector.
          Bitmap^ selected = nullptr;
          List<ToolStripButton^> buttons = gcnew List<ToolStripButton^>();

          for (std::vector<std::string>::const_iterator iterator = values.begin(); iterator != values.end(); iterator++)
          {
            String^ text = CppStringToNative(*iterator);
            ToolStripButton^ button = gcnew ToolStripButton();
            button->DisplayStyle = ToolStripItemDisplayStyle::Image;

            button->Text = text;
            button->Image = create_color_image(text);
            button->Tag = selector;
            button->ImageScaling = ToolStripItemImageScaling::None;
            button->Click += gcnew EventHandler(toolbar_item, &ToolBarItemImpl::OnColorItemActivation);

            buttons.Add(button);
          }
          selector->DropDown->Items->AddRange(buttons.ToArray());

        }
        break;
      }
    }
  }
}

//--------------------------------------------------------------------------------------------------

Bitmap^ ToolBarImpl::create_color_image(String^ color)
{
  SolidBrush^ brush= gcnew SolidBrush(Color::White);
  Pen^ pen = gcnew Pen(Color::LightGray);

  Bitmap^ image = gcnew Bitmap(60, 15);
  Graphics^ g = Graphics::FromImage(image);
  brush->Color = ColorTranslator::FromHtml(color);
  g->FillRectangle(brush, Drawing::Rectangle(1, 0, 58, 14));
  g->DrawRectangle(pen, Drawing::Rectangle(1, 0, 58, 14));

  return image;
}

//--------------------------------------------------------------------------------------------------
