/* 
 * Copyright (c) 2008, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#ifndef _WF_FILECHOOSER_H_
#define _WF_FILECHOOSER_H_

#include "mforms/mforms.h"

using namespace System;
using namespace Windows::Forms;

namespace MySQL {
  namespace Forms {

    public ref class FileChooserImpl : public ViewImpl
    {
      static bool create(::mforms::FileChooser *self, ::mforms::FileChooserType type, bool show_hidden)
      {
        // On Windows hiding/showing hidden files is a global switch in Explorer so we can just
        // ignore the show_hidden flag here. It is necessary for Linux/Mac.
        FileChooserImpl ^form= gcnew FileChooserImpl(self);

        if (form != nullptr)
        {
          Windows::Forms::CommonDialog^ dialog;
          switch (type)
          {
          case ::mforms::OpenFile:
            dialog= ViewImpl::create_dialog<OpenFileDialog>(self, form);
            ((FileDialog^) dialog)->RestoreDirectory= true;
            break;
          case ::mforms::SaveFile:
            dialog= ViewImpl::create_dialog<SaveFileDialog>(self, form);
            ((FileDialog^) dialog)->RestoreDirectory= true;
            break;
          case ::mforms::OpenDirectory:
            dialog= ViewImpl::create_dialog<FolderBrowserDialog>(self, form);
            break;
          default:
            return false;
          }
          return true;
        }
        return false;
      }

      static void set_title(::mforms::FileChooser *self, const std::string &title)
      {
        FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

        if (form != nullptr)
        {
          if (form->get_control<FileDialog>())
            form->get_control<FileDialog>()->Title= CppStringToNative(title);
          else
            form->get_control<FolderBrowserDialog>()->Description = CppStringToNative(title);
        }
      }

      static bool run_modal(::mforms::FileChooser *self)
      {
        FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

        if (form != nullptr)
        {
          if (form->get_control<FileDialog>())
          {
            Windows::Forms::FileDialog ^f= form->get_control<FileDialog>();
            return f->ShowDialog() == DialogResult::OK;
          }else
          {
            Windows::Forms::FolderBrowserDialog ^f= form->get_control<FolderBrowserDialog>();
            return f->ShowDialog() == DialogResult::OK;
          }
        }
        return false;
      }


      static void set_directory(::mforms::FileChooser *self, const std::string &path)
      {
        FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

        if (form != nullptr)
        {
          if (form->get_control<FileDialog>())
          {
            Windows::Forms::FileDialog ^f= form->get_control<FileDialog>();
            f->InitialDirectory= CppStringToNative(path);
          }else
          {
            Windows::Forms::FolderBrowserDialog ^f= form->get_control<FolderBrowserDialog>();
            f->SelectedPath = CppStringToNative(path);
          }
        }
      }

      static std::string get_directory(::mforms::FileChooser *self)
      {
        FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

        if (form != nullptr)
        {
          Windows::Forms::FolderBrowserDialog ^f= form->get_control<FolderBrowserDialog>();

          return NativeToCppString(f->SelectedPath);
        }
        return "";
      }

      static std::string get_path(::mforms::FileChooser *self)
      {
        FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

        if (form != nullptr)
        {
          if (form->get_control<FileDialog>())
          {
            Windows::Forms::FileDialog ^f= form->get_control<FileDialog>();
            return NativeToCppString(f->FileName);
          }else
          {
            Windows::Forms::FolderBrowserDialog ^f= form->get_control<FolderBrowserDialog>();
            return NativeToCppString(f->SelectedPath);
          }
        }
        return "";
      }

      static void set_extensions(::mforms::FileChooser *self, const std::string &extensions, 
        const std::string &default_extension)
      {
        FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

        if (form != nullptr)
        {
          Windows::Forms::FileDialog ^f= form->get_control<FileDialog>();

          f->DefaultExt= CppStringToNative(default_extension);
          f->Filter= CppStringToNative(extensions);
        }
      }

      FileChooserImpl(::mforms::FileChooser *form)
        : ViewImpl(form)
      {
      }

    public:
      static void init(Manager ^mgr)
      {
        ::mforms::ControlFactory *f= ::mforms::ControlFactory::get_instance();

        DEF_CALLBACK3(bool, ::mforms::FileChooser*, ::mforms::FileChooserType, bool, mgr, f->_filechooser_impl, FileChooserImpl, create);
        DEF_CALLBACK2(void, ::mforms::FileChooser*, const std::string&, mgr, f->_filechooser_impl, FileChooserImpl, set_title);
        DEF_CALLBACK1(bool, ::mforms::FileChooser*, mgr, f->_filechooser_impl, FileChooserImpl, run_modal);
        DEF_CALLBACK3(void, ::mforms::FileChooser*, const std::string&, const std::string&, mgr, f->_filechooser_impl, FileChooserImpl, set_extensions);
        DEF_CALLBACK2(void, ::mforms::FileChooser*, const std::string&, mgr, f->_filechooser_impl, FileChooserImpl, set_directory);
        DEF_CALLBACK1(std::string, ::mforms::FileChooser*, mgr, f->_filechooser_impl, FileChooserImpl, get_directory);
        DEF_CALLBACK1(std::string, ::mforms::FileChooser*, mgr, f->_filechooser_impl, FileChooserImpl, get_path);
      }
    };

  };
};

#endif
