/* 
 * Copyright (c) 2009, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#ifndef _WF_Utilities_H_
#define _WF_Utilities_H_

#include <mforms/mforms.h>
#include "mforms/utilities.h"

#include "wf_mforms.h"

using namespace std;
using namespace mforms;
using namespace Windows::Forms;

namespace MySQL {
  namespace Forms {

    // Message type for the C# interface.
    public enum class MessageType
    {
      MessageInfo,
      MessageWarning,
      MessageError
    };

    /**
     * A custom message box for the output as there is no predefined dialog which allows to
     * have custom button captions.
     */
    public ref class CustomMessageBox : Windows::Forms::Form
    {
    private:
      // Constructor is private. CustomMessageBox should be accessed through the public Show() method
      CustomMessageBox();

      // GUI Elements, we have 3 buttons whose text can be customized.
      Windows::Forms::Label^ _messageLabel;
      Windows::Forms::Button^ _button1;
      Windows::Forms::Button^ _button2;
      Windows::Forms::Button^ _button3;
      Windows::Forms::PictureBox^ _picture;
      Windows::Forms::CheckBox^ _checkbox;

      void ComputeLayout();
      void ButtonClick(System::Object^ sender, EventArgs^ e);

      static mforms::DialogResult ShowVistaStyle(const std::string& title, const std::string& text,
        PCWSTR mainIcon, const std::string& buttonOK, const std::string& buttonCancel,
        const std::string& buttonOther, const std::string& checkbox, bool& checked);

      static mforms::DialogResult ShowTraditionalStyle(String^ title, String^ text, Drawing::Image^ image,
        String^ button1, String^ button2, String^ button3, String^ checkbox, bool& checked);
    public:
      // C++ interface
      static mforms::DialogResult Show(const std::string& title, const std::string& text,
        PCWSTR mainIcon, const std::string& buttonOK, const std::string& buttonCancel,
        const std::string& buttonOther, const std::string& checkbox, bool& checked);

      // C# interface
      static Windows::Forms::DialogResult Show(MessageType type, String^ title, String^ text, String^ buttonOK,
        String^ buttonCancel, String^ buttonOther, String^ checkbox, bool& checked);
      static Windows::Forms::DialogResult Show(MessageType type, String^ title, String^ text, String^ buttonOK);
    };

    public ref class UtilitiesImpl
    {
    private:
      static void load_passwords();
      static void unload_passwords(bool store);
    protected:
      UtilitiesImpl();

      static int show_message(const string &title, const string &text, const string &ok, 
        const string &cancel, const string &other);
      static int show_error(const string &title, const string &text, const string &ok, 
        const string &cancel, const string &other);
      static int show_warning(const string &title, const string &text, const string &ok, 
        const string &cancel, const string &other);
      static int show_message_with_checkbox(const string &title, const string &text, const string &ok, 
        const string &cancel, const string &other, const std::string &checkbox_text, bool &isChecked);
      static void show_wait_message(const string &title, const string &text);
      static bool hide_wait_message();

      static void set_clipboard_text(const string &content);
      static string get_clipboard_text();
      static string get_special_folder(FolderType type);

      static void open_url(const string &url);

      static void add_timeout(float interval, const sigc::slot<bool> &slot);

      static void store_password(const std::string &service, const std::string &account, const std::string &password);
      static bool find_password(const std::string &service, const std::string &account, std::string &password);
      static void forget_password(const std::string &service, const std::string &account);

    public:
      static void init(Manager ^mgr)
      {
        ::mforms::ControlFactory *f= ::mforms::ControlFactory::get_instance();

        DEF_CALLBACK5(int, const string&, const string&, const string&, const string&, const string&, 
          mgr, f->_utilities_impl, UtilitiesImpl, show_message);
        DEF_CALLBACK5(int, const string&, const string&, const string&, const string&, const string&, 
          mgr, f->_utilities_impl, UtilitiesImpl, show_error);
        DEF_CALLBACK5(int, const string&, const string&, const string&, const string&, const string&, 
          mgr, f->_utilities_impl, UtilitiesImpl, show_warning);
        DEF_CALLBACK7(int, const string&, const string&, const string&, const string&, const string&, 
          const string&, bool&, mgr, f->_utilities_impl, UtilitiesImpl, show_message_with_checkbox);
        DEF_CALLBACK2(void, const string&, const string&, mgr, f->_utilities_impl, UtilitiesImpl,
          show_wait_message);
        DEF_CALLBACK0(bool, mgr, f->_utilities_impl, UtilitiesImpl, hide_wait_message);
        DEF_CALLBACK1(void, const string&, mgr, f->_utilities_impl, UtilitiesImpl, set_clipboard_text);
        DEF_CALLBACK0(string, mgr, f->_utilities_impl, UtilitiesImpl, get_clipboard_text);
        DEF_CALLBACK1(void, const string&, mgr, f->_utilities_impl, UtilitiesImpl, open_url);
        DEF_CALLBACK2(void, float, const sigc::slot<bool>&, mgr, f->_utilities_impl, UtilitiesImpl, add_timeout);
        DEF_CALLBACK1(string, mforms::FolderType, mgr, f->_utilities_impl, UtilitiesImpl, get_special_folder);
        DEF_CALLBACK3(void, const string&, const string&, const string&, mgr, f->_utilities_impl, UtilitiesImpl, store_password);
        DEF_CALLBACK3(bool, const string&, const string&, string&, mgr, f->_utilities_impl, UtilitiesImpl, find_password);
        DEF_CALLBACK2(void, const string&, const string&, mgr, f->_utilities_impl, UtilitiesImpl, forget_password);
      }
    };
  };
};

#endif // _WF_Utilities_H_
