#include "stdafx.h"

#include "db_sql_editor_history_be.h"
#include "sqlide/recordset_data_storage.h"
#include "grt/common.h"
#include "base/string_utilities.h"
#include "base/util_functions.h"
#include "base/log.h"
#include "tinyxml.h"
#include <fstream>
#include <glib/gstdio.h>
#include <boost/foreach.hpp>
#include <stdlib.h>
#include <fstream>
#include <mforms/mforms.h>

ENABLE_LOG("sqlide-history")

using namespace bec;
using namespace grt;


const char *SQL_HISTORY_DIR_NAME= "sql_history";


DbSqlEditorHistory::DbSqlEditorHistory(GRTManager *grtm)
:
_grtm(grtm), _current_entry_index(-1)
{
  _entries_model= EntriesModel::create(_grtm);
  _details_model= DetailsModel::create(_grtm);
  _entries_model->reset_current_entry_index= sigc::bind(sigc::mem_fun(this, (void(DbSqlEditorHistory::*)(int))&DbSqlEditorHistory::current_entry), -1);
  load();
}


DbSqlEditorHistory::~DbSqlEditorHistory()
{
  save();
}


void DbSqlEditorHistory::reset()
{
  _details_model->reset();
  _entries_model->reset();
  _current_entry_index= -1;
}


void DbSqlEditorHistory::load()
{
  _entries_model->load();
}


void DbSqlEditorHistory::save()
{
  _entries_model->save();
}


void DbSqlEditorHistory::add_entry(const std::list<std::string> &statements)
{
  _entries_model->add_entry(statements);
  if (_entries_model->row_count() > 0)
    current_entry(_entries_model->row_count()-1);
  save();
}


void DbSqlEditorHistory::insert_entry(const std::tm &t)
{
  _entries_model->insert_entry(t);
}


void DbSqlEditorHistory::current_entry(int index)
{
  if (index == -1)
  {
    _details_model->reset();
    _details_model->refresh();
  }
  else
  {
    DetailsModel::Ref details_model= _entries_model->details_model(index);
    details_model->load();
    _details_model->load_from(details_model);
  }
  _current_entry_index= index;
  _entries_model->refresh();
}


std::string DbSqlEditorHistory::restore_sql_from_history(int entry_index, std::list<int> &detail_indexes)
{
  std::string sql;
  if (entry_index >= 0)
  {
    DetailsModel::Ref details_model= _entries_model->details_model(entry_index);
    NodeId node;
    node.index->push_back(int());
    std::string statement;
    BOOST_FOREACH (int row, detail_indexes)
    {
      node[0]= row;
      details_model->get_field(node, 1, statement);
      sql+= statement + ";\n";
    }
  }
  return sql;
}


void DbSqlEditorHistory::EntriesModel::reset()
{
  VarGridModel::reset();

  {
    GStaticRecMutexLock data_mutex(_data_mutex);
    reinit(_details_models);
  }

  _readonly= true;

  add_column("Date", std::string());

  boost::shared_ptr<sqlite::connection> data_swap_db= this->data_swap_db();
  Recordset_data_storage::create_data_swap_tables(data_swap_db.get(), _column_names, _column_types);

  refresh_ui();
}

void DbSqlEditorHistory::EntriesModel::load()
{
  std::string sql_history_dir= make_path(_grtm->get_user_datadir(), SQL_HISTORY_DIR_NAME);
  g_mkdir_with_parents(sql_history_dir.c_str(), 0700);
  {
    GError *error= NULL;
    GDir *dir= g_dir_open(sql_history_dir.c_str(), 0, &error);
    if (!dir)
    {
      _grtm->get_grt()->send_error(_("Can't open SQL history directory"), (error ? error->message : sql_history_dir.c_str()));
      return;
    }
    // files are not read in alpha-order, so we need to sort them before inserting
    std::set<std::string> entries;
    ScopeExitTrigger on_scope_exit(sigc::bind(sigc::ptr_fun(&g_dir_close), dir));
    while (const char *name_= g_dir_read_name(dir))
    {
      // file name is expected in "YYYY-MM-DD" format
      std::string name(name_);
      if (name.size() != 10)
        continue;
      name[4]= '\0';
      name[7]= '\0';
      entries.insert(name);
    }

    for (std::set<std::string>::const_iterator name= entries.begin(); name != entries.end(); ++name)
    {
      tm t;
      memset(&t, 0, sizeof(t));
      t.tm_year= atoi(&(*name)[0])-1900;
      t.tm_mon= atoi(&(*name)[5])-1;
      t.tm_mday= atoi(&(*name)[8]);
      if (t.tm_year != 0)
        insert_entry(t);
    }
  }
}


void DbSqlEditorHistory::EntriesModel::save()
{
  GStaticRecMutexLock data_mutex(_data_mutex);

  for (RowId row= 0; row < _row_count; ++row)
  {
    DetailsModel::Ref details_model= _details_models[row];
    if (!details_model->loaded())
      continue;
    details_model->save();
  }
}


void DbSqlEditorHistory::EntriesModel::add_entry(const std::list<std::string> &statements)
{
  if (statements.empty())
    return;

  std::tm timestamp = local_timestamp();
  insert_entry(timestamp);

  std::string time= format_time(timestamp, "%X");
  std::list<std::string> timed_statements;

  BOOST_FOREACH(std::string statement, statements)
  {
    timed_statements.push_back(time);
    timed_statements.push_back(base::strip_text(statement));
  }

  DetailsModel::Ref details_model= _details_models.back();
  details_model->add_entries(timed_statements);

  refresh_ui();
}


void DbSqlEditorHistory::EntriesModel::insert_entry(const std::tm &t)
{
  std::string last_date;
  if (_row_count > 0)
    get_field(NodeId(_row_count-1), 0, last_date);
  std::string date= format_time(t, "%x");
  if (date != last_date)
  {
    GStaticRecMutexLock data_mutex(_data_mutex);
    _data.reserve(_data.size() + _column_count);
    _data.push_back(date);
    DetailsModel::Ref details_model= DetailsModel::create(_grtm);
    details_model->datestamp(t);
    _details_models.push_back(details_model);
    ++_row_count;
    ++_data_frame_end;
  }
}


bec::MenuItemList DbSqlEditorHistory::EntriesModel::get_popup_items_for_nodes(const std::vector<bec::NodeId> &nodes)
{
  bec::MenuItemList items;
  bec::MenuItem item;

  item.name = "delete_selection";
  item.caption = "Delete Selected Date(s) Log";
  item.enabled = nodes.size() > 0;
  items.push_back(item);
  
  item.name = "delete_all";
  item.caption = "Delete All Logs";
  item.enabled = true;
  items.push_back(item);
  
  return items;
}


bool DbSqlEditorHistory::EntriesModel::activate_popup_item_for_nodes(const std::string &action, const std::vector<bec::NodeId> &orig_nodes)
{
  if (action == "delete_selection")
  {
    for (size_t n= 0, count= orig_nodes.size(); n < count; ++n)
    {
      std::vector<int> rows;
      rows.reserve(orig_nodes.size());
      BOOST_FOREACH (const bec::NodeId &node, orig_nodes)
        rows.push_back(node[0]);
      delete_entries(rows);
      return true;
    }
  }
  else if (action == "delete_all")
  {
    delete_all_entries();
    return true;
  }

  return false;
}


void DbSqlEditorHistory::EntriesModel::delete_all_entries()
{
  if (mforms::Utilities::show_message("Clear History",
                                      "Do you really want to delete the entire query history?\nThis operation cannot be undone.",
                                      "Delete All", "Cancel", "") == mforms::ResultCancel)
    return;
  
  std::vector<int> rows;
  rows.reserve(_row_count);
  for (RowId row= 0; row < _row_count; ++row)
    rows.push_back(row);
  delete_entries(rows);  
}

void DbSqlEditorHistory::EntriesModel::delete_entries(const std::vector<int> &rows)
{
  if (rows.empty())
    return;
  {
    std::vector<int> sorted_rows= rows;
    std::sort(sorted_rows.begin(), sorted_rows.end());
    GStaticRecMutexLock data_mutex(_data_mutex);
    BOOST_REVERSE_FOREACH (RowId row, sorted_rows)
    {
      if (row >= _row_count)
        continue;
      
      DetailsModel::Ref details_model= _details_models[row];
      std::string storage_file_path= details_model->storage_file_path();
      g_remove(storage_file_path.c_str());

      Cell row_begin= _data.begin() + row * _column_count;
      _data.erase(row_begin, row_begin + _column_count);
      _details_models.erase(_details_models.begin() + row);
      --_row_count;
    }
  }
  refresh_ui();
  reset_current_entry_index();
}


void DbSqlEditorHistory::DetailsModel::reset()
{
  VarGridModel::reset();

  _loaded= false;
  _last_loaded_row= -1;
  _last_timestamp = std::string("");
  _last_statement = std::string("");
  _datestamp= std::tm();

  _readonly= true;

  add_column("Time", std::string());
  add_column("SQL", std::string());

  boost::shared_ptr<sqlite::connection> data_swap_db= this->data_swap_db();
  Recordset_data_storage::create_data_swap_tables(data_swap_db.get(), _column_names, _column_types);

  refresh_ui();
}


std::string DbSqlEditorHistory::DetailsModel::storage_file_path() const
{
  std::string storage_file_path= make_path(_grtm->get_user_datadir(), SQL_HISTORY_DIR_NAME);
  storage_file_path= make_path(storage_file_path, format_time(_datestamp, "%Y-%m-%d"));
  return storage_file_path;
}

void DbSqlEditorHistory::DetailsModel::load()
{
  if (_loaded)
    return;

  std::string storage_file_path= this->storage_file_path();
  
  char *file_name=g_filename_from_utf8(storage_file_path.c_str(), -1, NULL, NULL, NULL);
  if (file_name)
  {
    if (g_file_test(file_name, G_FILE_TEST_EXISTS))
    {
      std::ifstream history_xml(file_name);

      if (history_xml.is_open())
      {
        GStaticRecMutexLock data_mutex(_data_mutex);
        _data.reserve(_data.size() + _column_count);

        std::string line;

        // Skips the first line in the file as is the xml header
        std::getline(history_xml, line);

        const char *errptr;
        int erroffs=0;
        char *pattern = "<ENTRY( timestamp\\=\'(.*)\')?>(.*)<\\/ENTRY>";
        int patres[64];

        pcre *patre= pcre_compile(pattern, 0, &errptr, &erroffs, NULL);
        if (!patre)
          throw std::logic_error("error compiling regex "+std::string(errptr));

        // process the file
        _row_count = 0;
        while (history_xml.good())
        {
          std::string timestamp;
          std::string statement;
          const char *value;

          std::getline(history_xml, line);

          // executes the regexp against the new line
          int rc = pcre_exec(patre, NULL, line.c_str(), line.length(), 0, 0, patres, sizeof(patres)/sizeof(int));

          if ( rc > 0 )
          {
            // gets the values timestamp and 
            pcre_get_substring(line.c_str(), patres, rc, 2, &value);
            timestamp = value ? std::string(value) : "";

            pcre_get_substring(line.c_str(), patres, rc, 3, &value);
            statement = value ? std::string(value) : "";

            // decides whether to use or not the existing data
            if (timestamp != _last_timestamp.repr() && timestamp != "~")
              _last_timestamp = timestamp;

            if (statement != _last_statement.repr() && statement != "~")
              _last_statement = statement;

            _data.push_back(_last_timestamp);
            _data.push_back(_last_statement);
        
            _row_count++;
          }
        }

        pcre_free(patre);

        history_xml.close();

        _data_frame_end= _row_count;

        _last_loaded_row= _row_count-1;

        _loaded= true;
      }
      else
        log_error("Can't open SQL history file %s", storage_file_path.c_str());
    }
    else
      _loaded = true; //Should be marked as loaded as a new file will be created
  }
  else
    log_error("Error converting file name %s to utf8", storage_file_path.c_str());

}

void DbSqlEditorHistory::DetailsModel::save()
{
  std::string storage_file_path= this->storage_file_path();
  std::ofstream ofs;
  {
    std::string storage_file_dir= make_path(_grtm->get_user_datadir(), SQL_HISTORY_DIR_NAME);
    if (g_mkdir_with_parents(storage_file_dir.c_str(), 0700) != -1)
    {
      bool is_file_new= (g_file_test(storage_file_path.c_str(), G_FILE_TEST_EXISTS) == 0);
      if (is_file_new || g_file_test(storage_file_path.c_str(), G_FILE_TEST_IS_REGULAR))
      {
        ofs.open(storage_file_path.c_str(), std::ios_base::app);
        if (is_file_new)
          ofs << "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n";
      }
    }
  }

  if (!ofs.is_open() || !ofs)
  {
    _grtm->get_grt()->send_error("Can't write to SQL history file", storage_file_path);
    return;
  }

  {
    GStaticRecMutexLock data_mutex(_data_mutex);
    std::string last_saved_timestamp;
    std::string last_saved_statement;
    get_field(NodeId(_last_loaded_row), 0, last_saved_timestamp);
    get_field(NodeId(_last_loaded_row), 1, last_saved_statement);

    for (RowId row= _last_loaded_row+1; row < _row_count; ++row)
    {
      std::string time, sql;
      get_field(NodeId(row), 0, time);
      get_field(NodeId(row), 1, sql);

      if (time == last_saved_timestamp)
        time = "~";
      else
        last_saved_timestamp = time;

      if (sql == last_saved_statement)
        sql = "~";
      else
        last_saved_statement = sql;

      std::string xml_time, xml_sql;
      TiXmlBase::EncodeString(time, &xml_time); 
      TiXmlBase::EncodeString(sql, &xml_sql); 
      ofs << "<ENTRY timestamp=\'" << xml_time << "\'>" << xml_sql << "</ENTRY>\n";
    }
    _last_loaded_row= _row_count - 1;
  }
  ofs.flush();
}


void DbSqlEditorHistory::DetailsModel::add_entries(const std::list<std::string> &statements)
{
  if (statements.empty())
    return;

  if (!_loaded)
    load();

  {
    GStaticRecMutexLock data_mutex(_data_mutex);

    _data.reserve(_data.size() + _column_count);

    try
    {
      int index=0;
      BOOST_FOREACH (std::string statement, statements)
      {
        if (index % 2)
        {
          // decides whether to use or not the existing data
          if (statement != _last_statement.repr())
            _last_statement = statement;
        
          _data.push_back(_last_statement);
        }
        else
        {
          if (statement != _last_timestamp.repr())
            _last_timestamp = statement;

          _data.push_back(_last_timestamp);
        }
      }

      index++;
    }
    catch(...)
    {
      _data.resize(_row_count * _column_count);
      throw;
    }

    _row_count+= statements.size()/2;
    _data_frame_end= _row_count;

  }

  refresh_ui();
}


void DbSqlEditorHistory::DetailsModel::load_from(DbSqlEditorHistory::DetailsModel::Ref details_model)
{
  {
    GStaticRecMutexLock data_mutex(_data_mutex);

    _data= details_model->data();
    _row_count= details_model->row_count();
    _data_frame_end= _row_count;
  }

  refresh_ui();
}
