/* 
 * Copyright (c) 2007, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
#ifndef _DB_SQL_EDITOR_BE_H_
#define _DB_SQL_EDITOR_BE_H_

// renamed from db_sql_editor_be.h

#include "workbench/wb_backend_public_interface.h"
#include "sqlide/recordset_be.h"
#include "sqlide/sql_editor_be.h"
#include "sqlide/db_sql_editor_log_be.h"
#include "sqlide/db_sql_editor_history_be.h"
#include "cppdbc.h"
#include "grts/structs.workbench.h"
#include "grts/structs.db.mgmt.h"
#include "base/file_utilities.h"
#include "base/ui_form.h"
#include "grt/refresh_ui.h"
#include "grt/action_list.h"
#include "sqlide/wb_context_sqlide.h"
#include "sqlide/wb_live_schema_tree.h"
#include "sqlide/wb_overview_live_physical.h"
#include <boost/enable_shared_from_this.hpp>

#include "mforms/task_sidebar.h"
#include "mforms/textbox.h"
#include "mforms/splitter.h"

namespace bec
{
  class DBObjectEditorBE;
}

class MYSQLWBBACKEND_PUBLIC_FUNC Db_sql_editor : public bec::UIForm, public bec::RefreshUI, public boost::enable_shared_from_this<Db_sql_editor>
{
public:
  enum
  {
    RefreshSidebar, // update the sidebar
//deprecated    RefreshActiveSchemaSelector, // refresh active schema selector
//deprecated    RefreshSchemaTree, // refresh live schema tree
    RefreshEditor, // refresh the text editor using data from backend
    RefreshEditorBackend, // save text editor contents to backend so that it can be saved to file
    RefreshEditorTitle,  // refresh the caption of active editor
    RefreshSnippets, // snippets list
    RefreshSnippetCollections, // snippet collections list
    RefreshOverview, // refresh the live overview
    RefreshRecordsetTitle, // refresh caption of active recordset
    RunCurrentScript, // running sql needs to be initiated by front-end
    RunCurrentStatement // run only statement under cursor position (if there any statement at all)
  };

public:
  typedef boost::shared_ptr<Db_sql_editor> Ref;
  typedef boost::weak_ptr<Db_sql_editor> Ptr;
  static Db_sql_editor::Ref create(wb::WBContextSQLIDE *wbsql, const db_mgmt_ConnectionRef &conn);
  
protected:
  Db_sql_editor(wb::WBContextSQLIDE *wbsql, const db_mgmt_ConnectionRef &conn);
public:
  virtual ~Db_sql_editor();

public:
  virtual bool close();
  virtual bool is_main_form() { return true; }
  virtual std::string get_form_context_name() const;
  
  virtual mforms::MenuBar *get_menubar();
  virtual mforms::ToolBar *get_toolbar();
  
  void auto_save();
  void save_workspace(const std::string &workspace_name, bool is_autosave);
  bool load_workspace(const std::string &workspace_name);
  
public:
  bec::GRTManager * grt_manager() const { return _grtm; }
  wb::WBContextSQLIDE *wbsql() const { return _wbsql; }
  void context_ui(wb::WBContextUI *val);

  void update_menu_and_toolbar();
private:
  wb::WBContextSQLIDE *_wbsql;
  wb::WBContextUI *_context_ui;
  bec::GRTManager *_grtm;
  mforms::MenuBar *_menu;
  mforms::ToolBar *_toolbar;
  grt::DictRef _options;
  std::string _connection_info;
  base::LockFile *_autosave_lock;
  std::string _autosave_path;
  bool _loading_workspace;
  bool _close_done;

  void activate_command(const std::string &command);
public:
  ::ActionList & action_list();
private:
  ::ActionList _action_list;
private:
  void register_default_actions();

public:
  db_mgmt_RdbmsRef rdbms();

public:
  Sql_editor::Ref sql_editor();
  Sql_editor::Ref sql_editor(int index);
  std::string sql_editor_path(int index) { return _sql_editors[index].filename; }
  std::string sql_editor_caption(int index=-1);
  bool sql_editor_dirty(int index) { return _sql_editors[index].dirty && !_sql_editors[index].is_scratch; }
  bool sql_editor_is_scratch(int index) { return _sql_editors[index].is_scratch; }
  void sql_editor_dirty(int index, bool flag);
  bool sql_editor_will_close(int index);
  long long active_recordset_for_sql_editor(int index);
  int sql_editor_index_for_recordset(long long rset);
private:
  struct Sql_editor_info {
    std::string filename;
    std::string orig_encoding;
    std::string caption;
    Sql_editor::Ref editor;
    std::vector<long long> resultsets;
    long long active_resultset;
    bool dirty;
    bool is_scratch;
  };
  typedef std::vector<Sql_editor_info> Sql_editors;
  Sql_editors _sql_editors;
  int _sql_editors_serial;
  int _scratch_editors_serial;
  GMutex *_sql_editors_mutex;
public:
  sigc::signal<void, Sql_editor::Ref, bool> sql_editor_list_changed;

  int add_sql_editor(bool scratch = false); // returns index of the added sql_editor
  void remove_sql_editor(int index);
  void add_empty_recordset_for_sql_editor(int editor);
  int sql_editor_count();
  int active_sql_editor_index() const { return _active_sql_editor_index; }
  void active_sql_editor_index(int val) { _active_sql_editor_index= val; }
private:
  int _active_sql_editor_index;
  int _updating_sql_editor;

private:
  void on_sql_editor_text_change();
  void on_sql_editor_text_selection_change();
  void set_sql_editor_text(const std::string &sql, bool run_sql);
public:
  std::string caption() const;

private:
  std::map<std::string, std::string> _connection_details;

  grt::StringRef do_connect(grt::GRT *grt);
  grt::StringRef do_disconnect(grt::GRT *grt);
public:
  void connect();
  bool connected() const;
  void finish_startup();
  void cancel_query();
  void reset();
  void commit();
  void rollback();
  bool auto_commit();
  void auto_commit(bool value);
private:
  void do_commit();
public:  
  db_mgmt_ConnectionRef connection_descriptor() const { return _connection; }

private:
  bool get_session_variable(sql::Dbc_connection_handler::Ref &dbc_conn, const std::string &name, std::string &value);
  void cache_sql_mode();
private:
  std::string _sql_mode;

private:
  void create_connection(sql::Dbc_connection_handler::Ref &dbc_conn, db_mgmt_ConnectionRef db_mgmt_conn, bool autocommit_mode);
  void init_connection(sql::Connection* dbc_conn_ref, const db_mgmt_ConnectionRef& connectionProperties, sql::Dbc_connection_handler::Ref& dbc_conn);
  void close_connection(sql::Dbc_connection_handler::Ref &dbc_conn);
  bec::GMutexLock ensure_valid_dbc_connection(sql::Dbc_connection_handler::Ref &dbc_conn, GMutex *dbc_conn_mutex);
  bec::GMutexLock ensure_valid_aux_connection();
  bec::GMutexLock ensure_valid_usr_connection();

private:
  bec::TimerActionThread *_keep_alive_thread;
  GMutex *_keep_alive_thread_mutex;
private:
  void send_message_keep_alive();
  void reset_keep_alive_thread();
  
  db_mgmt_ConnectionRef _connection;
  // connection for maintenance operations, fetching schema contents & live editors (DDL only)
  sql::Dbc_connection_handler::Ref _aux_dbc_conn;
  GMutex *_aux_dbc_conn_mutex;
  // connection for running sql scripts
  sql::Dbc_connection_handler::Ref _usr_dbc_conn;
  GMutex *_usr_dbc_conn_mutex;

public:
  typedef std::vector<Recordset::Ref> Recordsets;

  void exec_sql(std::string &sql, Sql_editor::Ref editor, bool sync, bool wrap_with_non_std_delimiter= false, bool dont_add_limit_clause= false);
  void exec_sql_retaining_editor_contents(const std::string &sql_script, bool sync, bool dont_add_limit_clause= false);

  Recordsets exec_sql_returning_results(const std::string &sql_script, bool dont_add_limit_clause);
  
  bool explain_sql(bool only_check_if_applicable);
  void explain_current_statement();
  bool is_running_query();
private:
  enum ExecFlags {
    Retaining = (1<<0), 
    Wrap_with_non_std_delimiter = (1<<1),
    Dont_add_limit_clause = (1<<2),
    Show_warnings = (1<<3)
  };
  grt::StringRef do_exec_sql(grt::GRT *grt, Ptr self_ptr, std::string &sql, Sql_editor::Ref editor, ExecFlags flags, Recordsets *result_list);
  void do_explain_sql(const std::string &sql);
public:
  GrtThreadedTask::Ref exec_sql_task;
private:
  int on_exec_sql_finished();
  bool _is_running_query;

public:
  bool continue_on_error() { return _continue_on_error; }
  void continue_on_error(bool val);
private:
  bool _continue_on_error;

public:
  int recordset_count();
  Recordset::Ref recordset(int index);
  void recordsets_are_pinned_by_default(bool value);
  bool recordsets_are_pinned_by_default() { return _recordsets_are_pinned_by_default; }
  void active_recordset(Recordset::Ref value);
  Recordset::Ref active_recordset() { return _active_recordset; }
private:
  Recordsets _recordsets;
  GMutex *_recordsets_mutex;
  bool _recordsets_are_pinned_by_default;
  Recordset::Ref _active_recordset;
public:
  sigc::signal<void, Recordset::Ref, bool> recordset_list_changed;
  //delme sigc::signal<int, long long> close_recordset_ui;
private:
  void on_close_recordset(Recordset::Ptr rs_ptr);
  int _rs_sequence;
private:
  void apply_changes_to_recordset(Recordset::Ptr rs_ptr);
  bool run_data_changes_commit_wizard(Recordset::Ptr rs_ptr);
  void apply_data_changes_commit(std::string &sql_script_text, Recordset::Ptr rs_ptr);
  void save_active_recordset_edits();
  void discard_active_recordset_edits();
  void recall_recordset_query(Recordset::Ptr rs_ptr);

  void show_export_recordset(Recordset::Ptr rs_ptr);
public:
  bool can_close();
  bool can_close_(bool interactive);

public:
  typedef sigc::signal<int, const std::string&> SqlEditorTextInsertSignal;
  SqlEditorTextInsertSignal sql_editor_text_insert_signal;

public:
  void new_sql_script_file();
  void new_sql_scratch_area();
  void open_sql_script_file(const std::string &file_path, bool run_immediately, bool run_only_skip_editor = false,
                            const std::string &encoding = "");
  bool save_sql_script_file(const std::string &file_path, int editor_index);
  void revert_sql_script_file();

public:
  sigc::signal<int, int> sql_editor_new_ui;
//  sigc::signal<int, std::string> sql_editor_caption_ui;

public:
  void get_schemata(std::list<std::string> &schemata);
  void active_schema(const std::string &value);
  std::string active_schema() const;
  int active_schema_index() const;
private:
  void cache_active_schema_name();

public:
  wb::LiveSchemaTree *get_schema_tree();
  void request_refresh_schema_tree();
  
private:
  std::list<std::string> fetch_live_schema_list();
  bool fetch_live_schema_contents(bec::NodeId node, const std::string &schema_name, const wb::LiveSchemaTree::SchemaContentArrivedSlot &arrived_slot);
  grt::StringRef do_fetch_live_schema_contents(grt::GRT *grt, Ptr self_ptr, bec::NodeId node, std::string schema_name, wb::LiveSchemaTree::SchemaContentArrivedSlot arrived_slot);
  void notify_of_fetched_schema_contents(Ptr self_ptr, bec::NodeId node, db_SchemaRef schema, bool schemata_tree_needs_refresh, bool live_physical_overview_schema_node_needs_refresh, bool live_physical_overview_needs_refresh);
  bool fetch_object_details(wb::LiveSchemaTree::ObjectType obj_type, const std::string &schema_name, const std::string &obj_name, const wb::LiveSchemaTree::ObjectDetailsArrivedSlot &arrived_slot);
private:
  wb::LiveSchemaTree _schema_tree;
  GrtThreadedTask::Ref live_schema_fetch_task;
  GMutex *_schema_contents_mutex;

  mforms::TaskSidebar* _side_bar;
  mforms::Splitter _side_splitter;
  mforms::TextBox _info_box;
private:
  void do_alter_live_object(wb::LiveSchemaTree::ObjectType type, const std::string &schema_name, const std::string &obj_name);
  void do_create_live_object(wb::LiveSchemaTree::ObjectType type, const std::string &schema_name, const std::string &obj_name);
  void do_drop_live_object(wb::LiveSchemaTree::ObjectType type, const std::string schema_name, const std::string obj_name);
  std::string get_object_ddl_script(wb::LiveSchemaTree::ObjectType type, const std::string &schema_name, const std::string &obj_name);
  void apply_object_alter_script(std::string &alter_script, bec::DBObjectEditorBE* obj_editor);
  void refresh_live_object_in_editor(bec::DBObjectEditorBE* obj_editor, bool using_old_name);
  void refresh_live_object_in_overview(wb::LiveSchemaTree::ObjectType type, const std::string schema_name, const std::string old_obj_name, const std::string new_obj_name);
  
  bool parse_ddl_into_catalog(db_mgmt_RdbmsRef rdbms, db_CatalogRef client_state_catalog, const std::string &obj_descr, const std::string &ddl_script, std::string sql_mode);
  
  db_SchemaRef create_new_schema(db_CatalogRef owner);
  db_TableRef create_new_table(db_SchemaRef owner);
  db_ViewRef create_new_view(db_SchemaRef owner);
  db_RoutineRef create_new_routine(db_SchemaRef owner);
public:
  void schema_object_activated(const std::string &action, wb::LiveSchemaTree::ObjectType type, const std::string &schema, const std::string &name);
  bool apply_changes_to_object(bec::DBObjectEditorBE* obj_editor, bool dry_run);
  virtual void notify_of_ui_form_creation(bec::UIForm *form);
  bool run_live_object_alteration_wizard(std::string &alter_script, bec::DBObjectEditorBE* obj_editor);
public:
  void create_live_table_stubs(bec::DBObjectEditorBE *table_editor);
  bool expand_live_table_stub(bec::DBObjectEditorBE *table_editor, const std::string &schema_name, const std::string &obj_name);

private:
  typedef sigc::signal<int, long long, const std::string&, const std::string&> Error_cb;
  typedef sigc::signal<int, float> Batch_exec_progress_cb;
  typedef sigc::signal<int, long, long> Batch_exec_stat_cb;
  Error_cb on_sql_script_run_error;
  Batch_exec_progress_cb on_sql_script_run_progress;
  Batch_exec_stat_cb on_sql_script_run_statistics;

public:
  wb::LivePhysicalOverviewBE * live_physical_overview();
  bool activate_live_object(GrtObjectRef object);
  bool create_live_object(GrtObjectRef object_type, std::string owner_name, std::string obj_name);
  bool drop_live_object(GrtObjectRef object_type, std::string owner_name, std::string obj_name);
  void refresh_physical_overview(bool sync);
  bool is_physical_overview_enabled() const;
private:
  grt::StringRef do_refresh_physical_overview(grt::GRT *grt, Ptr self_ptr, bool hard_refresh);
  void do_refresh_physical_overview(bool hard_refresh);
  void do_refresh_physical_overview(std::list<std::string> schema_list, bool hard_refresh);
private:
  wb::LivePhysicalOverviewBE * _live_physical_overview;
  bool _is_physical_overview_enabled;
  GrtThreadedTask::Ref live_schemata_refresh_task;

public:
  DbSqlEditorLog::Ref log() { return _log; }
  DbSqlEditorHistory::Ref history() { return _history; }
  std::string restore_sql_from_history(int entry_index, std::list<int> &detail_indexes);
  int exec_sql_error_count() { return _exec_sql_error_count; }
  
  sigc::slot<void, std::string, bool> output_text_slot;
protected:
  DbSqlEditorLog::Ref _log;
  DbSqlEditorHistory::Ref _history;

public:
  void get_log_event_details(int log_event_no, int &log_event_type_code, std::string &log_event_time, std::string &log_event_action, std::string &log_event_message);
protected:
  RowId add_log_message(int msg_type, const std::string &msg, const std::string &context, const std::string &duration);
  void set_log_message(RowId log_message_index, int msg_type, const std::string &msg, const std::string &context, const std::string &duration);
  void refresh_log_messages(bool ignore_last_message_timestamp);
private:
  bool _has_pending_log_messages;
  time_t _last_log_message_timestamp;
  int _exec_sql_error_count;

protected:
  int _progress_status_update_interval;

private:
  bec::MenuItemList get_live_catalog_menu_items(const std::string &schema, const std::string &object, 
                                                const std::string &type);
  bool call_live_catalog_menu_item(const std::string &item,
                                   const std::string &schema, const std::string &object, 
                                   const std::string &type);

  bec::MenuItemList recordset_popup_menu_items(const std::vector<int> &rows, int column, Recordset::Ptr rs_ptr);
  bool call_recordset_popup_menu_item(const std::string &item, const std::vector<int> &rows,
                                      int column, Recordset::Ptr rs_ptr);

  void main_form_changed(bec::UIForm *form);
  void setup_sidebar();
  void sidebar_command(const std::string& command);
  
  void schema_row_selected();
public:
  bool save_snippet(const std::string &text);

  //mforms::TaskSidebar* get_sidebar();
  mforms::View *get_sidebar() { return &_side_splitter; }
};


#endif /* _DB_SQL_EDITOR_BE_H_ */
