/* 
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "inserts_export_form.h"
#include "sqlide/recordset_text_storage.h"
#include "sqlide/recordset_sql_storage.h"
#include "base/string_utilities.h"

using namespace mforms;
using namespace base;
using namespace MySQL::Geometry;

InsertsExportForm::InsertsExportForm(Recordset::Ref rset)
: Form(NULL), _record_set(rset), _vbox(false), _path_box(true), _button_box(true)
{
  set_content(&_vbox);
  
  _vbox.set_padding(MF_WINDOW_PADDING);
  _vbox.set_spacing(MF_TABLE_ROW_SPACING);
  _vbox.add(&_table, true, true);
  _vbox.add(&_button_box, false, true);
  
  _table.set_row_spacing(MF_TABLE_ROW_SPACING);
  _table.set_column_spacing(MF_TABLE_COLUMN_SPACING);
  _table.set_row_count(4);
  _table.set_column_count(2);
  
  _button_box.set_spacing(MF_BUTTON_SPACING);
  
  _type_label.set_text_align(mforms::MiddleRight);
  _type_label.set_text(_("File Format:"));
  _table.add(&_type_label, 0, 1, 1, 2, mforms::HFillFlag);
  _table.add(&_type_sel, 1, 2, 1, 2, mforms::HFillFlag|mforms::HExpandFlag);

  
  _type_sel.clear();
  _type_sel.signal_changed().connect(sigc::mem_fun(this, &InsertsExportForm::change_type));
  std::vector<std::string> formats_descs;
  _record_set->data_storages_for_export(_format_names, formats_descs);
  for (size_t i= 0; i < formats_descs.size(); i++)
  {
    _type_sel.add_item(formats_descs[i]);
      
    char *extension= g_utf8_strdown(_format_names[i].data(), _format_names[i].length());
    char *sep;
    sep = strchr(extension, '_');
    if (sep) *sep= 0;
    sep = strchr(extension, ' ');
    if (sep) *sep= 0;
      
    _format_exts.push_back(std::string(".").append(extension));
    g_free(extension);
  }
  
  _path_label.set_text_align(mforms::MiddleRight);
  _path_label.set_text(_("File Path:"));
  _table.add(&_path_label, 0, 1, 2, 3, mforms::HFillFlag);
  _path_button.set_text("...");
  _path_button.signal_clicked().connect(sigc::mem_fun(this, &InsertsExportForm::browse_path));
  _path_box.set_spacing(4);
  _path_box.add(&_path_text, true, true);
  _path_box.add(&_path_button, false, false);
  _table.add(&_path_box, 1, 2, 2, 3, mforms::HFillFlag|mforms::HExpandFlag);

  _ok_button.set_text(_("Export"));
  _ok_button.set_padding(Padding(4, 0, 4, 0));
  _cancel_button.set_text(_("Cancel"));
  _cancel_button.set_padding(Padding(4, 0, 4, 0));
  
  Utilities::add_end_ok_cancel_buttons(&_button_box, &_ok_button, &_cancel_button);
  
  set_title(_("Export Inserts Data to File"));
  set_size(420, 220);
  center();
}

std::string InsertsExportForm::add_extension(const std::string &orig_path)
{
  std::string path = orig_path;
  if (!path.empty())
  {    
    // strip old extension
    for (std::vector<std::string>::const_iterator e = _format_exts.begin(); e != _format_exts.end(); ++e)
    {
      if (path.size() > e->size() && path.substr(path.size()-e->size()) == *e)
      {
        path = path.substr(0, path.size()-e->size());
        break;
      }
    }
    
    path.append(_format_exts[_type_sel.get_selected_index()]);
  }
  return path;
}

void InsertsExportForm::browse_path()
{
  mforms::FileChooser chooser(mforms::SaveFile);
  
  chooser.set_title(_("Export to File"));
  chooser.set_directory(_path_text.get_string_value());
  if (chooser.run_modal())
  {
    std::string path = add_extension(chooser.get_path());
    _path_text.set_value(path);
  }
}

void InsertsExportForm::change_type()
{
  std::string path = add_extension(_path_text.get_string_value());
  _path_text.set_value(path);
}


void InsertsExportForm::run()
{  
retry:
  if (run_modal(&_ok_button, &_cancel_button))
  {
    int i = _type_sel.get_selected_index();
    std::string format= _format_names[i < 0 ? 0 : i];
    
    if (g_file_test(_path_text.get_string_value().c_str(), G_FILE_TEST_EXISTS))
    {
      if (mforms::Utilities::show_warning(_("Export Inserts Data"),
        base::strfmt(_("File %s already exists, do you want to replace it?"),
        _path_text.get_string_value().c_str()), _("Replace"), _("Cancel"), "") != mforms::ResultOk)
        goto retry;
    }
 
    Recordset_data_storage::Ref dataStorage= _record_set->data_storage_for_export(format);
    
    if (dynamic_cast<Recordset_text_storage*>(dataStorage.get()))
    {
      Recordset_text_storage *textStorage= dynamic_cast<Recordset_text_storage*> (dataStorage.get());
      textStorage->data_format(format);
      textStorage->file_path(_path_text.get_string_value());

      Recordset_sql_storage* storage = dynamic_cast<Recordset_sql_storage*>(_record_set->data_storage().get());
      textStorage->parameter_value("GENERATOR_QUERY", _record_set->generator_query());
      textStorage->parameter_value("GENERATE_DATE", bec::fmttime(time(NULL), DATETIME_FMT));
      textStorage->parameter_value("TABLE_NAME", storage->table_name());
    }
    
    try
    {
      dataStorage->serialize(_record_set);
    }
    catch (std::exception &exc)
    {
      mforms::Utilities::show_error(_("Export Inserts Data"), exc.what(), _("OK"));
    }
  }
}
