#include "plugin_editor_base.h"
#include "grt/editor_base.h"
#include "sql_editor_fe.h"
#include "base/string_utilities.h"

using base::strfmt;


#define TEXT_UPDATE_TIMER 500

//------------------------------------------------------------------------------
PluginEditorBase::PluginEditorBase(grt::Module *module, bec::GRTManager *grtm, const grt::BaseListRef &args, const char* glade_file)
  : GUIPluginBase(module), _grtm(grtm)
  , _xml(0)
  , _editor_notebook(0)
  , _live_object_editor_decorator_xml(0)
  , _live_object_editor_decorator_control(0)
{
  _refreshing= false;
  set_shadow_type(Gtk::SHADOW_NONE);

  if (glade_file)
    _xml= Gtk::Builder::create_from_file(grtm->get_data_file_path(glade_file));
}

//------------------------------------------------------------------------------
PluginEditorBase::~PluginEditorBase()
{  
}

//------------------------------------------------------------------------------
void PluginEditorBase::refresh_form_data()
{
  if (_refreshing)
    return;

  _refreshing= true;

  decorate_object_editor();

  do_refresh_form_data();
  
  _refreshing= false;
}

//------------------------------------------------------------------------------
bool PluginEditorBase::is_editing_live_object()
{
  return get_be()->is_editing_live_object();
}
//------------------------------------------------------------------------------
void PluginEditorBase::decorate_object_editor()
{
  if (!_editor_notebook)
    return;

  if (is_editing_live_object())
  {
    if (!_live_object_editor_decorator_control)
    {
      _live_object_editor_decorator_xml= Gtk::Builder::create_from_file(_grtm->get_data_file_path("modules/data/live_editor_decoration.glade"));
      _live_object_editor_decorator_xml->get_widget("live_editor_decorator_vpane", _live_object_editor_decorator_control);
      _live_object_editor_decorator_xml->get_widget("live_editor_placeholder", _live_editor_placeholder);

      _live_object_editor_decorator_xml->get_widget("live_editor_log_text", _live_editor_log_text);
      _live_editor_log_text->get_buffer()->set_text("DBMS feedback messages will go here upon applying changes.");
      //_live_object_editor_decorator_xml->get_widget("live_editor_progressbar", _live_editor_progressbar);

      Gtk::Button *apply_live_edtior_button= 0;
      _live_object_editor_decorator_xml->get_widget("apply_live_edtior_button", apply_live_edtior_button);
      apply_live_edtior_button->signal_pressed().connect(sigc::mem_fun(this, &PluginEditorBase::apply_changes_to_live_object));
      Gtk::Button *revert_live_edtior_button= 0;
      _live_object_editor_decorator_xml->get_widget("revert_live_edtior_button", revert_live_edtior_button);
      revert_live_edtior_button->signal_pressed().connect(sigc::mem_fun(this, &PluginEditorBase::revert_changes_to_live_object));
      Gtk::Button *close_live_edtior_button= 0;
      _live_object_editor_decorator_xml->get_widget("close_live_edtior_button", close_live_edtior_button);
      close_live_edtior_button->signal_pressed().connect(sigc::mem_fun(this, &PluginEditorBase::close_live_object_editor));
    }

    if (_editor_notebook->get_parent() != _live_object_editor_decorator_control)
    {
      _editor_notebook->reparent(*_live_editor_placeholder);
      _live_object_editor_decorator_control->reparent(*this);
      _live_object_editor_decorator_control->show();
    }
  }
  else
  {
    if (_editor_notebook->get_parent() != this)
    {
      if (_editor_notebook->get_parent() == _live_object_editor_decorator_control)
        _live_object_editor_decorator_control->unparent();
      _editor_notebook->reparent(*this);
    }
  }
}
//------------------------------------------------------------------------------
void PluginEditorBase::apply_changes_to_live_object()
{
  _live_editor_log_text->get_buffer()->set_text("");
  _live_editor_log_text->modify_text(Gtk::STATE_NORMAL, Gdk::Color("black"));
  //_live_editor_progressbar->set_fraction(0);
  bec::BaseEditor *editor= get_be();
  editor->on_live_object_change_error= sigc::mem_fun(this, &PluginEditorBase::on_live_object_change_error);
  editor->on_live_object_change_progress= sigc::mem_fun(this, &PluginEditorBase::on_live_object_change_progress);
  editor->on_live_object_change_statistics= sigc::mem_fun(this, &PluginEditorBase::on_live_object_change_statistics);
  editor->apply_changes_to_live_object();
}
//------------------------------------------------------------------------------
void PluginEditorBase::revert_changes_to_live_object()
{
  get_be()->refresh_live_object();
}
//------------------------------------------------------------------------------
void PluginEditorBase::close_live_object_editor()
{
  if (get_be()->can_close())
  {
    Gtk::Notebook *notebook= dynamic_cast <Gtk::Notebook *> (get_parent());
    if (notebook)
    {
      hide();
      notebook->remove_page(*this);
      bool visible= false;
      for (int c= notebook->get_n_pages(), i= 0; i < c; i++)
      {
        if (notebook->get_nth_page(i)->is_visible())
        {
          visible= true;
          break;
        }
      }
      if (!visible)
        notebook->hide();
    }
    else
    {
      delete get_toplevel();
    }
  }
}
//------------------------------------------------------------------------------
int PluginEditorBase::on_live_object_change_error(long long err_code, const std::string &err_msg, const std::string &sql)
{
  _live_editor_log_text->modify_text(Gtk::STATE_NORMAL, Gdk::Color("red"));
  std::string err_code_msg= (err_code == -1) ? "" : strfmt("SQL Error %lli: ", err_code);
  std::string message= strfmt("%s%s\nSQL: %s\n\n", err_code_msg.c_str(), err_msg.c_str(), sql.c_str());
  std::string text= _live_editor_log_text->get_buffer()->get_text("");
  _live_editor_log_text->get_buffer()->set_text(text + message);
  return 0;
}
//------------------------------------------------------------------------------
int PluginEditorBase::on_live_object_change_progress(float progress)
{
  //_live_editor_progressbar->set_fraction(progress);
  return 0;
}
//------------------------------------------------------------------------------
int PluginEditorBase::on_live_object_change_statistics(long success_count, long fail_count)
{
  //_live_editor_progressbar->set_fraction(1.);
  if (0 == fail_count)
    _live_editor_log_text->get_buffer()->set_text("Changes have been applied OK");
  return 0;
}
//------------------------------------------------------------------------------
void PluginEditorBase::add_option_combo_change_handler(Gtk::ComboBox *combo, const std::string &option, const sigc::slot<void,std::string,std::string> &setter)
{
  combo->signal_changed().connect(sigc::bind(sigc::mem_fun(this, &PluginEditorBase::combo_changed), combo, option, setter));
}

//------------------------------------------------------------------------------
sigc::connection PluginEditorBase::add_entry_change_timer(Gtk::Entry *entry, const sigc::slot<void,std::string> &setter)
{
  TextChangeTimer timer;

  timer.commit= sigc::bind(sigc::mem_fun(this, &PluginEditorBase::entry_timeout), entry);
  timer.setter= setter;
  _timers[entry]= timer;

  return entry->signal_changed().connect(sigc::bind(sigc::mem_fun(this, &PluginEditorBase::entry_changed), entry));
}

//------------------------------------------------------------------------------
sigc::connection PluginEditorBase::add_text_change_timer(Gtk::TextView *text, const sigc::slot<void,std::string> &setter)
{
  TextChangeTimer timer;

  timer.commit= sigc::bind(sigc::mem_fun(this, &PluginEditorBase::text_timeout), text);
  timer.setter= setter;
  _timers[text]= timer;

  return text->get_buffer()->signal_changed().connect(sigc::bind(sigc::mem_fun(this, &PluginEditorBase::text_changed), text));
}

//------------------------------------------------------------------------------
sigc::connection PluginEditorBase::add_sqleditor_text_change_timer(SqlEditorFE *text, const sigc::slot<void,std::string> &setter)
{
  TextChangeTimer timer;

  timer.commit= sigc::bind(sigc::mem_fun(this, &PluginEditorBase::sqleditor_text_timeout), text);
  timer.setter= setter;
  _timers[&text->widget()]= timer;

  return text->signal_text_changed().connect(sigc::bind(sigc::mem_fun(this, &PluginEditorBase::sqleditor_text_changed), text));
}

//------------------------------------------------------------------------------
bool PluginEditorBase::entry_timeout(Gtk::Entry *entry)
{
  _timers[entry].setter(entry->get_text());
  return false;
}

//------------------------------------------------------------------------------
bool PluginEditorBase::text_timeout(Gtk::TextView *text)
{
  _timers[text].setter(text->get_buffer()->get_text());
  return false;
}

//------------------------------------------------------------------------------
bool PluginEditorBase::sqleditor_text_timeout(SqlEditorFE *text)
{
  _timers[&text->widget()].setter(text->get_text());
  return false;
}

//------------------------------------------------------------------------------
void PluginEditorBase::entry_changed(Gtk::Entry *entry)
{
  if (!_refreshing)
  {
    if (_timers[entry].conn)
      _timers[entry].conn.disconnect();

    _timers[entry].conn= Glib::signal_timeout().connect(_timers[entry].commit, TEXT_UPDATE_TIMER);
  }
}

//------------------------------------------------------------------------------
void PluginEditorBase::text_changed(Gtk::TextView *text)
{
  if (!_refreshing)
  {
    if (_timers[text].conn)
      _timers[text].conn.disconnect();

    _timers[text].conn= Glib::signal_timeout().connect(_timers[text].commit, TEXT_UPDATE_TIMER);
  }
}

//------------------------------------------------------------------------------
void PluginEditorBase::sqleditor_text_changed(SqlEditorFE *text)
{
  if (!_refreshing)
  {
    Gtk::Widget *sql_widget = &text->widget();
    if (_timers[sql_widget].conn)
      _timers[sql_widget].conn.disconnect();

    _timers[sql_widget].conn= Glib::signal_timeout().connect(_timers[sql_widget].commit, TEXT_UPDATE_TIMER);
  }
}

//------------------------------------------------------------------------------
void PluginEditorBase::commit_text_changes()
{
  for (std::map<Gtk::Widget*,TextChangeTimer>::iterator iter= _timers.begin();
       iter != _timers.end(); ++iter)
  {
    if (iter->second.conn)
    {
      iter->second.commit();
      iter->second.conn.disconnect();
    }
  }
}


//------------------------------------------------------------------------------
void PluginEditorBase::combo_changed(Gtk::ComboBox *combo, const std::string &option, const sigc::slot<void,std::string,std::string> &setter)
{
  if (!_refreshing)
  {
    Gtk::TreeModel::iterator iter = combo->get_active();
    if (iter)
    {
      Gtk::TreeRow row= *iter;
      Glib::ustring text;
      row.get_value(0, text);
      setter(option, text);
    }
  }
}

bool PluginEditorBase::should_close_on_delete_of(const std::string &oid)
{
  return get_be()->should_close_on_delete_of(oid); 
}
