#include "sqlide/recordset_view.h"
#include "sqlide/utils_fe.h"
#include "linux_utilities/gtk_helpers.h"
#include "base/string_utilities.h"
#include "linux_utilities/toolbar_manager.h"

using base::strfmt;


RecordsetView * RecordsetView::create(Recordset::Ref model, Gtk::Container *parent)
{
  RecordsetView *view= Gtk::manage(new RecordsetView(model));
  view->init();
  if (parent)
    parent->add(*view);
  return view;
}

RecordsetView::RecordsetView(Recordset::Ref model)
:
_grid(NULL)
{
  _filter_entry= 0;
  this->model(model);
}

RecordsetView::~RecordsetView()
{
}

void RecordsetView::init()
{
  bec::GRTManager *grtm= _model->grtm();
  Glib::RefPtr<Gtk::Builder> xml(Gtk::Builder::create_from_file(grtm->get_data_file_path("recordset_view.glade")));

  Gtk::Container *recordset_view = 0;
  xml->get_widget("recordset_view", recordset_view);
  Gtk::ScrolledWindow *recordset_placeholder = 0;
  xml->get_widget("recordset_placeholder", recordset_placeholder);

  _grid= GridView::create(_model);
  
  // get_selected() doesn't work work with SELECTION_MULTIPLE
  _grid->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
  
  recordset_placeholder->add(*_grid);

  recordset_view->reparent(*this);
  set_shadow_type(Gtk::SHADOW_NONE);

  //WIDGET_ (xml, messages_view) // auto-replaced
  xml->get_widget("messages_view", _messages_view);
  //WIDGET_ (xml, messages_window) // auto-replaced
  xml->get_widget("messages_window", _messages_window);
  _messages_window->property_height_request()= 1;

  _toolbar_box = 0;
  xml->get_widget("recordset_toolbar", _toolbar_box);
  
  ActionList &action_list= _model->action_list();
  action_list.register_action("record_first", sigc::mem_fun(this, &RecordsetView::on_goto_first_row_btn_clicked));
  action_list.register_action("record_back", sigc::mem_fun(this, &RecordsetView::on_record_prev));
  action_list.register_action("record_next", sigc::mem_fun(this, &RecordsetView::on_record_next));
  action_list.register_action("record_last", sigc::mem_fun(this, &RecordsetView::on_goto_last_row_btn_clicked));
  //action_list.register_action("record_fetch_all", sigc::mem_fun(this, &RecordsetView::));
  //action_list.register_action("record_wrap_vertical", sigc::mem_fun(this, &RecordsetView::));
  action_list.register_action("record_sort_asc", sigc::mem_fun(this, &RecordsetView::on_record_sort_asc));
  action_list.register_action("record_sort_desc", sigc::mem_fun(this, &RecordsetView::on_record_sort_desc));
  action_list.register_action("record_edit", sigc::mem_fun(this, &RecordsetView::on_record_edit));
  action_list.register_action("record_add", sigc::mem_fun(this, &RecordsetView::on_record_add));
  action_list.register_action("record_del", sigc::mem_fun(this, &RecordsetView::on_record_del));
  action_list.register_action("record_save", sigc::mem_fun(this, &RecordsetView::on_commit_btn_clicked));
  action_list.register_action("record_discard", sigc::mem_fun(this, &RecordsetView::on_rollback_btn_clicked));
  //action_list.register_action("record_refresh", sigc::mem_fun(this, &RecordsetView::));

  _grid->signal_event().connect(sigc::mem_fun(this, &RecordsetView::on_event));

  update_toolbar();
  
  show();
}

void RecordsetView::model(Recordset::Ref value)
{
  _model= value;
  _model->refresh_ui_cb= sigc::mem_fun(this, &RecordsetView::refresh);
  _model->refresh_ui_status_bar_signal.connect(sigc::mem_fun(this, &RecordsetView::update_toolbar));
  _model->task->msg_cb(sigc::mem_fun(this, &RecordsetView::process_task_msg));

  if (_grid)
    _grid->model(_model);
}

int RecordsetView::process_task_msg(int msg_type, const std::string &message, const std::string &detail)
{
  _messages_view->get_buffer()->set_text(message);

  _messages_window->property_height_request()= message.empty() ? 1 :
    ((1 < _messages_window->get_height()) ? _messages_window->get_height() : 26);

  switch ((grt::MessageType)msg_type)
  {
    case grt::ErrorMsg:
      _messages_view->modify_text(Gtk::STATE_NORMAL, Gdk::Color("red"));
      break;
    case grt::WarningMsg:
    case grt::InfoMsg:
      _messages_view->modify_text(Gtk::STATE_NORMAL, Gdk::Color("black"));
      break;
    default:
      break;
  }

  return 0;
}


Gtk::Widget *RecordsetView::create_toolbar_item(const bec::ToolbarItem &item)
{
  if (item.command == "filter")
  {
    Gtk::Entry *entry= _filter_entry = Gtk::manage(new Gtk::Entry());
    entry->set_size_request(100, -1);
    entry->signal_key_release_event().connect(sigc::bind(sigc::mem_fun(this, &RecordsetView::on_data_search_entry_key_pressed), entry));
    return entry;
  }
  
  return 0;
}


void RecordsetView::update_toolbar()
{
  Glib::ustring text;

  // force a repaint of the records in case the contents have changed too
  _grid->queue_draw();


  if (_filter_entry)
    text = _filter_entry->get_text();
  
  ToolbarManager::rebuild_toolbar(_toolbar_box,
                                  _model->get_toolbar_items(),
                                  sigc::mem_fun(this, &RecordsetView::create_toolbar_item),
                                  sigc::mem_fun(this, &RecordsetView::activate_toolbar_item)
                                 );
  
  if (_filter_entry)
    _filter_entry->set_text(text);
}


bool RecordsetView::activate_toolbar_item(const std::string &action)
{
  bool r = _model->action_list().trigger_action(action);
  if (r)
    update_toolbar();
  return r;
}


void RecordsetView::reset()
{
  _model->reset();
}

int RecordsetView::refresh()
{
  _grid->refresh(true);
  update_toolbar();

  return 0;
}

bool RecordsetView::on_event(GdkEvent *event)
{
  bool processed= false;

  if (GDK_BUTTON_PRESS == event->type)
  {
    if (3 == event->button.button)
    {
      Gtk::StateType state= _model->is_readonly() ? Gtk::STATE_INSENSITIVE : Gtk::STATE_NORMAL;
      std::vector<int> rows;
      Gtk::TreePath path;
      Gtk::TreeViewColumn *tcolumn;
      int cell_x, cell_y;

      if (_grid->get_path_at_pos(event->button.x, event->button.y, path, tcolumn, cell_x, cell_y))
      {
        _grid->get_selection()->select(path);
        int column = 0;

        std::vector<Gtk::TreeViewColumn*> columns(_grid->get_columns());
        for (int i = 0; i < (int)columns.size(); i++)
        {
          if (columns[i] == tcolumn)
          {
            column = i;
            break;
          }
        }
        int row = path.front();//_grid->get_selection()->get_selected();
        if (row >= 0 && column > 0)
        {
          rows.push_back(row);
          
          run_popup_menu(_model->get_popup_menu_items(rows, column-1),
                         event->button.time,
                         sigc::hide_return(sigc::bind(sigc::mem_fun(_model.get(), &Recordset::activate_popup_menu_item), rows, column-1)));
        }
      }
      processed= true;
    }
  }

  if (!processed)
    processed= Gtk::Frame::on_event(event);

  return processed;
}



bool RecordsetView::on_data_search_entry_key_pressed(GdkEventKey *event, Gtk::Entry *search_entry)
{
  switch (event->keyval)
  {
  case GDK_Return:
  case GDK_KP_Enter:
  case GDK_ISO_Enter:
  case GDK_3270_Enter:
  std::string value= search_entry->get_text();
  if (value.empty())
    _model->reset_data_search_string();
  else
    _model->set_data_search_string(value);
    return true;
  }
  return false;
}

void RecordsetView::on_commit_btn_clicked()
{
  save_changes();
}

void RecordsetView::on_rollback_btn_clicked()
{
  revert_changes();
}

bool RecordsetView::has_changes()
{
  return _model->has_pending_changes();
}

void RecordsetView::save_changes()
{
  _model->apply_changes();
}

void RecordsetView::revert_changes()
{
  _model->rollback();
}

void RecordsetView::on_goto_first_row_btn_clicked()
{
  if (_model->row_count() == 0)
    return;
    
  Gtk::TreePath tree_path(1);
  tree_path[0]= 0;
  _grid->set_cursor(tree_path);
}

void RecordsetView::on_goto_last_row_btn_clicked()
{
  Gtk::TreePath tree_path(1);
  size_t row_count= _model->row_count();
  if (row_count == 0)
    return;
  tree_path[0] = row_count - 1;
  _grid->set_cursor(tree_path);
}

void RecordsetView::on_record_prev()
{
  Gtk::TreeModel::Path path;
  Gtk::TreeViewColumn *column= NULL;
  _grid->get_cursor(path, column);
  if (!column)
    return;
  path.prev();
  _grid->set_cursor(path, *column);
}

void RecordsetView::on_record_next()
{
  Gtk::TreeModel::Path path;
  Gtk::TreeViewColumn *column= NULL;
  _grid->get_cursor(path, column);
  if (!column)
    return;
  path.next();
  _grid->set_cursor(path, *column);
}

void RecordsetView::on_record_edit()
{
  if (_model->is_readonly())
    return;
  Gtk::TreeModel::Path path;
  Gtk::TreeViewColumn *column= NULL;
  _grid->get_cursor(path, column);
  if (!column)
    return;
  _grid->set_cursor(path, *column, true);
}

void RecordsetView::on_record_add()
{
  if (_model->is_readonly())
    return;
  Gtk::TreePath tree_path(1);
  size_t row_count= _model->row_count();
  if (row_count == 0)
    return;
  tree_path[0] = row_count;
  _grid->set_cursor(tree_path);
  on_record_edit();
}

void RecordsetView::on_record_del()
{
  if (_model->is_readonly())
    return;
  _grid->delete_selected_rows();
}

void RecordsetView::on_record_sort_asc()
{
  int row, col;
  _grid->current_cell(row, col);
  if (col < 0)
    return;
  _grid->sort_by_column(col, -1, true);
}

void RecordsetView::on_record_sort_desc()
{
  int row, col;
  _grid->current_cell(row, col);
  if (col < 0)
    return;
  _grid->sort_by_column(col, 1, true);
}
