/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Text.RegularExpressions;
using System.Windows.Forms;

using MySQL.Grt;
using MySQL.Utilities.Properties;


namespace MySQL.Controls
{
  public class CustomDataGridViewTextBoxColumn : DataGridViewTextBoxColumn
  {
    public delegate int AdditionalColumnWidthDelegate(DataGridViewColumn sender);
    public AdditionalColumnWidthDelegate AdditionalColumnWidth;
    public override int GetPreferredWidth(DataGridViewAutoSizeColumnMode autoSizeColumnMode, bool fixedHeight)
    {
      int prefferedWidth = base.GetPreferredWidth(autoSizeColumnMode, fixedHeight);
      if (AdditionalColumnWidth != null)
        prefferedWidth += AdditionalColumnWidth(this);
      return prefferedWidth;
    }
  }

  public class GridView : DataGridView
  {
    private GridModel model;
    private bool refreshing = false;
    private Bitmap fieldNullBitmap;
    private Bitmap fieldBlobBitmap;
    private bool wrapMode = false;
    private bool autoScroll = false;
    private bool allowAutoResizeColumns = true;
    // defines the grid color
    private Color gridColor = Color.FromArgb(240, 240, 240);
    // defines the alternating row color
    private Color gridAlternateRowColor = Color.FromArgb(248, 248, 248);

    public CustomDataGridViewTextBoxColumn.AdditionalColumnWidthDelegate AdditionalColumnWidth;

    protected override void Dispose(bool disposing)
    {
      if (disposing && (model != null))
      {
        model.Dispose();
        model = null;
      }
      base.Dispose(disposing);
    }

    public GridView(GridModel model)
    {
      fieldNullBitmap = Resources.field_overlay_null;
      fieldBlobBitmap = Resources.field_overlay_blob;

      Model = model;

      Dock = DockStyle.Fill;
      BorderStyle = BorderStyle.None;
      BackgroundColor = SystemColors.Window;

      EditMode = DataGridViewEditMode.EditOnKeystrokeOrF2;
      VirtualMode = true;
      AllowUserToOrderColumns = true;
      AllowUserToResizeRows = false;
      ColumnHeadersHeightSizeMode = DataGridViewColumnHeadersHeightSizeMode.DisableResizing;
      ShowCellToolTips = true;

      RowHeadersWidth -= 15;

			// Set grid color
			GridColor = gridColor;
    }


    public GridModel Model
    {
      set { model = value; }
      get { return model; }
    }


    public void ProcessModelChange()
    {
      SuspendLayout();
      refreshing = true;

      ReadOnly = model.is_readonly();

      CustomDataGridViewTextBoxColumn[] columns = new CustomDataGridViewTextBoxColumn[model.get_column_count()];
      for (int i= 0; i < model.get_column_count(); i++)
      {
        CustomDataGridViewTextBoxColumn column= new CustomDataGridViewTextBoxColumn();
        column.AdditionalColumnWidth = AdditionalColumnWidth;
        column.HeaderText = model.get_column_caption(i);
        Type columnValueType;
        switch (model.get_column_type(i))
        {
          case GridModel.ColumnType.DatetimeType:
            columnValueType = typeof(DateTime);//! needs corresponding set_field or conversion to string. see PushValue
            column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.TopLeft;
            break;
          case GridModel.ColumnType.NumericType:
            columnValueType = typeof(long);
            column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.TopRight;
            break;
          case GridModel.ColumnType.FloatType:
            columnValueType = typeof(double);
            column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.TopRight;
            break;
          default:
            columnValueType = typeof(string);
            column.DefaultCellStyle.Alignment = DataGridViewContentAlignment.TopLeft;
            break;
        }
        if (null != columnValueType)
          column.ValueType = columnValueType;

        column.SortMode = DataGridViewColumnSortMode.Programmatic;
        column.FillWeight = 0.001F;

        columns[i]= column;
      }

      try
      {
        Columns.Clear();
      }
      catch (System.InvalidOperationException)
      {
        CancelEdit();
        Columns.Clear();
      }

      Columns.AddRange(columns);

      // restore sorting glyphs
      {
        List<int> indexes;
        List<int> orders;
        model.sort_columns(out indexes, out orders);
        int l = 0;
        foreach (int n in indexes)
        {
          DataGridViewColumn column = Columns[n];
          column.HeaderCell.SortGlyphDirection = (orders[l] == 1) ? SortOrder.Ascending : SortOrder.Descending;
          ++l;
        }
      }

      refreshing = false;
      ProcessModelRowsChange();
      ResumeLayout();
    }


    public int ProcessModelRowsChange()
    {
      SuspendLayout();
      refreshing = true;

      int firstDisplayedScrollingRowIndex = AutoScroll ? -1 : 0;
      if (AutoScroll && (RowCount > 0))
      {
        if (Rows[RowCount-1].Displayed)
          firstDisplayedScrollingRowIndex = FirstDisplayedScrollingRowIndex;
        else
          firstDisplayedScrollingRowIndex = -1;
      }

      RowCount = model.count();
      if (allowAutoResizeColumns)
        AutoResizeColumns(DataGridViewAutoSizeColumnsMode.DisplayedCells, true);
      if (wrapMode)
        AutoResizeRows(DataGridViewAutoSizeRowsMode.DisplayedCells, true);

      if (0 != firstDisplayedScrollingRowIndex)
      {
        if (FirstDisplayedScrollingRowIndex >= RowCount)
          firstDisplayedScrollingRowIndex = -1;
        if ((-1 == firstDisplayedScrollingRowIndex) && (RowCount > 0))
        {
          firstDisplayedScrollingRowIndex = RowCount - 1;
          FirstDisplayedScrollingRowIndex = firstDisplayedScrollingRowIndex;
        }
      }

      ClearSelection();

      Refresh();
      refreshing = false;
      ResumeLayout();
      return 0;
    }


    public void SetRowSelected(int rowIndex)
    {
      if (0 == RowCount)
        return;
      DataGridViewRow row = Rows[rowIndex];
      if (0 == row.Cells.Count)
        return;
      int columnIndex = (null != CurrentCell) ? CurrentCell.ColumnIndex : 0;
      CurrentCell = row.Cells[columnIndex];

      if (0 == rowIndex)
        VerticalScrollBar.Value = VerticalScrollBar.Minimum;
      else if ((RowCount-1) == rowIndex)
        VerticalScrollBar.Value = VerticalScrollBar.Maximum;
    }


    public bool AllowAutoResizeColumns
    {
      get { return allowAutoResizeColumns; }
      set { allowAutoResizeColumns = value; }
    }


    public bool WrapMode
    {
      get { return wrapMode; }
      set
      {
        if (value == wrapMode)
          return;
        wrapMode = value;
        if (wrapMode)
          RowPrePaint += new DataGridViewRowPrePaintEventHandler(OnRowPrePaint);
        else
          RowPrePaint -= OnRowPrePaint;
        DefaultCellStyle.WrapMode = wrapMode ? DataGridViewTriState.True : DataGridViewTriState.False;
      }
    }


    public bool AutoScroll
    {
      get { return autoScroll; }
      set { autoScroll = value; }
    }

    private void OnRowPrePaint(Object sender, DataGridViewRowPrePaintEventArgs e)
    {
      AutoResizeRow(e.RowIndex);
    }


    protected override void OnCellPainting(DataGridViewCellPaintingEventArgs e)
    {
      base.OnCellPainting(e);

      if ((0 > e.ColumnIndex) || (0 > e.RowIndex))
        return;

      int iconId = model.get_field_icon(new NodeId(e.RowIndex), e.ColumnIndex, IconSize.Icon16);
      Image icon = GrtIconManager.get_instance().get_icon(iconId);
      if (null != icon)
      {
        Rectangle rect = e.CellBounds;
        rect.Height = icon.Height;
        rect.Width = icon.Width;
        switch (e.CellStyle.Alignment)
        {
          case DataGridViewContentAlignment.TopRight:
          case DataGridViewContentAlignment.MiddleRight:
          case DataGridViewContentAlignment.BottomRight:
            rect.X += e.CellBounds.Width - rect.Width;
            rect.X -= 1; // cell border width (required only for right alignment)
            break;
          case DataGridViewContentAlignment.TopCenter:
          case DataGridViewContentAlignment.MiddleCenter:
          case DataGridViewContentAlignment.BottomCenter:
            rect.X += (e.CellBounds.Width - rect.Width) / 2;
            break;
        }
        e.PaintBackground(e.CellBounds, true);
        e.Graphics.DrawImageUnscaledAndClipped(icon, rect);
        e.Handled = true;
      }
    }

    protected override void OnCellFormatting(DataGridViewCellFormattingEventArgs e)
    {
      base.OnCellFormatting(e);

			// Alter background color for every odd row
      if (1 == e.RowIndex % 2)
				e.CellStyle.BackColor = gridAlternateRowColor;
    }

    protected override void OnReadOnlyChanged(EventArgs e)
    {
      base.OnReadOnlyChanged(e);

      AllowUserToAddRows = !ReadOnly;
      AllowUserToDeleteRows = !ReadOnly;
    }

    protected override void OnRowsRemoved(DataGridViewRowsRemovedEventArgs e)
    {
      base.OnRowsRemoved(e);

      if (refreshing || model == null)
        return;
      for (int i = 0; i < e.RowCount; i++)
        model.delete_node(new NodeId(e.RowIndex));
    }

    protected override void OnNewRowNeeded(DataGridViewRowEventArgs e)
    {
      base.OnNewRowNeeded(e);

      if (refreshing)
        return;
    }

    protected override void OnCancelRowEdit(QuestionEventArgs e)
    {
      base.OnCancelRowEdit(e);

      e.Response = (model.count() < RowCount);
    }

    protected override void OnCellBeginEdit(DataGridViewCellCancelEventArgs e)
    {
      base.OnCellBeginEdit(e);

      switch (model.get_column_type(e.ColumnIndex))
      {
        case GridModel.ColumnType.BlobType:
          e.Cancel = true;
          break;
      }
      if (!e.Cancel)
        model.set_edited_field(e.RowIndex, e.ColumnIndex);
    }

    protected override void OnCellEndEdit(DataGridViewCellEventArgs e)
    {
      base.OnCellEndEdit(e);

      model.set_edited_field(-1, -1);
    }

    protected override void OnCellValuePushed(DataGridViewCellValueEventArgs e)
    {
      base.OnCellValuePushed(e);

      if (refreshing)
        return;

      if (null == e.Value)
      {
        if (typeof(string) != Columns[e.ColumnIndex].ValueType)
        {
          model.set_field_null(new NodeId(e.RowIndex), e.ColumnIndex);
        }
        else
          e.Value = "";
      }

      if (null != e.Value)
      {
        Type t = e.Value.GetType();
        if (typeof(string) == t)
          model.set_field(new NodeId(e.RowIndex), e.ColumnIndex, (string)e.Value);
        else if (typeof(double) == t)
          model.set_field(new NodeId(e.RowIndex), e.ColumnIndex, (double)e.Value);
        else if (typeof(int) == t)
          model.set_field(new NodeId(e.RowIndex), e.ColumnIndex, (int)e.Value);
        else if (typeof(long) == t)
          model.set_field(new NodeId(e.RowIndex), e.ColumnIndex, (long)e.Value);
      }
    }

    protected override void OnCellValueNeeded(DataGridViewCellValueEventArgs e)
    {
      base.OnCellValueNeeded(e);

      String value;
      model.get_field_repr(new NodeId(e.RowIndex), e.ColumnIndex, out value);
      e.Value = value;
    }

    protected override void OnCellToolTipTextNeeded(DataGridViewCellToolTipTextNeededEventArgs e)
    {
      base.OnCellToolTipTextNeeded(e);

      if (wrapMode || (0 > e.RowIndex) || (0 > e.ColumnIndex))
        return;

      String value;
      model.get_field_repr(new NodeId(e.RowIndex), e.ColumnIndex, out value);

      DataGridViewCell cell = Rows[e.RowIndex].Cells[e.ColumnIndex];
      if ((cell.Size.Width < cell.PreferredSize.Width)
          || (cell.Size.Height < cell.PreferredSize.Height))
      {
        e.ToolTipText = value;
      }
      else
      {
        Regex r = new Regex("\n");
        Match m = r.Match(value);
        if (m.Success)
          e.ToolTipText = value;
      }
    }

    protected override void OnDataError(bool displayErrorDialogIfNoHandler, DataGridViewDataErrorEventArgs e)
    {
      base.OnDataError(displayErrorDialogIfNoHandler, e);

      e.Cancel = true;
    }

    protected override void OnCellLeave(DataGridViewCellEventArgs e)
    {
      base.OnCellLeave(e);

      if (IsCurrentCellInEditMode)
        EditMode = DataGridViewEditMode.EditOnEnter;
      else
        EditMode = DataGridViewEditMode.EditOnKeystrokeOrF2;
    }

    protected override bool ProcessDataGridViewKey(KeyEventArgs e)
    {
      switch (e.KeyCode)
      {
        case Keys.Home:
          FirstDisplayedScrollingColumnIndex = 0;
          break;

        case Keys.End:
          FirstDisplayedScrollingColumnIndex = Columns.Count - 1;
          break;
      }
      return base.ProcessDataGridViewKey(e);
    }

    protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
    {
      Keys key = keyData & Keys.KeyCode;

      if (CurrentCell != null && CurrentCell.IsInEditMode)
      {
        switch (key)
        {
          case Keys.Escape:
            CancelEdit();
            EndEdit();
            EditMode = DataGridViewEditMode.EditOnKeystrokeOrF2;
            return true;
        }
      } 
      
      return base.ProcessCmdKey(ref msg, keyData);
    }

    public List<NodeId> SelectedNodes()
    {
      List<DataGridViewRow> selectedRows = GetSelectedRows();
      List<NodeId> nodes = new List<NodeId>(selectedRows.Count);
      foreach (DataGridViewRow row in selectedRows)
        nodes.Add(new NodeId(row.Index));
      return nodes;
    }

    public List<DataGridViewRow> GetSelectedRows()
    {
      SortedDictionary<int, int> rowIndeces = new SortedDictionary<int, int>();
      int indecesEnd = RowCount;
      if (!ReadOnly)
        --indecesEnd;
      foreach (DataGridViewCell cell in SelectedCells)
      {
        int rowIndex = cell.RowIndex;
        if (rowIndex < indecesEnd)
          rowIndeces[rowIndex] = 0;
      }
      List<DataGridViewRow> res = new List<DataGridViewRow>(rowIndeces.Count);
      foreach (KeyValuePair<int, int> pair in rowIndeces)
        res.Add(Rows[pair.Key]);
      return res;
    }

    public List<int> GetSelectedRowsCol(ref int column)
    {
      List<DataGridViewRow> rows = GetSelectedRows();
      List<int> indexes = new List<int>(rows.Count);
      foreach (DataGridViewRow row in rows)
      {
        indexes.Add(row.Index);
      }
      column = (0 < SelectedColumns.Count) ? SelectedColumns[SelectedColumns.Count - 1].Index : -1;
      if (-1 == column)
      {
        if (0 < SelectedCells.Count)
        {
          if (!SelectedCells[SelectedCells.Count - 1].OwningRow.Selected)
            column = SelectedCells[SelectedCells.Count - 1].ColumnIndex;
        }
        else
        {
          column = -1;
        }
      }
      return indexes;
    }

    /// <summary>
    /// Copies the currently selected cells to the clipboard.
    /// </summary>
    public void Copy()
    {
      IDataObject data = GetClipboardContent();
      Clipboard.SetDataObject(data);
    }

    /// <summary>
    /// Determines if anything can be copied from the grid to the clipboard.
    /// </summary>
    public bool CanCopy()
    {
      return SelectedCells.Count > 0;
    }
  }
}
