/* 
 * Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include <mforms/mforms.h>
#include "base/string_utilities.h"

#import "MFCodeEditor.h"
#import "NSString_extras.h"

#import "SciLexer.h"

using namespace mforms;

// Marker ID assignments. Markers with higher number overlay lower ones.
#define CE_STATEMENT_MARKER 0
#define CE_ERROR_MARKER 1
#define CE_BREAKPOINT_MARKER 2
#define CE_BREAKPOINT_HIT_MARKER 3
#define CE_CURRENT_LINE_MARKER 4

extern NSString *SCIMarginClickNotification;

//--------------------------------------------------------------------------------------------------

/**
 * This function loads the style name map from an external file which allows us to map
 * style names to their associated ids.
 */
void load_style_names(std::string path, NSMutableDictionary* style_entries)
{
  NSError* error = nil;
  NSString* content = [NSString stringWithContentsOfFile: [NSString stringWithCPPString: path]
                                                encoding: NSASCIIStringEncoding
                                                   error: &error];

  // Ignore errors. We might have non-existing files.
  if (error == nil)
  {
    unsigned length = [content length];
    unsigned paraStart = 0, paraEnd = 0, contentsEnd = 0;

    NSRange currentRange;

    while (paraEnd < length)
    {
      [content getParagraphStart: &paraStart end: &paraEnd
                     contentsEnd: &contentsEnd forRange: NSMakeRange(paraEnd, 0)];
      currentRange = NSMakeRange(paraStart, contentsEnd - paraStart);
      NSArray* entries = [[content substringWithRange: currentRange] componentsSeparatedByString: @"="];
      if ([entries count] == 2)
      {
        NSString* style = [[entries objectAtIndex: 0] stringByTrimmingCharactersInSet: [NSCharacterSet whitespaceCharacterSet]];
        NSString* number = [[entries objectAtIndex: 1] stringByTrimmingCharactersInSet: [NSCharacterSet whitespaceCharacterSet]];
        if ([style length] > 0 && [number length] > 0)
          [style_entries setObject: [NSNumber numberWithInt: [number intValue]] forKey: style];
      }
    }
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * Superordinated style map loader which controls the load_style_names function.
 */
NSDictionary* get_style_map(NSString* language)
{
  NSMutableDictionary* result = [NSMutableDictionary dictionaryWithCapacity: 100];
  
  // These are global styles and always available.
	[result setObject: [NSNumber numberWithInt: STYLE_BRACEBAD] forKey: @"BRACEBAD"];
	[result setObject: [NSNumber numberWithInt: STYLE_BRACELIGHT] forKey: @"BRACELIGHT"];
	[result setObject: [NSNumber numberWithInt: STYLE_CALLTIP] forKey: @"CALLTIP"];
	[result setObject: [NSNumber numberWithInt: STYLE_CONTROLCHAR] forKey: @"CONTROLCHAR"];
	[result setObject: [NSNumber numberWithInt: STYLE_DEFAULT] forKey: @"DEFAULT"];
	[result setObject: [NSNumber numberWithInt: STYLE_LINENUMBER] forKey: @"LINENUMBER"];
  
  load_style_names(mforms::App::get()->get_resource_path("default.txt"), result);
  if (![language isSameAs: @"Null"])
  {  
    NSString* style_file = [language stringByAppendingString: @".txt"];
    std::string full_path = mforms::App::get()->get_resource_path([style_file UTF8String]);
    load_style_names(full_path, result);
  }
  
  return result;
}

//--------------------------------------------------------------------------------------------------

bool string_as_bool(NSString* value)
{
  return ([value isSameAs: @"1"]) || ([value isSameAs: @"true"]) || ([value isSameAs: @"yes"]);
}

//--------------------------------------------------------------------------------------------------

bool node_as_bool(NSXMLNode* node)
{
  return string_as_bool([node stringValue]);
}

//--------------------------------------------------------------------------------------------------

int node_as_int(NSXMLNode* node)
{
  return [[node stringValue] intValue];
}

//--------------------------------------------------------------------------------------------------

NSString* attribute_value(NSXMLElement* node, NSString* name)
{
  NSXMLNode* attribute = [node attributeForName: name];
  if (attribute == nil)
    return @"";
  return [[attribute stringValue] stringByTrimmingCharactersInSet: [NSCharacterSet whitespaceCharacterSet]];
}

//--------------------------------------------------------------------------------------------------

void read_style_attribute(NSXMLElement* node, NSString* name, NSMutableDictionary* style_entry)
{
  NSString* value = attribute_value(node, name);
  if ([value length] > 0)
    [style_entry setObject: value forKey: name];
}

//--------------------------------------------------------------------------------------------------

/**
 * Loads the configuration file for the given language and fills the given style map with the loaded
 * style data. Every other setting (tab usage, keywords etc.) will be applied
 * to the editor on the way. Styles need a bit more processing so the caller will take for that.
 */
void load_language_settings(ScintillaView* scintilla, NSString* language, NSMutableDictionary* style_map)
{
  NSString* config_file = [language stringByAppendingString: @".xml"];
  std::string native_config_path = mforms::App::get()->get_resource_path([config_file UTF8String]);
  
  NSError* error = nil;
  NSURL* url = [NSURL fileURLWithPath: [NSString stringWithCPPString: native_config_path]];
  NSXMLDocument* document = [[NSXMLDocument alloc] initWithContentsOfURL: url options: 0 error: &error];
  if (error != nil)
  {
    NSLog(@"%@ (%@)", error, url);
  }
  else
  {
    NSXMLElement* element = [document rootElement];
    if (![[element name] isEqualToString: @"ScintillaNET"])
    {
      NSLog(@"Invalid syntax highlighter configuration file (expected 'ScintillaNET' got '%@')", [element name]);
    }
    else
    {
      // Read only the block for the given language.
      // For now we don't support sublanguages.
      NSArray* children = [element elementsForName: @"Language"];
      element = nil;
      NSEnumerator* enumerator = [children objectEnumerator];
      while (element = [enumerator nextObject])
      {
        if ([attribute_value(element, @"Name") isSameAs: language])
          break;
      }
      
      if (element != nil)
      {
        // Indentation settings.
        children = [element elementsForName: @"Indentation"];
        if ([children count] > 0)
        {
          NSXMLElement* child = [children objectAtIndex: 0];
          NSXMLNode* attribute = [child attributeForName: @"TabWidth"];
          if (attribute != nil)
            [scintilla setGeneralProperty: SCI_SETTABWIDTH value: node_as_int(attribute)];
          attribute = [child attributeForName: @"UseSpaces"];
          if (attribute != nil)
            [scintilla setGeneralProperty: SCI_SETUSETABS value: !node_as_bool(attribute)];
        }
        
        // Keywords and lexer properties.
        children = [element elementsForName: @"Lexer"];
        if ([children count] > 0)
        {
          // For now we stick with the first lexer entry. Not sure if we ever need more than that.
          NSXMLElement* lexer_element = [children objectAtIndex: 0];
          children = [lexer_element elementsForName: @"Keywords"];

          NSEnumerator* enumerator = [children objectEnumerator];
          while (NSXMLElement* child = [enumerator nextObject])
          {
            int list = 0;
            NSXMLNode* attribute = [child attributeForName: @"List"];
            if (attribute != nil)
              list = node_as_int(attribute);
            [scintilla setReferenceProperty: SCI_SETKEYWORDS
                                  parameter: list
                                      value: [[child stringValue] UTF8String]];            
          }
          
          children = [lexer_element elementsForName: @"Properties"];

          enumerator = [children objectEnumerator];
          while (NSXMLElement* properties = [enumerator nextObject])
          {
            NSArray* property_list = [properties elementsForName: @"Property"];
            NSEnumerator* property_enumerator = [property_list objectEnumerator];
            while (NSXMLElement* property = [property_enumerator nextObject])
            {
              NSString* name;
              NSString* value;
              NSXMLNode* attribute = [property attributeForName: @"Name"];
              if (attribute != nil)
                name = [attribute stringValue];
              attribute = [property attributeForName: @"Value"];
              if (attribute != nil)
                value = [attribute stringValue];
              if ([name length] > 0 && [value length] > 0)
                [scintilla setLexerProperty: name value: value];
            }
          }
        }
        
        // Styles setup.
        children = [element elementsForName: @"Styles"];
        if ([children count] > 0)
        {
          enumerator = [children objectEnumerator];
          while (NSXMLElement* styles = [enumerator nextObject])
          {
            NSArray* style_list = [styles elementsForName: @"Style"];
            NSEnumerator* style_enumerator = [style_list objectEnumerator];
            while (NSXMLElement* style = [style_enumerator nextObject])
            {
              NSString* name;
              NSXMLNode* attribute = [style attributeForName: @"Name"];
              if (attribute != nil)
                name = [[attribute stringValue] stringByTrimmingCharactersInSet: [NSCharacterSet whitespaceCharacterSet]];
              
              if ([name length] == 0)
                continue;
              
              NSMutableDictionary* style_entry = [NSMutableDictionary dictionaryWithCapacity: 7];
              read_style_attribute(style, @"ForeColor", style_entry);
              read_style_attribute(style, @"BackColor", style_entry);
              read_style_attribute(style, @"Bold", style_entry);
              read_style_attribute(style, @"FontName", style_entry);
              read_style_attribute(style, @"Italic", style_entry);
              read_style_attribute(style, @"Size", style_entry);
              read_style_attribute(style, @"Underline", style_entry);
              
              // Set the new value in the style map replacing an eventually already set value
              // creating so kinda inheritance for style settings.
              [style_map setObject: style_entry forKey: name];
            }
          }
          
        }
      }
    }
  }
  
  [document release];
}

//--------------------------------------------------------------------------------------------------

void language_setup(ScintillaView* scintilla, NSString* language)
{
  // In opposition to the Windows version of this implementation we do not have automatic language
  // setup, so we have to take care for keywords and colors manually here.
  [scintilla setStringProperty: SCI_SETLEXERLANGUAGE parameter: nil value: language];

  NSMutableDictionary* styles = [NSMutableDictionary dictionaryWithCapacity: 100];
  load_language_settings(scintilla, @"default", styles);
  
  if (![language isSameAs: @"Null"])
    load_language_settings(scintilla, language, styles);

  NSDictionary* style_map = get_style_map(language);

  NSEnumerator* name_enumerator = [styles keyEnumerator];
  while (NSString* name = [name_enumerator nextObject])
  {
    int style_id = -1;
    NSNumber* number = [style_map objectForKey: name];
    if (number != nil)
      style_id = [number intValue];
    
    if (style_id < 0)
      continue;
    
    NSDictionary* style = [styles objectForKey: name];
    NSEnumerator* style_enumerator = [style keyEnumerator];
    while (NSString* attribute = [style_enumerator nextObject])
    {
      int property = -1;
      if ([attribute isEqualToString: @"ForeColor"])
        property = SCI_STYLESETFORE;
      else
        if ([attribute isEqualToString: @"BackColor"])
          property = SCI_STYLESETBACK;
        else
          if ([attribute isEqualToString: @"Bold"])
            property = SCI_STYLESETBOLD;
          else
            if ([attribute isEqualToString: @"Italic"])
              property = SCI_STYLESETITALIC;
            else
              if ([attribute isEqualToString: @"Underline"])
                property = SCI_STYLESETUNDERLINE;
              else
                if ([attribute isEqualToString: @"FontName"])
                  property = SCI_STYLESETFONT;
                else
                  if ([attribute isEqualToString: @"Size"])
                    property = SCI_STYLESETSIZE;

      switch (property)
      {
        case SCI_STYLESETFORE:
        case SCI_STYLESETBACK:
          [scintilla setColorProperty: property parameter: style_id fromHTML: [style valueForKey: attribute]];
          break;
          
        case SCI_STYLESETFONT:
          [scintilla setStringProperty: property parameter: style_id value: [style valueForKey: attribute]];
          break;
          
        case SCI_STYLESETBOLD:
        case SCI_STYLESETITALIC:
        case SCI_STYLESETUNDERLINE:
          [scintilla setGeneralProperty: property parameter: style_id value: string_as_bool([style valueForKey: attribute])];
          break;

        case SCI_STYLESETSIZE:
          [scintilla setGeneralProperty: property parameter: style_id value: [[style valueForKey: attribute] intValue]];
          break;
      }
    }
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * Helper to setup an individual marker in the code editor.
 */
void setup_marker(NSBundle* bundle, ScintillaView* scintilla, int marker_id, const char* icon_path, NSColor* background)
{
  std::string full_path = mforms::App::get()->get_resource_path(icon_path);
  NSError* error = nil;
  NSString* xpm = [NSString stringWithContentsOfFile: [NSString stringWithCPPString: full_path]
                                            encoding: NSASCIIStringEncoding error: &error];
  if (error != nil)
  {
    NSLog(@"%@ (%@)", error, icon_path);
    
    [scintilla setGeneralProperty: SCI_MARKERDEFINE parameter: marker_id value: SC_MARK_BACKGROUND];
    [scintilla setColorProperty: SCI_MARKERSETBACK parameter: marker_id value: background];
    [scintilla setColorProperty: SCI_MARKERSETFORE parameter: marker_id value: [NSColor whiteColor]];
  }
  else
    [scintilla setStringProperty: SCI_MARKERDEFINEPIXMAP parameter: marker_id value: xpm];
}

//--------------------------------------------------------------------------------------------------

/**
 * Helper to set up the code editor with usually used settings.
 */
void setup_editor(ScintillaView* scintilla, bool use_tabs, int indentation, NSString* language)
{
  // Some values as default before reading the config file.
  [scintilla setGeneralProperty: SCI_SETUSETABS value: use_tabs ? 1 : 0];
  [scintilla setGeneralProperty: SCI_SETTABWIDTH value: indentation];
  [scintilla setGeneralProperty: SCI_SETTABINDENTS value: 1];
  [scintilla setGeneralProperty: SCI_SETINDENT value: indentation];

  language_setup(scintilla, language);
  
  [scintilla setGeneralProperty: SCI_SETCARETLINEVISIBLE value: 1];
  [scintilla setColorProperty: SCI_SETCARETLINEBACK
                    value: [NSColor colorWithDeviceRed: 248.0 / 255 green: 200.0 / 255 blue: 0 alpha: 1]];
  [scintilla setGeneralProperty: SCI_SETCARETLINEBACKALPHA value: 20];

  [scintilla setGeneralProperty: SCI_SETBACKSPACEUNINDENTS value: 1];
  int guide_type = (language == @"python") ? SC_IV_LOOKFORWARD : SC_IV_LOOKBOTH;
  [scintilla setGeneralProperty: SCI_SETINDENTATIONGUIDES value: guide_type];

  // Margin: Line number style.
  [scintilla setColorProperty: SCI_STYLESETFORE parameter: STYLE_LINENUMBER fromHTML: @"#404040"];
  [scintilla setColorProperty: SCI_STYLESETBACK parameter: STYLE_LINENUMBER fromHTML: @"#E0E0E0"];
  
  [scintilla setGeneralProperty: SCI_SETMARGINTYPEN parameter: 0 value: SC_MARGIN_NUMBER];
  int lineNumberStyleWidth = [scintilla getGeneralProperty: SCI_TEXTWIDTH parameter: STYLE_LINENUMBER extra: (int)"_99999"];
	[scintilla setGeneralProperty: SCI_SETMARGINWIDTHN parameter: 0 value: lineNumberStyleWidth];
  [scintilla setGeneralProperty: SCI_SETMARGINSENSITIVEN parameter: 0 value: 1]; // Make Scintilla send a notification for clicks in this margin.
  
  // Margin: Markers.
  [scintilla setGeneralProperty: SCI_SETMARGINWIDTHN parameter: 1 value: 16];
  [scintilla setGeneralProperty: SCI_SETMARGINSENSITIVEN parameter: 1 value: 1];
  
  // Margin: Indicators (folders).
  [scintilla setColorProperty: SCI_SETFOLDMARGINCOLOUR
                    parameter: 1
                    value: [NSColor colorWithDeviceWhite: 230.0 / 255 alpha: 1]];
  [scintilla setGeneralProperty: SCI_SETMARGINWIDTHN parameter: 2 value: 16];
  [scintilla setGeneralProperty: SCI_SETMARGINSENSITIVEN parameter: 2 value: 1];
  
  [scintilla setGeneralProperty: SCI_SETEOLMODE value: SC_EOL_LF];

  // Marker definitions.
  NSBundle* bundle = [NSBundle bundleForClass: [scintilla class]];
  setup_marker(bundle, scintilla, CE_STATEMENT_MARKER, "editor_statement.xpm", [NSColor blueColor]);
  setup_marker(bundle, scintilla, CE_ERROR_MARKER, "editor_error.xpm",
               [NSColor colorWithDeviceRed: 50.0 / 255 green: 222.0 / 255 blue: 46.0 / 255 alpha: 1]);
  setup_marker(bundle, scintilla, CE_BREAKPOINT_MARKER, "editor_breakpoint.xpm",
               [NSColor colorWithDeviceRed: 169.0 / 255 green: 68.0 / 255 blue: 62.0 / 255 alpha: 1]);
  setup_marker(bundle, scintilla, CE_BREAKPOINT_HIT_MARKER, "editor_breakpoint_hit.xpm",
               [NSColor colorWithDeviceRed: 169.0 / 255 green: 68.0 / 255 blue: 62.0 / 255 alpha: 1]);
  setup_marker(bundle, scintilla, CE_CURRENT_LINE_MARKER, "editor_current_pos.xpm",
               [NSColor colorWithDeviceRed: 169.0 / 255 green: 68.0 / 255 blue: 62.0 / 255 alpha: 1]);
  
  // Folder setup.
  [scintilla setGeneralProperty: SCI_SETMARGINWIDTHN parameter: 2 value: 16];
  [scintilla setGeneralProperty: SCI_SETMARGINMASKN parameter: 2 value: SC_MASK_FOLDERS];
  [scintilla setGeneralProperty: SCI_SETMARGINSENSITIVEN parameter: 2 value: 1];
  [scintilla setGeneralProperty: SCI_MARKERDEFINE parameter: SC_MARKNUM_FOLDEROPEN value: SC_MARK_BOXMINUS];
  [scintilla setGeneralProperty: SCI_MARKERDEFINE parameter: SC_MARKNUM_FOLDER value: SC_MARK_BOXPLUS];
  [scintilla setGeneralProperty: SCI_MARKERDEFINE parameter: SC_MARKNUM_FOLDERSUB value: SC_MARK_VLINE];
  [scintilla setGeneralProperty: SCI_MARKERDEFINE parameter: SC_MARKNUM_FOLDERTAIL value: SC_MARK_LCORNER];
  [scintilla setGeneralProperty: SCI_MARKERDEFINE parameter: SC_MARKNUM_FOLDEREND value: SC_MARK_BOXPLUSCONNECTED];
  [scintilla setGeneralProperty: SCI_MARKERDEFINE parameter: SC_MARKNUM_FOLDEROPENMID value: SC_MARK_BOXMINUSCONNECTED];
  [scintilla setGeneralProperty: SCI_MARKERDEFINE parameter: SC_MARKNUM_FOLDERMIDTAIL value: SC_MARK_TCORNER];
  for (int n= 25; n < 32; ++n) // Markers 25..31 are reserved for folding.
  {
    [scintilla setColorProperty: SCI_MARKERSETFORE parameter: n value: [NSColor whiteColor]];
    [scintilla setColorProperty: SCI_MARKERSETBACK parameter: n fromHTML: @"#404040"];
  }
  
  // Init markers & indicators for highlighting of syntax errors.
  [scintilla setColorProperty: SCI_INDICSETFORE parameter: 0 fromHTML: @"#D01921"];
  [scintilla setGeneralProperty: SCI_INDICSETUNDER parameter: 0 value: 1];
  [scintilla setGeneralProperty: SCI_INDICSETSTYLE parameter: 0 value: INDIC_SQUIGGLE];
}

//--------------------------------------------------------------------------------------------------

@implementation MFCodeEditor

- (id) initWithFrame: (NSRect) frame
          codeEditor: (CodeEditor*) codeEditor
{
  self = [super initWithFrame: frame];
  if (self)
  {
    mfOwner = codeEditor;
    mfOwner->set_data(self);

    [[NSNotificationCenter defaultCenter] addObserver: self
                                             selector: @selector(marginClick:)
                                                 name: SCIMarginClickNotification
                                               object: self];
    
    [[NSNotificationCenter defaultCenter] addObserver: self
                                             selector: @selector(textChanged:)
                                                 name: NSTextDidChangeNotification
                                               object: self];
  }
  return self;
}

- (void) dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];
  [super dealloc];
}

- (void)setTag:(NSInteger)tag
{
  mTag = tag;
}


- (NSInteger)tag
{
  return mTag;
}

/**
 * Notification from the scintilla if a margin was clicked. Translate and forward that call
 * to the backend for further processing.
 */
- (void) marginClick: (NSNotification*) aNotification
{
  NSDictionary* info = [aNotification userInfo];
  
  int margin = 0;
  NSNumber* number = [info objectForKey: @"margin"];
  if (number != nil)
    margin = [number intValue];
  
  int line = 0;
  number = [info objectForKey: @"line"];
  if (number != nil)
    line = [number intValue];
  
  mforms::ModifierKey modifiers = mforms::ModifierNoModifier;
  int sci_modifiers = 0;
  number = [info objectForKey: @"modifiers"];
  if (number != nil)
  {
    sci_modifiers = [number intValue];
    if ((sci_modifiers & SCI_SHIFT) != 0)
      modifiers = modifiers | mforms::ModifierShift;
    if ((sci_modifiers & SCI_CTRL) != 0)
      modifiers = modifiers | mforms::ModifierControl;
    if ((sci_modifiers & SCI_ALT) != 0)
      modifiers = modifiers | mforms::ModifierAlt;
  }
  mfOwner->gutter_clicked(margin, line, modifiers);
}


- (void) textChanged: (NSNotification*) aNotification
{
  NSDictionary* info = [aNotification userInfo];
  
  int linesAdded = 0;
  NSNumber* number = [info objectForKey: @"linesAdded"];
  if (number != nil)
    linesAdded = [number intValue];
  
  int line = 0;
  number = [info objectForKey: @"line"];
  if (number != nil)
    line = [number intValue];
  
  mfOwner->text_changed(line, linesAdded);
}


static bool ce_create(CodeEditor* self)
{
  [[[MFCodeEditor alloc] initWithFrame: NSMakeRect(0, 0, 100, 100)
                            codeEditor: self] autorelease];
  
  return true;
}


static void ce_set_text(CodeEditor* self, const std::string& text)
{
  ScintillaView *editor = self->get_data();
  if (editor)
  {
    [editor setString: [NSString stringWithCPPString: text]];
  }
}


static const std::string ce_get_text(CodeEditor* self, bool selection_only)
{
  ScintillaView *editor = self->get_data();
  if (editor)
    return [[editor string] UTF8String];
  return "";
}


static void ce_get_selection(CodeEditor* self, int &start, int &length)
{
  ScintillaView *editor = self->get_data();
  if (editor)
  {
    NSRange sel = [editor selectedRange];
    start = sel.location;
    length = sel.length;
  }
  else
  {
    start = 0;
    length= 0;
  }
}


static void ce_set_selection(CodeEditor* self, int start, int length)
{
  ScintillaView *editor = self->get_data();
  if (editor)
  {
    [editor setGeneralProperty: SCI_GOTOPOS value: start];
    //[editor setGeneralProperty: SCI_SETSEL parameter: start value: length];
  }
}


static bool ce_get_range_of_line(CodeEditor* self, int line, int &start, int &length)
{
  ScintillaView *editor = self->get_data();
  if (editor)
  {
    start = [editor getGeneralProperty: SCI_POSITIONFROMLINE parameter: line];
    length = [editor getGeneralProperty: SCI_LINELENGTH parameter: line];
    return true;
  }
  return false;
}


static void ce_set_language(CodeEditor* self, SyntaxHighlighterLanguage language)
{
  ScintillaView *editor = self->get_data();
  if (editor)
  {
    switch (language)
    {
      case mforms::LanguageCpp:
        setup_editor(editor, false, 2, @"cpp"); // Currently string based. Will later use enum.
        break;

      case mforms::LanguageLua:
        setup_editor(editor, false, 2, @"lua");
        break;

      case mforms::LanguagePython:
        setup_editor(editor, false, 2, @"python");
        break;

      case mforms::LanguageMySQL:
        setup_editor(editor, false, 2, @"mysql");
        break;

      default:
        setup_editor(editor, false, 2, @"Null");
    }
  }
}


static void ce_set_read_only(CodeEditor* self, bool flag)
{
  ScintillaView *editor = self->get_data();
  if (editor)
    [editor setEditable: !flag];
}

static void ce_show_markup(CodeEditor* self, LineMarkup markup, int line)
{
  ScintillaView *editor = self->get_data();
  if (editor)
  {
    // The marker mask contains one bit for each set marker (0..31).
    unsigned int marker_mask = [editor getGeneralProperty: SCI_MARKERGET parameter: line];
    unsigned int new_marker_mask = 0;
    if ((markup & mforms::LineMarkupStatement) != 0)
    {
      unsigned int mask = 1 << CE_STATEMENT_MARKER;
      if ((marker_mask & mask) != mask)
        new_marker_mask |= mask;
    }
    if ((markup & mforms::LineMarkupError) != 0)
    {
      unsigned int mask = 1 << CE_ERROR_MARKER;
      if ((marker_mask & mask) != mask)
        new_marker_mask |= mask;
    }
    if ((markup & mforms::LineMarkupBreakpoint) != 0)
    {
      unsigned int mask = 1 << CE_BREAKPOINT_MARKER;
      if ((marker_mask & mask) != mask)
        new_marker_mask |= mask;
    }
    if ((markup & mforms::LineMarkupBreakpointHit) != 0)
    {
      unsigned int mask = 1 << CE_BREAKPOINT_HIT_MARKER;
      if ((marker_mask & mask) != mask)
        new_marker_mask |= mask;
    }
    if ((markup & mforms::LineMarkupCurrent) != 0)
    {
      unsigned int mask = 1 << CE_CURRENT_LINE_MARKER;
      if ((marker_mask & mask) != mask)
        new_marker_mask |= mask;
    }

    [editor setGeneralProperty: SCI_MARKERADDSET parameter: line value: new_marker_mask];
  }
}

static void ce_remove_markup(CodeEditor* self, LineMarkup markup, int line)
{
  ScintillaView *editor = self->get_data();
  if (editor)
  {
    if (markup == mforms::LineMarkupAll)
      [editor setGeneralProperty: SCI_MARKERDELETEALL value: line];
    else
    {
      if ((markup & mforms::LineMarkupStatement) != 0)
        [editor setGeneralProperty: SCI_MARKERDELETE parameter: line value: CE_STATEMENT_MARKER];
      if ((markup & mforms::LineMarkupError) != 0)
        [editor setGeneralProperty: SCI_MARKERDELETE parameter: line value: CE_ERROR_MARKER];
      if ((markup & mforms::LineMarkupBreakpoint) != 0)
        [editor setGeneralProperty: SCI_MARKERDELETE parameter: line value: CE_BREAKPOINT_MARKER];
      if ((markup & mforms::LineMarkupBreakpointHit) != 0)
        [editor setGeneralProperty: SCI_MARKERDELETE parameter: line value: CE_BREAKPOINT_HIT_MARKER];
      if ((markup & mforms::LineMarkupCurrent) != 0)
        [editor setGeneralProperty: SCI_MARKERDELETE parameter: line value: CE_CURRENT_LINE_MARKER];
    }
  }
}

static int ce_line_count(CodeEditor* self)
{
  ScintillaView *editor = self->get_data();
  if (editor)
    return [editor getGeneralProperty: SCI_GETLINECOUNT];
  return 0;
}

static void ce_set_font(CodeEditor* self, const std::string& fontDescription)
{
  ScintillaView *editor = self->get_data();
  if (editor)
  {
    std::string name;
    int size;
    bool bold;
    bool italic;
    if (base::parse_font_description(fontDescription, name, size, bold, italic))
    {
      [editor setFontName: [NSString stringWithCPPString: name] size: size bold: bold italic: italic];
    }
  }
}
  
void cf_codeeditor_init()
{
  ::mforms::ControlFactory *f = ::mforms::ControlFactory::get_instance();

  f->_code_editor_impl.create = ce_create;
  f->_code_editor_impl.set_text = ce_set_text;
  f->_code_editor_impl.get_text = ce_get_text;
  f->_code_editor_impl.get_selection = ce_get_selection;
  f->_code_editor_impl.set_selection = ce_set_selection;
  f->_code_editor_impl.get_range_of_line = ce_get_range_of_line;
  f->_code_editor_impl.set_language = ce_set_language;
  f->_code_editor_impl.set_read_only = ce_set_read_only;
  f->_code_editor_impl.show_markup = ce_show_markup;
  f->_code_editor_impl.remove_markup = ce_remove_markup;
  f->_code_editor_impl.line_count = ce_line_count;
  f->_code_editor_impl.set_font = ce_set_font;
}

@end
