/* 
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
#include "../lf_menubar.h"
#include <gtkmm/menushell.h>
#include <gtkmm/menubar.h>
#include <gtkmm/menu.h>
#include <gtkmm/menuitem.h>
#include <gtkmm/accelgroup.h>
#include <gtkmm/separatormenuitem.h>
#include <mforms.h>
#include "base/wb_iterators.h"
#include "base/string_utilities.h"
#include "base/log.h"
ENABLE_LOG("mforms.linux")

//------------------------------------------------------------------------------
namespace
{
template <typename T>
T cast(void *ptr)
{
  return dynamic_cast<T>((Gtk::Widget*)ptr);
}
}

static Glib::RefPtr<Gtk::AccelGroup> accel_group;

//------------------------------------------------------------------------------
static void process_click(Gtk::MenuItem* mi, mforms::MenuItem* item)
{
  const int ignore_signal = (long)mi->get_data("ignore_signal");
  if (!ignore_signal && mi)
    item->callback();
}

//------------------------------------------------------------------------------
void mforms::gtk::set_accel_group(Glib::RefPtr<Gtk::AccelGroup> ag)
{
  accel_group = ag;
}

//------------------------------------------------------------------------------
Gtk::MenuBar* mforms::gtk::widget_for_menubar(mforms::MenuBar* self)
{
  Gtk::MenuBar* mb = self->get_data<Gtk::MenuBar>();
  return mb;
}

//------------------------------------------------------------------------------
bool mforms::gtk::MenuItemImpl::create_menu_bar(mforms::MenuBar *item)
{
  Gtk::MenuBar* mb = cast<Gtk::MenuBar*>(item->get_data_ptr());
  if (mb)
    delete mb;
  mb = new Gtk::MenuBar();
  mb->show();
  item->set_data(Gtk::manage(mb));
  return mb;
}

//------------------------------------------------------------------------------
bool mforms::gtk::MenuItemImpl::create_menu_item(mforms::MenuItem *item, const std::string &label, const mforms::MenuItemType type)
{
  Gtk::MenuItem* mi = cast<Gtk::MenuItem*>(item->get_data_ptr());

  if (mi)
  {
    item->set_data(0);
    delete mi;
  }

  if (type == mforms::SeparatorMenuItem)  
    item->set_data(Gtk::manage(new Gtk::SeparatorMenuItem()));
  else
  {
    if (type == mforms::CheckedMenuItem)
    {
      Gtk::CheckMenuItem* ci = new Gtk::CheckMenuItem(label);
      item->set_data(Gtk::manage(ci));
    }
    else
      item->set_data(Gtk::manage(new Gtk::MenuItem(label)));
  }

  mi = cast<Gtk::MenuItem*>(item->get_data_ptr());
  if (mi)
  {
    mi->show();
    if (type != mforms::SeparatorMenuItem)
    {
      mi->set_use_underline(true);
      mi->signal_activate().connect(sigc::bind(sigc::ptr_fun(process_click), mi, item));
    }
  }

  return mi;
}

//------------------------------------------------------------------------------
void mforms::gtk::MenuItemImpl::set_title(mforms::MenuItem *item, const std::string& label)
{
  Gtk::MenuItem* mi = cast<Gtk::MenuItem*>(item->get_data_ptr());
  if (mi)
    mi->set_label(label);
}

//------------------------------------------------------------------------------
std::string mforms::gtk::MenuItemImpl::get_title(mforms::MenuItem *item)
{
  std::string ret;
  Gtk::MenuItem* mi = cast<Gtk::MenuItem*>(item->get_data_ptr());
  if (mi)
    ret = mi->get_label();
  return ret;
}

//------------------------------------------------------------------------------
void mforms::gtk::MenuItemImpl::set_shortcut(mforms::MenuItem *item, const std::string& item_shortcut)
{
  if (!item_shortcut.empty())
  {
    Gdk::ModifierType   accel_mods = (Gdk::ModifierType)0;
    guint                accel_key = 0;

    // convert the accelerator format from Control+X to <control>x which is recognized by gtk
    std::vector<std::string> parts(base::split(item_shortcut, "+"));

    if (parts.size() > 0)
    {
      std::string shortcut = parts.back();
      parts.pop_back();

      while (parts.size() > 0)
      {
        std::string mod = parts.back();
        parts.pop_back();
        std::transform(mod.begin(), mod.end(), mod.begin(), (int(*)(int))std::tolower);
        if ("modifier" == mod)
          mod = "control";
        shortcut = "<"+mod+">"+shortcut;
      }
      if (!shortcut.empty())
        Gtk::AccelGroup::parse(shortcut, accel_key, accel_mods);
    }

    Gtk::MenuItem* mi = cast<Gtk::MenuItem*>(item->get_data_ptr());
    if (accel_key != 0 && mi)
    {
      if (accel_group)
        mi->add_accelerator("activate", accel_group, accel_key, accel_mods, Gtk::ACCEL_VISIBLE);
      else
        log_error("AccelGroup was not set for menubar");
    }
  }
}

//------------------------------------------------------------------------------
void mforms::gtk::MenuItemImpl::set_enabled(mforms::MenuBase *item, bool is_on)
{
  Gtk::Widget* mb = item->get_data<Gtk::Widget>();
  if (mb)
    mb->set_sensitive(is_on);
}

//------------------------------------------------------------------------------
bool mforms::gtk::MenuItemImpl::get_enabled(mforms::MenuBase *item)
{
  bool ret = false;
  Gtk::Widget* mb = item->get_data<Gtk::Widget>();
  if (mb)
    ret = mb->get_sensitive();
  return ret;
}

//------------------------------------------------------------------------------
void mforms::gtk::MenuItemImpl::set_checked(mforms::MenuItem *item, bool on)
{
  Gtk::CheckMenuItem* mi = cast<Gtk::CheckMenuItem*>(item->get_data_ptr());
  if (mi)
  {
    mi->set_data("ignore_signal", (void*)1);
    mi->set_active(on);
    mi->set_data("ignore_signal", 0);
  }
  else
    log_error("Passed MenuItem '%s' does not have CheckMenuItem at %p", mforms::gtk::MenuItemImpl::get_title(item).c_str(), item->get_data_ptr());
}

//------------------------------------------------------------------------------
bool mforms::gtk::MenuItemImpl::get_checked(mforms::MenuItem *item)
{
  bool ret = false;
  Gtk::CheckMenuItem* mi = cast<Gtk::CheckMenuItem*>(item->get_data_ptr());
  if (mi)
  {
    ret = mi->get_active();
  }
  else
    log_error("Passed MenuItem '%s' does not have CheckMenuItem at %p", mforms::gtk::MenuItemImpl::get_title(item).c_str(), item->get_data_ptr());
  return ret;
}

//------------------------------------------------------------------------------
void mforms::gtk::MenuItemImpl::insert_item(mforms::MenuBase *menu, int index, mforms::MenuItem *item)
{
  Gtk::MenuShell* menu_shell = 0;
  Gtk::MenuItem* item_to_insert = cast<Gtk::MenuItem*>(item->get_data_ptr());

  Gtk::MenuBar* mb = cast<Gtk::MenuBar*>(menu->get_data_ptr());
  if (mb)
  {
    menu_shell = mb;
  }
  else
  {
    Gtk::MenuItem* mi = cast<Gtk::MenuItem*>(menu->get_data_ptr());
    if (mi)
    {
      Gtk::Menu* menu = 0;
      if (mi->has_submenu())
        menu = mi->get_submenu();
      else
      {
        menu = Gtk::manage(new Gtk::Menu());
        mi->set_submenu(*menu);
        menu->show();
      }
      menu_shell = menu;
    }
    else
      log_error("Passed MenuBase %p does not contain neither Gtk::MenuBar nor Gtk::MenuItem", menu);
  }

  if (menu_shell && item_to_insert)
    menu_shell->insert(*item_to_insert, index);
}

//------------------------------------------------------------------------------
void mforms::gtk::MenuItemImpl::remove_item(mforms::MenuBase *menu, mforms::MenuItem *item)
{
  Gtk::MenuShell* menu_shell = 0;

  Gtk::MenuBar* mb = cast<Gtk::MenuBar*>(menu->get_data_ptr());
  if (mb)
  {
    menu_shell = mb;
  }
  else
  {
    Gtk::MenuItem* mi = cast<Gtk::MenuItem*>(menu->get_data_ptr());
    if (mi)
    {
      if (mi->has_submenu())
        menu_shell = mi->get_submenu();
      else
        log_error("Requesting to remove MenuItem from Menu with no sub menu");
    }
    else
      log_error("Passed MenuBase %p does not contain neither Gtk::MenuBar nor Gtk::MenuItem", menu);
  }

  Gtk::MenuItem* item_to_remove = item ? cast<Gtk::MenuItem*>(item->get_data_ptr()) : 0;
  if (menu_shell)
  {
    if (item_to_remove)
      menu_shell->remove(*item_to_remove); // May not work needs to be tested
    else
    {
      typedef Glib::ListHandle<Gtk::Widget*>    WList;
      WList list = menu_shell->get_children();
      for (base::const_range<WList> it(list); it; ++it)
        menu_shell->remove(*(*it));
    }
  }
}

//------------------------------------------------------------------------------
void mforms::gtk::lf_menubar_init()
{
  ::mforms::ControlFactory *f = ::mforms::ControlFactory::get_instance();

  f->_menu_item_impl.create_menu_bar  = mforms::gtk::MenuItemImpl::create_menu_bar;
  f->_menu_item_impl.create_menu_item = mforms::gtk::MenuItemImpl::create_menu_item;
  f->_menu_item_impl.set_title        = mforms::gtk::MenuItemImpl::set_title;
  f->_menu_item_impl.get_title        = mforms::gtk::MenuItemImpl::get_title;
  f->_menu_item_impl.set_shortcut     = mforms::gtk::MenuItemImpl::set_shortcut;
  f->_menu_item_impl.set_enabled      = mforms::gtk::MenuItemImpl::set_enabled;
  f->_menu_item_impl.get_enabled      = mforms::gtk::MenuItemImpl::get_enabled;
  f->_menu_item_impl.set_checked      = mforms::gtk::MenuItemImpl::set_checked;
  f->_menu_item_impl.get_checked      = mforms::gtk::MenuItemImpl::get_checked;

  f->_menu_item_impl.insert_item      = mforms::gtk::MenuItemImpl::insert_item;
  f->_menu_item_impl.remove_item      = mforms::gtk::MenuItemImpl::remove_item;
}

