/* 
 * Copyright (c) 2008, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
#include "../lf_view.h"
#include "mforms/simplegrid.h"

#include <boost/shared_ptr.hpp>
#include <gtkmm/treeview.h>
#include <gtkmm/treestore.h>
#include <gtkmm/liststore.h>
#include <gtkmm/celleditable.h>
#include <gtkmm/eventbox.h>
#include <gtkmm/box.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/comboboxentrytext.h>
#include <gdk/gdkkeysyms.h>

namespace
{

static Gtk::TreePath cast_path(const mforms::SimpleGridPath& s)
{
  Gtk::TreePath p;
  const int size = s.size();
  for (int i = 0; i < size; ++i)
    p.push_back(s[i]);

  return p;
}

static mforms::SimpleGridPath cast_path(const Gtk::TreePath& s)
{
  mforms::SimpleGridPath p;
  const int size = s.size();
  for (int i = 0; i < size; ++i)
    p.append(s[i]);

  return p;
}

//==============================================================================
//
//==============================================================================
class GridCell
{
  public:
    typedef Gtk::TreeModelColumn<GridCell*>   Column;
    typedef std::vector<std::string>      EnumDef;
    typedef boost::shared_ptr<EnumDef>  EnumDefRef;

    ~GridCell();
    void reset();

    GridCell();
    GridCell(const std::string& atext);
    explicit GridCell(const bool v);

    void set(const std::string& s);
    void set(const bool v);

    const std::string& text() const;
    bool get_value(bool *b) const;
    bool get_value(std::string *s) const;
    std::string as_string() const;

    void set_editable(const bool is_editable) {_editable = is_editable;}
    bool is_editable() const {return _editable;}

    void set_type(const mforms::CellType ct);
    mforms::CellType type() const {return _type;}

    void set_enum_def(EnumDef* list);
    const EnumDefRef get_enum_def() const {return _enum_def;}

    void set_fg(const double red, const double green, const double blue);
    Gdk::Color& fg() {return _fg;}
    bool fg_set() const {return _fg_set;}

    void set_bg(const double red, const double green, const double blue);
    Gdk::Color& bg() {return _bg;}
    bool bg_set() const {return _bg_set;}

    void set_underline(const bool on) {_underline = on;}
    bool underline() const {return _underline;}

    void set_shade(const mforms::Shade cs) {_shade |= (1 << cs);}
    void reset_shade(const mforms::Shade cs) {_shade = 1 << cs;}
    void unset_shade(const mforms::Shade cs) {_shade &= ~(1 << cs);}
    bool has_shade(const mforms::Shade cs) const {return _shade & (1 << cs);}

  private:
    mforms::CellType                     _type;
    std::string                  _text;
    bool                         _bool;
    bool                         _editable;
    EnumDefRef                   _enum_def;
    bool                         _fg_set;
    Gdk::Color                   _fg;
    bool                         _bg_set;
    Gdk::Color                   _bg;
    bool                         _underline;
    int                          _shade;
};


//==============================================================================
class GridModel : public Glib::Object, public Gtk::TreeModel, 
                         virtual public sigc::trackable
{
  protected:
    GridModel(Gtk::TreeView *treeview, const std::string& name);

  public:
    typedef std::vector<GridCell>     Cells;

    ~GridModel();

    static Glib::RefPtr<GridModel> create(Gtk::TreeView *treeview, const std::string& name)
    {
      return Glib::RefPtr<GridModel>(new GridModel(treeview, name));
    }

    void add_column();
    int columns_count() const {return _ncols;}

    mforms::SimpleGridPath  append_row(const std::string& gid);
    mforms::SimpleGridPath  append_row(const mforms::SimpleGridPath& path);

    GridCell* cell(const int gid, const int rid, const int column);
    const GridCell* cell(const int gid, const int rid, const int column) const;
    GridCell* cell(const Gtk::TreePath& path, const int col_id, Gtk::TreeIter* iter);
    GridCell* cell(const mforms::SimpleGridPath& path, const int col_id);
    GridCell* cell(const Gtk::TreeIter& iter, const int col_id);

    int group_index(const std::string& group_id);
    Cells* row_from_iter(const Gtk::TreeIter& iter);
    std::string group_name(const Gtk::TreeIter& iter);

  protected:
    virtual Gtk::TreeModelFlags get_flags_vfunc() const;
    virtual int get_n_columns_vfunc() const;
    virtual GType get_column_type_vfunc(int index) const;

    virtual bool iter_next_vfunc(const iterator& iter, iterator& iter_next) const;
    virtual bool get_iter_vfunc(const Path& path, iterator& iter) const;
    virtual bool iter_children_vfunc(const iterator& parent, iterator& iter) const;
    virtual bool iter_parent_vfunc(const iterator& child, iterator& iter) const;
    virtual bool iter_nth_child_vfunc(const iterator& parent, int n, iterator& iter) const;
    virtual bool iter_nth_root_child_vfunc(int n, iterator& iter) const;
    virtual bool iter_has_child_vfunc(const iterator& iter) const;
    virtual int iter_n_children_vfunc(const iterator& iter) const;
    virtual int iter_n_root_children_vfunc() const;
    virtual TreeModel::Path get_path_vfunc(const iterator& iter) const;
    virtual bool iter_is_valid(const iterator& iter) const;
    virtual void get_value_vfunc(const iterator& iter, int column, Glib::ValueBase& value) const;
    virtual void set_value_impl(const iterator& row, int column, const Glib::ValueBase& value);
    virtual void get_value_impl(const iterator& row, int column, Glib::ValueBase& value) const;

  protected:
    void reset_iter(iterator& iter) const throw();

  private:
    bool fill(const int gid, const int rid, Gtk::TreeIter* iter, Gtk::TreePath *path);
    int                                           _stamp;
    int                                           _ncols;

    struct GridRow
    {
      std::string            header_text;
      Cells                  row;
      std::vector<Cells>     rows;
    };

    mutable std::vector<GridRow>     _root;

    void setup_group(GridRow* row, const std::string& group_name);
    void set_group_name(GridRow* row, const std::string& name);
    std::string group_name(GridRow* row) const;
};

//==============================================================================
//
//==============================================================================
class GridCellEditable : public Gtk::EventBox, public Gtk::CellEditable
{
  public:
    GridCellEditable();

    void set_type(const mforms::CellType type) {_type = type;}
    void set_enum_def(const GridCell::EnumDefRef def);

    void set_text(const std::string& s);
    std::string get_text();
    void set_path(const Glib::ustring& path) {_path = path;}
    Glib::ustring path() const {return _path;};
    Gtk::TreePath gtk_path() const {return Gtk::TreePath(_path);};

  protected:
    virtual void start_editing_vfunc(GdkEvent* event);
    virtual void on_editing_done();

  private:
    bool on_key_release(GdkEventKey* ev);

    mforms::CellType                    _type;
    Gtk::ComboBoxEntryText      _combo;
    Glib::ustring               _path;
};

class GridView;

//==============================================================================
//
//==============================================================================
class GridCellRenderer : public Gtk::CellRendererText
{
  public:
    GridCellRenderer(const int model_column, Gtk::TreeView* tv, GridView* gv);

    void cell_data(Gtk::CellRenderer*, const Gtk::TreeModel::iterator& it);

    void activate(const Gtk::TreePath& path);

    bool hits_click_area(const int x, const int y, const Gdk::Rectangle& area, const mforms::CellType type);

    int column() const {return _colid;}

    GridCell* cell_from(const Glib::ustring& path);
    GridCell* cell_from(const Gtk::TreePath& path);
    GridCell* cell_from(const Gtk::TreeIter& path);
  protected:
    virtual void on_edited(const Glib::ustring& path, const Glib::ustring& new_text);
    virtual void on_editing_canceled();

    #ifdef GLIBMM_VFUNCS_ENABLED
    virtual void render_vfunc(const Glib::RefPtr<Gdk::Drawable>& window, Gtk::Widget& widget, const Gdk::Rectangle& background_area, const Gdk::Rectangle& cell_area, const Gdk::Rectangle& expose_area, Gtk::CellRendererState flags);
    virtual Gtk::CellEditable* start_editing_vfunc(GdkEvent* event, Gtk::Widget& widget, const Glib::ustring& path, const Gdk::Rectangle& background_area, const Gdk::Rectangle& cell_area, Gtk::CellRendererState flags);
    virtual void get_size_vfunc(Gtk::Widget& widget, const Gdk::Rectangle* cell_area, int* x_offset, int* y_offset, int* width, int* height) const;
    #endif //GLIBMM_VFUNCS_ENABLED

  private:
    void do_shading(const Glib::RefPtr<Gdk::Drawable>& window, const Gdk::Rectangle& background_area);
    void combo_item_selected(const Glib::ustring& path);
    void editable_edit_done();
    const int                     _colid;
    Gtk::TreeView                *_tv;
    Glib::RefPtr<GridModel>       _store;
    GridCellEditable              _editable;
    GridCell                     *_cell;
    GridView                     *_gv;
    Gtk::TreeModel::iterator      _iter;
};

//==============================================================================
//
//==============================================================================
class GridView : public mforms::gtk::ViewImpl
{
  protected:
    virtual Gtk::Widget* get_outer() const { return &_top_box; }
  public:
    typedef mforms::SimpleGridPath  Path;
    typedef std::string     Value;

    GridView(mforms::SimpleGrid* self);

    int add_column(const std::string& name);

    Path append_row(const std::string& group_name);
    Path append_row(const Path& path);
    bool set_value(const Path& rid, const int col_id, const std::string& cv, const bool editable);
    bool set_value(const Path& rid, const int col_id, const char* cv, const bool editable);
    bool set_value(const Path& rid, const int col_id, bool cv, const bool editable);
    std::string get_value(const Path& rid, const int col_id);
    std::string get_value(const Path& rid, const int col_id, mforms::CellType* type);

    bool set_fg(const Path& rid, const int col_id, const double r, const double g, const double b);
    bool set_bg(const Path& rid, const int col_id, const double r, const double g, const double b);
    bool set_underline(const Path& rid, const int col_id, const bool is_on);

    // ownership of the vector @list will be passed to the GridCell
    bool set_enum_def(const Path& rid, const int col_id, std::vector<std::string>* list);
    // ownership of the char** @list will not be passed to the GridCell
    bool set_enum_def(const Path& rid, const int col_id, const char** const list);

    void shade(const Path& rid, const mforms::Shade shade, const int col_id = -1);
    void unshade(const Path& rid, const mforms::Shade shade, const int col_id = -1);
    bool has_shade(const Path& rid, const int col_id, const mforms::Shade s);
    void scroll_to_row(const Path& rid);

    //void set_search_key(const std::string& text);
    //void find_all();

    void content_edited(const Path& p, const int col)
    {
      _mgrid->signal_content_edited().emit(p, col);
    }
  private:
    //bool find_all_process_iter(const Gtk::TreeIter& it);
    //void copy_search_key();
    void _do_init();
    bool key_release_slot(GdkEventKey*);
    bool button_release_slot(GdkEventButton*);

    mforms::SimpleGrid                 *_mgrid;

    Glib::RefPtr<GridModel>             _model;
    mutable Gtk::VBox                   _top_box;
    Gtk::ScrolledWindow                 _scroll;
    Gtk::TreeView                       _tree;
    Gtk::HBox                           _search_panel;
    std::string                         _search_key;
    bool                                _init_done;

    void text_cell_data(Gtk::CellRenderer* cr, const Gtk::TreeModel::iterator& it, const int colid);

    void edited_slot(const Glib::ustring& path, const Glib::ustring& new_value, const int col);
    void edited_bool_slot(const Glib::ustring& path, const int col);
    void row_activated_slot(const Gtk::TreeModel::Path&, Gtk::TreeViewColumn*);

    void show_search_panel();
};


#define dprintf(...)
//#define dprintf     printf

//------------------------------------------------------------------------------
GridCell::GridCell()
     : _type(mforms::CellInvalid)
     , _fg_set(false)
     , _bg_set(false)
     , _underline(false)
     , _shade(mforms::ShadeNone)
{}

//------------------------------------------------------------------------------
GridCell::GridCell(const std::string& atext)
     : _type(mforms::CellText)
     , _text(atext)
     , _fg_set(false)
     , _bg_set(false)
     , _underline(false)
     , _shade(mforms::ShadeNone)
{}

//------------------------------------------------------------------------------
GridCell::GridCell(const bool v)
     : _type(mforms::CellBool)
     , _bool(v)
     , _fg_set(false)
     , _bg_set(false)
     , _underline(false)
     , _shade(mforms::ShadeNone)
{}

//------------------------------------------------------------------------------
GridCell::~GridCell()
{
  reset();
}

//------------------------------------------------------------------------------
void GridCell::reset()
{
  _enum_def.reset();
  _text.clear();
  _bool = 0;
  _type = mforms::CellInvalid;
}

//------------------------------------------------------------------------------
void GridCell::set_type(const mforms::CellType ct)
{
  _type = ct;
  if (_type == mforms::CellHeader)
  {
    _editable = false;
  }
}

//------------------------------------------------------------------------------
void GridCell::set(const std::string& s)
{
  _text = s;
  if (_type != mforms::CellEnum)
    _type = mforms::CellText;
}

//------------------------------------------------------------------------------
void GridCell::set(const bool s)
{
  _type = mforms::CellBool;
  _bool = s;
  _enum_def.reset();
}

//------------------------------------------------------------------------------
const std::string& GridCell::text() const
{
  return _text;
}

//------------------------------------------------------------------------------
bool GridCell::get_value(bool *b) const
{
  const bool ret = _type == mforms::CellBool;
  if (ret)
    *b = _bool;
  return ret;
}

//------------------------------------------------------------------------------
bool GridCell::get_value(std::string *s) const
{
  bool ret = (_type != mforms::CellInvalid);
  switch (_type)
  {
    case mforms::CellText:
    case mforms::CellEnum:
      *s = _text;
      break;
    default:
      ret = false;
      break;
  }
  return ret;
}

//------------------------------------------------------------------------------
std::string GridCell::as_string() const
{
  std::string ret;
  switch (_type)
  {
    case mforms::CellText:
    case mforms::CellEnum:
    case mforms::CellHeader:
      ret = _text;
      break;
    case mforms::CellBool:
      ret = _bool ? "1" : "0";
      break;
    default:
      break;
  }
  return ret;
}

//------------------------------------------------------------------------------
void GridCell::set_enum_def(GridCell::EnumDef* list)
{
  _enum_def = EnumDefRef(list);
  _type = mforms::CellEnum;
  _editable = true;
}

//------------------------------------------------------------------------------
void GridCell::set_fg(const double red, const double green, const double blue)
{
  _fg.set_rgb_p(red, green, blue);
  _fg_set = true;
}

//------------------------------------------------------------------------------
void GridCell::set_bg(const double red, const double green, const double blue)
{
  _bg.set_rgb_p(red, green, blue);
  _bg_set = true;
}




//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
GridModel::GridModel(Gtk::TreeView *treeview, const std::string& name)
     : Glib::ObjectBase(typeid(GridModel))
     , Glib::Object()
     , Gtk::TreeModel()
     , _stamp(3)
     , _ncols(0)
{}

//------------------------------------------------------------------------------
GridModel::~GridModel()
{}

//------------------------------------------------------------------------------
GridModel::Cells* GridModel::row_from_iter(const Gtk::TreeIter& iter)
{
  GridModel::Cells* ret = 0;
  const GtkTreeIter* const it = iter.gobj();
  if (it && it->stamp == _stamp)
  {
    const int gid = (long)it->user_data;
    GridRow* root_row = ((int)_root.size() > gid && gid >= 0) ? &(_root[gid]) : 0;
    if (root_row)
    {
      const int rid = (long)it->user_data2;
      if (rid >= 0)
      {
        if (rid < (int)root_row->rows.size())
          ret = &(root_row->rows[rid]);
      }
      else
        ret = &(root_row->row);
    }
  }
  return ret;
}


//------------------------------------------------------------------------------
void GridModel::add_column()
{
  ++_ncols;
  const int root_size = _root.size();
  for (int i = 0; i < root_size; ++i)
  {
    _root[i].row.resize(_ncols);
    std::vector<Cells>& rows = _root[i].rows;
    const int rows_count = rows.size();
    for (int j = 0; j < rows_count; ++j)
      rows[j].resize(_ncols);
  }
}

//------------------------------------------------------------------------------
Gtk::TreeModelFlags GridModel::get_flags_vfunc() const
{
  return Gtk::TreeModelFlags(0);
}

//------------------------------------------------------------------------------
int GridModel::get_n_columns_vfunc() const
{
  return _ncols;
}

//------------------------------------------------------------------------------
GType GridModel::get_column_type_vfunc(int index) const
{
  return G_TYPE_STRING;
}

//------------------------------------------------------------------------------
bool GridModel::iter_next_vfunc(const iterator& iter, iterator& iter_next) const
{
  bool ret = false;
  const GtkTreeIter* const it = iter.gobj();
  if (it && it->stamp == _stamp)
  {
    GtkTreeIter* itn = iter_next.gobj();
    itn->stamp = 0; // set stamp to initial invalid value
    int gid = (long)it->user_data;
    int rid = (long)it->user_data2;
    if (gid >= 0)
    {
      itn->user_data3 = (void*)-1;
      const GridRow* const row = (int)_root.size() > gid ? &(_root[gid]) : 0;
      // advance row first, if we have row id != -1, otherwise advance gid
      if (rid >= 0 && row)
      {
        const std::vector<Cells>& rows = row->rows;
        if (++rid < (int)rows.size())
        {
          itn->stamp = _stamp;
          itn->user_data  = it->user_data;
          itn->user_data2 = (void*)rid;
          ret = true;
        }
      }
      else if (++gid < (int)_root.size())
      {
        itn->stamp = _stamp;
        itn->user_data  = (void*)gid;
        itn->user_data2 = (void*)-1;
        ret = true;
      }
    }
  }
  dprintf("_stamp=%i, iter_next_vfunc(iter = '%s', iter_next='%s') = %i\n", _stamp, iter2str(iter).c_str(), iter2str(iter_next).c_str(), ret);
  return ret;
}

//------------------------------------------------------------------------------
bool GridModel::get_iter_vfunc(const Path& path, iterator& iter) const
{
  bool ret = false;
  GtkTreeIter* it = iter.gobj();
  const int path_size = path.size();

  if (path_size > 0)
  {
    const int gid = path[0];
    if (gid >= 0 && gid < (int)_root.size())
    {
      it->stamp      = _stamp;
      it->user_data  = (void*)gid;
      it->user_data3 = (void*)-1;
      const GridRow* const row = &(_root[gid]);
      if (path_size > 1)
        it->user_data2 = (void*)(path[1] < (int)row->rows.size() ? path[1]: -1);
      else
        it->user_data2 = (void*)-1;
      ret = true;
    }
  }

  dprintf("_stamp=%i, get_iter_vfunc(path = '%s', iter='%s') = %i\n", _stamp, path.to_string().c_str(), iter2str(iter).c_str(), ret);
  return ret;
}

//------------------------------------------------------------------------------
bool GridModel::iter_children_vfunc(const iterator& parent, iterator& iter) const
{
  bool ret = false;
  const GtkTreeIter* const pit = parent.gobj();
  GtkTreeIter* it = iter.gobj();
  it->stamp      = 0;
  it->user_data  = (void*)-1;
  it->user_data2 = (void*)-1;
  it->user_data3 = (void*)-1;

  if (pit && pit->stamp == _stamp)
  {
    const int gid = (long)pit->user_data;
    const int rid = (long)pit->user_data2;
    if (gid >= 0 && gid < (int)_root.size() && rid == -1)
    {
      it->stamp = _stamp;
      const GridRow* const root_row = &(_root[gid]);
      if (root_row->rows.size() > 0)
      {
        it->user_data  = pit->user_data;
        it->user_data2 = 0;
        it->user_data3 = (void*)-1;
        ret = true;
      }
      else
        it->stamp = 0;
    }
  }
  dprintf("_stamp=%i, iter_children_vfunc(parent = '%s', iter='%s') = %i\n", _stamp, iter2str(parent).c_str(), iter2str(iter).c_str(), ret);
  return ret;
}

//------------------------------------------------------------------------------
bool GridModel::iter_parent_vfunc(const iterator& child, iterator& iter) const
{
  bool ret = false;
  const GtkTreeIter* const cit = child.gobj();
  if (cit && cit->stamp == _stamp)
  {
    GtkTreeIter* it = iter.gobj();
    if (it)
    {
      it->stamp = _stamp;
      it->user_data = cit->user_data;
      it->user_data2 = it->user_data3 = (void*)-1;
      ret = true;
    }
  }
  dprintf("_stamp=%i, iter_parent_vfunc(child = '%s', iter='%s') = %i\n", _stamp, iter2str(child).c_str(), iter2str(iter).c_str(), ret);
  return ret;
}

//------------------------------------------------------------------------------
bool GridModel::iter_nth_child_vfunc(const iterator& parent, int n, iterator& iter) const
{
  bool ret = false;
  const GtkTreeIter* const pit = parent.gobj();
  if (pit && pit->stamp == _stamp)
  {
    GtkTreeIter* it = iter.gobj();
    if (it)
    {
      const int gid = (long)pit->user_data;
      if (gid >= 0 && (int)_root.size() > gid && (int)_root[gid].rows.size() > n && n >= 0)
      {
        it->stamp      = _stamp;
        it->user_data  = pit->user_data;
        it->user_data2 = (void*)n;
        it->user_data3 = (void*)-1;
        ret = true;
      }
      else
        it->stamp = 0;
    }
  }
  dprintf("_stamp=%i, iter_nth_child_vfunc(parent = '%s', n=%i, iter='%s') = %i\n", _stamp, iter2str(parent).c_str(), n, iter2str(iter).c_str(), ret);
  return ret;
}

//------------------------------------------------------------------------------
bool GridModel::iter_nth_root_child_vfunc(int n, iterator& iter) const
{
  bool ret = false;
  GtkTreeIter* it = iter.gobj();
  if (it)
  {
    if (n >= 0 && n < (int)_root.size())
    {
      it->stamp      = _stamp;
      it->user_data  = (void*)n;
      it->user_data2 = (void*)-1;
      it->user_data3 = (void*)-1;
      ret = true;
    }
    else
      it->stamp = 0;
  }
  dprintf("_stamp=%i, iter_nth_root_children_vfunc(n = %i, iter='%s') = %i\n", _stamp, n, iter2str(iter).c_str(), ret);
  return ret;
}

//------------------------------------------------------------------------------
bool GridModel::iter_has_child_vfunc(const iterator& iter) const
{
  bool ret = false;
  const GtkTreeIter* const it = iter.gobj();
  if (it && it->stamp == _stamp)
  {
    const int gid = (long)it->user_data;
    const int rid = (long)it->user_data2;
    if (rid == -1)
      ret = gid >= 0 && (int)_root.size() > gid && _root[gid].rows.size() > 0;
  }
  dprintf("_stamp=%i, iter_has_child_vfunc(iter='%s') = %i\n", _stamp, iter2str(iter).c_str(), ret);
  return ret;
}

//------------------------------------------------------------------------------
int GridModel::iter_n_children_vfunc(const iterator& iter) const
{
  int ret = 0;
  const GtkTreeIter* const it = iter.gobj();
  if (it && it->stamp == _stamp)
  {
    const int gid = (long)it->user_data;
    const int rid = (long)it->user_data2;
    if (rid == -1)
    {
      if (gid > 0 && gid < (int)_root.size())
        ret = _root[gid].rows.size();
    }
  }
  dprintf("_stamp=%i, iter_n_children_vfunc(iter='%s') = %i\n", _stamp, iter2str(iter).c_str(), ret);
  return ret;
}

//------------------------------------------------------------------------------
int GridModel::iter_n_root_children_vfunc() const
{
  dprintf("_stamp=%i, iter_n_root_children_vfunc() = %i\n", _stamp, _root.size());
  return _root.size();
}

//------------------------------------------------------------------------------
Gtk::TreeModel::Path GridModel::get_path_vfunc(const iterator& iter) const
{
  Gtk::TreeModel::Path  path;
  const GtkTreeIter* const it = iter.gobj();
  if (it && it->stamp == _stamp)
  {
    const int gid = (long)it->user_data;
    if (gid >= 0 && (int)_root.size() > gid)
    {
      path.append_index(gid);
      const int rid = (long)it->user_data2;
      if (rid >= 0 && rid < (int)_root[gid].rows.size())
        path.append_index(rid);
    }
  }
  return path;
}


//------------------------------------------------------------------------------
bool GridModel::iter_is_valid(const iterator& iter) const
{
  bool ret = false;
  const GtkTreeIter* const it = iter.gobj();
  if (it && it->stamp == _stamp)
  {
    const int gid = (long)it->user_data;
    if (gid >= 0 && (int)_root.size() > gid)
    {
      ret = true;
      const int rid = (long)it->user_data2;
      if (rid >= 0 && rid >= (int)(_root[gid].rows.size()))
        ret = false;
    }
  }
  return ret;
}

//------------------------------------------------------------------------------
void GridModel::set_value_impl(const iterator& row, int column, const Glib::ValueBase& value)
{
  const GtkTreeIter* const it = row.gobj();
  if (it && it->stamp == _stamp)
  {
    const int gid = (long)it->user_data;
    const int rid = (long)it->user_data2;
    GridCell* gcell = cell(gid, rid, column);

    if (gcell)
    {
      const GValue* const gv = value.gobj();
      const GType type = G_VALUE_TYPE(gv);

      switch (type)
      {
        case G_TYPE_BOOLEAN:
          {
            const bool val = g_value_get_boolean(gv);
            gcell->set(val);
            break;
          }
        case G_TYPE_STRING:
          {
            const std::string val = g_value_get_string(gv);
            gcell->set(val);
            break;
          }
        default:
          {
            break;
          }
      }//case
    }// if(cell)
  }//if (it...
}

//------------------------------------------------------------------------------
void GridModel::get_value_vfunc(const iterator& iter, int column, Glib::ValueBase& value) const
{
  const GtkTreeIter* const it = iter.gobj();
  if (it && it->stamp == _stamp)
  {
    const int gid = (long)it->user_data;
    const int rid = (long)it->user_data2;
    const GridCell* gcell = cell(gid, rid, column);

    if (gcell)
    {
      GValue* gv = value.gobj();

      switch (gcell->type())
      {
        case mforms::CellBool:
          {
            bool bv = false;
            if (gcell->get_value(&bv))
            {
              g_value_init(gv, G_TYPE_BOOLEAN);
              g_value_set_boolean(gv, bv);
            }
            break;
          }
        case mforms::CellText:
        case mforms::CellEnum:
          {
            std::string sv;
            if (gcell->get_value(&sv))
            {
              g_value_init(gv, G_TYPE_STRING);
              g_value_set_string(gv, sv.c_str());
            }
            break;
          }
        case mforms::CellInvalid:
        case mforms::CellHeader:
          break;
      }
    }
  }
}

//------------------------------------------------------------------------------
void GridModel::get_value_impl(const iterator& row, int column, Glib::ValueBase& value) const
{
  get_value_vfunc(row, column, value);
}

//------------------------------------------------------------------------------
void GridModel::reset_iter(iterator& iter) const throw()
{
  GtkTreeIter* it = iter.gobj();
  if (it)
  {
    it->stamp = 0;
    it->user_data = it->user_data2 = it->user_data3 = (void*)-1;
  }
}


//------------------------------------------------------------------------------
void GridModel::setup_group(GridRow* row, const std::string& group_name)
{
  if (row)
  {
    row->row.resize(_ncols);
    set_group_name(row, group_name);

    GridCell* gc = 0;
    for (int i = ((int)(row->row.size())) - 1; i >= 0; --i)
    {
      if ((gc = &(row->row[i])))
      {
        gc->set(false);
        gc->set_type(mforms::CellHeader);
      }
    }
  }
}

//------------------------------------------------------------------------------
void GridModel::set_group_name(GridRow* row, const std::string& name)
{
  row->header_text = name;
}

//------------------------------------------------------------------------------
std::string GridModel::group_name(GridRow* row) const
{
  return row->header_text;
}

//------------------------------------------------------------------------------
std::string GridModel::group_name(const Gtk::TreeIter& iter)
{
  std::string ret;
  const GtkTreeIter* const it = iter.gobj();
  if (it && it->stamp == _stamp)
  {
    const int gid = (long)it->user_data;
    if (gid >= 0 && (int)_root.size() > gid)
      ret = group_name(&(_root[gid]));
  }
  return ret;
}

//------------------------------------------------------------------------------
int GridModel::group_index(const std::string& group_id)
{
  int index = -1;
  const int root_size = _root.size();
  for (int i = 0; i < root_size; ++i)
  {
    if (_root[i].header_text == group_id)
    {
      index = i;
      break;
    }
  }
  return index;
}

//------------------------------------------------------------------------------
mforms::SimpleGridPath GridModel::append_row(const mforms::SimpleGridPath& path)
{
  Gtk::TreeIter giter;
  Gtk::TreePath gpath;

  if (path.size() == 1)
  {
    const int gid = path[0];
    if (gid >= 0 && gid < (int)_root.size())
    {
      GridRow* row = &(_root[gid]);
      if (row)
      {
        row->rows.push_back(Cells(_ncols));
        const int rid = row->rows.size() - 1;
        fill(gid, rid, &giter, &gpath);
        row_inserted(gpath, giter);

        Gtk::TreeIter piter;
        Gtk::TreePath ppath;
        fill(gid, -1, &piter, &ppath);
        row_has_child_toggled(ppath, piter);
      }
    }
  }
  else if (path.size() == 0)
  {
    _root.push_back(GridRow());
    const int gid = _root.size() - 1;

    GridRow* row = &(_root.back());
    row->row.resize(_ncols);

    fill(gid, -1, &giter, &gpath);
    row_inserted(gpath, giter);
  }

  return cast_path(gpath);
}

//------------------------------------------------------------------------------
// To add group pass non-existing group name
mforms::SimpleGridPath GridModel::append_row(const std::string& gid)
{
  int gindex = group_index(gid);

  GridRow *row = 0;
  if (gindex < 0)
  {
    _root.push_back(GridRow());
    gindex = _root.size() - 1;

    row = &(_root.back());
    setup_group(row, gid);

    Gtk::TreeIter giter;
    Gtk::TreePath gpath;
    fill(gindex, -1, &giter, &gpath);
    row_inserted(gpath, giter);
  }

  Gtk::TreeIter giter;
  Gtk::TreePath rpath;
  fill(gindex, -1, &giter, &rpath);

  return cast_path(rpath);
}

//------------------------------------------------------------------------------
GridCell* GridModel::cell(const int gid, const int rid, const int column)
{
  GridCell* cell = 0;
  GridRow* row = ((int)_root.size() > gid && gid >= 0) ? &(_root[gid]) : 0;
  Cells* cells = 0;
  if (row)
  {
    if (rid >= 0)
    {
      if (rid < (int)row->rows.size())
        cells = &(row->rows[rid]);
    }
    else
      cell = &(row->row[column]);
  }

  if (cells && (int)cells->size() > column)
    cell = &((*cells)[column]);

  return cell;
}

//------------------------------------------------------------------------------
const GridCell* GridModel::cell(const int gid, const int rid, const int column) const
{
  const GridCell* cell = 0;
  const GridRow* row = ((int)_root.size() > gid && gid >= 0) ? &(_root[gid]) : 0;
  const Cells* cells = 0;
  if (row)
  {
    if (rid >= 0)
    {
      if (rid < (int)row->rows.size())
        cells = &(row->rows[rid]);
    }
    else
      cell = &(row->row[column]);
  }

  if (cells && (int)cells->size() > column)
    cell = &((*cells)[column]);

  return cell;
}

//------------------------------------------------------------------------------
GridCell* GridModel::cell(const Gtk::TreePath& path, const int column, Gtk::TreeIter* iter)
{
  GridCell* rcell = 0;

  if (get_iter_vfunc(path, *iter))
  {
    const int gid = path.size() > 0 ? path[0] : -1;
    const int rid = path.size() > 1 ? path[1] : -1;

    rcell = cell(gid, rid, column);

    if (rcell)
    {
      fill(-1, -1, iter, 0);

      GtkTreeIter* it = iter->gobj();
      it->stamp = 0;
    }
  }
  return rcell;
}

//------------------------------------------------------------------------------
GridCell* GridModel::cell(const mforms::SimpleGridPath& path, const int column)
{
  const int gid = path.size() > 0 ? path[0] : -1;
  const int rid = path.size() > 1 ? path[1] : -1;

  GridCell* rcell = cell(gid, rid, column);

  return rcell;
}

//------------------------------------------------------------------------------
GridCell* GridModel::cell(const Gtk::TreeIter& iter, const int column)
{
  GridCell* gcell = 0;

  const GtkTreeIter* const it = iter.gobj();
  if (it && it->stamp == _stamp)
  {
    const int gid = (long)it->user_data;
    const int rid = (long)it->user_data2;

    gcell = cell(gid, rid, column);
  }

  return gcell;
}

//------------------------------------------------------------------------------
bool GridModel::fill(const int gid, const int rid, Gtk::TreeIter* iter, Gtk::TreePath *path)
{
  GtkTreeIter* it = iter->gobj();
  it->stamp = _stamp;
  it->user_data = (void*)gid;
  it->user_data2 = (void*)rid;
  it->user_data3 = (void*)-1;

  if (path && gid >= 0)
    path->append_index(gid);
  if (path && rid >= 0)
    path->append_index(rid);

  return true;
}

#define CELLCHECK_XOFF  8
#define CELLCHECK_YOFF  4

//------------------------------------------------------------------------------
GridCellEditable::GridCellEditable()
     : Glib::ObjectBase("GridCellEditable")
{
  _combo.get_entry()->property_has_frame() = false;
  _combo.get_entry()->gobj()->is_cell_renderer = true;
  add(_combo);
  show_all_children();

  signal_key_release_event().connect(sigc::mem_fun(this, &GridCellEditable::on_key_release));
  _combo.signal_changed().connect(sigc::mem_fun(_combo.get_entry(), &Gtk::Entry::grab_focus));
}

//------------------------------------------------------------------------------
bool GridCellEditable::on_key_release(GdkEventKey* ev)
{
  if (ev->keyval == GDK_Escape || ev->keyval == GDK_Return)
    editing_done();
  return true;
}

//------------------------------------------------------------------------------
void GridCellEditable::start_editing_vfunc(GdkEvent* event)
{
  _combo.get_entry()->start_editing(event);
  _combo.get_entry()->grab_focus();
}

//------------------------------------------------------------------------------
void GridCellEditable::on_editing_done()
{
  hide();
}

//------------------------------------------------------------------------------
void GridCellEditable::set_text(const std::string& s)
{
  _combo.get_entry()->set_text(s);
}

//------------------------------------------------------------------------------
std::string GridCellEditable::get_text()
{
  return _combo.get_entry()->get_text();
}

//------------------------------------------------------------------------------
void GridCellEditable::set_enum_def(const GridCell::EnumDefRef def)
{
  if (def)
  {
    _combo.clear_items();
    const int size = def->size();
    for (int i = 0; i < size; ++i)
    {
      _combo.append_text((*def)[i]);
    }
  }
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
GridCellRenderer::GridCellRenderer(const int model_column, Gtk::TreeView* tv, GridView* gv)
                 : Glib::ObjectBase(typeid(GridCellRenderer))
                 , _colid(model_column)
                 , _tv(tv)
                 , _cell(0)
                 , _gv(gv)
{
  _editable.signal_editing_done().connect(sigc::mem_fun(this, &GridCellRenderer::editable_edit_done));
}

//------------------------------------------------------------------------------
GridCell* GridCellRenderer::cell_from(const Glib::ustring& path)
{
  GridCell *cell(0);

  if (!_store)
    _store = Glib::RefPtr<GridModel>::cast_static(_tv->get_model());

  if (_store)
  {
    Gtk::TreeIter iter;
    cell = _store->cell(Gtk::TreePath(path), _colid, &iter);
  }

  return cell;
}

//------------------------------------------------------------------------------
GridCell* GridCellRenderer::cell_from(const Gtk::TreePath& path)
{
  GridCell *cell(0);

  if (!_store)
    _store = Glib::RefPtr<GridModel>::cast_static(_tv->get_model());

  if (_store)
  {
    Gtk::TreeIter iter;
    cell = _store->cell(path, _colid, &iter);
  }

  return cell;
}

//------------------------------------------------------------------------------
GridCell* GridCellRenderer::cell_from(const Gtk::TreeIter& iter)
{
  GridCell *cell(0);

  if (!_store)
    _store = Glib::RefPtr<GridModel>::cast_static(_tv->get_model());

  if (_store)
  {
    cell = _store->cell(iter, _colid);
  }

  return cell;
}

//------------------------------------------------------------------------------
void GridCellRenderer::editable_edit_done()
{
  edited(_editable.path(), _editable.get_text());
  _gv->content_edited(cast_path(_editable.gtk_path()), _colid);
}

//------------------------------------------------------------------------------
void GridCellRenderer::on_editing_canceled()
{
}

//------------------------------------------------------------------------------
void GridCellRenderer::on_edited(const Glib::ustring& path, const Glib::ustring& new_text)
{
  GridCell *cell = cell_from(path);
  if (cell)
  {
    cell->set(new_text);
    _gv->content_edited(cast_path(Gtk::TreePath(path)), _colid);
  }
}

//------------------------------------------------------------------------------
void blurred_line(cairo_t* cr, const int x1, const int y1, const int x2, const int y2)
{
  const int    width[] = {5  , 1};
  const double color[] = {0.8, 0};
  const int size = sizeof(width) / sizeof(int);

  for (int i = 0; i < size; ++i)
  {
    cairo_set_line_width(cr, width[i]);
    cairo_set_source_rgb(cr, 1, color[i], color[i]);
    cairo_move_to(cr, x1, y1);
    cairo_line_to(cr, x2, y2);
    cairo_stroke(cr);
  }
}

//------------------------------------------------------------------------------
void GridCellRenderer::do_shading(const Glib::RefPtr<Gdk::Drawable>& window, const Gdk::Rectangle& background_area)
{
  Cairo::RefPtr<Cairo::Context> ctxmm = window->create_cairo_context();
  cairo_t* cr = ctxmm->cobj();
  cairo_save(cr);
  if (_cell->has_shade(mforms::ShadeFind))
  {
    const GridModel::Cells* row = _store->row_from_iter(_iter);
    if (row)
    {
      const GridCell* prev_cell = (row && (_colid - 1) >= 0)? &((*row)[_colid - 1]) : 0;
      const GridCell* next_cell = (row && ((_colid + 1) < (int)row->size()))? &((*row)[_colid + 1]) : 0;

      //cairo_set_source_rgba(cr, 1, 0, 0, 1);
      cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
      //cairo_set_line_width(cr, 1);

      const int x1 = background_area.get_x();
      const int y1 = background_area.get_y();
      const int x2 = x1 + background_area.get_width();
      const int y2 = y1 + background_area.get_height();
      blurred_line(cr, x1, y1 + 2, x2, y1 + 2);
      blurred_line(cr, x1, y2 - 2, x2, y2 - 2);

      bool draw_left = false;
      if (prev_cell)
      {
        if (!prev_cell->has_shade(mforms::ShadeFind))
          draw_left = true;
      }
      else
        draw_left = true;

      if (draw_left)
        blurred_line(cr, x1, y1 + 2, x1, y2 - 2);

      bool draw_right = false;
      if (next_cell)
      {
        if (!next_cell->has_shade(mforms::ShadeFind))
          draw_right = true;
      }
      else
        draw_right = true;

      if (draw_right)
        blurred_line(cr, x2, y1 + 2, x2, y2 - 2);

      cairo_stroke(cr);
    }
  }

  if (_cell->has_shade(mforms::ShadeFilter))
  {
    const Gdk::Color clr = _tv->get_style()->get_base(Gtk::STATE_NORMAL);
    cairo_set_source_rgba(cr, clr.get_red_p(), clr.get_green_p(), clr.get_blue_p(), 0.95);
    cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
    gdk_cairo_rectangle(cr, background_area.gobj());
    cairo_fill(cr);
  }
  cairo_restore(cr);
}

//------------------------------------------------------------------------------
void GridCellRenderer::get_size_vfunc(Gtk::Widget& widget, const Gdk::Rectangle* cell_area, int* x_offset, int* y_offset, int* width, int* height) const
{
  Gtk::CellRendererText::get_size_vfunc(widget, cell_area, x_offset, y_offset, width, height);
  if (height && _cell && _cell->type() == mforms::CellHeader)
    *height += 8;
}

//------------------------------------------------------------------------------
void GridCellRenderer::render_vfunc(const Glib::RefPtr<Gdk::Drawable>& window
                                      ,Gtk::Widget& widget
                                      ,const Gdk::Rectangle& background_area
                                      ,const Gdk::Rectangle& cell_area
                                      ,const Gdk::Rectangle& expose_area,
                                      Gtk::CellRendererState flags
                                      )
{
  bool needs_text_render = true;
  property_underline() = Pango::UNDERLINE_NONE;
  property_foreground_set() = false;
  property_background_set() = false;
  set_alignment(0, 0.01);
  property_scale() = 1;
  bool draw_bg = false;
  bool draw_header = false;
  Gdk::Color bg;
  std::string cont;
  mforms::CellType type = mforms::CellInvalid;

  if (_cell)
  {
    type = _cell->type();
    if (_cell->fg_set())
    {
      Gdk::Color &color = _cell->fg();
      widget.get_colormap()->alloc_color(color, true, true);
      property_foreground_gdk() = color;
    }

    if (_cell->bg_set() && !(flags & Gtk::CELL_RENDERER_SELECTED))
    {
      widget.get_colormap()->alloc_color(_cell->bg(), true, true);
      bg = _cell->bg();
      draw_bg = true;
    }

    if (_cell->underline())
      property_underline() = Pango::UNDERLINE_LOW;

    const bool header = (type == mforms::CellHeader);
    if (header)
    {
      bg = _tv->get_style()->get_base(Gtk::STATE_PRELIGHT);
      draw_bg = true;
      if (_colid == _store->columns_count() - 1)
      {
        property_text() = _store->group_name(_iter);

        Gdk::Color fg = _tv->get_style()->get_text(Gtk::STATE_PRELIGHT);
        property_foreground_gdk() = fg;
        draw_header = true;
      }
      needs_text_render = false;
    }

  }

  if (draw_bg)
  {
    Cairo::RefPtr<Cairo::Context> ctxmm = window->create_cairo_context();
    cairo_t* cr = ctxmm->cobj();
    cairo_set_source_rgba(cr, bg.get_red_p(), bg.get_green_p(), bg.get_blue_p(), 0.9);
    cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);

    gdk_cairo_rectangle(cr, background_area.gobj());

    cairo_fill(cr);
  }

  if (type == mforms::CellBool)
  {
    bool b = false;
    if (_cell->get_value(&b))
    {
      GtkWidget *wdg = widget.gobj();
      gtk_paint_check(wdg->style,
                      window->gobj(),
                      GTK_STATE_NORMAL, b ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
                      background_area.gobj(), wdg, "cellcheck",
                      background_area.get_x() + CELLCHECK_XOFF,
                      background_area.get_y() + CELLCHECK_YOFF,
                      13, 13);
      needs_text_render = false;
    }
  }

  if (needs_text_render)
    Gtk::CellRendererText::render_vfunc(window, widget, background_area, cell_area, expose_area, flags);

  if (draw_header)
  {
    Gdk::Rectangle r = background_area;
    r.set_x(20);
    r.set_width(_tv->get_width() - 20);
    property_underline() = Pango::UNDERLINE_LOW;
    property_scale() = 1.1;

    Gtk::CellRendererText::render_vfunc(window, widget, r, r, r, flags);
  }

  // paint shading
  do_shading(window, background_area);
}

//------------------------------------------------------------------------------
void GridCellRenderer::cell_data(Gtk::CellRenderer* cr, const Gtk::TreeModel::iterator& it)
{
  property_text() = "";
  property_editable() = false;
  _cell = 0;
  _iter = it;

  if (it)
  {
    _cell = cell_from(it);
    if (_cell)
    {
      const mforms::CellType type = _cell->type();
      if (type != mforms::CellInvalid)
      {
        std::string s;
        if (_cell->get_value(&s))
        {
          property_markup() = s;
          if (type == mforms::CellText || type == mforms::CellEnum)
            property_editable() = _cell->is_editable();
        }
      }
    }
  }
}

//------------------------------------------------------------------------------
bool GridCellRenderer::hits_click_area(const int x, const int y, const Gdk::Rectangle& area, const mforms::CellType type)
{
  bool ret = false;
  switch (type)
  {
    case mforms::CellBool:
    {
      const int xa = area.get_x() + CELLCHECK_XOFF;
      const int ya = area.get_y() + CELLCHECK_YOFF;
      ret = (x >= xa && y >= ya && x <= (xa + 13) && y <= (ya + 13));
      break;
    }
    case mforms::CellEnum:
    case mforms::CellText:
    {
      int xo = 0, yo = 0, w = 0, h = 0;
      get_size(*_tv, area, xo, yo, w, h);
      xo += area.get_x();
      yo += area.get_y();
      ret = (x >= xo && y >= yo && x <= (xo + w) && y <= (yo + h));
      break;
    }
    case mforms::CellInvalid:
    case mforms::CellHeader:
      break;
  }

  return ret;
}

//------------------------------------------------------------------------------
void GridCellRenderer::activate(const Gtk::TreePath& path)
{
  GridCell *cell = cell_from(path);
  if (cell && cell->type() == mforms::CellBool)
  {
    bool b = false;
    if (cell->get_value(&b))
    {
      cell->set(!b);
      _store->row_changed(path, _store->get_iter(path));
      _gv->content_edited(cast_path(path), _colid);
    }
  }
}

//------------------------------------------------------------------------------
Gtk::CellEditable* GridCellRenderer::start_editing_vfunc(GdkEvent* event,
                                                               Gtk::Widget& widget,
                                                               const Glib::ustring& path,
                                                               const Gdk::Rectangle& background_area,
                                                               const Gdk::Rectangle& cell_area,
                                                               Gtk::CellRendererState flags)
{
  Gtk::CellEditable*  ce(0);
  GridCell*     cell = cell_from(path);
  const bool can_edit = cell && !cell->has_shade(mforms::ShadeFilter);

  if (can_edit)
  {
    if (cell)
    {
      _editable.set_type(cell->type());
      switch (cell->type())
      {
        case mforms::CellEnum:
        {
          _editable.set_enum_def(cell->get_enum_def());
          std::string s;
          cell->get_value(&s);

          _editable.set_text(s);
          _editable.set_path(path);

          ce = &_editable;

          _editable.show_all();
          break;
        }
        default:
          break;
      }
    }
    else
      _editable.set_type(mforms::CellText);
  }

  if (can_edit && !ce)
    ce = Gtk::CellRendererText::start_editing_vfunc(event, widget, path, background_area, cell_area, flags);

  return ce;
}







//------------------------------------------------------------------------------
GridView::GridView(mforms::SimpleGrid* self)
         : mforms::gtk::ViewImpl(self)
         , _mgrid(self)
         , _init_done(false)
{
  _scroll.add(_tree);
  _top_box.pack_start(_scroll, true, true);
  _top_box.pack_start(_search_panel, false, false, 4);

  _tree.set_headers_visible(true);
  _tree.signal_button_release_event().connect(sigc::mem_fun(this, &GridView::button_release_slot));
  _tree.signal_row_activated().connect(sigc::mem_fun(this, &GridView::row_activated_slot));
  _tree.set_level_indentation(10);

  _tree.signal_key_release_event().connect(sigc::mem_fun(this, &GridView::key_release_slot));
  _scroll.show_all();
}

//------------------------------------------------------------------------------
void GridView::_do_init()
{
  _model = GridModel::create(&_tree, "");
  _tree.set_model(_model);
  _init_done = true;
}

//------------------------------------------------------------------------------
bool GridView::key_release_slot(GdkEventKey* e)
{
  if (e->keyval == GDK_f && e->state & GDK_CONTROL_MASK)
    show_search_panel();
  return false;
}

//------------------------------------------------------------------------------
int GridView::add_column(const std::string& name)
{
  if (!_init_done)
    _do_init();

  _model->add_column();
  const int colid = _model->columns_count() - 1;

  GridCellRenderer*    crt = Gtk::manage(new GridCellRenderer(colid, &_tree, this));
  Gtk::TreeViewColumn*     tvcol = Gtk::manage(new Gtk::TreeViewColumn(name, *crt));

  tvcol->set_cell_data_func(*crt, sigc::mem_fun(crt, &GridCellRenderer::cell_data));
  tvcol->set_resizable(true);

  _tree.append_column(*tvcol);
  return colid;
}

//------------------------------------------------------------------------------
void GridView::row_activated_slot(const Gtk::TreeModel::Path& path, Gtk::TreeViewColumn* col)
{
  GridCellRenderer* cr = dynamic_cast<GridCellRenderer*>(col->get_first_cell_renderer());
  if (cr)
  {
    GridCell* cell = cr->cell_from(path);
    if (cell)
    {
      if (cell->type() == mforms::CellHeader)
      {
        if (_tree.row_expanded(path))
          _tree.collapse_row(path);
        else
          _tree.expand_row(path, false);
      }
    }
  }
}

//------------------------------------------------------------------------------
bool GridView::button_release_slot(GdkEventButton* ev)
{
  if (ev->button == 1)
  {
    Gtk::TreePath path;
    Gtk::TreeViewColumn *col(0);
    int cell_x = 0;
    int cell_y = 0;
    if (_tree.get_path_at_pos(ev->x, ev->y, path, col, cell_x, cell_y))
    {
      if (col)
      {
        Gdk::Rectangle cell_rect;
        _tree.get_background_area(path, *col, cell_rect);
        GridCellRenderer* cr = dynamic_cast<GridCellRenderer*>(col->get_first_cell_renderer());
        if (cr)
        {
          GridCell* cell = cr->cell_from(path);
          if (cell && !cell->has_shade(mforms::ShadeFilter))
          {
            const bool hits = cr->hits_click_area(ev->x, ev->y, cell_rect, cell->type());
            switch (cell->type())
            {
              case mforms::CellBool:
              {
                if (hits)
                {
                  if (cell->is_editable())
                    cr->activate(path);
                  else
                    _mgrid->signal_ro_content_clicked().emit(cast_path(path), cr->column());
                }
                break;
              }
              case mforms::CellEnum:
              case mforms::CellText:
              {
                if (hits && !cell->is_editable())
                  _mgrid->signal_ro_content_clicked().emit(cast_path(path), cr->column());
                break;
              }
              case mforms::CellInvalid:
              case mforms::CellHeader:
                break;
            }
          }
        }
      }
    }
  }
  return false;
}

//------------------------------------------------------------------------------
GridView::Path GridView::append_row(const std::string& group_name)
{
  if (!_init_done)
    _do_init();

  return _model->append_row(group_name);
}

//------------------------------------------------------------------------------
GridView::Path GridView::append_row(const Path& path)
{
  if (!_init_done)
    _do_init();

  return _model->append_row(path);
}

//------------------------------------------------------------------------------
bool GridView::set_value(const GridView::Path& rid, const int col_id, const GridView::Value& cv, const bool editable)
{
  Gtk::TreeIter iter;
  const Gtk::TreePath path = cast_path(rid);
  GridCell *cell = _model->cell(path, col_id, &iter);
  if (cell)
  {
    cell->set_editable(editable);
    cell->set(cv);
    _model->row_changed(path, iter);
  }
  return cell;
}

//------------------------------------------------------------------------------
bool GridView::set_value(const GridView::Path& rid, const int col_id, const char* cv, const bool editable)
{
  return set_value(rid, col_id, std::string(cv), editable);
}

//------------------------------------------------------------------------------
bool GridView::set_value(const GridView::Path& rid, const int col_id, bool cv, const bool editable)
{
  Gtk::TreeIter iter;
  const Gtk::TreePath path = cast_path(rid);
  GridCell *cell = _model->cell(path, col_id, &iter);
  if (cell)
  {
    cell->set_editable(editable);
    cell->set(cv);
    _model->row_changed(path, iter);
  }
  return cell;
}

//------------------------------------------------------------------------------
std::string GridView::get_value(const Path& rid, const int col_id)
{
  std::string s;
  const GridCell *cell = _model->cell(rid, col_id);

  if (cell)
    s = cell->as_string();

  return s;
}

//------------------------------------------------------------------------------
std::string GridView::get_value(const Path& rid, const int col_id, mforms::CellType* type)
{
  std::string s;
  const GridCell *cell = _model->cell(rid, col_id);

  if (cell)
  {
    s = cell->as_string();
    if (type)
      *type = cell->type();
  }

  return s;
}

//------------------------------------------------------------------------------
bool GridView::set_fg(const Path& rid, const int col_id, const double r, const double g, const double b)
{
  GridCell *cell = _model->cell(rid, col_id);
  if (cell)
    cell->set_fg(r, g, b);
  return cell;
}

//------------------------------------------------------------------------------
bool GridView::set_bg(const Path& rid, const int col_id, const double r, const double g, const double b)
{
  GridCell *cell = _model->cell(rid, col_id);
  if (cell)
    cell->set_bg(r, g, b);
  return cell;
}

//------------------------------------------------------------------------------
bool GridView::set_underline(const Path& rid, const int col_id, const bool is_on)
{
  GridCell *cell = _model->cell(rid, col_id);
  if (cell)
    cell->set_underline(is_on);
  return cell;
}

//------------------------------------------------------------------------------
bool GridView::set_enum_def(const Path& rid, const int col_id, std::vector<std::string>* list)
{
  GridCell *cell = _model->cell(rid, col_id);
  if (cell)
    cell->set_enum_def(list);
  return cell;
}

//------------------------------------------------------------------------------
bool GridView::set_enum_def(const Path& rid, const int col_id, const char** const list)
{
  if (list)
  {
    std::auto_ptr<GridCell::EnumDef> def(new GridCell::EnumDef());

    for (const char** l = list; *l != 0; ++l)
      def->push_back(*l);

    if (def->size() > 0)
      set_enum_def(rid, col_id, def.release());
  }

  return list != 0;
}

//------------------------------------------------------------------------------
void GridView::shade(const Path& rid, const mforms::Shade shade, const int col_id)
{
  GridCell *cell;
  if (col_id >= 0)
  {
    cell = _model->cell(rid, col_id);
    if (cell)
      cell->set_shade(shade);
  }
  else
  {
    const int size = _model->get_n_columns();
    for (int i = 0; i < size; ++i)
    {
      cell = _model->cell(rid, i);
      if (cell)
        cell->set_shade(shade);
    }
  }
}

//------------------------------------------------------------------------------
void GridView::unshade(const Path& rid, const mforms::Shade shade, const int col_id)
{
  GridCell *cell;
  if (col_id >= 0)
  {
    cell = _model->cell(rid, col_id);
    if (cell)
      cell->set_shade(shade);
  }
  else
  {
    const int size = _model->get_n_columns();
    for (int i = 0; i < size; ++i)
    {
      cell = _model->cell(rid, i);
      if (cell)
        cell->unset_shade(shade);
    }
  }
}

//------------------------------------------------------------------------------
bool GridView::has_shade(const Path& rid, const int col_id, const mforms::Shade s)
{
  bool ret = false;
  GridCell *cell;
  if (col_id >= 0)
  {
    cell = _model->cell(rid, col_id);
    if (cell)
      ret = cell->has_shade(s);
  }
  return ret;
}

//------------------------------------------------------------------------------
void GridView::scroll_to_row(const Path& rid)
{
  _tree.scroll_to_row(cast_path(rid));
}

//------------------------------------------------------------------------------
void GridView::show_search_panel()
{
  if (0 == _search_panel.get_data("created"))
  {
    // emit signal, or call callback
  }
}

}


//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
bool create(mforms::SimpleGrid* self)
{
  GridView* gv = new GridView(self);
  return gv;
}

//------------------------------------------------------------------------------
int add_column(mforms::SimpleGrid* self, const std::string& name)
{
  GridView* cb = self->get_data<GridView>();
  return cb->add_column(name);
}

//------------------------------------------------------------------------------
mforms::SimpleGridPath append_header(mforms::SimpleGrid* self, const std::string& gid)
{
  GridView* cb = self->get_data<GridView>();
  return cb->append_row(gid);
}

//------------------------------------------------------------------------------
mforms::SimpleGridPath append_row(mforms::SimpleGrid* self, const mforms::SimpleGridPath& path)
{
  GridView* cb = self->get_data<GridView>();
  return cb->append_row(path);
}

//------------------------------------------------------------------------------
bool set_str_value(mforms::SimpleGrid* self, const mforms::SimpleGridPath& rid, const int col_id, const std::string& cv, const bool editable)
{
  GridView* cb = self->get_data<GridView>();
  return cb->set_value(rid, col_id, cv, editable);
}

//------------------------------------------------------------------------------
bool set_bool_value(mforms::SimpleGrid* self, const mforms::SimpleGridPath& rid, const int col_id, bool cv, const bool editable)
{
  GridView* cb = self->get_data<GridView>();
  return cb->set_value(rid, col_id, cv, editable);
}

//------------------------------------------------------------------------------
std::string get_value(mforms::SimpleGrid* self, const mforms::SimpleGridPath& rid, const int col_id, mforms::CellType* type)
{
  GridView* cb = self->get_data<GridView>();
  return cb->get_value(rid, col_id, type);
}

//------------------------------------------------------------------------------
bool set_fg(mforms::SimpleGrid* self, const mforms::SimpleGridPath& rid, const int col_id, const double r, const double g, const double b)
{
  GridView* cb = self->get_data<GridView>();
  return cb->set_fg(rid, col_id, r, g, b);
}

//------------------------------------------------------------------------------
bool set_bg(mforms::SimpleGrid* self, const mforms::SimpleGridPath& rid, const int col_id, const double r, const double g, const double b)
{
  GridView* cb = self->get_data<GridView>();
  return cb->set_bg(rid, col_id, r, g, b);
}

//------------------------------------------------------------------------------
bool set_underline(mforms::SimpleGrid* self, const mforms::SimpleGridPath& rid, const int col_id, const bool is_on)
{
  GridView* cb = self->get_data<GridView>();
  return cb->set_underline(rid, col_id, is_on);
}

//------------------------------------------------------------------------------
bool set_enum_def(mforms::SimpleGrid* self, const mforms::SimpleGridPath& rid, const int col_id, std::vector<std::string>* list)
{
  GridView* cb = self->get_data<GridView>();
  return cb->set_enum_def(rid, col_id, list);
}

//------------------------------------------------------------------------------
bool set_enum_def_c(mforms::SimpleGrid* self, const mforms::SimpleGridPath& rid, const int col_id, const char** const list)
{
  GridView* cb = self->get_data<GridView>();
  return cb->set_enum_def(rid, col_id, list);
}

//------------------------------------------------------------------------------
void shade(mforms::SimpleGrid* self, const mforms::SimpleGridPath& rid, const mforms::Shade shade, const int col_id)
{
  GridView* cb = self->get_data<GridView>();
  return cb->shade(rid, shade, col_id);
}

//------------------------------------------------------------------------------
void unshade(mforms::SimpleGrid* self, const mforms::SimpleGridPath& rid, const mforms::Shade shade, const int col_id)
{
  GridView* cb = self->get_data<GridView>();
  return cb->unshade(rid, shade, col_id);
}

//------------------------------------------------------------------------------
bool has_shade(mforms::SimpleGrid* self, const mforms::SimpleGridPath& rid, const int col_id, const mforms::Shade s)
{
  GridView* cb = self->get_data<GridView>();
  return cb->has_shade(rid, col_id, s);
}

//------------------------------------------------------------------------------
void scroll_to_row(mforms::SimpleGrid* self, const mforms::SimpleGridPath& rid)
{
  GridView* cb = self->get_data<GridView>();
  return cb->scroll_to_row(rid);
}

//------------------------------------------------------------------------------
void SimpleGrid_init()
{
  ::mforms::ControlFactory *f = ::mforms::ControlFactory::get_instance();
  f->_simple_grid_impl.create = &create;
  f->_simple_grid_impl.add_column = &add_column;
  f->_simple_grid_impl.append_header = append_header;
  f->_simple_grid_impl.append_row = append_row;
  f->_simple_grid_impl.set_str_value = set_str_value;
  f->_simple_grid_impl.set_bool_value = set_bool_value;
  f->_simple_grid_impl.get_value = get_value;
  f->_simple_grid_impl.set_fg = set_fg;
  f->_simple_grid_impl.set_bg = set_bg;
  f->_simple_grid_impl.set_underline = set_underline;
  f->_simple_grid_impl.set_enum_def = set_enum_def;
  f->_simple_grid_impl.set_enum_def_c = set_enum_def_c;
  f->_simple_grid_impl.shade = shade;
  f->_simple_grid_impl.unshade = unshade;
  f->_simple_grid_impl.has_shade = has_shade;
  f->_simple_grid_impl.scroll_to_row = &scroll_to_row;
}
