/* 
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#ifndef _MFORMS_TOOLBAR_H_
#define _MFORMS_TOOLBAR_H_

#include <mforms/box.h>
#include <mforms/app.h>
#include <vector>
#include <sigc++/sigc++.h>

namespace mforms {
  class ToolBarItem;
  class ToolBar;

  enum ToolBarType
  {
    MainToolBar,
    ToolPickerToolBar, // only toggle and separator items are required to work
    OptionsToolBar
  };
    
  enum ToolBarItemType
  {
    LabelItem,
    ActionItem,
    ToggleItem,
    SearchFieldItem,
    SelectorItem,
    ColorSelectorItem,
    SeparatorItem,
    ExpanderItem,
    ImageBoxItem
  };
  
  struct MFORMS_EXPORT ToolBarImplPtrs
  {
    bool (__stdcall *create_tool_bar)(ToolBar *item, ToolBarType type);
    void (__stdcall *insert_item)(ToolBar *toolbar, int index, ToolBarItem *item);
    void (__stdcall *remove_item)(ToolBar *toolbar, ToolBarItem *item);

    bool (__stdcall *create_tool_item)(ToolBarItem *item, ToolBarItemType type);
    void (__stdcall *set_item_icon)(ToolBarItem *item, const std::string&);
    void (__stdcall *set_item_alt_icon)(ToolBarItem *item, const std::string&);
    void (__stdcall *set_item_text)(ToolBarItem *item, const std::string&);
    std::string (__stdcall *get_item_text)(ToolBarItem *item);
    void (__stdcall *set_item_enabled)(ToolBarItem *item, bool);
    bool (__stdcall *get_item_enabled)(ToolBarItem *item);
    void (__stdcall *set_item_checked)(ToolBarItem *item, bool);
    bool (__stdcall *get_item_checked)(ToolBarItem *item);
    void (__stdcall *set_item_tooltip)(ToolBarItem *item, const std::string&);

    // For selector items only.
    void (__stdcall *set_selector_items)(ToolBarItem* item, const std::vector<std::string>& values);
  }; 
  
  
  // base abstract class for menuitem and menubase
  class MFORMS_EXPORT ToolBar : public View, public sigc::trackable
  {
  protected:    
    ToolBarImplPtrs *_impl;
    std::vector<ToolBarItem*> _items;
    ToolBarType _type;
    
  public:
    ToolBar(ToolBarType type = MainToolBar);
    virtual ~ToolBar();
    
    ToolBarType get_type() { return _type; }
    
    std::vector<ToolBarItem*> &get_items() { return _items; }
    
    ToolBarItem *find_item(const std::string &name);
    
    void set_item_enabled(const std::string &name, bool flag);
    void set_item_checked(const std::string &name, bool flag);
    
    void add_item(ToolBarItem *item);
    void insert_item(int index, ToolBarItem *item);
    void remove_all();
    void remove_item(ToolBarItem *item);
    
    void validate();
  };
  
  class MFORMS_EXPORT ToolBarItem : public Object
  {
  public:
    ToolBarItem(ToolBarItemType type = ActionItem);
    ToolBarItemType get_type() { return _type; }
    
    // Caption for labels, selected entry in drop down items, search text in search fields.
    void set_text(const std::string &text);
    std::string get_text();

    void set_tooltip(const std::string &text);

    void set_icon(const std::string &path);
    void set_alt_icon(const std::string &path);
    
    void set_enabled(bool flag);
    bool get_enabled();

    void set_checked(bool flag);
    bool get_checked();

    sigc::signal<void> &signal_activated() { return _clicked_signal; }
    
    void set_name(const std::string &name);
    std::string get_name() { return _name; }

    void set_selector_items(const std::vector<std::string>& values);

    void set_validator(const sigc::slot<bool> &slot);
    
  public:
    void callback();
    
    void validate();
  private:
    ToolBarImplPtrs *_impl;
    std::string _name;
    ToolBarItemType _type;
    sigc::signal<void> _clicked_signal;
    sigc::slot<bool> _validate;
  };
};

#endif
