/* 
 * Copyright (c) 2008, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

/**
 * Implementation of the mforms view, which is the base for most of the visual controls in mforms.
 */

#include "stdafx.h"

#include <mforms/view.h>
#include <mforms/mforms.h>
#include <mforms/app.h>
#include <stdio.h>

using namespace mforms;
using namespace MySQL;

//--------------------------------------------------------------------------------------------------

View::View()
{
  _parent= 0;
  _view_impl= &ControlFactory::get_instance()->_view_impl;
  _resize_mode= ResizeBoth;
  _layout_dirty= true;
}

//--------------------------------------------------------------------------------------------------

View::~View()
{
  App::get()->view_destroyed(this);
  
  set_destroying();
  if (_parent)
    _parent->remove_subview(this);
  
  std::list<std::pair<View*,bool> >::iterator iter;
  
  while ((iter = _subviews.begin()) != _subviews.end())
  {
    bool managed = iter->second;
    View *sv = iter->first;
    
    _subviews.erase(iter);
    if (managed)
    {
      sv->release();
    }
  }

  // Let the frontend delete all resources it allocated.
  if (_view_impl->destroy)
    _view_impl->destroy(this);
}

//--------------------------------------------------------------------------------------------------

void View::set_managed()
{
  Object::set_managed();
  if (_parent)
  {
    for (std::list<std::pair<View*,bool> > ::iterator iter= _parent->_subviews.begin(); iter != _parent->_subviews.end(); ++iter)
    {
      if (iter->first == this)
      {
        iter->second = true;
        break;
      }
    }    
  }
}

//--------------------------------------------------------------------------------------------------

void View::add_subview(View *sv)
{
  _subviews.push_back(std::make_pair(sv, sv->_managed));
}

//--------------------------------------------------------------------------------------------------

void View::remove_subview(View *sv)
{
  if (!is_destroying())
  {
    for (std::list<std::pair<View*,bool> > ::iterator iter= _subviews.begin(); iter != _subviews.end(); ++iter)
    {
      if (iter->first == sv)
      {
        _subviews.erase(iter);
        break;
      }
    }
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * Searches for a subview with the given name in this view or any of its subviews using a depth-first search.
 */
View *View::find_subview(const std::string &name)
{
  for (std::list<std::pair<View*,bool> > ::const_iterator iter= _subviews.begin(); iter != _subviews.end(); ++iter)
  {
    if (iter->first->get_name() == name)
      return iter->first;
    
    View *sv= iter->first->find_subview(name);
    if (sv)
      return sv;
  }
  return 0;  
}

//--------------------------------------------------------------------------------------------------

int View::get_subview_index(View *sv)
{
  int i= 0;
  for (std::list<std::pair<View*,bool> > ::const_iterator iter= _subviews.begin(); iter != _subviews.end(); ++iter, ++i)
  {
    if (iter->first == sv)
      return i;
  }
  return -1;  
}

//--------------------------------------------------------------------------------------------------

View *View::get_subview_at_index(int index)
{
  int i= 0;
  for (std::list<std::pair<View*,bool> > ::const_iterator iter= _subviews.begin(); iter != _subviews.end(); ++iter, ++i)
  {
    if (i == index)
      return iter->first;
  }
  return 0;  
}

//--------------------------------------------------------------------------------------------------

/**
 * Returns true if the given subview is a direct child of this view.
 */
bool View::contains_subview(View* subview)
{
  return get_subview_index(subview) >= 0;
}

//--------------------------------------------------------------------------------------------------

AutoResizeMode View::get_resize_mode()
{ 
  return _resize_mode;
}

//--------------------------------------------------------------------------------------------------

void View::set_resize_mode(AutoResizeMode mode)
{
  _resize_mode= mode;
}

//--------------------------------------------------------------------------------------------------

void View::set_name(const std::string &name)
{
  _name= name;

  // Optional implementation.
  if (_view_impl->set_name)
    _view_impl->set_name(this, name);
}

//--------------------------------------------------------------------------------------------------

void View::set_tooltip(const std::string &text)
{
  _view_impl->set_tooltip(this, text);
}

//--------------------------------------------------------------------------------------------------

void View::set_font(const std::string &fontDescription)
{
  _view_impl->set_font(this, fontDescription);
}

//--------------------------------------------------------------------------------------------------

std::string View::get_name()
{ 
  return _name; 
}

//--------------------------------------------------------------------------------------------------

void View::set_parent(View *parent)
{
  _parent= parent;
  if (_managed)
    set_managed();
}

//--------------------------------------------------------------------------------------------------

View* View::get_parent() const
{
  return _parent; 
}

//--------------------------------------------------------------------------------------------------

Form *View::get_parent_form() const
{
  View *parent= get_parent();
  Form *form= 0;
  while (parent && (form = dynamic_cast<Form*>(parent)) == 0)
    parent= parent->get_parent();

  return form;
}

//--------------------------------------------------------------------------------------------------

int View::get_width()
{
  return (*_view_impl->get_width)(this);
}

//--------------------------------------------------------------------------------------------------

int View::get_height()
{
  return (*_view_impl->get_height)(this);
}

//--------------------------------------------------------------------------------------------------

int View::get_preferred_width()
{
  return (*_view_impl->get_preferred_width)(this);
}

//--------------------------------------------------------------------------------------------------

int View::get_preferred_height()
{
  return (*_view_impl->get_preferred_height)(this);
}

//--------------------------------------------------------------------------------------------------

int View::get_x()
{
  return (*_view_impl->get_x)(this);
}

//--------------------------------------------------------------------------------------------------

int View::get_y()
{
  return (*_view_impl->get_y)(this);
}

//--------------------------------------------------------------------------------------------------

void View::set_position(int x, int y)
{
  (*_view_impl->set_position)(this, x, y);
}

//--------------------------------------------------------------------------------------------------

void View::set_size(int width, int height)
{
  set_layout_dirty(true);
  (*_view_impl->set_size)(this, width, height);
}

//--------------------------------------------------------------------------------------------------

void View::set_padding(const Geometry::Padding& padding)
{
  set_layout_dirty(true);
  if (_view_impl->set_padding)
    (*_view_impl->set_padding)(this, padding);
}

//--------------------------------------------------------------------------------------------------

void View::client_to_screen(Geometry::Point& position)
{
  (*_view_impl->client_to_screen)(this, position);
}

//--------------------------------------------------------------------------------------------------

void View::show(bool flag)
{
  (*_view_impl->show)(this, flag);
}

//--------------------------------------------------------------------------------------------------

bool View::is_shown()
{
  return (*_view_impl->is_shown)(this);
}

//--------------------------------------------------------------------------------------------------

void View::set_enabled(bool flag)
{
  (*_view_impl->set_enabled)(this, flag);
}

//--------------------------------------------------------------------------------------------------

void View::set_needs_repaint()
{
  _view_impl->set_needs_repaint(this);
}

//--------------------------------------------------------------------------------------------------

void View::needs_resize()
{
  if (_parent)
    _parent->needs_resize();
}

//--------------------------------------------------------------------------------------------------

void View::set_layout_dirty(bool value)
{
  _layout_dirty= value;
  if (_parent != NULL && value)
    _parent->set_layout_dirty(true);
}

//--------------------------------------------------------------------------------------------------

bool View::is_layout_dirty()
{
  return _layout_dirty;
}

//--------------------------------------------------------------------------------------------------

void View::relayout()
{
  if (_parent != NULL)
    _parent->relayout();
  (*_view_impl->relayout)(this);
}

//--------------------------------------------------------------------------------------------------

void View::on_resize()
{
  needs_resize();
  _signal_resized.emit();
}

//--------------------------------------------------------------------------------------------------

void View::suspend_layout()
{
  if (_view_impl->suspend_layout)
    _view_impl->suspend_layout(this, true);
}

//--------------------------------------------------------------------------------------------------

void View::resume_layout()
{
  if (_view_impl->suspend_layout)
    _view_impl->suspend_layout(this, false);
}

//--------------------------------------------------------------------------------------------------

void View::set_front_color(const std::string &color)
{
  _view_impl->set_front_color(this, color);
}

//--------------------------------------------------------------------------------------------------

std::string View::get_front_color()
{
  return _view_impl->get_front_color(this);
}

//--------------------------------------------------------------------------------------------------

void View::set_back_color(const std::string &color)
{
  _view_impl->set_back_color(this, color);
}

//--------------------------------------------------------------------------------------------------

std::string View::get_back_color()
{
  return _view_impl->get_back_color(this);
}

//--------------------------------------------------------------------------------------------------

void View::set_back_image(const std::string &path, ImageLayout layout)
{
  _view_impl->set_back_image(this, path, layout);
}

//--------------------------------------------------------------------------------------------------

void View::show_retain_counts(int depth)
{
  printf("%*s '%s' (%i)\n", depth, "--", get_name().c_str(), retain_count());

  for (std::list<std::pair<View*,bool> >::const_iterator iter= _subviews.begin();
       iter != _subviews.end(); ++iter)
  {
    iter->first->show_retain_counts(depth+1);
  }
}

//--------------------------------------------------------------------------------------------------

void View::flush_events()
{
  if (_view_impl->flush_events)
    _view_impl->flush_events(this);
}

//--------------------------------------------------------------------------------------------------

