/* 
 * Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

/**
 * Contains the implementation of the .NET wrapper for the mforms application class.
 */

#include "stdafx.h"

#include "base/log.h"
#include "wf_app.h"
#include "GrtTemplates.h"

using namespace System::Threading;

using namespace MySQL::Forms;
using namespace MySQL::Grt;
using namespace MySQL::Geometry;
using namespace MySQL::Drawing;

//--------------------------------------------------------------------------------------------------

ENABLE_LOG("mforms.wr")

AppImpl::AppImpl(MySQL::Forms::Manager^ mgr,
  StrStrStrStrDelegate^ show_file_dialog, 
  VoidStrDelegate^ show_status_text,
  BoolStrStrFloatDelegate^ show_progress,
  BoolStrIntStrPtrDelegate^ request_input,
  CanvasViewStringStringIntPtrDelegate^ create_diagram,
  VoidCanvasViewDelegate^ destroy_view,
  VoidCanvasViewDelegate^ switched_view,
  VoidCanvasViewDelegate^ tool_changed,
  IntPtrGRTManagerModuleStrStrGrtListFlagsDelegate^ open_editor,
  VoidIntPtrDelegate^ show_editor,
  VoidIntPtrDelegate^ hide_editor,
  VoidRefreshTypeStringIntPtrDelegate^ refresh_gui,
  VoidBoolDelegate^ lock_gui,
  VoidStrDelegate^ perform_command,
  VoidDelegate^ quit_application,
  StringAppCommandAppViewStringDelegate^ app_command)
  : _callbacks(new ::wb::WBFrontendCallbacks())
{
  // Store a reference to this class in the backend.
  _cb_handle = GCHandle::Alloc(this);
  IntPtr pointer= GCHandle::ToIntPtr(_cb_handle);
  mforms::App::get()->set_data(pointer.ToPointer());

  set_show_file_dialog(show_file_dialog);
  set_show_status_text(show_status_text);
  set_show_progress(show_progress);
  set_request_input(request_input);

  set_create_diagram(create_diagram);
  set_destroy_view(destroy_view);
  set_switched_view(switched_view);
  set_tool_changed(tool_changed);

  set_open_editor(open_editor);
  set_show_editor(show_editor);
  set_hide_editor(hide_editor);

  set_refresh_gui(refresh_gui);
  set_perform_command(perform_command);

  set_lock_gui(lock_gui);

  set_quit_application(quit_application);

  // This delegate is served by the mforms app impl structure, hence we don't need to create
  // an additional hook like we do for the other delegates.
  app_command_delegate= app_command;
}

//--------------------------------------------------------------------------------------------------

AppImpl::~AppImpl()
{
  _cb_handle.Free();
  delete _callbacks;
}

//--------------------------------------------------------------------------------------------------

::wb::WBFrontendCallbacks* AppImpl::get_callbacks()
{
  return _callbacks;
}

//--------------------------------------------------------------------------------------------------

bool AppImpl::request_input_wrapper(const std::string &title, int flags, std::string& res)
{
  String^ result= nullptr;
  bool flag;

  flag= request_input_delegate(CppStringToNative(title), flags, result);
  if (result != nullptr)
    res= NativeToCppString(result);

  return flag;
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_show_file_dialog(StrStrStrStrDelegate^ dt)
{
  show_file_dialog_delegate= dt;
  show_file_dialog_wrapper_delegate= 
    gcnew StrStrStrStrWrapperDelegate(this, &AppImpl::show_file_dialog_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(show_file_dialog_wrapper_delegate);
  STR_STR_STR_STR_CB cb = static_cast<STR_STR_STR_STR_CB>(ip.ToPointer());
  _callbacks->show_file_dialog= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_show_status_text(VoidStrDelegate^ dt)
{
  show_status_text_delegate= dt;
  show_status_text_wrapper_delegate= 
    gcnew VoidStrWrapperDelegate(this, &AppImpl::show_status_text_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(show_status_text_wrapper_delegate);
  VOID_STR_CB cb = static_cast<VOID_STR_CB>(ip.ToPointer());
  _callbacks->show_status_text= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_show_progress(BoolStrStrFloatDelegate^ dt)
{
  show_progress_delegate= dt;
  show_progress_wrapper_delegate= 
    gcnew BoolStrStrFloatWrapperDelegate(this, &AppImpl::show_progress_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(show_progress_wrapper_delegate);
  BOOL_STR_STR_FLOAT_CB cb = static_cast<BOOL_STR_STR_FLOAT_CB>(ip.ToPointer());
  _callbacks->show_progress= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_request_input(BoolStrIntStrPtrDelegate^ dt)
{
  request_input_delegate= dt;
  request_input_wrapper_delegate= 
    gcnew BoolStrIntStrPtrWrapperDelegate(this, &AppImpl::request_input_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(request_input_wrapper_delegate);
  BOOL_STR_INT_STRPTR_CB cb = static_cast<BOOL_STR_INT_STRPTR_CB>(ip.ToPointer());
  _callbacks->request_input= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_create_diagram(CanvasViewStringStringIntPtrDelegate^ dt)
{
  create_diagram_delegate= dt;
  create_diagram_wrapper_delegate=
    gcnew CanvasViewDiagramWrapperDelegate(this, &AppImpl::create_diagram_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(create_diagram_wrapper_delegate);
  CANVASVIEW_DIAGRAM_CB cb = static_cast<CANVASVIEW_DIAGRAM_CB>(ip.ToPointer());
  _callbacks->create_diagram= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_destroy_view(VoidCanvasViewDelegate^ dt)
{
  destroy_view_delegate= dt;
  destroy_view_wrapper_delegate=
    gcnew VoidCanvasViewWrapperDelegate(this, &AppImpl::destroy_view_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(destroy_view_wrapper_delegate);
  VOID_CANVASVIEW_CB cb = static_cast<VOID_CANVASVIEW_CB>(ip.ToPointer());
  _callbacks->destroy_view= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_switched_view(VoidCanvasViewDelegate^ dt)
{
  switched_view_delegate= dt;
  switched_view_wrapper_delegate=
    gcnew VoidCanvasViewWrapperDelegate(this, &AppImpl::switched_view_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(switched_view_wrapper_delegate);
  VOID_CANVASVIEW_CB cb = static_cast<VOID_CANVASVIEW_CB>(ip.ToPointer());
  _callbacks->switched_view= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_tool_changed(VoidCanvasViewDelegate^ dt)
{
  tool_changed_delegate= dt;
  tool_changed_wrapper_delegate=
    gcnew VoidCanvasViewWrapperDelegate(this, &AppImpl::tool_changed_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(tool_changed_wrapper_delegate);
  VOID_CANVASVIEW_CB cb = static_cast<VOID_CANVASVIEW_CB>(ip.ToPointer());
  _callbacks->tool_changed= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_open_editor(IntPtrGRTManagerModuleStrStrGrtListFlagsDelegate^ dt)
{
  open_editor_delegate= dt;
  open_editor_wrapper_delegate=
    gcnew IntGRTManagerModuleStrStrGrtListFlagsWrapperDelegate(this, &AppImpl::open_editor_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(open_editor_wrapper_delegate);
  INT_GRTMANAGER_MODULE_STR_STR_GRTLIST_FLAGS_CB cb = static_cast<INT_GRTMANAGER_MODULE_STR_STR_GRTLIST_FLAGS_CB>(ip.ToPointer());
  _callbacks->open_editor= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_show_editor(VoidIntPtrDelegate^ dt)
{
  show_editor_delegate= dt;
  show_editor_wrapper_delegate= 
    gcnew VoidIntPtrWrapperDelegate(this, &AppImpl::show_editor_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(show_editor_wrapper_delegate);
  VOID_INT64_CB cb = static_cast<VOID_INT64_CB>(ip.ToPointer());
  _callbacks->show_editor= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_hide_editor(VoidIntPtrDelegate^ dt)
{
  hide_editor_delegate= dt;
  hide_editor_wrapper_delegate= 
    gcnew VoidIntPtrWrapperDelegate(this, &AppImpl::hide_editor_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(hide_editor_wrapper_delegate);
  VOID_INT64_CB cb = static_cast<VOID_INT64_CB>(ip.ToPointer());
  _callbacks->hide_editor= sigc::ptr_fun(cb);
}


//--------------------------------------------------------------------------------------------------

void AppImpl::set_refresh_gui(VoidRefreshTypeStringIntPtrDelegate ^ dt)
{
  refresh_gui_delegate= dt;
  refresh_gui_wrapper_delegate= 
    gcnew VoidRefreshTypeStringIntPtrWrapperDelegate(this, &AppImpl::refresh_gui_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(refresh_gui_wrapper_delegate);
  VOID_REFRESHTYPE_STRING_INTPTR_CB cb = static_cast<VOID_REFRESHTYPE_STRING_INTPTR_CB>(ip.ToPointer());
  _callbacks->refresh_gui= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_perform_command(VoidStrDelegate ^ dt)
{
  perform_command_delegate= dt;
  perform_command_wrapper_delegate= 
    gcnew VoidStrWrapperDelegate(this, &AppImpl::perform_command_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(perform_command_wrapper_delegate);
  VOID_STR_CB cb = static_cast<VOID_STR_CB>(ip.ToPointer());
  _callbacks->perform_command = sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_lock_gui(VoidBoolDelegate^ dt)
{
  lock_gui_delegate= dt;
  lock_gui_wrapper_delegate=
    gcnew VoidBoolWrapperDelegate(this, &AppImpl::lock_gui_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(lock_gui_wrapper_delegate);
  VOID_BOOL_CB cb = static_cast<VOID_BOOL_CB>(ip.ToPointer());
  _callbacks->lock_gui= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_quit_application(VoidDelegate^ dt)
{
  quit_application_delegate= dt;
  quit_application_wrapper_delegate=
    gcnew VoidWrapperDelegate(this, &AppImpl::quit_application_wrapper);
  IntPtr ip = Marshal::GetFunctionPointerForDelegate(quit_application_wrapper_delegate);
  VOID_CB cb = static_cast<VOID_CB>(ip.ToPointer());
  _callbacks->quit_application= sigc::ptr_fun(cb);
}

//--------------------------------------------------------------------------------------------------

std::string AppImpl::show_file_dialog_wrapper(const std::string& str1, const std::string& str2,
  const std::string& str3)
{
  return NativeToCppString(show_file_dialog_delegate(CppStringToNative(str1), CppStringToNative(str2),
    CppStringToNative(str3)));
}

//--------------------------------------------------------------------------------------------------

void AppImpl::show_status_text_wrapper(const std::string& str1)
{
  show_status_text_delegate(CppStringToNative(str1));
}

//--------------------------------------------------------------------------------------------------

bool AppImpl::show_progress_wrapper(const std::string& str1, const std::string& str2, float f3)
{
  return show_progress_delegate(CppStringToNative(str1), CppStringToNative(str2), f3);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::shell_output_wrapper(const std::string& str1)
{
  shell_output_delegate(CppStringToNative(str1));
}

//--------------------------------------------------------------------------------------------------

::mdc::CanvasView* AppImpl::create_diagram_wrapper(const model_DiagramRef& model)
{
  BaseWindowsCanvasView ^cv = create_diagram_delegate(CppStringToNative(model->id()),
    CppStringToNative(model->name()), IntPtr(&model.content()));
  return cv->get_unmanaged_object();
}

//--------------------------------------------------------------------------------------------------

void AppImpl::destroy_view_wrapper(::mdc::CanvasView *canvas_view)
{
  BaseWindowsCanvasView^ wcv = 
    BaseWindowsCanvasView::GetFromFixedId((IntPtr)canvas_view->get_user_data());
  destroy_view_delegate(wcv);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::switched_view_wrapper(::mdc::CanvasView *canvas_view)
{
  BaseWindowsCanvasView^ wcv = 
    BaseWindowsCanvasView::GetFromFixedId((IntPtr)canvas_view->get_user_data());
  switched_view_delegate(wcv);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::tool_changed_wrapper(::mdc::CanvasView *canvas_view)
{
  BaseWindowsCanvasView^ wcv = 
    BaseWindowsCanvasView::GetFromFixedId((IntPtr)canvas_view->get_user_data());
  tool_changed_delegate(wcv);
}

//--------------------------------------------------------------------------------------------------

uintptr_t AppImpl::open_editor_wrapper(::bec::GRTManager* grt_manager, grt::Module *module,
  const std::string& str2, const std::string& str3, const grt::BaseListRef &grt_list,
  bec::GUIPluginFlags flags)
{
  return (uintptr_t)open_editor_delegate(gcnew GrtManager(grt_manager), gcnew GrtModule(module), 
    CppStringToNative(str2), CppStringToNative(str3),
    gcnew GrtValue(grt_list), (GUIPluginFlags)flags).ToPointer();
}

//--------------------------------------------------------------------------------------------------

void AppImpl::show_editor_wrapper(uintptr_t native_handle)
{
  show_editor_delegate(IntPtr((void*)native_handle));
}

//--------------------------------------------------------------------------------------------------

void AppImpl::hide_editor_wrapper(uintptr_t native_handle)
{
  hide_editor_delegate(IntPtr((void*)native_handle));
}

//--------------------------------------------------------------------------------------------------

void AppImpl::refresh_gui_wrapper(::wb::RefreshType refresh, const std::string &str, uintptr_t ptr)
{
  refresh_gui_delegate((RefreshType)refresh, CppStringToNative(str), IntPtr((void*)ptr));
}

//--------------------------------------------------------------------------------------------------

void AppImpl::perform_command_wrapper(const std::string &command)
{
  perform_command_delegate(CppStringToNative(command));
}

//--------------------------------------------------------------------------------------------------

void AppImpl::lock_gui_wrapper(bool flag)
{
  lock_gui_delegate(flag);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::quit_application_wrapper()
{
  quit_application_delegate();
}

//--------------------------------------------------------------------------------------------------

std::string AppImpl::app_command_wrapper(AppCommand command, mforms::AppView* view, const std::string &str)
{
  AppViewImpl^ native_view= nullptr;
  if (view != NULL)
    native_view= (AppViewImpl^) ObjectImpl::FromUnmanaged(view);
  String^ result= app_command_delegate(command, native_view, CppStringToNative(str));
  return NativeToCppString(result);
}

//--------------------------------------------------------------------------------------------------

Object^ AppImpl::GetFromFixedId(IntPtr ip)
{
  GCHandle gcHandle = GCHandle::FromIntPtr(ip);
  return (Object^) gcHandle.Target;
}

//--------------------------------------------------------------------------------------------------

void AppImpl::dock_view(mforms::App *app, mforms::AppView *view, const std::string &position)
{
  AppImpl^ application= (AppImpl^) GetFromFixedId((IntPtr) app->get_data_ptr());
  application->app_command_wrapper(AppCommand::AppDockView, view, position);
}

//--------------------------------------------------------------------------------------------------

bool AppImpl::select_view(mforms::App *app, const std::string &identifier)
{
  AppImpl^ application= (AppImpl^) GetFromFixedId((IntPtr) app->get_data_ptr());
  return !application->app_command_wrapper(AppCommand::AppSelectView, nullptr, identifier).empty();
}

//--------------------------------------------------------------------------------------------------

void AppImpl::undock_view(mforms::App *app, mforms::AppView *view)
{
  AppImpl^ application= (AppImpl^) GetFromFixedId((IntPtr) app->get_data_ptr());
  application->app_command_wrapper(AppCommand::AppUndockView, view, "");
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_view_title(mforms::App *app, mforms::AppView *view, const std::string &title)
{
  AppImpl^ application= (AppImpl^) GetFromFixedId((IntPtr) app->get_data_ptr());
  application->app_command_wrapper(AppCommand::AppSetViewTitle, view, title);
}

//--------------------------------------------------------------------------------------------------

std::string AppImpl::get_resource_path(mforms::App *app, const std::string &file)
{
  AppImpl^ application= (AppImpl^) GetFromFixedId((IntPtr) app->get_data_ptr());
  return application->app_command_wrapper(AppCommand::AppGetResourcePath, nullptr, file);
}

//--------------------------------------------------------------------------------------------------

void AppImpl::set_status_text(mforms::App *app, const std::string &text)
{
  AppImpl^ application= (AppImpl^) GetFromFixedId((IntPtr) app->get_data_ptr());
  application->app_command_wrapper(AppCommand::AppSetStatusText, nullptr, text);
}

//--------------------------------------------------------------------------------------------------

std::string AppImpl::get_bounds(mforms::App *app)
{
  AppImpl^ application= (AppImpl^) GetFromFixedId((IntPtr) app->get_data_ptr());
  return application->app_command_wrapper(AppCommand::AppGetBounds, nullptr, "");
}

//--------------------------------------------------------------------------------------------------

static int message_loop_exit_code = MININT; // Can stay unguarded. We only use it from the main thread.

/**
 * Establishes a local event loop which is needed by the python debugger. This is potentially a very
 * dangerous and hard-to-debug thing so use it carefully.
 * Additionally, if you fail to call exit_event_loop the call will never return and block closing
 * the application.
 */
int AppImpl::enter_event_loop(mforms::App *app, float max_wait_time)
{
  message_loop_exit_code = -MININT;
  int remaining_milliseconds;
  if (max_wait_time <= 0)
    remaining_milliseconds = MAXINT;
  else
    remaining_milliseconds = (int) (1000 * max_wait_time);
  while (message_loop_exit_code == -MININT && remaining_milliseconds > 0)
  {
    Application::DoEvents();
    Thread::Sleep(100);
    remaining_milliseconds -= 100;
  }
  if (remaining_milliseconds == 0 || message_loop_exit_code == -MININT)
    return -1;
  return message_loop_exit_code;
}

//--------------------------------------------------------------------------------------------------

void AppImpl::exit_event_loop(mforms::App *app, int ret_code)
{
  message_loop_exit_code = ret_code;
}

//--------------------------------------------------------------------------------------------------

MySQL::Drawing::Color AppImpl::get_system_color(mforms::SystemColor type)
{
  switch (type)
  {
  case mforms::SystemHighlight:
    {
      System::Drawing::Color^ color = System::Drawing::Color::FromKnownColor(KnownColor::Highlight);
      return NativeToColor(color);
    }
  default:
#ifdef _DEBUG
    throw new std::runtime_error("mforms::App: Invalid system color enumeration given.");
#else
    log_error("mforms::App: Invalid system color enumeration given.");
    return MySQL::Drawing::Color::Black();
#endif
  }
}

//--------------------------------------------------------------------------------------------------

