/* 
 * Copyright (c) 2008, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "changefactory.h"
#include "changeobjects.h"
#include "grts/structs.h"

namespace grt
{

bool is_any(ValueRef& s);


DiffChange* ChangeFactory::create_value_added_change(DiffChange *parent, const ValueRef &source, const ValueRef &target)
{
  return new ValueAddedChange(ValueAdded, target);
}


DiffChange* ChangeFactory::create_value_removed_change(DiffChange *parent, const ValueRef &source, const ValueRef &target)
{
  return new ValueRemovedChange;
}


DiffChange* ChangeFactory::create_object_attr_modified_change(DiffChange *parent, const ObjectRef &source, const ObjectRef &target, const std::string &attr, DiffChange *change)
{
  if (change)
    return new ObjectAttrModifiedChange(attr, change);
  return NULL;
}


DiffChange* ChangeFactory::create_object_modified_change(DiffChange *parent, const ObjectRef &source, const ObjectRef &target, ChangeSet &changes)
{
  if (!changes.empty())
    return new MultiChange(ObjectModified, changes);
  return NULL;
}

template<class T> inline bool check(ValueRef source, ValueRef target)
{
  return T::cast_from(source) == T::cast_from(target);
}

DiffChange * ChangeFactory::create_simple_value_change( DiffChange *parent, const ValueRef &source, const ValueRef &target )
{
  grt::Type t= UnknownType;
  if (source.is_valid())
    t= source.type();
  else if (target.is_valid())
    t= target.type();

  switch(t)
  {

  case IntegerType:
    if(check<grt::IntegerRef>(source, target))
      return NULL;
    break;
  case DoubleType:
    if(check<grt::DoubleRef>(source, target))
      return NULL;
    break;
  case StringType:
      if(check<grt::StringRef>(source, target))
        return NULL;
    break;

  case ObjectType:

    if(GrtObjectRef::can_wrap(source) && GrtObjectRef::can_wrap(target))
    {
      StringRef n1(
        source.is_valid() ? GrtObjectRef::cast_from(source)->name() : "");
      StringRef n2(
        target.is_valid() ? GrtObjectRef::cast_from(target)->name() : "");
      if(n1 == n2)
        return NULL;
//      std::cout<<"Obj Diff"<<n1.c_str()<<"  "<<n2.c_str()<<std::endl;
    }
    break;
  case ListType:
  case DictType:
  case UnknownType:
    return NULL;
    break;

  default:
    assert(0);
  }
  SimpleValueChange *ret= new SimpleValueChange(source, target);
  return ret;
}


DiffChange* ChangeFactory::create_dict_change(DiffChange *parent, const DictRef &source, const DictRef &target, ChangeSet &changes)
{
  if (changes.empty())
    return NULL;

  return new MultiChange(DictModified, changes);
}

DiffChange* ChangeFactory::create_dict_item_added_change(DiffChange *parent, const DictRef &source, const DictRef &target, const std::string &key, ValueRef v)
{
  return new DictItemAddedChange(key, v);
}

DiffChange* ChangeFactory::create_dict_item_modified_change(DiffChange *parent, const DictRef &source, const DictRef &target, const std::string &key, DiffChange *change)
{
  if (change)
    return new DictItemModifiedChange(key, change);
  return NULL;
}

DiffChange* ChangeFactory::create_dict_item_removed_change(DiffChange *parent, const DictRef &source, const DictRef &target, const std::string &key)
{
  return new DictItemRemovedChange(key);
}

}

