/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "grtui/grt_wizard_plugin.h"
#include "grtui/wizard_view_text_page.h"

#include "grtui/grtdb_object_filter.h"

#include "db_mysql_sql_script_sync.h"
#include "base/string_utilities.h"
#include "mforms/fs_object_selector.h"

using namespace grtui;
using namespace mforms;

namespace GenerateAlter {

  
class WbPluginSQLExportAlter : public WizardPlugin
{
  DbMySQLScriptSync _be;
  
public:
  WbPluginSQLExportAlter(grt::Module *module);

  DbMySQLScriptSync *get_be() { return &_be; }
};

  
  

class ExportInputPage : public WizardPage
{
public:
  ExportInputPage(WizardPlugin *form)
    : WizardPage(form, "options")//, _options(mforms::TitledBoxPanel)
  {
    set_title(_("Forward Engineer ALTER Script Options"));
    set_short_title(_("Options"));

    //_options.set_title(_("Options"));

    //_options.add(&_options_box);
    _options_box.set_padding(12);
    _options_box.set_row_spacing(8);
    _options_box.set_column_spacing(4);
    
    _options_box.set_column_count(3);
    _options_box.set_row_count(5);

    _options_box.add(&_in_descr, 0, 3, 0, 1, (TableItemFlags)HFillFlag);
    _options_box.add(&_in_label, 0, 1, 1, 2, (TableItemFlags)HFillFlag);
    _options_box.add(&_in_entry, 1, 2, 1, 2, (TableItemFlags)(HExpandFlag | HFillFlag));
    _options_box.add(&_in_button, 2, 3, 1, 2, (TableItemFlags)0);
    
    _in_descr.set_text(_("Pick the SQL script file to be compared with the current model. Script must include schema qualifiers."));
    _in_label.set_text(_("Input File:"));
    
    std::string input= form->module()->document_string_data("input_filename", "");
    _input_selector= mforms::manage(new FsObjectSelector(&_in_button, &_in_entry));
    _input_selector->initialize(input, mforms::OpenFile, "SQL Files (*.sql)|*.sql", _("Browse..."),
      false, sigc::mem_fun(this, &WizardPage::validate));
    _input_selector->signal_changed().connect(sigc::mem_fun(this, &ExportInputPage::validate));

    _options_box.add(&_out_descr, 0, 3, 2, 3, (TableItemFlags)HFillFlag);
    _options_box.add(&_out_label, 0, 1, 3, 4, (TableItemFlags)HFillFlag);
    _options_box.add(&_out_entry, 1, 2, 3, 4, (TableItemFlags)(HExpandFlag|HFillFlag));
    _options_box.add(&_out_button, 2, 3, 3, 4, (TableItemFlags)0);
    _options_box.add(&_skip_out_label, 1, 2, 4, 5, (TableItemFlags)HFillFlag);
    
    _skip_out_label.set_text(_("Leave blank to view generated script but not save to a file."));
    _skip_out_label.set_style(SmallHelpTextStyle);
    
    _out_descr.set_text(_("Enter the path for the ALTER script to be created."));
    _out_label.set_text(_("Output File:"));
    _out_button.set_text(_("Browse..."));

    std::string output= form->module()->document_string_data("output_filename", "");
    _output_selector= mforms::manage(new FsObjectSelector(&_out_button, &_out_entry));
    _output_selector->initialize(output, mforms::OpenFile, "SQL Files (*.sql)|*.sql", _("Browse..."),
      false, sigc::mem_fun(this, &WizardPage::validate));
    _output_selector->signal_changed().connect(sigc::mem_fun(this, &ExportInputPage::validate));

    add(&_options_box, false, false);


    signal_leave().connect(sigc::mem_fun(this, &ExportInputPage::gather_options));
  }


  void gather_options(bool advancing)
  {
    if (advancing)
    {
      ((WbPluginSQLExportAlter*)_form)->get_be()->set_option("InputFileName1", _in_entry.get_string_value());
      
      values().gset("InputPath", _in_entry.get_string_value());
      values().gset("OutputPath", _out_entry.get_string_value());

      grt::Module *module= ((WizardPlugin*)_form)->module();
      
      module->set_document_data("output_filename", _out_entry.get_string_value());
      module->set_document_data("input_filename", _in_entry.get_string_value());

    }
  }

  virtual bool allow_next()
  {    
    return g_file_test(_in_entry.get_string_value().c_str(), (GFileTest)(G_FILE_TEST_EXISTS|G_FILE_TEST_IS_REGULAR));
  }

  virtual bool advance()
  {
    if (_confirmed_overwrite_for != _out_entry.get_string_value() && !FsObjectSelector::check_and_confirm_file_overwrite(&_out_entry))
      return false;
    _confirmed_overwrite_for= _out_entry.get_string_value();
    return WizardPage::advance();
  }

protected:
//  Panel _options;
  std::string _confirmed_overwrite_for;
  Table _options_box;

  Label _in_label;
  Label _in_descr;
  TextEntry _in_entry;
  Button _in_button;
  FsObjectSelector* _input_selector;

  Label _out_label;
  Label _out_descr;
  TextEntry _out_entry;
  Button _out_button;
  FsObjectSelector* _output_selector;
  Label _skip_out_label;
};



class PreviewScriptPage : public ViewTextPage
{
public:
  PreviewScriptPage(WizardForm *form)
    : ViewTextPage(form, "preview", (ViewTextPage::Buttons)(ViewTextPage::CopyButton|ViewTextPage::SaveButton), "SQL Scripts (*.sql)|*.sql")
  {
    set_title(_("Review Generated Script"));
    set_short_title(_("Review SQL Script"));
    
    set_editable(true);
  }


  virtual void enter(bool advancing)
  {
    if (advancing)
    {
      try
      {
        set_text(((WbPluginSQLExportAlter*)_form)->get_be()->generate_diff_tree_script());
        _form->clear_problem();
      }
      catch (std::exception &exc)
      {
        set_text(std::string(_("Could not generate ALTER script.")).append("\n").append(exc.what()));
        _form->set_problem(_("Error generating script."));
      }
    }
  }


  virtual bool advance()
  {
    std::string path= values().get_string("OutputPath");
    if (!path.empty())
    {
      save_text_to(path);
        
      _form->grtm()->push_status_text(base::strfmt(_("Wrote ALTER Script to '%s'"), path.c_str()));
      _form->grtm()->get_grt()->send_info(base::strfmt(_("Wrote ALTER Script to '%s'"), path.c_str()));
    }
    return true;
  }
  
  virtual std::string next_button_caption()
  {
    return execute_caption();
  }

  virtual bool next_closes_wizard() { return true; }
};

//--------------------------------------------------------------------------------

WbPluginSQLExportAlter::WbPluginSQLExportAlter(grt::Module *module)
    : WizardPlugin(module), _be(grtm())
{
  set_title(_("Forward Engineer an ALTER Script"));

  add_page(mforms::manage(new ExportInputPage(this)));
  add_page(mforms::manage(new PreviewScriptPage(this)));
}

}

//--------------------------------------------------------------------------------

grtui::WizardPlugin *createExportALTERScriptWizard(grt::Module *module, db_CatalogRef catalog)
{
  return new GenerateAlter::WbPluginSQLExportAlter(module);
}

//--------------------------------------------------------------------------------

