/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "wb_overview_live_physical.h"
#include "wb_overview_live_physical_schema.h"
#include "base/string_utilities.h"

using namespace bec;
using namespace wb;
using namespace wb::internal;
using namespace base;
using namespace grt;

//------------------------------------------------------------------------


LivePhysicalSchemataNode::LivePhysicalSchemataNode(workbench_physical_ModelRef amodel)
: PhysicalSchemataNode(amodel)
{  
  label= _("Overview");
}


OverviewBE::Node * LivePhysicalSchemataNode::create_child_node(db_SchemaRef schema)
{
  LivePhysicalSchemaNode *node= new LivePhysicalSchemaNode(schema);
  node->init();
  return node;
}


bool LivePhysicalSchemataNode::add_object(WBContext *wb)
{
  return wb->create_live_object(db_SchemaRef(object.get_grt()), "", "");
}


void LivePhysicalSchemataNode::delete_object(WBContext *wb)
{
  wb->drop_live_object(db_SchemaRef(object.get_grt()), "", *object->name());
}


//------------------------------------------------------------------------


class LivePhysicalRootNode : public OverviewBE::ContainerNode
{
public:
  LivePhysicalRootNode(workbench_physical_ModelRef model, PhysicalOverviewBE *owner)
    : ContainerNode(OverviewBE::ODivision)
  {
    type= OverviewBE::ORoot;
    if (model->rdbms().is_valid())
      label= strfmt("%s Overview", model->rdbms()->caption().c_str());
    expanded= true;
    object= model;
    display_mode= OverviewBE::MSmallIcon;

    {
      LivePhysicalSchemataNode *node= new LivePhysicalSchemataNode(model);
      node->init();
      children.push_back(node);
    }
  }
};


//------------------------------------------------------------------------


LivePhysicalOverviewBE::LivePhysicalOverviewBE(WBContext *wb)
: PhysicalOverviewBE(wb)
{
  _schemata_node_index= 0;
  _model_mutex= g_mutex_new();
}


LivePhysicalOverviewBE::~LivePhysicalOverviewBE()
{
  g_mutex_free(_model_mutex);
}


OverviewBE::ContainerNode * LivePhysicalOverviewBE::create_root_node(workbench_physical_ModelRef model, PhysicalOverviewBE *owner)
{
  return new LivePhysicalRootNode(model, owner);
}


std::string LivePhysicalOverviewBE::identifier() const
{
  return "overview.physical.live";
}


std::string LivePhysicalOverviewBE::get_form_context_name() const
{
  return "Live Physical Overview";
}


std::string LivePhysicalOverviewBE::get_title()
{
  return "Overview";
}


LivePhysicalOverviewBE::Node * LivePhysicalOverviewBE::do_get_node(const bec::NodeId &node) const
{
  GMutexLock model_mutex_lock(_model_mutex);
  return wb::PhysicalOverviewBE::do_get_node(node);
}


bec::ToolbarItemList LivePhysicalOverviewBE::get_toolbar_items(const bec::NodeId &node)
{
  bec::ToolbarItemList items;
  if (node.depth() == 1 && node[0] == 0)
  {
    bec::ToolbarItem item;
    bec::IconManager *im = bec::IconManager::get_instance();

    item.icon = im->get_icon_id("tiny_refresh.png");
    item.caption = "Refresh";
    item.name = "refresh";
    item.command = "refresh";
    item.tooltip = "Refresh the Schema List";
    item.type = ToolbarAction;
    item.enabled = true;

    items.insert(items.begin(), item);

    item.type = ToolbarSeparator;
    items.push_back(item);

    item.icon = im->get_icon_id("tiny_new_schema.png");
    item.caption = "New Schema";
    item.name = "new_schema";
    item.command = "new_schema";
    item.tooltip = "Create a New Schema";
    item.type = ToolbarAction;
    item.enabled = true;
    items.push_back(item);

    item.icon = im->get_icon_id("tiny_new_table.png");
    item.caption = "New Table";
    item.name = "new_table";
    item.command = "new_table";
    item.tooltip = "Create a New Table";
    item.type = ToolbarAction;
    item.enabled = true;
    items.push_back(item);

    item.icon = im->get_icon_id("tiny_new_view.png");
    item.caption = "New View";
    item.name = "new_view";
    item.command = "new_view";
    item.tooltip = "Create a New View";
    item.type = ToolbarAction;
    item.enabled = true;
    items.push_back(item);

    item.icon = im->get_icon_id("tiny_new_routine.png");
    item.caption = "New Routine";
    item.name = "new_routine";
    item.command = "new_routine";
    item.tooltip = "Create a New Routine";
    item.type = ToolbarAction;
    item.enabled = true;
    items.push_back(item);
  }
  return items;
}


bool LivePhysicalOverviewBE::activate_toolbar_item(const bec::NodeId &node, const std::string &name)
{
  if (node.depth() == 1 && node[0] == 0)
  {
    bec::ToolbarItem item;
    std::string schema = "";

    get_field(get_focused_child(bec::NodeId(0)), wb::OverviewBE::Label, schema);

    if (name == "refresh")
    {
      refresh_signal();
    }
    else if (name == "new_schema")
    {
      _wb->create_live_object(db_SchemaRef(_wb->get_grt()), "", "");
    }
    else if (name == "new_table")
    {
      _wb->create_live_object(db_TableRef(_wb->get_grt()), schema, "");
    }
    else if (name == "new_view")
    {
      _wb->create_live_object(db_ViewRef(_wb->get_grt()), schema, "");
    }
    else if (name == "new_routine")
    {
      _wb->create_live_object(db_RoutineRef(_wb->get_grt()), schema, "");
    }
    return true;
  }
  return false;
}


bec::MenuItemList LivePhysicalOverviewBE::get_popup_items_for_nodes(const std::vector<bec::NodeId> &nodes)
{
  bec::MenuItemList items;

  bec::MenuItem active_schema_item;
  active_schema_item.caption= _("Set as Default Schema");
  active_schema_item.name= "set_active_schema";
  active_schema_item.enabled= false;

  bec::MenuItem clipboard_copy_item;
  clipboard_copy_item.caption= _("Copy to Clipboard");
  clipboard_copy_item.name= "clipboard_copy";
  clipboard_copy_item.type= MenuCascade;
  clipboard_copy_item.enabled= true;

  bec::MenuItem editor_insert_item;
  editor_insert_item.caption= _("Send to SQL Editor");
  editor_insert_item.name= "editor_insert";
  editor_insert_item.type= MenuCascade;
  editor_insert_item.enabled= true;

  bec::MenuItem object_name_short_item;
  object_name_short_item.caption= _("Name (short)");
  object_name_short_item.name= "object_name_short";
  object_name_short_item.enabled= true;

  bec::MenuItem object_name_long_item;
  object_name_long_item.caption= _("Name (long)");
  object_name_long_item.name= "object_name_long";
  object_name_long_item.enabled= true;

  bec::MenuItem table_select_statement_item;
  table_select_statement_item.caption= _("Select All Statement");
  table_select_statement_item.name= "table_select_statement";
  table_select_statement_item.enabled= false;

  bec::MenuItem table_insert_statement_item;
  table_insert_statement_item.caption= _("Insert Statement");
  table_insert_statement_item.name= "table_insert_statement";
  table_insert_statement_item.enabled= false;

  bec::MenuItem table_update_statement_item;
  table_update_statement_item.caption= _("Update Statement");
  table_update_statement_item.name= "table_update_statement";
  table_update_statement_item.enabled= false;

  bec::MenuItem table_delete_statement_item;
  table_delete_statement_item.caption= _("Delete Statement");
  table_delete_statement_item.name= "table_delete_statement";
  table_delete_statement_item.enabled= false;

  bec::MenuItem table_create_statement_item;
  table_create_statement_item.caption= _("Create Statement");
  table_create_statement_item.name= "table_create_statement";
  table_create_statement_item.enabled= false;

  bec::MenuItem table_column_info_statement_item;
  table_column_info_statement_item.caption= _("Show Column Info Statement");
  table_column_info_statement_item.name= "table_column_info_statement";
  table_column_info_statement_item.enabled= false;
  
  bec::MenuItem table_column_names_item;
  table_column_names_item.caption= _("Column Names");
  table_column_names_item.name= "table_column_names";
  table_column_names_item.enabled= false;
    
  if (nodes.size() == 1)
  {
    Node *node= get_node(nodes[0]);
    GrtObjectRef object= node->object;
    std::string schema_name;
    std::string object_type;
    std::string unlocalized_object_type;
    bool allow_view_data= false;
    bool allow_edit_data= false;
    if (db_SchemaRef::can_wrap(object))
    {
      object_type= "Schema";
      unlocalized_object_type= "schema";
      schema_name= object->name();
      active_schema_item.enabled= true;
      table_create_statement_item.enabled= true;
      object_name_long_item.caption= _("Name");
      object_name_long_item.name= "object_name_long";
      object_name_short_item.enabled= false;
    }
    else if (db_TableRef::can_wrap(object))
    {
      object_type= "Table";
      unlocalized_object_type= "table";
      schema_name= object->owner()->name();
      allow_view_data= true;
      allow_edit_data= true;
      table_column_names_item.enabled= true;
      table_column_info_statement_item.enabled= true;
      table_select_statement_item.enabled= true;
      table_insert_statement_item.enabled= true;
      table_update_statement_item.enabled= true;
      table_delete_statement_item.enabled= true;
      table_create_statement_item.enabled= true;
    }
    else if (db_ViewRef::can_wrap(object))
    {
      object_type= "View";
      unlocalized_object_type= "view";
      schema_name= object->owner()->name();
      allow_view_data= true;
      table_column_names_item.enabled= true;
      table_select_statement_item.enabled= true;
      table_insert_statement_item.enabled= true;
      table_update_statement_item.enabled= true;
      table_delete_statement_item.enabled= true;
      table_create_statement_item.enabled= true;
    }
    else if (db_RoutineRef::can_wrap(object))
    {
      object_type= "Routine";
      unlocalized_object_type= "routine";
      schema_name= object->owner()->name();
    }

    if (!object_type.empty())
    {
      if (allow_edit_data)
      {
        bec::MenuItem item;
        item.caption= _("Edit Table Data");
        item.name= "edit_data";
        items.push_back(item);
      }
            
      if (allow_view_data)
      {
        bec::MenuItem item;
        {
          std::string caption= _("Select Rows");
          {
            DictRef options= DictRef::cast_from(get_wb()->get_grt()->get("/wb/options/options"));
            bool limit_rows= (0 != options.get_int("SqlEditor:LimitRows"));
            int limit_rows_count= options.get_int("SqlEditor:LimitRowsCount");
            if (limit_rows && (limit_rows_count <= 0))
              limit_rows= false;
            if (limit_rows)
              caption+= strfmt(_(" - Limit %i"), limit_rows_count);
          }
          item.caption= caption;
        }
        item.name= "select_data";
        items.push_back(item);
      }

      if (active_schema_item.enabled)
      {
        items.push_back(active_schema_item);
        bec::MenuItem item;
        item.type = MenuSeparator;
        items.push_back(item);
      }

      {
        MenuItemList subitems;
        subitems.push_back(object_name_short_item);
        subitems.push_back(object_name_long_item);
        subitems.push_back(table_column_names_item);
        subitems.push_back(table_select_statement_item);
        subitems.push_back(table_insert_statement_item);
        subitems.push_back(table_update_statement_item);
        subitems.push_back(table_delete_statement_item);
        subitems.push_back(table_create_statement_item);
        subitems.push_back(table_column_info_statement_item);
        for (MenuItemList::iterator i= subitems.begin(), end= subitems.end(); i != end; ++i)
        {
          if (!i->enabled)
            continue;
          i->name= "clipboard_" + i->name;
          clipboard_copy_item.subitems.push_back(*i);
        }
        items.push_back(clipboard_copy_item);
      }

      {
        MenuItemList subitems;
        subitems.push_back(object_name_short_item);
        subitems.push_back(object_name_long_item);
        subitems.push_back(table_column_names_item);
        subitems.push_back(table_select_statement_item);
        subitems.push_back(table_insert_statement_item);
        subitems.push_back(table_update_statement_item);
        subitems.push_back(table_delete_statement_item);
        subitems.push_back(table_create_statement_item);
        subitems.push_back(table_column_info_statement_item);
        for (MenuItemList::iterator i= subitems.begin(), end= subitems.end(); i != end; ++i)
        {
          if (!i->enabled)
            continue;
          i->name= "editor_" + i->name;
          editor_insert_item.subitems.push_back(*i);
        }
        items.push_back(editor_insert_item);
      }

      {
        bec::MenuItem item;
        item.type = MenuSeparator;
        items.push_back(item);
      }

      {
        bec::MenuItem item;
        item.caption= strfmt(_("Alter %s..."), object_type.c_str());
        item.name= "alter";
        items.push_back(item);
      }

      {
        bec::MenuItem item;
        item.caption= strfmt(_("Drop %s..."), object_type.c_str());
        item.name= "drop";
        items.push_back(item);
      }

      {
        bec::MenuItem item;
        item.type = MenuSeparator;
        items.push_back(item);
      }
    }

    try
    {
      bec::MenuItem item;

      if (!unlocalized_object_type.empty())
      {
        // plugin items
        bec::MenuItemList plugin_items(object_plugin_items_slot(schema_name,
                                                                object->name(),
                                                                unlocalized_object_type));
        
        if (!plugin_items.empty())
        {
          items.insert(items.end(), plugin_items.begin(), plugin_items.end());
          bec::MenuItem item;
          item.type = MenuSeparator;
          items.push_back(item);    
        }
      }
    }
    catch (const std::exception &exc)
    {
      g_message("exception fetching context menu items for live tree: %s", exc.what());
    }    
    
    {
      bec::MenuItem item;
      item.caption= _("Refresh All");
      item.name= "refresh";
      items.push_back(item);
    }
  }

  return items;
}


bool LivePhysicalOverviewBE::activate_popup_item_for_nodes(const std::string &name, const std::vector<bec::NodeId> &nodes)
{
  Node *node= get_node(nodes[0]);
  GrtObjectRef object= node->object;

  if ((name == "alter") || (name == "drop"))
  {
    LiveObjectSignal &live_object_signal= (name == "drop") ? drop_live_object_signal : alter_live_object_signal;
    if (db_SchemaRef::can_wrap(object))
      live_object_signal(LiveSchemaTree::Schema, "", *db_SchemaRef::cast_from(object)->name());
    else if (db_TableRef::can_wrap(object))
      live_object_signal(LiveSchemaTree::Table, object->owner()->name(), *object->name());
    else if (db_ViewRef::can_wrap(object))
      live_object_signal(LiveSchemaTree::View, object->owner()->name(), *object->name());
    else if (db_RoutineRef::can_wrap(object))
      live_object_signal(LiveSchemaTree::Routine, object->owner()->name(), *object->name());
    return true;
  }
  else if (name == "select_data" || name == "edit_data" || name == "copy_column_names" ||
    (0 == name.find("clipboard_")) || (0 == name.find("editor_")))
  {
    if (db_SchemaRef::can_wrap(object))
      activate_object_signal(name, LiveSchemaTree::Schema, "", *object->name());
    else if (db_TableRef::can_wrap(object))
      activate_object_signal(name, LiveSchemaTree::Table, object->owner()->name(), *object->name());
    else if (db_ViewRef::can_wrap(object))
      activate_object_signal(name, LiveSchemaTree::View, object->owner()->name(), *object->name());
    else if (db_RoutineRef::can_wrap(object))
      activate_object_signal(name, LiveSchemaTree::Routine, object->owner()->name(), *object->name());
    return true;
  }
  else if (name == "set_active_schema")
  {
    if (db_SchemaRef::can_wrap(object))
      active_schema_change_signal(*object->name());
    return true;
  }
  else if (name == "refresh")
  {
    refresh_signal();
    return true;
  }
  else
  {
    bool ret= false;
    if (db_SchemaRef::can_wrap(object))
      ret= call_object_plugin_items_slot(name, *object->name(), *object->name(), "schema");
    else if (db_TableRef::can_wrap(object))
      ret= call_object_plugin_items_slot(name, object->owner()->name(), *object->name(), "table");
    else if (db_ViewRef::can_wrap(object))
      ret= call_object_plugin_items_slot(name, object->owner()->name(), *object->name(), "view");
    else if (db_RoutineRef::can_wrap(object))
      ret= call_object_plugin_items_slot(name, object->owner()->name(), *object->name(), "routine");
    
    if (ret)
      return true;
  }

  return false;
}


void LivePhysicalOverviewBE::refresh_schemata_node()
{
  refresh_node(NodeId(_schemata_node_index), true);
}


void LivePhysicalOverviewBE::load_schema_contents(db_SchemaRef schema)
{
  schema_fetch_slot(bec::NodeId(), *schema->name(), ::LiveSchemaTree::SchemaContentArrivedSlot());
}


void LivePhysicalOverviewBE::refresh_schema_node(db_SchemaRef schema)
{
  try
  {
    NodeId schema_node_id= get_node_child_for_object(NodeId(_schemata_node_index), schema);
    OverviewBE::ContainerNode *schema_node= dynamic_cast<OverviewBE::ContainerNode*> (get_node(schema_node_id));
    schema_node->clear_children();
    schema_node->init();
  }
  catch (const std::exception &exc)
  {
    g_warning("refresh_schema_node: %s", exc.what());
  }
}


int LivePhysicalOverviewBE::get_default_tab_page_index()
{
  return active_schema_index();
}
