/* 
 * Copyright (c) 2007, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include <sstream>

#include "base/config_file.h"
#include "wb_config.h"
#include "wb_module.h"
#include "wb_overview.h"
#include "model/wb_model_diagram_form.h"
#include "wb_context_ui.h"
#include "wb_context.h"
#include "model/wb_component.h"

#include "mdc_back_layer.h"

#include "grt/common.h"

#include "base/util_public_interface.h"

#include "wbcanvas/model_figure_impl.h"

#include <mforms/mforms.h>

#include "user_defined_type_editor.h"
#include "preferences_form.h"
#include "document_properties_form.h"
#include "grt_shell_window.h"
#include "plugin_manager_window.h"

#include "server_instance_editor.h"
#include "grtui/grtdb_connection_editor.h"
#include "base/string_utilities.h"
#include "grtdb/db_helpers.h"

using namespace wb;
using namespace grt;
using namespace base;
using namespace MySQL::Geometry;

//--------------------------------------------------------------------------------------------------

WorkbenchImpl::WorkbenchImpl(CPPModuleLoader *loader)
: super(loader), _wb(0)
{
#ifdef _WIN32
  _last_wmi_session_id= 1;
  _last_wmi_monitor_id= 1;
#endif
  _current_user_type_editor= NULL;
}

//--------------------------------------------------------------------------------------------------

WorkbenchImpl::~WorkbenchImpl()
{

}

//--------------------------------------------------------------------------------------------------

void WorkbenchImpl::set_context(WBContext *wb)
{
  _wb= wb;
}


#define def_plugin(group, aName, type, aCaption, descr)\
  {\
    app_PluginRef plugin(get_grt());\
    plugin->name("wb."group"."aName);\
    plugin->caption(aCaption);\
    plugin->description(descr);\
    plugin->moduleName("Workbench");\
    plugin->moduleFunctionName(aName);\
    plugin->pluginType(type);\
    plugin->groups().insert("Application/Workbench");\
    list.insert(plugin);\
  }

#define def_object_plugin(group, klass, aName, aCaption, descr)\
  {\
    app_PluginRef plugin(get_grt());\
    plugin->name("wb."group"."aName);\
    plugin->caption(aCaption);\
    plugin->description(descr);\
    plugin->moduleName("Workbench");\
    plugin->moduleFunctionName(aName);\
    plugin->pluginType("normal");\
    plugin->groups().insert("Application/Workbench");\
    app_PluginObjectInputRef input(get_grt());\
    input->owner(plugin);\
    input->objectStructName(klass::static_class_name());\
    plugin->inputValues().insert(input);\
    list.insert(plugin);\
  }


#define def_view_plugin(group, aName, aCaption, descr)\
  {\
    app_PluginRef plugin(get_grt());\
    plugin->name("wb."group"."aName);\
    plugin->caption(aCaption);\
    plugin->description(descr);\
    plugin->moduleName("Workbench");\
    plugin->moduleFunctionName(aName);\
    plugin->pluginType("normal");\
    plugin->groups().insert("Application/Workbench");\
    app_PluginObjectInputRef input(get_grt());\
    input->owner(plugin);\
    input->name("activeDiagram");\
    input->objectStructName(model_Diagram::static_class_name());\
    plugin->inputValues().insert(input);\
    list.insert(plugin);\
  }

#define def_model_plugin(group, aName, aCaption, descr)\
  {\
    app_PluginRef plugin(get_grt());\
    plugin->name("wb."group"."aName);\
    plugin->caption(aCaption);\
    plugin->description(descr);\
    plugin->moduleName("Workbench");\
    plugin->moduleFunctionName(aName);\
    plugin->pluginType("normal");\
    plugin->groups().insert("Application/Workbench");\
    app_PluginObjectInputRef input(get_grt());\
    input->owner(plugin);\
    input->name("activeModel");\
    input->objectStructName(model_Model::static_class_name());\
    plugin->inputValues().insert(input);\
    list.insert(plugin);\
  }

#define def_form_model_plugin(group, aName, aCaption, descr)\
  {\
    app_PluginRef plugin(get_grt());\
    plugin->name("wb."group"."aName);\
    plugin->caption(aCaption);\
    plugin->description(descr);\
    plugin->moduleName("Workbench");\
    plugin->moduleFunctionName(aName);\
    plugin->pluginType("standalone");\
    plugin->groups().insert("Application/Workbench");\
    app_PluginObjectInputRef input(get_grt());\
    input->owner(plugin);\
    input->name("activeModel");\
    input->objectStructName(model_Model::static_class_name());\
    plugin->inputValues().insert(input);\
    list.insert(plugin);\
  }

#define def_form_plugin(group, aName, aCaption, descr)\
  {\
    app_PluginRef plugin(get_grt());\
    plugin->name("wb."group"."aName);\
    plugin->caption(aCaption);\
    plugin->description(descr);\
    plugin->moduleName("Workbench");\
    plugin->moduleFunctionName(aName);\
    plugin->pluginType("standalone");\
    plugin->groups().insert("Application/Workbench");\
    list.insert(plugin);\
  }


#define def_arg_plugin(group, aName, type, aCaption, descr)\
  {\
    app_PluginRef plugin(get_grt());\
    app_PluginInputDefinitionRef pdef(get_grt());\
    plugin->name("wb."group"."aName);\
    plugin->caption(aCaption);\
    plugin->description(descr);\
    plugin->moduleName("Workbench");\
    plugin->moduleFunctionName(aName);\
    plugin->pluginType(type);\
    pdef->owner(plugin);\
    pdef->name("string");\
    plugin->inputValues().insert(pdef);\
    plugin->groups().insert("Application/Workbench");\
    list.insert(plugin);\
  }

#define def_model_arg_plugin(group, aName, type, aCaption, descr)\
  {\
    app_PluginRef plugin(get_grt());\
    plugin->name("wb."group"."aName);\
    plugin->caption(aCaption);\
    plugin->description(descr);\
    plugin->moduleName("Workbench");\
    plugin->moduleFunctionName(aName);\
    plugin->pluginType(type);\
    app_PluginInputDefinitionRef pdef(get_grt());\
      pdef->owner(plugin);\
      pdef->name("string");\
      plugin->inputValues().insert(pdef);\
    app_PluginObjectInputRef model(get_grt());\
      model->owner(plugin);\
      model->name("activeModel");\
      model->objectStructName(model_Model::static_class_name());\
      plugin->inputValues().insert(model);\
    plugin->groups().insert("Application/Workbench");\
    list.insert(plugin);\
  }

#define def_file_plugin(group, aName, ptype, aCaption, descr, aDialogCaption, aType, aExtensions)\
  {\
    app_PluginRef plugin(get_grt());\
    app_PluginFileInputRef pdef(get_grt());\
    plugin->name("wb."group"."aName);\
    plugin->caption(aCaption);\
    plugin->description(descr);\
    plugin->moduleName("Workbench");\
    plugin->moduleFunctionName(aName);\
    plugin->pluginType(ptype);\
    pdef->owner(plugin);\
    pdef->dialogTitle(aDialogCaption);\
    pdef->dialogType(aType);\
    pdef->fileExtensions(aExtensions);\
    plugin->inputValues().insert(pdef);\
    plugin->groups().insert("Application/Workbench");\
    list.insert(plugin);\
  }


ListRef<app_Plugin> WorkbenchImpl::getPluginInfo()
{
  ListRef<app_Plugin> list(get_grt());

  def_plugin      ("file", "newDocument", "internal", "New Model", "New Document");
  def_plugin      ("file", "newDocumentFromDB", "internal", "Reverse Engineer Database", "Reverse Engineer");
  def_file_plugin ("file", "openModel", "internal", "Open Model", "Open Model from File", "Open Model", "open", "mwb");
  def_plugin      ("file", "saveModel", "internal", "Save Model", "Save Model to Current File");
  def_arg_plugin  ("file", "openRecentModel", "internal", "Open Model", "Open Model");
  def_file_plugin ("file", "saveModelAs", "internal", "Save As", "Save Model to a New File", "Save Model", "save", "mwb");
  def_plugin      ("file", "exit", "internal", "Exit", "Exit Workbench");
    
  def_file_plugin ("export", "exportPNG", "normal", "Export as PNG", "Export Current Diagram as PNG", "Export as PNG", "save", "png");
  def_file_plugin ("export", "exportPDF", "normal", "Export as PDF", "Export Current Diagram as PDF", "Export as PDF", "save", "pdf");
  def_file_plugin ("export", "exportSVG", "normal", "Export as SVG", "Export Current Diagram as SVG", "Export as SVG", "save", "svg");
  def_file_plugin ("export", "exportPS", "normal", "Export as PS", "Export Current Diagram as PostScript", "Export as PostScript", "save", "ps");

  def_plugin("edit", "selectAll", "internal", "Select All", "Select All Objects in Diagram");
  def_plugin("edit", "selectSimilar", "internal", "Select Similar Figures", "Select Similar Figures in Diagram");
  def_plugin("edit", "selectConnected", "internal", "Select Connected Figures", "Select Figures Connected to the Selected One");

  def_view_plugin("edit", "raiseSelection", "Bring to Front", "Bring Selected Object to Front");
  def_view_plugin("edit", "lowerSelection", "Send to Back", "Send Selected Object to Back");

  def_view_plugin("edit", "toggleGridAlign", "Align to Grid", "Align Objects to Grid");
  def_view_plugin("edit", "toggleGrid", "Toggle Grid", "Toggle Grid");
  def_view_plugin("edit", "togglePageGrid", "Toggle Page Guides", "Toggle Page Guides");
  
  def_view_plugin("edit", "editSelectedFigure", "Edit Selected Objects", "Edit Selected Objects");
  def_view_plugin("edit", "editSelectedFigureInNewWindow", "Edit Selected Objects in New Window", "Edit Selected Objects in New Window");
  
  def_object_plugin("edit", GrtObject, "editObject", "Edit Selected Object", "Edit Selected Object");
  def_object_plugin("edit", GrtObject, "editObjectInNewWindow", "Edit Selected Object in New Window", "Edit Selected Object in New Window");

  def_plugin("edit", "goToNextSelected", "internal", "Go to Next Selected", "Go to Next Selected Object");
  def_plugin("edit", "goToPreviousSelected", "internal", "Go to Previous Selected", "Go to Previous Selected Object");

  def_plugin("view", "zoomIn", "internal", "Zoom In", "Zoom In Diagram");
  def_plugin("view", "zoomOut", "internal", "Zoom Out", "Zoom Out Diagram");
  def_plugin("view", "zoomDefault", "internal", "Zoom 100%", "Zoom Back Diagram to Default");

  def_arg_plugin("view", "goToMarker", "internal", "Go to Marker", "Go to Previously Set Diagram Marker");
  def_arg_plugin("view", "setMarker", "internal", "Set a Marker", "Set a Diagram Marker to the Current Location");

  def_model_arg_plugin("view", "setFigureNotation", "internal", "Set Objects Notation", "Set Object Notation");
  def_model_arg_plugin("view", "setRelationshipNotation", "internal", "Set Relationships Notation", "Set Relationship Notation");

  def_model_plugin("model", "newDiagram", "Add New Diagram", "Add a New Diagram to the Model");

  def_file_plugin("tools", "runScriptFile", "normal", "Run Script File", "Select a Script File to Execute", "Open Script and Execute", "open", "py,lua");
  def_file_plugin("tools", "installModuleFile", "standalone", "Install Plugin", "Select a Module or Plugin File to Install", "Select Module to Install", "open", "py,lua,mwbplugin,mwbpluginz");

  def_form_model_plugin("form", "showUserTypeEditor", "User Types Editor", "Open User Defined Types Editor");
  def_form_model_plugin("form", "showModelOptions", "Show Model Options", "Open Model Options Window");
  def_form_model_plugin("form", "showDocumentProperties", "Document Properties", "Open Document Properties Window");
  def_form_plugin("form", "showOptions", "Preferences", "Open Options Window");
  def_form_plugin("form", "showConnectionManager", "Manage Database Connections", "Open DB Connection Manager");
  def_form_plugin("form", "showInstanceManager", "Manage Server Instance Profiles", "Open Server Profile Manager");
  def_form_plugin("form", "showQueryConnectDialog", "Query Database...", "Connect to and Query a Database Server");
  def_form_plugin("form", "showGRTShell", "Show GRT Shell...", "Show Workbench Script Development Shell");
  def_plugin("form", "newGRTFile", "standalone", "New Script File...", "Create a new Workbench script/plugin file");
  def_plugin("form", "openGRTFile", "standalone", "Open Script File...", "Open an existing Workbench script/plugin file");
  def_form_plugin("form", "showPluginManager", "Plugin Manager...", "Show Workbench Plugin Manager Window");
  
  def_plugin("debug", "debugValidateGRT", "normal", "Validate GRT Tree", "Validate Consistency of GRT Tree");
  def_plugin("debug", "debugShowInfo", "normal", "Show Debugging Info", "Show various system and application information");
  def_plugin("debug", "debugGrtStats", "normal", "Show GRT Debugging Info", "Show various internal GRT stats");

  return list;
}


int WorkbenchImpl::copyToClipboard(const std::string &astr)
{
  _wb->get_grt_manager()->get_dispatcher()
    ->call_from_main_thread<void>(boost::bind(mforms::Utilities::set_clipboard_text, astr), true, false);

  return 1;
}


int WorkbenchImpl::hasUnsavedChanges()
{
  return _wb->has_unsaved_changes() ? 1 : 0;
}



int WorkbenchImpl::newDocument()
{
  _wb->new_document();

  return 0;
}


int WorkbenchImpl::newDocumentFromDB()
{
  // if there is a model open, do plain reveng, otherwise create one 1st

  if (!_wb->get_document().is_valid())
    _wb->new_document();

  bec::ArgumentPool args;
  _wb->update_plugin_arguments_pool(args);
  args.add_entries_for_object("activeCatalog", _wb->get_document()->physicalModels()[0]->catalog(), "db.Catalog");
  _wb->execute_plugin("db.plugin.database.rev_eng", args);
  
  return 0;
}


int WorkbenchImpl::openModel(const std::string &filename)
{
  _wb->open_document(filename);

  return 0;
}


int WorkbenchImpl::openRecentModel(const std::string &index)
{
  _wb->open_recent_document(atoi(index.c_str()));

  return 0;
}


int WorkbenchImpl::saveModel()
{
  _wb->save_as(_wb->get_filename());

  return 0;
}


int WorkbenchImpl::saveModelAs(const std::string &filename)
{
  _wb->save_as(bec::append_extension_if_needed(filename, ".mwb"));

  return 0;
}


int WorkbenchImpl::exportPNG(const std::string &filename)
{
  _wb->export_png(bec::append_extension_if_needed(filename, ".png"));

  return 0;
}

int WorkbenchImpl::exportPDF(const std::string &filename)
{
  _wb->export_pdf(bec::append_extension_if_needed(filename, ".pdf"));

  return 0;
}


int WorkbenchImpl::exportSVG(const std::string &filename)
{
  _wb->export_svg(bec::append_extension_if_needed(filename, ".svg"));

  return 0;
}


int WorkbenchImpl::exportPS(const std::string &filename)
{
  _wb->export_ps(bec::append_extension_if_needed(filename, ".ps"));

  return 0;
}


static void quit(WBContext *wb)
{
  if (wb->get_ui()->request_quit())
    wb->get_ui()->perform_quit();
}

int WorkbenchImpl::exit()
{
  _wb->get_grt_manager()->get_dispatcher()->
  call_from_main_thread<void>(boost::bind(quit, _wb), false, false);

  return 0;
}



int WorkbenchImpl::selectAll()
{
  if (dynamic_cast<ModelDiagramForm*>(_wb->get_active_form()))
  {
    _wb->get_active_form()->select_all();
  }
  return 0;
}


int WorkbenchImpl::selectSimilar()
{
  if (!dynamic_cast<ModelDiagramForm*>(_wb->get_active_form())) return 0;

  model_DiagramRef view(dynamic_cast<ModelDiagramForm*>(_wb->get_active_form())->get_model_diagram());
  std::string figure_type;

  if (view->selection().count() != 1)
    return 0;

  model_ObjectRef object(view->selection().get(0));

  figure_type= object.class_name();

  view->unselectAll();

  if (model_FigureRef::can_wrap(object))
  {
    for (size_t c= view->figures().count(), i= 0; i < c; i++)
    {
      model_FigureRef figure(view->figures().get(i));

      if (figure.is_instance(figure_type))
        view->selectObject(figure);
    }
  }
  else if (model_ConnectionRef::can_wrap(object))
  {
    for (size_t c= view->connections().count(), i= 0; i < c; i++)
    {
      model_ConnectionRef conn(view->connections().get(i));

      if (conn.is_instance(figure_type))
        view->selectObject(conn);
    }
  }
  else if (model_LayerRef::can_wrap(object))
  {
    for (size_t c= view->layers().count(), i= 0; i < c; i++)
    {
      model_LayerRef layer(view->layers().get(i));

      if (layer.is_instance(figure_type))
        view->selectObject(layer);
    }
  }

  return 0;
}



int WorkbenchImpl::selectConnected()
{
  if (!dynamic_cast<ModelDiagramForm*>(_wb->get_active_form())) return 0;

  model_DiagramRef view(dynamic_cast<ModelDiagramForm*>(_wb->get_active_form())->get_model_diagram());
  std::string figure_type;
  model_FigureRef figure;

  if (view->selection().count() != 1)
    return 0;

  if (model_FigureRef::can_wrap(view->selection().get(0)))
    figure= model_FigureRef::cast_from(view->selection().get(0));

  if (figure.is_valid())
  {
    std::set<std::string> added;

    added.insert(figure.id());

    for (size_t c= view->connections().count(), i= 0; i < c; i++)
    {
      model_ConnectionRef conn(view->connections().get(i));

      if (conn->startFigure() == figure)
      {
        if (added.find(conn->endFigure()->id()) == added.end())
        {
          added.insert(conn->endFigure()->id());
          view->selectObject(conn->endFigure());
        }
      }
      else if (conn->endFigure() == figure)
      {
        if (added.find(conn->startFigure()->id()) == added.end())
        {
          added.insert(conn->startFigure()->id());
          view->selectObject(conn->startFigure());
        }
      }
    }
  }

  return 0;
}


static void activate_object(WBComponent *compo, const model_ObjectRef &object, bool newwindow)
{
  if (compo->handles_figure(object))
    compo->activate_canvas_object(object, newwindow);
}


int WorkbenchImpl::editSelectedFigure(const model_DiagramRef &view)
{
  ModelDiagramForm *form= dynamic_cast<ModelDiagramForm*>(_wb->get_active_form());
  if (form)
  {
    ListRef<model_Object> list(form->get_selection());

    for (size_t c= list.count(), i= 0; i < c; i++)
    {
      _wb->foreach_component(boost::bind(activate_object, _1, list.get(i), false));
    }
  }
  return 0;
}

int WorkbenchImpl::editSelectedFigureInNewWindow(const model_DiagramRef &view)
{
  ModelDiagramForm *form= dynamic_cast<ModelDiagramForm*>(_wb->get_active_form());
  if (form)
  {
    ListRef<model_Object> list(form->get_selection());

    for (size_t c= list.count(), i= 0; i < c; i++)
    {
      _wb->foreach_component(boost::bind(activate_object, _1, list.get(i), true));
    }
  }
  return 0;
}


int WorkbenchImpl::editObject(const GrtObjectRef &object)
{
  _wb->open_object_editor(object, bec::NoFlags);
  return 0;
}


int WorkbenchImpl::editObjectInNewWindow(const GrtObjectRef &object)
{
  _wb->open_object_editor(object, bec::ForceNewWindowFlag);
  return 0;
}



int WorkbenchImpl::goToNextSelected()
{
  ModelDiagramForm *form= dynamic_cast<ModelDiagramForm*>(_wb->get_active_form());
  if (!form) return 0;

  model_DiagramRef view(form->get_model_diagram());

  if (view->selection().count() == 0)
    return 0;

  for (size_t c= view->selection().count(), i= 0; i < c; i++)
  {
    model_Figure::ImplData *figure= model_FigureRef::cast_from(view->selection().get(i))->get_data();
    if (figure && figure->get_canvas_item())
    {
      if (form->get_view()->get_focused_item() == figure->get_canvas_item())
      {
        if (i < view->selection().count()-1)
        {
          form->focus_and_make_visible(view->selection().get(i+1), false);
          return 0;
        }
        break;
      }
    }
  }

  // focus 1st
  form->focus_and_make_visible(view->selection().get(0), false);
  
  return 0;
}


int WorkbenchImpl::goToPreviousSelected()
{
  ModelDiagramForm *form= dynamic_cast<ModelDiagramForm*>(_wb->get_active_form());
  if (!form) return 0;

  model_DiagramRef view(form->get_model_diagram());
  if (view->selection().count() == 0)
    return 0;

  for (size_t c= view->selection().count(), i= 0; i < c; i++)
  {
    model_Figure::ImplData *figure= model_FigureRef::cast_from(view->selection().get(i))->get_data();
    if (figure && figure->get_canvas_item())
    {
      if (form->get_view()->get_focused_item() == figure->get_canvas_item())
      {
        if (i > 0)
        {
          form->focus_and_make_visible(view->selection().get(i-1), false);
          return 0;
        }
        break;
      }
    }
  }
  form->focus_and_make_visible(view->selection().get(view->selection().count()-1), false);
  
  return 0;
}



int WorkbenchImpl::newDiagram(const model_ModelRef &model)
{
  model->addNewDiagram(false);

  return 0;
}


// canvas manipulation

int WorkbenchImpl::raiseSelection(const model_DiagramRef &view)
{
  for (size_t c= view->selection().count(), i= 0; i < c; i++)
  {
    if (view->selection().get(i).is_instance(model_Figure::static_class_name()))
    {
      model_FigureRef figure(model_FigureRef::cast_from(view->selection()[i]));

      figure->layer()->raiseFigure(figure);
    }
  }
  return 0;
}


int WorkbenchImpl::lowerSelection(const model_DiagramRef &view)
{
  for (size_t c= view->selection().count(), i= 0; i < c; i++)
  {
    if (view->selection().get(i).is_instance(model_Figure::static_class_name()))
    {
      model_FigureRef figure(model_FigureRef::cast_from(view->selection()[i]));
      
      figure->layer()->lowerFigure(figure);
    }
  }
  return 0;
}



int WorkbenchImpl::toggleGridAlign(const model_DiagramRef &view)
{
  ModelDiagramForm *form= _wb->get_model_context()->get_diagram_form_for_diagram_id(view->id());
  if (form)
  {
    form->get_view()->set_grid_snapping(!form->get_view()->get_grid_snapping());

    view->options().gset("AlignToGrid", form->get_view()->get_grid_snapping()?1:0);
  }
  return 0;
}


int WorkbenchImpl::toggleGrid(const model_DiagramRef &view)
{
  ModelDiagramForm *form= _wb->get_model_context()->get_diagram_form_for_diagram_id(view->id());
  if (form)
  {    
    form->get_view()->get_background_layer()->set_grid_visible(!form->get_view()->get_background_layer()->get_grid_visible());

    view->options().gset("ShowGrid", form->get_view()->get_background_layer()->get_grid_visible()?1:0);
  }
  return 0;
}


int WorkbenchImpl::togglePageGrid(const model_DiagramRef &view)
{
  ModelDiagramForm *form= _wb->get_model_context()->get_diagram_form_for_diagram_id(view->id());
  if (form)
  {    
    form->get_view()->get_background_layer()->set_paper_visible(!form->get_view()->get_background_layer()->get_paper_visible());
    
    view->options().gset("ShowPageGrid", form->get_view()->get_background_layer()->get_paper_visible()?1:0);
  }
  return 0;
}


int WorkbenchImpl::goToMarker(const std::string &marker)
{
  model_ModelRef model(_wb->get_model_context()->get_active_model(true));

  if (model.is_valid())
  {
    model_MarkerRef mk;
    
    for (size_t c= model->markers().count(), i= 0; i < c; i++)
      if (*model->markers().get(i)->name() == marker)
      {
        mk= model->markers().get(i);
        break;
      }
    
    if (mk.is_valid())
    {
      model_DiagramRef diagram = model_DiagramRef::cast_from(mk->diagram());
      diagram->zoom(mk->zoom());
      diagram->x(mk->x());
      diagram->y(mk->y());

      _wb->get_grt_manager()->get_dispatcher()->call_from_main_thread<void>(
        boost::bind(&WBContext::switch_diagram, _wb, diagram), false, false);
    }
  }

  return 0;
}


int WorkbenchImpl::setMarker(const std::string &marker)
{
  ModelDiagramForm *form= dynamic_cast<ModelDiagramForm*>(_wb->get_ui()->get_active_main_form());

  if (form)
  {
    model_MarkerRef mk(get_grt());
    
    model_ModelRef model(form->get_model_diagram()->owner());

    for (size_t c= model->markers().count(), i= 0; i < c; i++)
      if (*model->markers().get(i)->name() == marker)
      {
        model->markers().remove(i);
        break;
      }
    
    mk->owner(model);
    mk->name(marker);
    mk->diagram(form->get_model_diagram());
    mk->zoom(form->get_view()->get_zoom());
    mk->x(form->get_view()->get_viewport().left());
    mk->y(form->get_view()->get_viewport().top());

    model->markers().insert(mk);
  }
  return 0;
}


int WorkbenchImpl::setFigureNotation(const std::string &name, workbench_physical_ModelRef model)
{
//  model_ModelRef model(_wb->get_ui()->get_active_model(true));

  if (model.is_valid() && model.is_instance<workbench_physical_Model>())
    workbench_physical_ModelRef::cast_from(model)->figureNotation(name);

  return 0;
}


int WorkbenchImpl::setRelationshipNotation(const std::string &name, workbench_physical_ModelRef model)
{
//  model_ModelRef model(_wb->get_ui()->get_active_model(true));

  if (model.is_valid() && model.is_instance<workbench_physical_Model>())
    workbench_physical_ModelRef::cast_from(model)->connectionNotation(name);

  return 0;
}


int WorkbenchImpl::zoomIn()
{
  ModelDiagramForm *form= dynamic_cast<ModelDiagramForm*>(_wb->get_active_main_form());
  if (!form) return 0;

  form->zoom_in();
  return 0;
}


int WorkbenchImpl::zoomOut()
{
  ModelDiagramForm *form= dynamic_cast<ModelDiagramForm*>(_wb->get_active_main_form());
  if (!form) return 0;

  form->zoom_out();
  
  return 0;
}


int WorkbenchImpl::zoomDefault()
{
  ModelDiagramForm *form= dynamic_cast<ModelDiagramForm*>(_wb->get_active_main_form());
  if (!form) return 0;

  model_DiagramRef view(form->get_model_diagram());

  view->zoom(1.0);

  return 0;
}



int WorkbenchImpl::startTrackingUndo()
{
  get_grt()->begin_undoable_action();
  return 0;
}


int WorkbenchImpl::finishTrackingUndo(const std::string &description)
{
  get_grt()->end_undoable_action(description);
  return 0;
}


int WorkbenchImpl::cancelTrackingUndo()
{
  get_grt()->cancel_undoable_action();
  return 0;
}



int WorkbenchImpl::addUndoListAdd(const BaseListRef &list)
{
  get_grt()->get_undo_manager()->add_undo(new grt::UndoListInsertAction(list));
  return 0;
}

int WorkbenchImpl::addUndoListRemove(const BaseListRef &list, int index)
{
  get_grt()->get_undo_manager()->add_undo(new grt::UndoListRemoveAction(list, index));
  return 0;
}

int WorkbenchImpl::addUndoObjectChange(const ObjectRef &object, const std::string &member)
{
  get_grt()->get_undo_manager()->add_undo(new grt::UndoObjectChangeAction(object, member));
  return 0;
}

int WorkbenchImpl::addUndoDictSet(const DictRef &dict, const std::string &key)
{
  _wb->get_grt()->get_undo_manager()->add_undo(new grt::UndoDictSetAction(dict, key));
  return 0;
}

int WorkbenchImpl::beginUndoGroup()
{
  _wb->get_grt()->get_undo_manager()->begin_undo_group();
  return 0;
}

int WorkbenchImpl::endUndoGroup()
{
  _wb->get_grt()->get_undo_manager()->end_undo_group();
  return 0;
}

int WorkbenchImpl::setUndoDescription(const std::string &text)
{
  _wb->get_grt()->get_undo_manager()->set_action_description(text);
  return 0;
}


std::string WorkbenchImpl::createAttachedFile(const std::string &group, 
                                              const std::string &tmpl)
{
  return _wb->create_attached_file(group, tmpl);
}


int WorkbenchImpl::setAttachedFileContents(const std::string &filename, 
                                           const std::string &text)
{
  _wb->save_attached_file_contents(filename, text.data(), text.size());
  return 0;
}


std::string WorkbenchImpl::getAttachedFileContents(const std::string &filename)
{
  return _wb->get_attached_file_contents(filename);
}


std::string WorkbenchImpl::getAttachedFileTmpPath(const std::string &filename)
{
  return _wb->get_attached_file_tmp_path(filename);
}


int WorkbenchImpl::exportAttachedFileContents(const std::string &filename, 
                                              const std::string &export_to)
{
  return _wb->export_attached_file_contents(filename, export_to);
}

workbench_DocumentRef WorkbenchImpl::openModelFile(const std::string &path)
{
  return _wb->openModelFile(path);
}

int WorkbenchImpl::closeModelFile()
{
  return _wb->closeModelFile();
}

std::string WorkbenchImpl::getTempDir()
{
    return _wb->getTempDir();
}

std::string WorkbenchImpl::getDbFilePath()
{
  return _wb->getDbFilePath();
}

int WorkbenchImpl::runScriptFile(const std::string &filename)
{
  get_grt()->make_output_visible();
  get_grt()->send_output("Executing script "+filename+"...\n");

  _wb->get_grt_manager()->push_status_text("Executing script...");
  
  grt::AutoUndo undo(get_grt());
  
  _wb->get_grt_manager()->get_shell()->run_script_file(filename);

  char *basename= g_path_get_basename(filename.c_str());
  undo.end_or_cancel_if_empty(strfmt("Execute Script %s", basename));
  g_free(basename);
  
  get_grt()->send_output("\nScript finished.\n");
  _wb->get_grt_manager()->pop_status_text();
  
  return 0;
}


int WorkbenchImpl::installModuleFile(const std::string &filename)
{
  _wb->install_module_file(filename);
  return 0;
}


static int traverse_value(GRT *grt, const ObjectRef &owner, const std::string &member, const ValueRef &value);

static bool traverse_member(const MetaClass::Member *member, const ObjectRef &owner, const ObjectRef &object, GRT *grt)
{      
  std::string k= member->name;
  ValueRef v= object->get_member(k);

  if (!v.is_valid())
  {
    if((member->type.base.type == ListType) || (member->type.base.type == DictType))
      grt->send_output(strfmt("%s[%s] (type: %s, name: '%s', id: %s), has NULL list or dict member: '%s'\n", 
                                owner.class_name().c_str(), k.c_str(), 
                                object.class_name().c_str(), object.get_string_member("name").c_str(), object.id().c_str(),
                                k.c_str()));
  }

  if (k == "owner")
  {
    if (ObjectRef::cast_from(v) != owner)
    {
      if (!v.is_valid())
        grt->send_output(strfmt("%s[%s] (type: %s, name: '%s', id: %s), has no owner set\n", 
                                  owner.class_name().c_str(), member->name.c_str(),
                                  object.class_name().c_str(), object->get_string_member("name").c_str(), object.id().c_str()));
      else
        grt->send_output(strfmt("%s[%s] (type: %s, name: '%s', id: %s), has bad owner (or missing attr:dontfollow)\n", 
                                  owner.class_name().c_str(), member->name.c_str(), 
                                  object.class_name().c_str(), object->get_string_member("name").c_str(), object.id().c_str()));
    }
  }

  if (member->owned_object)
    traverse_value(grt, object, k, v);
  
  return true;
}


static int traverse_value(GRT *grt, const ObjectRef &owner, const std::string &member, const ValueRef &value)
{
  switch (value.type())
  {
  case DictType:
    {
      DictRef dict(DictRef::cast_from(value));

      for (DictRef::const_iterator iter= dict.begin(); iter != dict.end(); ++iter)
      {
        std::string k= iter->first;
        ValueRef v= iter->second;

        traverse_value(grt, owner, k, v);
      }
    }
    break;
  case ListType:
    {
      BaseListRef list(BaseListRef::cast_from(value));

      for (size_t c= list.count(), i= 0; i < c; i++)
      {
        ValueRef v;

        v= list.get(i);

        traverse_value(grt, owner, strfmt("%i", (int) i), v);
      }
    }
    break;
  case ObjectType:
    {
      ObjectRef object(ObjectRef::cast_from(value));
      MetaClass *gstruct= object->get_metaclass();
     
      gstruct->foreach_member(boost::bind(traverse_member, _1, owner, object, grt));
    }
    break;

  default:
    break;
  }
  return 0;
}


int WorkbenchImpl::debugValidateGRT()
{
  ValueRef root(get_grt()->root());
  ObjectRef owner;

  get_grt()->send_output("Validating GRT Tree...\n");

//QQQ  get_grt()->lock_tree_read();

  // make sure that all nodes have their owner set to their parent object
  // make sure that all refs that are not owned are marked dontfollow
  traverse_value(get_grt(), owner, "root", root);

//QQQ  get_grt()->unlock_tree_read();

  get_grt()->send_output("GRT Tree Validation Finished.\n");

  return 0;
}


int WorkbenchImpl::debugGrtStats()
{
  return 0;
}

//--------------------------------------------------------------------------------------------------

int WorkbenchImpl::debugShowInfo()
{
  grt::GRT *grt= get_grt();

  grt->make_output_visible();

#ifdef EDITION_OSS
#define EDITION_NAME "CE"
#elif defined(EDITION_SE)
#define EDITION_NAME "SE"
#else
#define EDITION_NAME "unknown edition"
#endif

#ifdef _WIN32
#define PLATFORM_NAME "Windows"
#define NATIVE_RENDERER "GDI"
#elif defined(__APPLE__)
#define PLATFORM_NAME "Mac OS X"
#define NATIVE_RENDERER "Quartz"
#else
#define PLATFORM_NAME "Linux/Unix"
#define NATIVE_RENDERER "Native"
#endif

  grt->send_output(strfmt("MySQL Workbench "EDITION_NAME" for "PLATFORM_NAME" version %i.%i.%i\n", APP_MAJOR_NUMBER, APP_MINOR_NUMBER, APP_RELEASE_NUMBER));

  grt->send_output(strfmt("Configuration Directory: %s\n", _wb->get_grt_manager()->get_user_datadir().c_str()));
  grt->send_output(strfmt("Data Directory: %s\n", _wb->get_grt_manager()->get_basedir().c_str()));
  
  int cver= cairo_version();
  grt->send_output(strfmt("Cairo Version: %i.%i.%i\n", (cver/10000)%100, (cver/100)%100, cver%100));

  gchar *tmp;
  const gchar *render_mode = "Unknown";
  const gchar *ogl_state;
  if (_wb->is_opengl_available())
    ogl_state= "OpenGL is available on this system";
  else
    ogl_state= "OpenGL is not available on this system";

  if (_wb->use_opengl())
  {
    if (_wb->is_opengl_available())
      render_mode= "so OpenGL is used for rendering.";
    else
      render_mode= "but there is an INTERNAL ERROR, contact the authors."; // Should never happen. Must be a coding mistake.
  }
  else
  {
    if (_wb->is_opengl_available())
      render_mode=  ("but "NATIVE_RENDERER" was requested. Using "NATIVE_RENDERER" for rendering.");
    else
      render_mode=  ("so "NATIVE_RENDERER" is used for rendering.");
  }

  grt->send_output(strfmt("Rendering Mode: %s, %s\n", ogl_state, render_mode));
  if (_wb->use_opengl())
  {
    if (_wb->is_opengl_available())
      grt->send_output(strfmt("OpenGL Driver Version: %s\n", _wb->_opengl_version.c_str()));
    else
      grt->send_output(strfmt("OpenGL Driver Version: Not Detected\n"));
  }
  grt->send_output(strfmt("OS: %s\n", tmp= get_local_os_name()));
  g_free(tmp);
  grt->send_output(strfmt("CPU: %s\n", tmp= get_local_hardware_info()));
  g_free(tmp);

#ifdef _WIN32
  grt->send_output(getFullVideoAdapterInfo());

  int locale_buffer_size = GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_SLANGUAGE, NULL, 0);
  if (locale_buffer_size > 0)
  {
    TCHAR* buffer = new TCHAR[locale_buffer_size];
    GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_SLANGUAGE, buffer, locale_buffer_size);

    CW2A converted_buffer(buffer, CP_UTF8);
    delete [] buffer;

    grt->send_output(strfmt("Current user language: %s\n", converted_buffer));
  }
  else
  {
    // Very unlikely the system cannot return the current user language, but just in case.
    grt->send_output("Unable to determine current user language.\n");
  }

#elif defined(__APPLE__)
//not needed? maybe get just machine model info #warning NOT IMPLEMENTED
#else
#warning not implemented
#endif

  grt->send_output("\n");

  return 0;
}

//--------------------------------------------------------------------------------------------------

std::string WorkbenchImpl::request_input_m(const std::string &caption)
{
  std::string result;

  if (_wb->request_input(caption, 0, result))
    return result;
  return "";
}


std::string WorkbenchImpl::input(const std::string &caption)
{
  return _wb->get_grt_manager()->get_dispatcher()->call_from_main_thread<std::string>(
    boost::bind(&WorkbenchImpl::request_input_m, this, caption), true, false);
}


int WorkbenchImpl::confirm(const std::string &title, const std::string &caption)
{
  return _wb->get_grt_manager()->get_dispatcher()->call_from_main_thread<int>(
    boost::bind(mforms::Utilities::show_message, title, caption, _("OK"), _("Cancel"), ""), true, false);
}


std::string WorkbenchImpl::requestFileOpen(const std::string &caption, const std::string &extensions)
{
  return _wb->get_grt_manager()->get_dispatcher()->
  call_from_main_thread<std::string>(boost::bind(_wb->show_file_dialog, "open", caption, extensions), true, false);
}


std::string WorkbenchImpl::requestFileSave(const std::string &caption, const std::string &extensions)
{
  return _wb->get_grt_manager()->get_dispatcher()->
  call_from_main_thread<std::string>(boost::bind(_wb->show_file_dialog, "save", caption, extensions), true, false);
}

//--------------------------------------------------------------------------------------------------

int WorkbenchImpl::showUserTypeEditor(const workbench_physical_ModelRef &model)
{
  if ( _current_user_type_editor == NULL)
  {
    _current_user_type_editor = new UserDefinedTypeEditor(_wb->get_ui(), model);
    scoped_connect(_current_user_type_editor->signal_closed(),boost::bind(&WorkbenchImpl::userTypeEditorClosed, this));
  }
  _current_user_type_editor->show_modal(NULL, NULL);
  
  return 0;
}


int WorkbenchImpl::showGRTShell()
{  
  _wb->get_ui()->get_shell_window()->show();

  return 0;
}

int WorkbenchImpl::newGRTFile()
{
  _wb->get_ui()->get_shell_window()->show();
  _wb->get_ui()->get_shell_window()->add_new_script();
  
  return 0;
}

int WorkbenchImpl::openGRTFile()
{
  _wb->get_ui()->get_shell_window()->show();
  _wb->get_ui()->get_shell_window()->open_script_file();
  
  return 0;
}

/**
 * Called by the front end when the user data type editor has been closed. We can then remove our reference to it.
 */
void WorkbenchImpl::userTypeEditorClosed()
{
  _current_user_type_editor= NULL;
}

int WorkbenchImpl::showDocumentProperties()
{
  DocumentPropertiesForm props(_wb->get_ui());
    
  props.show();
  
  return 0;
}


int WorkbenchImpl::showModelOptions(const workbench_physical_ModelRef &model)
{
  PreferencesForm prefs(_wb->get_ui(), model.id());
  prefs.show();
  
  return 0;
}


int WorkbenchImpl::showOptions()
{
  PreferencesForm prefs(_wb->get_ui());
  prefs.show();
  
  return 0;
}


int WorkbenchImpl::showConnectionManager()
{
  grtui::DbConnectionEditor editor(_wb->get_root()->rdbmsMgmt());
  _wb->show_status_text("Connection Manager Opened.");  
  editor.run();
  _wb->show_status_text("");  
  _wb->get_ui()->refresh_home_connections();
  
  return 0;
}


int WorkbenchImpl::showInstanceManager()
{
  ServerInstanceEditor editor(_wb->get_grt_manager(), _wb->get_root()->rdbmsMgmt());
  _wb->show_status_text("Server Profile Manager Opened.");
  db_mgmt_ServerInstanceRef instance(editor.run());
  _wb->show_status_text("");
  // save instance list now
  _wb->save_app_options();
  _wb->get_ui()->refresh_home_instances((int)_wb->get_root()->rdbmsMgmt()->storedInstances().get_index(instance));
  return 0;
}


int WorkbenchImpl::showQueryConnectDialog()
{
  _wb->add_new_query_window(db_mgmt_ConnectionRef());
  
  return 0;
}


int WorkbenchImpl::showPluginManager()
{
  PluginManagerWindow pm(_wb);
  
  pm.run();
  return 0;
}

//--------------------------------------------------------------------------------------------------

#ifdef _WIN32

/**
 * Opens a new wmi session, which is essentially a connection to a computer that can later be used to
 * query that box, manipulate service states or monitor values.
 *
 * @server The name or IP address of the machine to connect to. Leave empty for localhost.
 * @user The user name for the connection. Leave empty for localhost.
 * @password The password for the given user. Ignored when connecting to localhost.
 * @return A unique id for the new session. Use that for any further call and don't forgot to close the session
 *         once you don't need it anymore or you will get a memory leak. Returns 0 on error.
 */
int WorkbenchImpl::wmiOpenSession(const std::string server, const std::string& user, const std::string& password)
{
  wmi::WmiServices* services = new wmi::WmiServices(server, user, password);
  _wmi_sessions[++_last_wmi_session_id] = services;
#ifdef DEBUG
  _thread_for_wmi_session[_last_wmi_session_id] = g_thread_self();
#endif
  return _last_wmi_session_id;
}


#ifdef DEBUG
#define check_wmi_thread(session) {\
  if (_thread_for_wmi_session.find(session) != _thread_for_wmi_session.end())\
  {\
    if (g_thread_self() != _thread_for_wmi_session[session])\
    {\
      g_warning("%s called from invalid thread for WMI session %i", __FUNCTION__, session);\
      throw std::logic_error("WMI access from invalid thread"); \
    }\
  }\
}
#else
#define check_wmi_thread(s)
#endif

//--------------------------------------------------------------------------------------------------

/**
 * Closes the wmi session opened with wmiOpenSession. Necessary to avoid memory leaks. If there are
 * pending monitors for that session they will be closed implicitly.
 *
 * @param session The session that should be closed.
 * @return 1 if the session was successfully closed, -1 if the session is invalid.
 */
int WorkbenchImpl::wmiCloseSession(int session)
{
  if (_wmi_sessions.find(session) == _wmi_sessions.end())
    return -1;

  wmi::WmiServices* services = _wmi_sessions[session];
  std::map<int, wmi::WmiMonitor*>::iterator next, i= _wmi_monitors.begin(); 
  while (i != _wmi_monitors.end())
  {
    next = i;
    ++next;
    if (i->second->owner() == services)
      wmiStopMonitoring(i->first);
    i = next;
  }
  delete services;

  _wmi_sessions.erase(session);

  return 1;
}

//--------------------------------------------------------------------------------------------------

/**
 * Executes a query against the given server session. The query must be in WQL format.
 *
 * @param session The session to work against.
 * @param query The query to execute. For further info see http://msdn.microsoft.com/en-us/library/aa394606%28VS.85%29.aspx.
 * @return A list of GRT dicts containing the objects returned by the query, that is, name/value pairs
 *         of object properties.
 */
grt::DictListRef WorkbenchImpl::wmiQuery(int session, const std::string& query)
{
  if (_wmi_sessions.find(session) == _wmi_sessions.end())
    return grt::DictListRef();

  check_wmi_thread(session);

  wmi::WmiServices* services = _wmi_sessions[session];
  
  return services->query(_wb->get_grt(), query);
}

//--------------------------------------------------------------------------------------------------

/**
 * Used to query or control a service on a target machine.
 *
 * @param session The session describing the target connection.
 * @param service The name of the service on that machine to work with.
 * @param action The action to be executed. Allowed values are: status, start, stop.
 * @return A result describing the outcome of the action. Can be:
 *   - completed
 *   - error
 *   - already-running
 *   - already-stopped
 *   - already-starting
 *   - already-stopping
 *   - stopping
 *   - starting
 */
std::string WorkbenchImpl::wmiServiceControl(int session, const std::string& service, const std::string& action)
{
  if (_wmi_sessions.find(session) == _wmi_sessions.end())
    return "error - Invalid wmi session";
 
  check_wmi_thread(session);

  wmi::WmiServices* services = _wmi_sessions[session];
 
  return services->serviceControl(service, action);
}

//--------------------------------------------------------------------------------------------------

/**
 * Queries the target machine for certain system statistics.
 *
 * @param session The session describing the target connection.
 * @param what The value to query. Can be: TotalVisibleMemorySize, FreePhysicalMemory.
 * @return The asked for value. If what is invalid then the result is simply 0. The returned value
 *         is formatted as string to cater for different types of return values.
 */
std::string WorkbenchImpl::wmiSystemStat(int session, const std::string& what)
{
  if (_wmi_sessions.find(session) == _wmi_sessions.end())
    return "error - Invalid wmi session";
 
  check_wmi_thread(session);

  wmi::WmiServices* services = _wmi_sessions[session];
  
  return services->systemStat(what);
}

//--------------------------------------------------------------------------------------------------

/**
 * Starts a monitoring session for a given connection. It will set up a monitoring context to allow
 * quick queries for a single value. The monitor must be freed with wmiStopMonitoring to avoid errors and
 * memory leaks.
 *
 * @param session The session describing the target connection.
 * @param what The property/value to monitor. Supported values are: LoadPercentage.
 * @return A unique id describing the new monitor.
 */
int WorkbenchImpl::wmiStartMonitoring(int session, const std::string& what)
{
  if (_wmi_sessions.find(session) == _wmi_sessions.end())
    return -1;
              
  check_wmi_thread(session);

  wmi::WmiServices* services = _wmi_sessions[session];
  wmi::WmiMonitor* monitor = services->startMonitoring(what);
  _wmi_monitors[++_last_wmi_monitor_id] = monitor;
  
  return _last_wmi_monitor_id;
}

//--------------------------------------------------------------------------------------------------

/**
 * Reads the current value of the monitored property.
 *
 * @param monitor The monitor set up with wmiStartMonitoring.
 * @return The current value formatted as string.
 */
std::string WorkbenchImpl::wmiReadValue(int monitor_id)
{
  if (_wmi_monitors.find(monitor_id) == _wmi_monitors.end())
    return "error - Invalid wmi session";
               
  wmi::WmiMonitor* monitor = _wmi_monitors[monitor_id];
  
  return monitor->readValue();
}

//--------------------------------------------------------------------------------------------------

/**
 * Stops the giving monitor. After this call the monitor is no longer valid.
 *
 * @param monitor The monitor to stop.
 * @return -1 if monitor_id is invalid, else 1
 */
int WorkbenchImpl::wmiStopMonitoring(int monitor_id)
{
  if (_wmi_monitors.find(monitor_id) == _wmi_monitors.end())
    return -1;
               
  wmi::WmiMonitor* monitor = _wmi_monitors[monitor_id];
 
  delete monitor;
  _wmi_monitors.erase(monitor_id);

  return 1;
}

#endif

//--------------------------------------------------------------------------------------------------

static const char *DEFAULT_RDBMS_ID= "com.mysql.rdbms.mysql";

/**
 * Creates a new connection ref and adds it to the stored connections collection.
 * The new connection is also returned.
 */
db_mgmt_ConnectionRef WorkbenchImpl::create_connection(const std::string& host, const std::string& user,
                                                       const std::string socket_or_pipe_name,
                                                       bool can_use_networking,
                                                       bool can_use_socket_or_pipe,
                                                       int port,
                                                       const std::string& name)
{
  db_mgmt_RdbmsRef rdbms = find_object_in_list(_wb->get_root()->rdbmsMgmt()->rdbms(), DEFAULT_RDBMS_ID);
  grt::ListRef<db_mgmt_Connection> connections(_wb->get_root()->rdbmsMgmt()->storedConns());

  db_mgmt_ConnectionRef connection = db_mgmt_ConnectionRef(_wb->get_grt());
  db_mgmt_DriverRef driver;
  if (can_use_networking)
    driver = rdbms->defaultDriver();
  else
  {
    // If networking is enabled but sockets/named pipes aren't enabled then the server
    // is actually not accessible. We play nice though and use the default driver in that case.
    if (can_use_socket_or_pipe)
    {
      driver = find_object_in_list(rdbms->drivers(), "com.mysql.rdbms.mysql.driver.native_socket");
      if (!driver.is_valid())
        driver = rdbms->defaultDriver();
      else
        connection->parameterValues().gset("socket", socket_or_pipe_name);
    }
    else
      driver = rdbms->defaultDriver();
  }

  connection->driver(driver);
  connection->name(name);

  connection->parameterValues().gset("hostName", "localhost");
  connection->parameterValues().gset("userName", "root");
  connection->parameterValues().gset("port", port);

  connection->hostIdentifier(bec::get_host_identifier_for_connection(connection));
  connections.insert(connection);

  return connection;
}

//--------------------------------------------------------------------------------------------------

/**
 * Returns a list of Dicts with data for each locally installed MySQL server.
 */
grt::DictListRef WorkbenchImpl::getLocalServerList()
{
  grt::DictListRef entries;
  try
  {
#ifdef _WIN32
    int session = wmiOpenSession("", "", "");
    entries = wmiQuery(session,
      "select * from Win32_Service where (Name like \"%mysql%\" or DisplayName like \"%mysql%\")");
    wmiCloseSession(session);
#else
    // TODO: other OSes go here.
#endif
  }
  catch (...)
  {
    // If for some reason (e.g. insufficient rights) the retrieval fails then we return an empty list.
  }

  return entries;
}

//--------------------------------------------------------------------------------------------------

/**
 * Creates a list of new connections to all local servers found.
 */
int WorkbenchImpl::createConnectionsFromLocalServers()
{
  grt::DictListRef servers = getLocalServerList();
  if (!servers.is_valid())
    return -1;

  db_mgmt_RdbmsRef rdbms = find_object_in_list(_wb->get_root()->rdbmsMgmt()->rdbms(), DEFAULT_RDBMS_ID);
  grt::ListRef<db_mgmt_Connection> connections(_wb->get_root()->rdbmsMgmt()->storedConns());
  size_t count = servers->count();
  for (size_t i = 0; i < count; i++)
  {
    grt::DictRef server(servers[i]);

    std::string service_display_name = server.get_string("DisplayName", "invalid");
    std::string path = server.get_string("PathName", "invalid");
    std::string config_file = base::extract_option_from_command_line("--defaults-file", path);

    if (g_file_test(config_file.c_str(), G_FILE_TEST_EXISTS))
    {
      ConfigurationFile key_file(config_file, base::AutoCreateSections | base::AutoCreateKeys);

      // Early out if the default section is not there.
      if (!key_file.has_section("mysqld"))
        continue;

      bool can_use_networking = true;
      bool can_use_socket_or_pipe = false;
      std::string socket_or_pipe_name;

      int port = key_file.get_int("port", "mysqld");
      if (port == INT_MIN)
        port = 3306;
      if (key_file.has_key("skip-networking", "mysqld"))
        can_use_networking = false;
      if (key_file.has_key("enable-named-pipe", "mysqld"))
        can_use_socket_or_pipe = true;

      if (can_use_socket_or_pipe)
      {
        socket_or_pipe_name = key_file.get_string("socket", "mysqld");
        if (socket_or_pipe_name.size() == 0)
          socket_or_pipe_name = "MySQL"; 
      }

      create_connection("localhost", "root", socket_or_pipe_name, can_use_networking,
        can_use_socket_or_pipe, port, _("Local ") + service_display_name);
    }
  }
  return 0;
}

extern std::string get_admin_script_for_os(const std::string &os);
//--------------------------------------------------------------------------------------------------

/**
 * Creates a list of server instance entries for all local servers found.
 */
int WorkbenchImpl::createInstancesFromLocalServers()
{

  try
  {
    grt::DictListRef servers = getLocalServerList();
    if (!servers.is_valid())
      return -1;

    db_mgmt_RdbmsRef rdbms = find_object_in_list(_wb->get_root()->rdbmsMgmt()->rdbms(), DEFAULT_RDBMS_ID);
    grt::ListRef<db_mgmt_ServerInstance> instances(_wb->get_root()->rdbmsMgmt()->storedInstances());
    size_t count = servers->count();
    for (size_t i = 0; i < count; i++)
    {
      grt::DictRef entry(servers[i]);

      std::string service_display_name = entry.get_string("DisplayName", "invalid");
      std::string service_name = entry.get_string("Name", "invalid");
      std::string path = entry.get_string("PathName", "invalid");
      std::string config_file = base::extract_option_from_command_line("--defaults-file", path);

      if (g_file_test(config_file.c_str(), G_FILE_TEST_EXISTS))
      {
        ConfigurationFile key_file(config_file, base::AutoCreateSections | base::AutoCreateKeys);

        // Early out if the default section is not there.
        if (!key_file.has_section("mysqld"))
          continue;

        bool can_use_networking = true;
        bool can_use_socket_or_pipe = false;
        std::string socket_or_pipe_name;

        int port = key_file.get_int("port", "mysqld");
        if (port == INT_MIN)
          port = 3306;
        if (key_file.has_key("skip-networking", "mysqld"))
          can_use_networking = false;
        if (key_file.has_key("enable-named-pipe", "mysqld"))
          can_use_socket_or_pipe = true;

        if (can_use_socket_or_pipe)
        {
          socket_or_pipe_name = key_file.get_string("socket", "mysqld");
          if (socket_or_pipe_name.size() == 0)
            socket_or_pipe_name = "MySQL"; 
        }

        db_mgmt_ServerInstanceRef instance(_wb->get_grt());

        instance->owner(_wb->get_root()->rdbmsMgmt());

        instance->serverInfo().gset("sys.system", "Windows");
        instance->serverInfo().gset("windowsAdmin", 1);
        instance->loginInfo().gset("wmi.userName", ""); // Only used for remote connections.
        instance->loginInfo().gset("wmi.hostName", ""); // Only used for remote connections.

        instance->name("Local " + service_display_name);

        instance->serverInfo().gset("sys.preset", "Custom");
        instance->serverInfo().gset("sys.script", get_admin_script_for_os("Windows"));

        instance->serverInfo().gset("sys.config.path", config_file);
        instance->serverInfo().gset("sys.config.section", "mysqld");
        instance->serverInfo().gset("sys.mysqld.service_name", service_name);

        // Finally look up a connection or create one we can use for this instance.
        db_mgmt_ConnectionRef connection;
        grt::ListRef<db_mgmt_Connection> connections(_wb->get_root()->rdbmsMgmt()->storedConns());
        for (grt::ListRef<db_mgmt_Connection>::const_iterator end = connections.end(),
          iterator = connections.begin(); iterator != end; ++iterator)
        {
          grt::DictRef parameters((*iterator)->parameterValues());

          // Must be a localhost connection.
          std::string parameter = parameters.get_string("hostName");
          if (!parameter.empty() && parameter != "localhost" && parameter != "127.0.0.1")
            continue;

          // For now we only consider connections with a root user. We might later add support
          // for any user.
          parameter = parameters.get_string("userName");
          if (parameter != "root")
            continue;

          if (can_use_networking)
          {
            int other_port = parameters.get_int("port");
            if (other_port != port)
              continue;

            // Only native tcp/ip network connections for now.
            if ((*iterator)->driver()->id() != "com.mysql.rdbms.mysql.driver.native")
              continue;
          }
          else
            if (can_use_socket_or_pipe)
            {
              parameter = parameters.get_string("socket");
              if (parameter.size() == 0)
                parameter = "MySQL"; // Default pipe/socket name.

              if (parameter != socket_or_pipe_name)
                continue;

              // Only native socket/pipe connections for now.
              if ((*iterator)->driver()->id() != "com.mysql.rdbms.mysql.driver.native_socket")
                continue;
            }

            // All parameters are ok. This is a connection we can use.
            connection = *iterator;
            break;
        }

        // If we did not find a connection for this instance then create a new one.
        if (!connection.is_valid())
          connection = create_connection("localhost", "root", socket_or_pipe_name, can_use_networking,
          can_use_socket_or_pipe, port, _("Local instance ") + service_display_name);

        instance->connection(connection);
        instances.insert(instance);
      }
    }
  }
  catch (...)
  {
    // If for some reason (e.g. insufficient rights) the wmi session throws an exception
    // we don't want to see it. We simply do not create the list in that case.
  }

  return 0;
}

//--------------------------------------------------------------------------------------------------

/**
 * Returns a short string describing the currently active video adapter, especially the used chipset.
 */
std::string WorkbenchImpl::getVideoAdapter()
{
  std::string result = _("Unknown");
  try
  {
#ifdef _WIN32
    int session = wmiOpenSession("", "", "");
    grt::DictListRef entries = wmiQuery(session,
      "select Availability, VideoProcessor from Win32_VideoController");
    wmiCloseSession(session);

    size_t count = entries->count();
    for (size_t i = 0; i < count; i++)
    {
      grt::DictRef entry(entries[i]);

      if (entry.get_string("Availability", "0") != "3")
        continue;

      // Return the first active adapter we find.
      result = entry.get_string("VideoProcessor", "Unknown video processor");
      break;
    }
#else
    // TODO: other OSes go here.
#endif
  }
  catch (...)
  {
    // If for some reason (e.g. insufficient rights) the retrieval fails then we return an empty list.
  }
  return result;
}

//--------------------------------------------------------------------------------------------------

/**
 * Returns all available info about the currently active video adapter in human readable format.
 */
std::string WorkbenchImpl::getFullVideoAdapterInfo()
{
  std::stringstream result;
  try
  {
#ifdef _WIN32
    int session = wmiOpenSession("", "", "");
    grt::DictListRef entries = wmiQuery(session, "select * from Win32_VideoController");
    wmiCloseSession(session);

    size_t count = entries->count();
    for (size_t i = 0; i < count; i++)
    {
      grt::DictRef entry(entries[i]);

      if (entry.get_string("Availability", "0") != "3")
        continue;

      // Return the first active adapter we find.
      result << "Active video adapter: " << entry.get_string("Caption", "") << '\n';
      std::string value = entry.get_string("AdapterRAM", "");
      if (value.size() > 0)
      {
        int size = atoi(value.c_str()) / 1024 / 1024;
        result << "Installed video RAM: " << size << " MB\n";
      }
      else
        result << "Installed video RAM: unknown\n";

      result << "Current video mode: " << entry.get_string("VideoModeDescription", "unknown") << '\n';

      value = entry.get_string("CurrentBitsPerPixel", "");
      if (value.size() > 0)
        result << "Used bit depth: " << value << '\n';
      else
        result << "Used bit depth: unknown\n";
      
      result << "Driver version: " << entry.get_string("DriverVersion", "unknown") << '\n';
      result << "Installed display drivers: " << entry.get_string("InstalledDisplayDrivers", "unknown") << '\n';

      break;
    }
#else
    // TODO: other OSes go here.
#endif
  }
  catch (...)
  {
    // If for some reason (e.g. insufficient rights) the retrieval fails then we return an empty list.
  }
  return result.str();
}

//--------------------------------------------------------------------------------------------------
