/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
#ifndef _EDITOR_BASE_H_
#define _EDITOR_BASE_H_


#include "base/ui_form.h"
#include <grtpp.h>
#include "grt/grt_manager.h"
#include <grtpp_undo_manager.h>
#include "sqlide/sql_editor_be.h"
#include "refresh_ui.h"

#include "wbpublic_public_interface.h"


class GrtObject;


namespace bec {
  struct ChangeSet;



  class WBPUBLICBACKEND_PUBLIC_FUNC UndoObjectChangeGroup : public grt::UndoGroup
  {
    std::string _object_id;
    std::string _member;
    
  public:
    UndoObjectChangeGroup(const std::string &object_id, const std::string &member);

    virtual bool matches_group(UndoGroup *group) const;
  };


  class WBPUBLICBACKEND_PUBLIC_FUNC BaseEditor : public UIForm, public RefreshUI
  {
  public:
    BaseEditor(GRTManager *grtm, const grt::Ref<GrtObject> &object);
    virtual ~BaseEditor();
    
    virtual std::string get_form_context_name() const;

    virtual Sql_editor::Ref get_sql_editor() { return Sql_editor::Ref(); }

    virtual bool should_close_on_delete_of(const std::string &oid) { return get_object().id() == oid; }

    // must return a copy of the object
    virtual grt::Ref<GrtObject> get_object()= 0;

    virtual bool is_editing_live_object() { return false; }
    virtual void apply_changes_to_live_object() {}
    virtual void refresh_live_object() {}
    typedef boost::function<int (long long, const std::string&, const std::string&)> Live_object_change_error_cb;
    Live_object_change_error_cb on_live_object_change_error;
    typedef boost::function<int (float)> Live_object_change_progress_cb;
    Live_object_change_progress_cb on_live_object_change_progress;
    typedef boost::function<int (long, long)> Live_object_change_statistics_cb;
    Live_object_change_statistics_cb on_live_object_change_statistics;

    grt::GRT *get_grt() { return _grtm->get_grt(); }
    GRTManager *get_grt_manager() { return _grtm; }

    virtual void on_object_changed();

  protected:
    GRTManager *_grtm;
    boost::signals2::scoped_connection _ui_refresh_conn;

    std::set<std::string> _ignored_object_fields_for_ui_refresh;

    void add_listeners(const grt::Ref<GrtObject> &object);

    void run_from_grt(const boost::function<void ()> &slot);

  private:
    friend class AutoUndoEdit;

    grt::Ref<GrtObject> _object;
    void object_member_changed(const std::string &member, const grt::ValueRef &ovalue);
    
    void undo_applied();
  };

  
  class WBPUBLICBACKEND_PUBLIC_FUNC AutoUndoEdit : public grt::AutoUndo
  {
    
    static void undo_applied(grt::UndoAction *applied, grt::UndoGroup *group, BaseEditor *editor)
    {
      if (group == applied)
        editor->undo_applied();
    }
    
  public:
    AutoUndoEdit(BaseEditor *editor)
    // if editing a live object, this should be a no-op
      : grt::AutoUndo(editor->get_grt(), editor->is_editing_live_object())
    {
      if (group)
      {
        /*
        //sigc::connection conn1, conn2;
        conn1= editor->get_grt()->get_undo_manager()->
        signal_undo().connect(sigc::bind(sigc::ptr_fun(&AutoUndoEdit::undo_applied), group, editor));
          signal_undo().connect(sigc::bind(sigc::ptr_fun(&AutoUndoEdit::undo_applied), group, editor));
        signal_undo().connect(boost::bind(\&AutoUndoEdit::undo_applied, _1, group, editor));

        conn2= editor->get_grt()->get_undo_manager()->
        signal_redo().connect(boost::bind(&AutoUndoEdit::undo_applied, _1, group, editor));

        mark_for_delete_when_destroyed(editor, conn1, conn2);
        */
        editor->scoped_connect(editor->get_grt()->get_undo_manager()->
          signal_undo(),boost::bind(&AutoUndoEdit::undo_applied, _1, group, editor));
        editor->scoped_connect(editor->get_grt()->get_undo_manager()->
          signal_redo(),boost::bind(&AutoUndoEdit::undo_applied,_1, group, editor));

      }
    }

    AutoUndoEdit(BaseEditor *editor, const grt::ObjectRef &object, const std::string &member)
      : grt::AutoUndo(editor->get_grt(), new UndoObjectChangeGroup(object.id(), member), editor->is_editing_live_object())
    {
      if (group)
      {
        /*
        //sigc::connection conn1, conn2;

        conn1= editor->get_grt()->get_undo_manager()->
        signal_undo().connect(boost::bind(&AutoUndoEdit::undo_applied, _1, group, editor));

        conn2= editor->get_grt()->get_undo_manager()->
        signal_redo().connect(boost::bind(&AutoUndoEdit::undo_applied, _1, group, editor));

        mark_for_delete_when_destroyed(editor, conn1, conn2);
        */
        editor->scoped_connect((editor->get_grt()->get_undo_manager()->
          signal_undo()),boost::bind(&AutoUndoEdit::undo_applied, _1, group, editor));
        editor->scoped_connect((editor->get_grt()->get_undo_manager()->
          signal_redo()),boost::bind(&AutoUndoEdit::undo_applied, _1, group, editor));

      }
    }
  };
};

#endif /* _EDITOR_BASE_H_ */
