/* 
 * Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#ifndef __WB_H__
#define __WB_H__

#include "wb_config.h"
#include "workbench/wb_context.h"
#include "workbench/wb_context_ui.h"
#include "model/wb_model_diagram_form.h"
#include "workbench/wb_history_tree.h"
#include "model/wb_diagram_options.h"
#include "model/wb_user_datatypes.h"
#include "model/wb_overview_physical.h"

#include "common/preferences_form.h"
#include "common/document_properties_form.h"

#include "GrtTemplates.h"
#include "SigCSlot.h"
#include "FindDialogBE.h"
#include "Overview.h"
#include "ModelViewForm.h"

#if WBContext_VERSION > 5
# error outdated wrapper Wb.h
#endif

using namespace System;
using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;

using namespace MySQL::Grt;
using namespace MySQL::GUI::Mdc;
using namespace MySQL::Forms;

namespace MySQL {
namespace Workbench {

public enum class Msg_type
{
  MT_warning=  grt::WarningMsg,
  MT_error=    grt::ErrorMsg,
  MT_info=     grt::InfoMsg,
  MT_progress= grt::ProgressMsg,
};

//--------------------------------------------------------------------------------------------------

public ref class PageSettings
{
public:
  PageSettings()
    : paper_type(nullptr), margin_top(0), margin_bottom(0), margin_left(0), margin_right(0), 
      orientation(nullptr)
  {
  }

  PageSettings(const app_PageSettingsRef &settings)
    : paper_type(CppStringToNative(settings->paperType().is_valid() ? settings->paperType()->name() : "")),
    margin_top(settings->marginTop()), margin_bottom(settings->marginBottom()),
    margin_left(settings->marginLeft()), margin_right(settings->marginRight()),
    orientation(CppStringToNative(settings->orientation()))
  {
  }

  void update_object(grt::ListRef<app_PaperType> paperTypes, app_PageSettingsRef settings)
  {
    settings->paperType(grt::find_named_object_in_list(paperTypes, NativeToCppString(paper_type)));

    settings->marginTop(grt::DoubleRef(margin_top));
    settings->marginBottom(grt::DoubleRef(margin_bottom));
    settings->marginLeft(grt::DoubleRef(margin_left));
    settings->marginRight(grt::DoubleRef(margin_right));

    settings->orientation(NativeToCppString(orientation));
  }

  String^ paper_type;
  double margin_top;
  double margin_bottom;
  double margin_left;
  double margin_right;
  String^ orientation;
};


public ref class PaperSize
{
public:
  PaperSize(const wb::WBPaperSize &paperSize)
    : name(CppStringToNative(paperSize.name)), 
    caption(CppStringToNative(paperSize.caption)), 
    margins_set(paperSize.margins_set),
    margin_top(paperSize.margin_top), margin_bottom(paperSize.margin_bottom), 
    margin_left(paperSize.margin_left), margin_right(paperSize.margin_right), 
    width(paperSize.width), height(paperSize.height),
    description(CppStringToNative(paperSize.description))
  {
  }

  String^ name;
  String^ caption;
  double width;
  double height;
  bool margins_set;
  double margin_top;
  double margin_bottom;
  double margin_left;
  double margin_right;
  String^ description;
};


// ----------------------------------------------------------------------------


public ref class WbOptions
{
  String^ basedir;
  String^ plugin_search_path;
  String^ struct_search_path;
  String^ module_search_path;
  String^ library_search_path;
  String^ user_data_dir;
  String^ open_at_startup;
  String^ open_at_startup_type;
  String^ run_at_startup;
  String^ run_at_startup_lang;
  bool force_sw_rendering;
  bool force_opengl_rendering;
  bool use_log_file;
  bool quit_when_done;

public:
  WbOptions(String^ basedir, String^ plugin_search_path, String^ struct_search_path,
    String^ module_search_path, String^ library_search_path, String^ user_data_dir, String^ open_at_startup, 
    String^ open_at_startup_type, String^ run_at_startup, String^ run_at_startup_lang,
    bool force_sw_rendering, bool force_opengl_rendering, bool use_log_file, bool quit_when_done) 
    : basedir(basedir),
    plugin_search_path(plugin_search_path),
    struct_search_path(struct_search_path),
    module_search_path(module_search_path),
    library_search_path(library_search_path),
    user_data_dir(user_data_dir),
    open_at_startup(open_at_startup),
    open_at_startup_type(open_at_startup_type),
    run_at_startup(run_at_startup),
    run_at_startup_lang(run_at_startup_lang),
    force_sw_rendering(force_sw_rendering),
    force_opengl_rendering(force_opengl_rendering),
    use_log_file(use_log_file),
    quit_when_done(quit_when_done)
  {
  }

  ::wb::WBOptions *get_unmanaged_object()
  {
    ::wb::WBOptions *opt= new ::wb::WBOptions;
    opt->basedir= NativeToCppString(basedir);
    opt->plugin_search_path= NativeToCppString(plugin_search_path);
    opt->struct_search_path= NativeToCppString(struct_search_path);
    opt->module_search_path= NativeToCppString(module_search_path);
    opt->library_search_path= NativeToCppString(library_search_path);
    opt->user_data_dir= NativeToCppString(user_data_dir);
    opt->open_at_startup= NativeToCppString(open_at_startup);
    opt->open_at_startup_type= NativeToCppString(open_at_startup_type);
    opt->run_at_startup= NativeToCppString(run_at_startup);
    opt->run_language= NativeToCppString(run_at_startup_lang);
    opt->force_sw_rendering= force_sw_rendering;
    opt->force_opengl_rendering= force_opengl_rendering;
    opt->use_log_file= use_log_file;
    opt->quit_when_done= quit_when_done;
    return opt;
  }
};


public ref class WbContext
{
  ::wb::WBContextUI *inner;

  ::MySQL::Grt::GrtManager^ manager;
  ::MySQL::Workbench::Overview^ physical_overview;
  ::MySQL::Grt::GrtValueTree^ catalog_tree;
  ::MySQL::Grt::TreeModel^ history_tree;
  ::MySQL::Grt::ListModel^ usertypes_tree;

  ::System::Collections::ArrayList open_editor_slot_wrappers;

  DelegateSlot0<void,void> ^undo_delegate;
  DelegateSlot0<bool,bool> ^can_undo_delegate;
  DelegateSlot0<void,void> ^redo_delegate;
  DelegateSlot0<bool,bool> ^can_redo_delegate;
  DelegateSlot0<void,void> ^copy_delegate;
  DelegateSlot0<bool,bool> ^can_copy_delegate;
  DelegateSlot0<void,void> ^cut_delegate;
  DelegateSlot0<bool,bool> ^can_cut_delegate;
  DelegateSlot0<void,void> ^paste_delegate;
  DelegateSlot0<bool,bool> ^can_paste_delegate;
  DelegateSlot0<void,void> ^select_all_delegate;
  DelegateSlot0<bool,bool> ^can_select_all_delegate;
  DelegateSlot0<void,void> ^delete_delegate;
  DelegateSlot0<bool,bool> ^can_delete_delegate;
  DelegateSlot0<void,void> ^search_delegate;
  DelegateSlot0<void,void> ^find_delegate;
  DelegateSlot0<bool,bool> ^can_find_delegate;
  DelegateSlot0<void,void> ^find_replace_delegate;
  DelegateSlot0<bool,bool> ^can_find_replace_delegate;

public:
  delegate void VoidStrUIFormDelegate(String^ str1, UIForm^ form);

private:
  // void (string, bec::UIFrom)
  [UnmanagedFunctionPointerAttribute(CallingConvention::Cdecl)]
  delegate void VoidStrUIFormWrapperDelegate(const std::string& str1, boost::shared_ptr<bec::UIForm> form);
  typedef void (*WbContext::VOID_STR_UIFORM_CB)(const std::string& str1, boost::shared_ptr<bec::UIForm> form);

  // TODO: implement differently, simpler!
  // Creating these views needs a lot of specialized knowledge, which requires to include many heavy-weight
  // classes (e.g. SQL editor). This should not be handled by a general callback but by the classes
  // that need to create a view (can't the wrapper create the frontend view class?).
  VoidStrUIFormDelegate^ create_main_form_view_delegate;
  VoidStrUIFormWrapperDelegate^ create_main_form_view_wrapper_delegate;
  void set_create_main_form_view(MySQL::Forms::AppImpl^ application, VoidStrUIFormDelegate^ dt);
  void create_main_form_view_wrapper(const std::string& view_name, boost::shared_ptr<bec::UIForm> form_be);

public:
  WbContext(bool verbose) : inner(new ::wb::WBContextUI(verbose)), manager(nullptr)
  { }

  ~WbContext()
  {
  }

  ::wb::WBContextUI *get_unmanaged_object() 
  { return inner; }

  bool init(MySQL::Forms::AppImpl^ application, WbOptions^ options,
    VoidStrUIFormDelegate^ create_main_form_view)
  {
    set_create_main_form_view(application, create_main_form_view);
    return inner->init(
      application->get_callbacks(), 
      options->get_unmanaged_object());
  }

  bool use_opengl()
  {
    return inner->get_wb()->use_opengl();
  }

  bool is_busy()
  {
    return inner->get_wb()->get_grt_manager()->get_dispatcher()->get_busy();
  }

  bool request_quit()
  {
    return inner->request_quit();
  }

  void perform_quit()
  {
    inner->perform_quit();
  }

  bool is_quitting()
  {
    return inner->is_quitting();
  }

  void finalize()
  {
    inner->finalize();
  }

  GrtManager^ get_grt_manager()
  {
    if (manager == nullptr)
      manager= gcnew GrtManager(inner->get_wb()->get_grt_manager());

    return manager; 
  }

  MenuStrip^ menu_for_form(MySQL::Grt::UIForm^ form);
  ToolStrip^ toolbar_for_form(MySQL::Grt::UIForm^ form);

  String^ get_title()
  { return CppStringToNative(inner->get_title()); }

  bool has_unsaved_changes()
  { return inner->get_wb()->has_unsaved_changes(); }

  void open_document(String^ file)
  {
    inner->get_wb()->open_document(NativeToCppString(file));
  }

  bool save_changes()
  { return inner->get_wb()->save_changes(); }

  void flush_idle_tasks()
  {
    try 
    {
      inner->get_wb()->flush_idle_tasks(); 
    }
    catch(std::exception *ex)
    {
      throw gcnew BackendException(ex);
    }
    catch(std::exception &ex)
    {
      throw gcnew BackendException(&ex);
    }
    catch(...)
    {
      throw; // Don't wrap the exception into something completely useless. The original exception
             // often is converted automatically into a proper managed exception.
    }
  }

  double delay_for_next_timer()
  {
    return inner->get_wb()->get_grt_manager()->delay_for_next_timeout();
  }

  void flush_timers()
  {
    return inner->get_wb()->get_grt_manager()->flush_timers();
  }


  // ----- Edit menu handling
  void validate_edit_menu()
  {
    (*inner->get_command_ui()->signal_validate_edit_menu_items())();
  }

  void edit_undo()
  {
    if (inner->get_active_main_form())
      inner->get_active_main_form()->undo();
  }

  bool edit_can_undo()
  {
    if (inner->get_active_main_form() && inner->get_active_main_form()->can_undo())
      return true;
    return false;
  }

  void edit_redo()
  {
    if (inner->get_active_main_form())
      inner->get_active_main_form()->redo();
  }

  bool edit_can_redo()
  {
    if (inner->get_active_main_form() && inner->get_active_main_form()->can_redo())
      return true;
    return false;
  }

  void edit_copy()
  {
    if (inner->get_active_form())
      inner->get_active_form()->copy();
  }

  bool edit_can_copy()
  {
    if (inner->get_active_form() && inner->get_active_form()->can_copy())
      return true;
    return false;
  }

  void edit_cut()
  {
    if (inner->get_active_form())
      inner->get_active_form()->cut();
  }

  bool edit_can_cut()
  {
    if (inner->get_active_form() && inner->get_active_form()->can_cut())
      return true;
    return false;
  }

  void edit_paste()
  {
    if (inner->get_active_form())
      inner->get_active_form()->paste();
  }

  bool edit_can_paste()
  {
    if (inner->get_active_form() && inner->get_active_form()->can_paste())
      return true;
    return false;
  }

  void edit_select_all()
  {
    if (inner->get_active_form())
      inner->get_active_form()->select_all();
  }

  bool edit_can_select_all()
  {
    if (inner->get_active_form() && inner->get_active_form()->can_select_all())
      return true;
    return false;
  }

  void edit_delete()
  {
    if (inner->get_active_form())
      inner->get_active_form()->delete_selection();
  }

  bool edit_can_delete()
  {
    if (inner->get_active_form() && inner->get_active_form()->can_delete())
      return true;
    return false;
  }


  bool try_searching_diagram(String ^text)
  {
    bec::UIForm *form= inner->get_active_main_form();
    if (form && dynamic_cast<wb::ModelDiagramForm*>(form))
    {
      dynamic_cast<wb::ModelDiagramForm*>(form)->search_and_focus_object(NativeToCppString(text));
      return true;
    }
    return false;
  }

  
  typedef DelegateSlot0<void, void> CommandActionDelegate;
  typedef DelegateSlot0<bool, bool> CommandValidateDelegate;
  

  void set_edit_menu_delegates(CommandActionDelegate::ManagedDelegate ^undo_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_undo_delegate,
                               CommandActionDelegate::ManagedDelegate ^redo_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_redo_delegate,
                               CommandActionDelegate::ManagedDelegate ^copy_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_copy_delegate,
                               CommandActionDelegate::ManagedDelegate ^cut_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_cut_delegate,
                               CommandActionDelegate::ManagedDelegate ^paste_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_paste_delegate,
                               CommandActionDelegate::ManagedDelegate ^select_all_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_select_all_delegate,
                               CommandActionDelegate::ManagedDelegate ^delete_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_delete_delegate,
                               CommandActionDelegate::ManagedDelegate ^search_delegate,
                               CommandActionDelegate::ManagedDelegate ^find_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_find_delegate,
                               CommandActionDelegate::ManagedDelegate ^find_replace_delegate,
                               CommandValidateDelegate::ManagedDelegate ^can_find_replace_delegate)
  {
    this->undo_delegate= gcnew CommandActionDelegate(undo_delegate);
    this->can_undo_delegate= gcnew CommandValidateDelegate(can_undo_delegate);
    this->redo_delegate= gcnew CommandActionDelegate(redo_delegate);
    this->can_redo_delegate= gcnew CommandValidateDelegate(can_redo_delegate);
    this->copy_delegate= gcnew CommandActionDelegate(copy_delegate);
    this->can_copy_delegate= gcnew CommandValidateDelegate(can_copy_delegate);
    this->cut_delegate= gcnew CommandActionDelegate(cut_delegate);
    this->can_cut_delegate= gcnew CommandValidateDelegate(can_cut_delegate);
    this->paste_delegate= gcnew CommandActionDelegate(paste_delegate);
    this->can_paste_delegate= gcnew CommandValidateDelegate(can_paste_delegate);
    this->select_all_delegate= gcnew CommandActionDelegate(select_all_delegate);
    this->can_select_all_delegate= gcnew CommandValidateDelegate(can_select_all_delegate);
    this->delete_delegate= gcnew CommandActionDelegate(delete_delegate);
    this->can_delete_delegate= gcnew CommandValidateDelegate(can_delete_delegate);
    this->search_delegate= gcnew CommandActionDelegate(search_delegate);
    this->find_delegate= gcnew CommandActionDelegate(find_delegate);
    this->can_find_delegate= gcnew CommandValidateDelegate(can_find_delegate);
    this->find_replace_delegate= gcnew CommandActionDelegate(find_replace_delegate);
    this->can_find_replace_delegate= gcnew CommandValidateDelegate(can_find_replace_delegate);

    inner->get_command_ui()->add_builtin_command("undo", 
        this->undo_delegate->get_slot(),
        this->can_undo_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("redo", 
        this->redo_delegate->get_slot(),
        this->can_redo_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("copy", 
        this->copy_delegate->get_slot(),
        this->can_copy_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("cut", 
        this->cut_delegate->get_slot(),
        this->can_cut_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("paste", 
        this->paste_delegate->get_slot(),
        this->can_paste_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("selectAll", 
        this->select_all_delegate->get_slot(),
        this->can_select_all_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("delete", 
        this->delete_delegate->get_slot(),
        this->can_delete_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("find", 
        this->find_delegate->get_slot(),
        this->can_find_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("search_replace", 
        this->find_replace_delegate->get_slot(),
        this->can_find_replace_delegate->get_slot());
    inner->get_command_ui()->add_builtin_command("searchbox", 
        this->search_delegate->get_slot());
  }
  


  // plugins
  void add_frontend_commands(List<String^>^ commands)
  {
    inner->get_command_ui()->add_frontend_commands(NativeToCppStringList2(commands));
  }

  void activate_command(String^ name)
  {
    std::string command= NativeToCppString(name);
    inner->get_command_ui()->activate_command(command);
  }

  virtual FindDialogBE^ get_find_dialog()
  {
    return gcnew FindDialogBE(inner->get_find_dialog());
  }

  // Overview

  Overview^ get_physical_overview()
  {
    if (physical_overview == nullptr)
      physical_overview= gcnew Overview(inner->get_physical_overview());
    return physical_overview;
  }

  ::MySQL::Grt::GrtValueTree^ get_catalog_tree()
  {
    if (catalog_tree == nullptr)
      catalog_tree= gcnew ::MySQL::Grt::GrtValueTree(inner->get_catalog_tree());

    return catalog_tree;
  }

  ::MySQL::Grt::TreeModel^ get_history_tree()
  {
    if (history_tree == nullptr)
      history_tree= gcnew ::MySQL::Grt::TreeModel(inner->get_history_tree());
    
    return history_tree;
  }

  ::MySQL::Grt::ListModel^ get_usertypes_tree()
  {
    if (usertypes_tree == nullptr)
      usertypes_tree= gcnew ::MySQL::Grt::ListModel(inner->get_usertypes_tree());
    
    return usertypes_tree;
  } 

  ::MySQL::Grt::GrtValueInspector^ get_inspector_for_selection(UIForm^ form, [Out] List<String^>^ %items)
  {
    std::vector<std::string> vitems;
    bec::ValueInspectorBE *insp= 
      inner->create_inspector_for_selection(form->get_unmanaged_object(), vitems);
    if (!insp)
      insp= inner->create_inspector_for_selection(vitems);
    if (insp)
    {
      items= CppStringListToNative(vitems);
      return gcnew ::MySQL::Grt::GrtValueInspector(insp, true);
    }
    return nullptr;
  }

  bool are_lists_equal(GrtValue^ v1, GrtValue^ v2)
  {
    if ((nullptr == v1) || (nullptr == v2))
      return (v1 == v2);
    ::grt::ValueRef v1_(v1->get_unmanaged_object());
    ::grt::ValueRef v2_(v2->get_unmanaged_object());
    ::grt::ObjectListRef l1_= ::grt::ObjectListRef::cast_from(v1_);
    ::grt::ObjectListRef l2_= ::grt::ObjectListRef::cast_from(v2_);

    return ::grt::compare_list_contents(l1_, l2_);
  }

  String^ get_description_for_selection(UIForm^ form, [Out] GrtValue^ %activeObjList, [Out] List<String^>^ %items)
  {
    std::vector<std::string> vitems;
    ::grt::ListRef<GrtObject> activeObjList_;
    std::string val= 
      inner->get_description_for_selection(form->get_unmanaged_object(), activeObjList_, vitems);
    activeObjList= gcnew GrtValue(activeObjList_);
    items= CppStringListToNative(vitems);
    return CppStringToNative(val);
  }

  String^ get_description_for_selection([Out] GrtValue^ %activeObjList, [Out] List<String^>^ %items)
  {
    std::vector<std::string> vitems;
    ::grt::ListRef<GrtObject> activeObjList_;
    std::string val= inner->get_description_for_selection(activeObjList_, vitems);
    activeObjList= gcnew GrtValue(activeObjList_);
    items= CppStringListToNative(vitems);
    return CppStringToNative(val);
  }

  void set_description_for_selection(GrtValue^ activeObjList, String^ val)
  {
    std::string val_= NativeToCppString(val);
    ::grt::ListRef<GrtObject> activeObjList_(::grt::ListRef<GrtObject>::cast_from(activeObjList->get_unmanaged_object()));
    inner->set_description_for_selection(activeObjList_, val_);
  }

  ::MySQL::Workbench::ModelViewForm^ get_diagram_form_for_diagram(String^ id)
  {
    return gcnew ModelViewForm(inner->get_wb()->get_model_context()->get_diagram_form_for_diagram_id(NativeToCppString(id)));
  }


  void set_active_form(UIForm^ uiform)
  {
    if (uiform == nullptr)
      inner->set_active_form(0);
    else
      inner->set_active_form(uiform->get_unmanaged_object());
  }

  void set_active_form_from_appview(MySQL::Forms::AppViewDockContent^ form)
  {
    if (form == nullptr)
      inner->set_active_form(0);
    else
      inner->set_active_form(form->get_unmanaged_object());
  }

  String ^get_active_context()
  { return CppStringToNative(inner->get_active_context()); }


  void show_output()
  {
    inner->show_output();
  }

  void close_gui_plugin(IntPtr handle)
  {
    inner->get_wb()->close_gui_plugin((NativeHandle)handle.ToPointer());
  }
  
  void execute_plugin(String^ name)
  {
    inner->get_wb()->execute_plugin(NativeToCppStringRaw(name));
  }

  // State support.
  String^ read_state(String^ name, String^ domain, String^ default_value);
  int read_state(String^ name, String^ domain, const int default_value);
  double read_state(String^ name, String^ domain, const double default_value);
  bool read_state(String^ name, String^ domain, const bool default_value);

  void save_state(String^ name, String^ domain, String^ value);
  void save_state(String^ name, String^ domain, const int value);
  void save_state(String^ name, String^ domain, const double value);
  void save_state(String^ name, String^ domain, const bool value);

  // Preferences support.
  String^ read_option_value(String^ model, String^ key, String^ default_value);

  // Paper.
  List<PaperSize^>^ get_paper_sizes()
  {
    return MySQL::Grt::CppListToObjectList<::wb::WBPaperSize,PaperSize>(inner->get_paper_sizes(false));
  }


  PageSettings^ get_page_settings()
  {
    app_PageSettingsRef settings(inner->get_page_settings());
    if (settings.is_valid())
      return gcnew PageSettings(settings);
    return nullptr;
  }


  void set_page_settings(PageSettings^ settings)
  {
    settings->update_object(inner->get_wb()->get_root()->options()->paperTypes(), inner->get_page_settings());
    inner->get_wb()->get_model_context()->update_page_settings();
  }

  /**
   * To be called by the front end once the main form is ready.
   */
  void finished_loading(WbOptions^ options)
  {
    inner->init_finish(options->get_unmanaged_object());
  }

  void close_document_finish()
  {
    inner->get_wb()->close_document_finish();

    // release overview object
    physical_overview = nullptr;
  }

  void new_model_finish()
  {
    inner->get_wb()->new_model_finish();
  }

  String^ get_filename()
  {
    return CppStringToNative(inner->get_wb()->get_filename());
  }  

};



public ref class DiagramOptionsBE
{
  wb::DiagramOptionsBE *inner;

  MySQL::Grt::DelegateSlot0<void, void> ^_delegate;

public:
  DiagramOptionsBE(WindowsGDICanvasView ^view, WbContext ^wbContext, MySQL::Grt::DelegateSlot0<void, void>::ManagedDelegate ^deleg)
  {
    _delegate= gcnew MySQL::Grt::DelegateSlot0<void, void>(deleg);

    inner= new wb::DiagramOptionsBE(view->get_unmanaged_object(), 
      wbContext->get_unmanaged_object()->get_wb()->get_model_context()->get_active_model_diagram(true), wbContext->get_unmanaged_object()->get_wb());

    inner->scoped_connect(inner->signal_changed(),_delegate->get_slot());
  }

  ~DiagramOptionsBE()
  {
    delete _delegate;
    delete inner;
  }

  String ^get_name()
  {
    return CppStringToNative(inner->get_name());
  }

  void set_name(String ^name)
  {
    inner->set_name(NativeToCppString(name));
  }

  int get_xpages() 
  {
    return inner->get_xpages();
  }

  int get_ypages()
  {
    return inner->get_ypages();
  }

  void set_xpages(int c)
  {
    inner->set_xpages(c);
  }

  void set_ypages(int c)
  {
    inner->set_ypages(c);
  }

  void commit()
  {
    inner->commit();
  }

  void update_size()
  {
    inner->update_size();
  }
};


} // namespace Workbench
} // namespace MySQL

#endif // __WB_H__
