#include "sqlide_form.h"
#include "sqlide/utils_fe.h"
#include "linux_utilities/gtk_helpers.h"
#include "sql_snippets_view.h"
#include "mforms/utilities.h"
#include <boost/foreach.hpp>
#include <fstream>
#include "base/string_utilities.h"
#include "linux_utilities/toolbar_manager.h"
#include "mforms/../gtk/lf_view.h"
#include "mforms/../gtk/lf_menubar.h"
#include "mforms/../gtk/lf_toolbar.h"

using base::strfmt;

void DbSqlEditorView::refresh_ui(const int what)
{
  switch (what)
  {
  case Db_sql_editor::RefreshSidebar:
    //_sidebar.refresh(true);
    break;
  case Db_sql_editor::RefreshEditor:
    active_editor()->set_text(_be->sql_editor()->sql());
    break;
  case Db_sql_editor::RefreshEditorBackend:
    _be->sql_editor()->sql(active_editor()->get_text());
    break;
  case Db_sql_editor::RefreshEditorTitle:
    editor_caption_changed();
    break;
  case Db_sql_editor::RefreshRecordsetTitle:
    recordset_caption_changed();
    break;
  case Db_sql_editor::RefreshSnippets:
    _sql_snippets_view->refresh();
    break;
  case Db_sql_editor::RunCurrentScript:
    execute_sql_script(false);
    break;
  case Db_sql_editor::RunCurrentStatement:
    execute_sql_script(true);
    break;
  case Db_sql_editor::RefreshOverview:
    if (_live_physical_overview)
      _live_physical_overview->refresh_active_group_node_children();
    break;
  }
}


DbSqlEditorView *DbSqlEditorView::create(Db_sql_editor::Ref editor_be)
{
  Glib::RefPtr<Gtk::Builder> xml = Gtk::Builder::create_from_file(editor_be->grt_manager()->get_data_file_path("db_sql_editor_view.glade"));
  
  DbSqlEditorView *v = 0;
  xml->get_widget_derived<DbSqlEditorView>("db_sql_editor_view", v);
  v = Gtk::manage(v);
  v->post_construct(editor_be);
  
  return v;
}


DbSqlEditorView::DbSqlEditorView(GtkVBox *vbox, Glib::RefPtr<Gtk::Builder> xml)
:
Gtk::VBox(vbox),
FormViewBase(),
_xml(xml),
_closing(false),
_live_physical_overview(0)
{
}


void DbSqlEditorView::post_construct(Db_sql_editor::Ref editor_be)
{
  Gtk::Widget *v = this;
  _be= editor_be;

  if (editor_be->wbsql()->get_wbui()->get_wb()->get_wb_options().get_int("Sidebar:RightAligned", 0))
  {
    Gtk::Paned *paned;
    _xml->get_widget("hpaned1", paned);
    swap_panned_children(paned, true);
  }

  {
    Gtk::Container *placeholder;
    _xml->get_widget("sql_snippets_placeholder", placeholder);

    wb::WBContextUI *wbui= editor_be->wbsql()->get_wbui();
    _sql_snippets_view= Gtk::manage(new SqlSnippetsView(wbui, _xml));
    placeholder->add(*_sql_snippets_view);
    _sql_snippets_view->show_all();
  }

  _be->exec_sql_task->progress_cb(sigc::mem_fun(this, &DbSqlEditorView::on_exec_sql_progress));
  _be->recordset_list_changed.connect(sigc::mem_fun(this, &DbSqlEditorView::recordset_list_changed));
  _be->output_text_slot= sigc::mem_fun(this, &DbSqlEditorView::output_text);
  
  _be->set_partial_refresh_ui_slot(sigc::mem_fun(this, &DbSqlEditorView::refresh_ui));
  
  _be->sql_editor_text_insert_signal.connect(sigc::mem_fun(this, &DbSqlEditorView::on_sql_editor_text_insert));

  _be->set_frontend_data(dynamic_cast<FormViewBase*>(this));

  mforms::View *sbview = _be->get_sidebar();
  Gtk::Widget *sidebar = mforms::gtk::ViewImpl::get_widget_for_view(sbview);
  Gtk::Container *place;
  _xml->get_widget("sidebar_placeholder", place);
  place->add(*sidebar);
  sidebar->show();
  
  _dispatch_rset_update.connect(sigc::mem_fun(this, &DbSqlEditorView::update_resultsets));
  
  // setup toolbar and menubar
  {
    Gtk::VBox *vbox = this;
    
    Gtk::Widget *w;
    mforms::ToolBar *tbar = _be->get_toolbar();
    w = mforms::gtk::widget_for_toolbar(tbar);
    if (w)
    {
      vbox->pack_start(*w, false, true);
      vbox->reorder_child(*w, 0);
    }
    
    mforms::MenuBar *menu = _be->get_menubar();
    w = mforms::gtk::widget_for_menubar(menu);
    if (w)
    {
      vbox->pack_start(*w, false, true);
      vbox->reorder_child(*w, 0);
    }
  }
}


DbSqlEditorView::~DbSqlEditorView()
{
}

//------------------------------------------------------------------------------
void DbSqlEditorView::dispose()
{
  if (_live_physical_overview)
    _live_physical_overview->reset();
  delete _live_physical_overview;
  _be->close();
  _be.reset();
  _xml.clear();
}

//------------------------------------------------------------------------------
bool DbSqlEditorView::toolbar_action(const std::string& item)
{
  _be->live_physical_overview()->activate_toolbar_item(bec::NodeId(0), item);
  return true;
}

//------------------------------------------------------------------------------
void DbSqlEditorView::init()
{
  _xml->get_widget("editor_note", _editor_note);
  _editor_note->signal_switch_page().connect(sigc::mem_fun(this, &DbSqlEditorView::editor_page_switched));
    
  _be->sql_editor_new_ui.connect(sigc::bind_return(sigc::hide(sigc::mem_fun(this, &DbSqlEditorView::add_editor_tab)), 0));

  _be->wbsql()->get_wbui()->get_command_ui()->add_builtin_command("query.explain",
        sigc::mem_fun(this, &DbSqlEditorView::explain_sql),
        sigc::mem_fun(this, &DbSqlEditorView::validate_explain_sql));

  _xml->get_widget("output_text", _output_text);
  {
    Pango::FontDescription font(_output_text->get_style()->get_font());
    font.set_family("Bitstream Vera Sans Mono");
    _output_text->modify_font(font);
  }
  Gtk::Button *clear_button;
  _xml->get_widget("output_clear_button", clear_button);
  clear_button->signal_clicked().connect(sigc::mem_fun(this, &DbSqlEditorView::clear_output));

  // tab control
  _xml->get_widget("db_sql_editor_tab_pages", _tab_pages);
  _xml->get_widget("log_page", _log_page);

  // this is the lower tabview
  _tab_pages->signal_switch_page().connect(sigc::mem_fun(this, &DbSqlEditorView::lower_page_switched));
  
  // toolbar buttons

  // live physical overview 
  {
    Glib::RefPtr<Gtk::AccelGroup> _accel_group= Gtk::AccelGroup::create();//!
    _live_physical_overview= new OverviewPanel(_be->wbsql()->get_wbui(), _be->live_physical_overview(), true);

    Gtk::HBox *live_physical_overview_toolbar = 0;
    _xml->get_widget("live_physical_overview_toolbar", live_physical_overview_toolbar);
    ToolbarManager::rebuild_toolbar(live_physical_overview_toolbar,
                                    _be->live_physical_overview()->get_toolbar_items(bec::NodeId(0)),
                                    sigc::slot<Gtk::Widget*,bec::ToolbarItem>(),
                                    sigc::mem_fun(this, &DbSqlEditorView::toolbar_action)
                                   );

    Gtk::Viewport *live_physical_overview_placeholder = 0;
    _xml->get_widget("live_physical_overview_placeholder", live_physical_overview_placeholder);
    live_physical_overview_placeholder->add(*_live_physical_overview);
    _live_physical_overview->set_data("FormViewBase", _live_physical_overview);
    _live_physical_overview->set_data("uiform", dynamic_cast<bec::UIForm*>(get_form())); //!_live_physical_overview->get_form()
    _live_physical_overview->show();
    _live_physical_overview->rebuild_all();
    //_live_physical_overview->select_default_group_page();

    if (!_be->is_physical_overview_enabled())
    {
      Gtk::Container *live_physical_overview_page;
      _xml->get_widget("live_physical_overview_page", live_physical_overview_page);
      int tab_page_index= _tab_pages->page_num(*live_physical_overview_page);
      _tab_pages->remove_page(tab_page_index);
    }
  }

  // log panel
  //WIDGET_DEF (_xml, Container, log_placeholder) // auto-replaced
  Gtk::Container *log_placeholder = 0;
  _xml->get_widget("log_placeholder", log_placeholder);
  _log_view= GridView::create(_be->log());
  log_placeholder->add(*_log_view);
  _log_view->row_numbers_visible(false);
  _be->log()->refresh_ui_cb=
    sigc::bind<bool>(sigc::mem_fun(_log_view, &GridView::refresh), false);
  _log_view->refresh(true);

  //Gtk::HBox *output_toolbar = _xml->get_widget("output_toolbar", output_toolbar);
  //ToolbarManager::rebuild_toolbar(output_toolbar,
  //                                _be->log()->get_toolbar_items(bec::NodeId(0)),
  //                                sigc::mem_fun(this, &DbSqlEditorView::output_toolbar_action)
  //                               );


  _xml->get_widget("log_event_detail_action_text_view", _log_event_detail_action_text_view);
  _xml->get_widget("log_event_detail_message_text_view", _log_event_detail_message_text_view);

  _xml->get_widget("output_note", _output_note);
  
  Gtk::RadioButton *radio;
  _xml->get_widget("output_radio1", radio);
  radio->signal_toggled().connect(sigc::bind(sigc::mem_fun(_output_note, &Gtk::Notebook::set_current_page), 0));
  _xml->get_widget("output_radio2", radio);
  _output_text_radio= radio;
  radio->signal_toggled().connect(sigc::bind(sigc::mem_fun(_output_note, &Gtk::Notebook::set_current_page), 1));
  _xml->get_widget("output_radio3", radio);
  radio->signal_toggled().connect(sigc::bind(sigc::mem_fun(_output_note, &Gtk::Notebook::set_current_page), 2));
  
  _log_view->get_selection()->signal_changed().connect(sigc::mem_fun(this, &DbSqlEditorView::on_log_view_selection_changed));
  on_log_view_selection_changed();

  // history panels
  Gtk::Container *history_entries_placeholder = 0;
  _xml->get_widget("history_entries_placeholder", history_entries_placeholder);
  Gtk::Container *history_details_placeholder = 0;
  _xml->get_widget("history_details_placeholder", history_details_placeholder);
  _history_entries_view= GridView::create(_be->history()->entries_model());
  _history_details_view= GridView::create(_be->history()->details_model());
  history_entries_placeholder->add(*_history_entries_view);
  history_details_placeholder->add(*_history_details_view);
  _be->history()->entries_model()->refresh_ui_cb=
    sigc::mem_fun(this, &DbSqlEditorView::on_history_entries_refresh);
  _be->history()->details_model()->refresh_ui_cb=
    sigc::bind<bool>(sigc::mem_fun(_history_details_view, &GridView::refresh), false);
  _history_entries_view->refresh(true);
  _history_details_view->refresh(true);

  //Gtk::HBox *history_toolbar = _xml->get_widget("history_toolbar", history_toolbar);
  //ToolbarManager::rebuild_toolbar(history_toolbar,
  //                                _be->history()->get_toolbar_items(bec::NodeId(0)),
  //                                sigc::mem_fun(this, &DbSqlEditorView::output_toolbar_action)
  //                               );


  _history_entries_view->signal_event().connect(sigc::mem_fun(this, &DbSqlEditorView::on_history_entries_view_event));
  _history_entries_view->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
  _on_history_entries_selection_changed_cb= sigc::mem_fun(this, &DbSqlEditorView::on_history_entries_selection_changed);
  _history_entries_view->get_selection()->signal_changed().connect(_on_history_entries_selection_changed_cb);
  _history_details_view->signal_event().connect(sigc::mem_fun(this, &DbSqlEditorView::on_history_details_view_event));
  _xml->get_widget("history_details_popup_menu", _history_details_popup_menu);
  {
    Gtk::MenuItem *w = 0;
    _xml->get_widget("append_to_sql_script_menuitem", w);
    w->signal_activate().connect(sigc::mem_fun(this, &DbSqlEditorView::on_append_to_sql_script_menuitem_activate));
  }
  {
    Gtk::MenuItem *w = 0;
    _xml->get_widget("replace_sql_script_menuitem", w);
    w->signal_activate().connect(sigc::mem_fun(this, &DbSqlEditorView::on_replace_sql_script_menuitem_activate));
  }
  
  // realize pre-existing editors
  for (int i = 0; i < _be->sql_editor_count(); i++)
  {
    _be->active_sql_editor_index(i);
    add_editor_tab();
  }
}

bool DbSqlEditorView::on_close()
{
  if (_be->can_close())
  {
    _closing = true;
    return true;
  }
  return false;
}

void DbSqlEditorView::editor_page_switched(GtkNotebookPage *page, guint index)
{
  _be->active_sql_editor_index(index);
  
  long long rskey = _be->active_recordset_for_sql_editor(index);
  if (rskey >= 0)
  {
    // activate the recordset for this editor
    for (int i= 0, rcount = _tab_pages->get_n_pages(); i < rcount; i++)
    {
      RecordsetView *rsview = dynamic_cast<RecordsetView*>(_tab_pages->get_nth_page(i));
      if (rsview)
      {
        if (rsview->model()->key() == rskey)
        {
          _tab_pages->set_current_page(i);
          break;
        }
      }
    }
  }

 // not sure the following line is needed --alfredo
 //!!! _be->request_refresh_of_menu_and_toolbar();
}

void DbSqlEditorView::lower_page_switched(GtkNotebookPage *page, guint index)
{
  if (_closing) return;
  RecordsetView *rs = active_recordset();
  if (rs)
  {
    _be->active_recordset(rs->model());

    int editor = _be->sql_editor_index_for_recordset(rs->model()->key());
    if (editor >= 0)
    {
      for (std::vector<boost::shared_ptr<SqlEditorFE> >::iterator iter = _sql_editors.begin();
        iter != _sql_editors.end(); ++iter)
      {
        if ((*iter)->be() == _be->sql_editor(editor))
        {
          _editor_note->set_current_page(_editor_note->page_num((*iter)->widget()));
          break;
        }
      }
    }
  }
  else
    _be->active_recordset(Recordset::Ref());

}

void DbSqlEditorView::close_editor_tab(boost::shared_ptr<SqlEditorFE> editor)
{
  int idx;
  
  for (idx = 0; idx < _be->sql_editor_count(); idx++)
  {
    if (_be->sql_editor(idx) == editor->be())
      break;
  }
  if (idx == _be->sql_editor_count())
    return;

  if (!_be->sql_editor_will_close(idx))
    return;

  _sql_editors.erase(_sql_editors.begin() + _editor_note->page_num(editor->widget()));
  _editor_note->remove_page(editor->widget());
  _be->remove_sql_editor(idx);
  
  _be->active_sql_editor_index(_editor_note->get_current_page());

  // auto-add a new buffer if all of them are closed
  if (_be->sql_editor_count() == 0)
    _be->new_sql_scratch_area();
}

void DbSqlEditorView::add_editor_tab()
{
  // sql editor
  boost::shared_ptr<SqlEditorFE> editor(new SqlEditorFE());
  
  editor->be(_be->sql_editor());
  editor->widget().set_size_request(-1, 1);

  editor->set_font(_be->wbsql()->get_wbui()->get_wb()->get_wb_options().get_string("workbench.general.Editor:Font"));
  
  editor->set_text(editor->be()->sql());

  Gtk::Label *label;
  _editor_note->append_page(editor->widget(), *create_closeable_tab(_be->sql_editor_caption(_be->active_sql_editor_index()),
                                                                    sigc::bind(sigc::mem_fun(this, &DbSqlEditorView::close_editor_tab),
                                                                               editor),
                                                                   &label));
  editor->widget().set_data("tab_label", label);
  
  _editor_note->set_current_page(_editor_note->get_n_pages()-1);
  editor->widget().grab_focus();

  editor->background_action_cb(sigc::bind(sigc::mem_fun(editor.get(), &SqlEditorFE::check_sql), false));
  
  editor->widget().signal_key_press_event().connect(sigc::mem_fun(*this, &DbSqlEditorView::on_sql_editor_key_press_event));

  editor->signal_selection_changed().connect(sigc::mem_fun(this, &DbSqlEditorView::on_sql_editor_selection_change));

  _sql_editors.push_back(editor);

  update_resultsets();
}


SqlEditorFE *DbSqlEditorView::active_editor()
{
  int page = _editor_note->get_current_page();
  if (page >= 0 && page < (int)_sql_editors.size())
    return _sql_editors[page].get();

  return 0;
}


RecordsetView *DbSqlEditorView::active_recordset()
{
  int page = _tab_pages->get_current_page();
  if (page >= 0)
    return dynamic_cast<RecordsetView*>(_tab_pages->get_nth_page(page));

  return 0;
}


void DbSqlEditorView::execute_sql_script(bool current_statement_only)
{
  std::string sql;
  if (current_statement_only)
  {
    active_editor()->check_sql(true);
    sql= active_editor()->current_sql_statement();
  }
  else
  {
    sql= active_editor()->get_selected_text();
    if (sql.empty())
      sql= active_editor()->get_text();
  }

  if (sql.empty())
    return;

 /*
  // close unpinned resultsets
  if (!_be->recordsets_are_pinned_by_default())
  {
    Recordset2View left_recordsets;
    for (int n = 0, count = _be->recordset_count(); n < count; ++n)
    {
      Recordset::Ref rs= _be->recordset(n);
      if (rs->pinned())
        left_recordsets[rs];
      else
        close_recordset(rs->key());
    }
    std::list<long long> recordsets_to_close;
    BOOST_FOREACH (Recordset2View::value_type &i, _recordset2view)
      if (left_recordsets.end() == left_recordsets.find(i.first))
        recordsets_to_close.push_back(i.first->key());
    BOOST_FOREACH (long long &key, recordsets_to_close)
      close_recordset(key);
  }*/

  _tab_pages->set_current_page(_tab_pages->page_num(*_log_page));

  _be->exec_sql(sql, active_editor()->be(), false, current_statement_only);
}


void DbSqlEditorView::update_resultsets()
{
  int non_rs_tabs = 0;
  std::list<Gtk::Widget*> leftover_tabs;
  for (int i = 0; i < _tab_pages->get_n_pages(); i++)
  {
    RecordsetView *rsview = dynamic_cast<RecordsetView*>(_tab_pages->get_nth_page(i));
    if (rsview && _view2recordset.find(rsview) != _view2recordset.end())
      leftover_tabs.push_back(rsview);
    else
      non_rs_tabs++;
  }

  for (int n= 0, rs_count= _be->recordset_count(); n < rs_count; ++n)
  {
    Recordset::Ref rs= _be->recordset(n);

    if (_recordset2view.end() == _recordset2view.find(rs))
    {
      Gtk::Label *label;
      Gtk::Widget *tab = create_closeable_tab(rs->caption(),
                                            sigc::hide_return(sigc::bind(sigc::mem_fun(this, &DbSqlEditorView::close_recordset), rs->key(), true)),
                                            &label);
      RecordsetView *rs_view= RecordsetView::create(rs, NULL);
      _tab_pages->insert_page(*rs_view, non_rs_tabs + n);
      _tab_pages->set_tab_label(*rs_view, *tab);

      rs_view->set_data("tab_label", label);

      _recordset2view[rs]= rs_view;
      _view2recordset[rs_view]= rs;

      rs_view->refresh();

      _tab_pages->set_current_page(non_rs_tabs+n);
    }
    else
    {
      std::list<Gtk::Widget*>::iterator iter = std::find(leftover_tabs.begin(), leftover_tabs.end(), _recordset2view[rs]);
      if (iter != leftover_tabs.end())
        leftover_tabs.erase(iter);
    }
  }

  // remove tabs that are gone
  for (std::list<Gtk::Widget*>::iterator iter = leftover_tabs.begin(); iter != leftover_tabs.end(); ++iter)
  {
    View2Recordset::iterator rsiter = _view2recordset.find(dynamic_cast<RecordsetView*>(*iter));
    if (rsiter != _view2recordset.end())
    {
      _recordset2view.erase(_recordset2view.find(rsiter->second));
      _view2recordset.erase(rsiter);
    }
    _tab_pages->remove_page(**iter);
  }
}


int DbSqlEditorView::after_exec_sql_finished()
{
  update_resultsets();
  //XXX select log tab if there are errors
 // _tab_pages->set_current_page((new_tab_pages_count) ? _tab_pages->get_n_pages()-1 : _tab_pages->page_num(*_log_page));

  refresh_log_view();
  on_history_entries_refresh(); //!_history_entries_view->refresh(false);
  _history_details_view->refresh(false);

  return 0;
}

int DbSqlEditorView::on_exec_sql_progress(float progress, const std::string &message)
{
  refresh_log_view();
  return 0;
}

void DbSqlEditorView::refresh_log_view()
{
  _log_view->refresh(false);

  int log_row_count= _log_view->row_count();
  if (log_row_count > 0)
  {
    Gtk::TreePath path;
    path.push_back(log_row_count-1);
    _log_view->scroll_to_row(path);
    _log_view->set_cursor(path);
  }
}

void DbSqlEditorView::recordset_list_changed(Recordset::Ref rset, bool added)
{
  _dispatch_rset_update.emit();
}

int DbSqlEditorView::close_recordset(long long key, bool confirm)
{
  Recordset2View::iterator i;
  for (i= _recordset2view.begin(); i != _recordset2view.end(); ++i)
  {
    if (i->first->key() == key)
      break;
  }
  if (_recordset2view.end() == i)
    return 0;

  if (confirm && i->second->has_changes())
  {
    mforms::Utilities::show_warning(_("Close Recordset"),
                                    _("The recordset has uncommited changes.\n"
                                      "Please Apply or Revert them before closing."),
                                    _("OK"));
    return false;
  }
  _tab_pages->remove_page(_tab_pages->page_num(*i->second));
  _view2recordset.erase(_view2recordset.find(i->second));
  _recordset2view.erase(i);

  return 0;
}

void DbSqlEditorView::on_history_entries_selection_changed()
{
  int row= _history_entries_view->current_row();
  if (-1 < row)
    _be->history()->current_entry(row);
}

bool DbSqlEditorView::on_history_details_view_event(GdkEvent *event)
{
  bool processed= false;

  if ((GDK_BUTTON_PRESS == event->type) && (3 == event->button.button))
  {
    _history_details_popup_menu->popup(event->button.button, event->button.time);
    processed= true;
  }

  return processed;
}

bool DbSqlEditorView::on_history_entries_view_event(GdkEvent *event)
{
  bool processed= false;

  if ((GDK_BUTTON_PRESS == event->type) && (3 == event->button.button))
  {
    std::vector<bec::NodeId> nodes;
    std::list<Gtk::TreePath> paths;
    
    paths= _history_entries_view->get_selection()->get_selected_rows();
    for (std::list<Gtk::TreePath>::const_iterator iter= paths.begin(); iter != paths.end(); ++iter)
      nodes.push_back(iter->front());

    bec::MenuItemList items(_be->history()->entries_model()->get_popup_items_for_nodes(nodes));
    
    run_popup_menu(items, event->button.time, 
                   sigc::hide_return(sigc::bind(sigc::mem_fun(*_be->history()->entries_model(), &DbSqlEditorHistory::EntriesModel::activate_popup_item_for_nodes), nodes)));

    processed= true;
  }

  return processed;

}

void DbSqlEditorView::on_append_to_sql_script_menuitem_activate()
{
  load_selected_history_items(false);
}

void DbSqlEditorView::on_replace_sql_script_menuitem_activate()
{
  load_selected_history_items(true);
}

void DbSqlEditorView::load_selected_history_items(bool overwrite)
{
  GridView::SelectedNodes nodes;
  _history_details_view->get_selected_nodes(nodes);
  std::list<int> sel_indexes;
  BOOST_FOREACH (GridView::SelectedNodes::value_type &i, nodes)
    sel_indexes.push_back(i.first);

  if (sel_indexes.empty() || _be->history()->current_entry() < 0)
    return;
  
  std::string sql= _be->restore_sql_from_history(_be->history()->current_entry(), sel_indexes);

  if (overwrite)
    active_editor()->set_text(sql);
  else
    active_editor()->set_text(active_editor()->get_text() + sql);
}

int DbSqlEditorView::on_history_entries_refresh()
{
  _on_history_entries_selection_changed_cb.disconnect();
  _history_entries_view->refresh(false);

  int row= _be->history()->current_entry();
  if (row >= 0)
  {
    Gtk::TreePath path;
    path.push_back(row);
    //_history_entries_view->get_selection()->select(path); //! crash here
  }
  _history_entries_view->get_selection()->signal_changed().connect(_on_history_entries_selection_changed_cb);

  return 0;
}

RecordsetView *DbSqlEditorView::get_active_recordset()
{
  int page= _tab_pages->get_current_page();
  
  return dynamic_cast<RecordsetView*>(_tab_pages->get_nth_page(page));
}

bool DbSqlEditorView::on_sql_editor_key_press_event(GdkEventKey *event)
{
  bool processed= false;

  if ((GDK_KEY_PRESS == event->type) && (event->state & GDK_CONTROL_MASK))
  {
    switch (event->keyval)
    {
    //case GDK_Enter:
    case GDK_KP_Enter:
      execute_sql_script(false);
      processed= true;
      break;
    }
  }

  return processed;
}

void DbSqlEditorView::on_sql_editor_selection_change()
{
  (*_be->wbsql()->get_wbui()->get_command_ui()->signal_validate_edit_menu_items())();
}

int DbSqlEditorView::on_sql_editor_text_insert(const std::string &text)
{
  active_editor()->insert_text(text);
  return 0;
}

bool DbSqlEditorView::validate_explain_sql()
{
  return true;
}

void DbSqlEditorView::explain_sql()
{
  // active_editor()->set_selected_range();
  _be->explain_sql(false);
}

void DbSqlEditorView::on_log_view_selection_changed()
{
  std::string log_event_header_text= "";
  std::string log_event_action_text= "";
  std::string log_event_message_text= "";
  Glib::ListHandle<Gtk::TreeModel::Path, Gtk::TreePath_Traits> selected_rows= _log_view->get_selection()->get_selected_rows();
  if (!selected_rows.empty())
  {
    Gtk::TreeModel::Path node_path= *selected_rows.begin();
    int log_event_no= node_path.back();
    int log_event_type_code= 0;
    std::string log_event_time;
    _be->get_log_event_details(log_event_no, log_event_type_code, log_event_time, log_event_action_text, log_event_message_text);
  }
  _log_event_detail_action_text_view->get_buffer()->set_text(log_event_action_text);
  _log_event_detail_message_text_view->get_buffer()->set_text(log_event_message_text);
}


void DbSqlEditorView::toggle_sidebar()
{
  Gtk::Widget *sidebar;
  
  _xml->get_widget("sidebar_placeholder", sidebar);
  
  if (sidebar->is_visible())
    sidebar->hide();
  else
    sidebar->show();
}


void DbSqlEditorView::editor_caption_changed()
{
  Gtk::Label *label = reinterpret_cast<Gtk::Label*>(active_editor()->widget().get_data("tab_label"));
  label->set_text(_be->sql_editor_caption());
}


void DbSqlEditorView::recordset_caption_changed()
{
  if (active_recordset())
  {
    Gtk::Label *label = reinterpret_cast<Gtk::Label*>(active_recordset()->get_data("tab_label"));
    if (label)
      label->set_text(active_recordset()->model()->caption());
  }
}


void DbSqlEditorView::find_text(const std::string &text)
{
  active_editor()->find_text(text);
}


void DbSqlEditorView::clear_output()
{
  switch (_output_note->get_current_page())
  {
  case 0:
    _be->log()->reset();
    break;
  case 1:
    _output_text->get_buffer()->set_text("");
    break;
  case 2:
    _be->history()->entries_model()->delete_all_entries();
    break;
  }
}


void DbSqlEditorView::output_text(const std::string &text, bool bring_to_front)
{
  if (bring_to_front)
  {
    _tab_pages->set_current_page(_tab_pages->page_num(*_log_page));
    _output_note->set_current_page(1);
    _output_text_radio->clicked();
  }
  
  Glib::RefPtr<Gtk::TextBuffer> buf = _output_text->get_buffer();
  buf->insert(buf->end(), text);

  // scroll to end
  Gtk::TextIter it = buf->end();
  _output_text->scroll_to(it, 0.3);
}
