/* 
 * Copyright (c) 2010, 2011 Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#import "MSQLEditorController.h"

#import "MScintillaView.h"
#import "MCPPUtilities.h"
#import "sql_editor_be.h"
#import "WBPluginEditorBase.h"

using namespace Scintilla;

//--------------------------------------------------------------------------------------------------

@implementation MSQLEditorController

- (void) setString: (NSString*) string
{
#if 0 // this doesn't seem to be right
  // First set the current eol policy (which might differ from the platform default, by user choice).
  std::string eol= mBackEnd->eol();
  int sci_eol= SC_EOL_LF;
  if ("\n" == eol)
    sci_eol= SC_EOL_CRLF;
  else
    if ("\r" == eol)
      sci_eol= SC_EOL_CR;
    else
      if ("\r\n" == eol)
        sci_eol= SC_EOL_CRLF;
  [mScintillaView setGeneralProperty: SCI_SETEOLMODE value: sci_eol];
#endif
  [mScintillaView setString: string];
}

//--------------------------------------------------------------------------------------------------

- (NSString*) string
{
  return [mScintillaView string];
}

- (NSString*) stringOrSelection
{
  if ([mScintillaView selectedRange].length > 0)
    return [mScintillaView selectedString];
  else
    return [mScintillaView string];
}

//--------------------------------------------------------------------------------------------------

- (void)insertText: (NSString*) text
{
  [mScintillaView insertText: text];

  // Make sure the new text follows our eol policy.
  std::string eol= mBackEnd->eol();
  long sci_eol= SC_EOL_LF;
  if ("\n" == eol)
    sci_eol= SC_EOL_CRLF;
  else
    if ("\r" == eol)
      sci_eol= SC_EOL_CR;
    else
      if ("\r\n" == eol)
        sci_eol= SC_EOL_CRLF;
  [mScintillaView setGeneralProperty: SCI_CONVERTEOLS value: sci_eol];
}  

static int replace_selected_text(const std::string &text, MSQLEditorController *self)
{
  NSRange selection = [self->mScintillaView selectedRange];
  
  [self->mScintillaView insertText: [NSString stringWithCPPString: text]];

  [self->mScintillaView setGeneralProperty: SCI_SETSELECTIONSTART
                     value: selection.location];
  [self->mScintillaView setGeneralProperty: SCI_SETSELECTIONEND
                     value: selection.location + text.length()];
  
  return 0;
}

//--------------------------------------------------------------------------------------------------

/**
 * Initialize scintilla editor (styles, colors, markers etc.) for MySQL.
 */
- (void) setupEditor
{  
  [[NSNotificationCenter defaultCenter] addObserver: self
                                           selector: @selector(textDidChange:)
                                               name: NSTextDidChangeNotification
                                             object: mScintillaView];
  [[NSNotificationCenter defaultCenter] addObserver: self
                                           selector: @selector(uiWasUpdated:)
                                               name: SCIUpdateUINotification
                                             object: mScintillaView];
  [WBPluginEditorBase setupCodeEditor: mScintillaView backend: mBackEnd withStatus: YES];  
  [mScintillaView setStatusText: @""/*"No errors found"*/];
  
  mBackEnd->replace_selected_text_slot= boost::bind(replace_selected_text, _1, self);
  
  std::string font = grt::StringRef::cast_from(mBackEnd->grtm()->get_app_option("workbench.general.Editor:Font"));
  
  std::string name;
  int size;
  bool bold, italic;
  
  if (base::parse_font_description(font, name, size, bold, italic))
    [mScintillaView setFontName:[NSString stringWithUTF8String:name.c_str()]
                           size:size 
                           bold:bold 
                         italic:italic];
  
  [mScintillaView setGeneralProperty: SCI_USEPOPUP value: 0];
  // CR in mac is from ancient, before OSX times
  //[mScintillaView setGeneralProperty: SCI_SETEOLMODE value: SC_EOL_CR];
  [mScintillaView setGeneralProperty: SCI_SETEOLMODE value: SC_EOL_LF];
}

//--------------------------------------------------------------------------------------------------

- (void)setView:(MScintillaView*)view
{
  mScintillaView = view;
}

//--------------------------------------------------------------------------------------------------

- (ScintillaView*)view
{
  return mScintillaView;
}

//--------------------------------------------------------------------------------------------------

static int process_sql_statement_border(int begin_lineno, int begin_line_pos, int end_lineno, int end_line_pos, 
                                        MSQLEditorController *self);
static int process_sql_error(const int err_tok_lineno, const int err_tok_line_pos, const int err_tok_len, 
                             const std::string &err_msg, MSQLEditorController *self);
static void process_sql_done(MSQLEditorController *self);
static void callCheckSql(MSQLEditorController *self);
static int insert_text(const std::string &text, MSQLEditorController *self);
static void change_selection(int, int, MSQLEditorController *self);
static void change_cursor_pos(int, MSQLEditorController *self);

//--------------------------------------------------------------------------------------------------

/**
 * Called when the UI in the SQL editor was updated (e.g. caret or selection change).
 */
- (void) uiWasUpdated: (NSNotification*) aNotification
{
  NSRange range= [[mScintillaView content] selectedRange];
  int start= range.location;
  int end= range.location + range.length;
  mBackEnd->set_selected_range(start, end);
  
  int currentPosition = [[[aNotification userInfo] valueForKey: @"currentPosition"] intValue];
  mBackEnd->set_cursor_pos(currentPosition);
};

//--------------------------------------------------------------------------------------------------

/**
 * Called when text was added or removed in the editor.
 */
- (void) textDidChange: (NSNotification*) aNotification
{  
  // Stop the timer in case it is about to trigger.
  [mSqlCheckTimer invalidate];
  
  mBackEnd->sql([[mScintillaView string] UTF8String]);
  
  // Set up a new timer.
  mSqlCheckTimer= [NSTimer scheduledTimerWithTimeInterval: 0.5
                                                   target: self
                                                 selector: @selector(checkSql:)
                                                 userInfo: nil
                                                  repeats: NO];
};

//--------------------------------------------------------------------------------------------------

- (void) setEditorBackEnd:(const Sql_editor::Ref&) be
{
  mBackEnd= be;
  [self setupEditor];
  
  mBackEnd->report_sql_statement_border= boost::bind(process_sql_statement_border, _1, _2, _3, _4, self);
  mBackEnd->sql_parser_err_cb(boost::bind(process_sql_error, _1, _2, _3, _4, self));
  mBackEnd->insert_text_slot= boost::bind(insert_text, _1, self);
  mBackEnd->sql_check_finish_signal.connect(boost::bind(process_sql_done, self));
  mBackEnd->change_cursor_pos_slot= boost::bind(change_cursor_pos, _1, self);
  mBackEnd->change_selected_range_slot= boost::bind(change_selection, _1, _2, self);
  
  [mScintillaView setString: [NSString stringWithCPPString: mBackEnd->sql()]];
}

//--------------------------------------------------------------------------------------------------

- (Sql_editor::Ref)backEnd
{
  return mBackEnd;
}

//--------------------------------------------------------------------------------------------------

/**
 * Remove all markers we set for previous sql errors.
 */
- (void) resetSqlCheckState
{
  [mScintillaView setStatusText: @""/*"No errors found"*/];
  
  int length= [mScintillaView getGeneralProperty: SCI_GETLENGTH parameter: 0];
  
  [mScintillaView setGeneralProperty: SCI_SETINDICATORCURRENT parameter: 0 value: 0];
  [mScintillaView setGeneralProperty: SCI_INDICATORCLEARRANGE parameter: 0 value: length];
  
  [mScintillaView setGeneralProperty: SCI_MARKERDELETEALL parameter: -1 value: 0];
}

//--------------------------------------------------------------------------------------------------

- (void)checkSqlInBackground:(BOOL)flag
{
  // Block drawing of the editor to avoid flickering when replacing markers.
  // However, this only works if we do synchronous sql syntax checks.
  [mScintillaView suspendDrawing: YES];
  [self resetSqlCheckState];
  
  NSString *text = [mScintillaView string];
  if (text)
  {
    mErrorCount = 0;
    mBackEnd->check_sql(YES);
  }
  [mScintillaView suspendDrawing: NO];
}

//--------------------------------------------------------------------------------------------------

/**
 * Triggered once a change in the editor content happend and there was a pause of at least 500 ms.
 */
- (void) checkSql:(NSTimer*) timer
{
  mSqlCheckTimer = nil;
  
  [self checkSqlInBackground: YES];
}

//--------------------------------------------------------------------------------------------------

/**
 * Triggered once a change in the editor content happend and there was a pause of at least 500 ms.
 */
- (NSString *) currentSqlStatement
{
  // if there is a pending checkSql, we cancel it and do it immediately
  if (mSqlCheckTimer)
  {
    [mSqlCheckTimer invalidate];
    mSqlCheckTimer = nil;
    [self checkSqlInBackground: NO];
  }
  
  int current_pos= [mScintillaView getGeneralProperty: SCI_GETCURRENTPOS parameter: 0];
  int current_line= [mScintillaView getGeneralProperty: SCI_LINEFROMPOSITION parameter: current_pos];
  int current_line_beginning_pos= [mScintillaView getGeneralProperty: SCI_POSITIONFROMLINE parameter: current_line];
  int current_line_pos= current_pos - current_line_beginning_pos;
  
  Sql_editor::SqlStatementBorder sql_statement_border= mBackEnd->get_sql_statement_border_by_line_pos(current_line+1, current_line_pos);
  if (sql_statement_border.begin_lineno == -1)
    return @"";
  CharacterRange cr;
  cr.cpMin= [mScintillaView getGeneralProperty: SCI_POSITIONFROMLINE parameter: sql_statement_border.begin_lineno - 1];
  cr.cpMin+= sql_statement_border.begin_line_pos;
  cr.cpMax= [mScintillaView getGeneralProperty: SCI_POSITIONFROMLINE parameter: sql_statement_border.end_lineno - 1];
  cr.cpMax+= sql_statement_border.end_line_pos;
  int doc_length= [mScintillaView getGeneralProperty: SCI_GETLENGTH parameter: 0];
  if (cr.cpMax > doc_length)
    cr.cpMax= doc_length;
  
  TextRange tr;
  tr.chrg= cr;
  tr.lpstrText= new char[cr.cpMax - cr.cpMin + 1];
  [mScintillaView getGeneralProperty: SCI_GETTEXTRANGE ref: &tr];
  NSString *sql= [NSString stringWithUTF8String: tr.lpstrText];
  delete [] tr.lpstrText;
  
  return sql;
}

//--------------------------------------------------------------------------------------------------

/**
 * Callback from the backend telling us about sql statement range.
 */
static int process_sql_statement_border(int begin_lineno, int begin_line_pos, int end_lineno, int end_line_pos, 
                                        MSQLEditorController *self)
{
  [self->mScintillaView setGeneralProperty: SCI_MARKERADD parameter: begin_lineno - 1 value: 0];
  return 0;
}

//--------------------------------------------------------------------------------------------------

/**
 * Callback from the backend telling us about sql errors it founds.
 */
static int process_sql_error(const int err_tok_lineno, const int err_tok_line_pos, const int err_tok_len,
                             const std::string &err_msg, MSQLEditorController *self)
{
  int line_start_pos= [self->mScintillaView getGeneralProperty: SCI_POSITIONFROMLINE parameter: err_tok_lineno - 1];
  
  [self->mScintillaView setGeneralProperty: SCI_SETINDICATORCURRENT parameter: 0 value: 0];
  [self->mScintillaView setGeneralProperty: SCI_INDICATORFILLRANGE parameter: line_start_pos + err_tok_line_pos value: err_tok_len];
  
  [self->mScintillaView setGeneralProperty: SCI_MARKERADD parameter: err_tok_lineno - 1 value: 1];
  
  ++self->mErrorCount;
  [self->mScintillaView setStatusText: [NSString stringWithFormat: @"%d error(s) found.", self->mErrorCount]];
  
  return 0;
}

//--------------------------------------------------------------------------------------------------

static void process_sql_done(MSQLEditorController *self)
{
}

//--------------------------------------------------------------------------------------------------

static int insert_text(const std::string &text, MSQLEditorController *self)
{
  NSString* sql_text = [NSString stringWithCPPString: text];
  if (sql_text == nil)
    sql_text= [NSString stringWithCString: text.c_str() encoding: NSISOLatin1StringEncoding];
  [self insertText: sql_text];
  return 0;
}

//--------------------------------------------------------------------------------------------------

static void change_selection(int start, int end, MSQLEditorController *self)
{
  [self->mScintillaView setGeneralProperty: SCI_SETSELECTIONSTART value: start];
  [self->mScintillaView setGeneralProperty: SCI_SETSELECTIONEND value: end];
}

//--------------------------------------------------------------------------------------------------

static void change_cursor_pos(int pos, MSQLEditorController *self)
{
  [self->mScintillaView setGeneralProperty: SCI_GOTOPOS value: pos];
}


@end
