/* 
 * Copyright (c) 2009, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#import "WBSQLQueryPanel.h"
#import "MResultsetViewer.h"
#import "GRTIconCache.h"
#import "MVerticalLayoutView.h"
#import "WBTabView.h"
#import "WBSplitView.h"
#import "MCPPUtilities.h"
#import "MContainerView.h"
#import "WBSplitViewUnbrokenizerDelegate.h"
#import "WBOverviewPanel.h"
#import "MScintillaView.h"
#import "WBMiniToolbar.h"
#import "GRTListDataSource.h"

#import "MSQLEditorController.h"

#include "sqlide/wb_sql_editor_snippets.h"
#include "mforms/../cocoa/MFView.h"

@interface ColorBallTextCell : NSTextFieldCell
{
}
@end

@implementation ColorBallTextCell

- (void)drawWithFrame:(NSRect)cellFrame inView:(NSView *)controlView 
{
  [[self backgroundColor] set];
  
  NSAttributedString *text = [self attributedStringValue];
  NSSize textSize = [text size];
  
  cellFrame.origin.y += (cellFrame.size.height - cellFrame.size.width) / 2;
  cellFrame.size.height = cellFrame.size.width;
  
  cellFrame = NSInsetRect(cellFrame, 2, 2);
  
  [[NSBezierPath bezierPathWithOvalInRect:cellFrame] fill];
  
  cellFrame.origin.y += (cellFrame.size.height - textSize.height) / 2;
  [text drawInRect:cellFrame];
}

@end


@interface ResultTabViewItem : NSTabViewItem
{
  MResultsetViewer *viewer;
}

- (void)setRSViewer: (MResultsetViewer*)viewer;
- (MResultsetViewer*)RSViewer;

@end


@implementation ResultTabViewItem

- (void)setRSViewer: (MResultsetViewer*)aViewer
{
  [viewer autorelease];
  viewer = [aViewer retain];
}

- (MResultsetViewer*)RSViewer
{
  return viewer;
}

- (void) dealloc
{
  [viewer release];
  [super dealloc];
}

@end




@implementation WBSQLQueryPanel

#pragma mark Table View support

- (NSInteger) numberOfRowsInTableView: (NSTableView*) tableView;
{
  if (tableView == mMessagesTable)
  {
    if (mBackEnd)
      return mBackEnd->log()->count();
  }
  else if (tableView == mHistoryTable)
  {
    if (mBackEnd)
      return mBackEnd->history()->entries_model()->count();
  }
  else if (tableView == mHistoryDetailsTable)
  {
    if (mBackEnd && [mHistoryTable selectedRow] >= 0)
      return mBackEnd->history()->details_model()->count();
  }

  return 0;
}



- (id) tableView: (NSTableView*) aTableView
objectValueForTableColumn: (NSTableColumn*) aTableColumn
             row: (NSInteger) rowIndex;
{
  if (aTableView == mMessagesTable)
  {
    std::string text;
    
    if ([[aTableColumn identifier] isEqual:@"0"])
    {
      bec::IconId icon_id= mBackEnd->log()->get_field_icon(rowIndex, 0, bec::Icon16);
      
      if (icon_id != 0)
        return [[GRTIconCache sharedIconCache] imageForIconId:icon_id];
      else
        return nil;
    }
    else
    {
      mBackEnd->log()->get_field(rowIndex, [[aTableColumn identifier] intValue], text);
      
      return [NSString stringWithUTF8String: text.c_str()];
    }
  }
  else if (aTableView == mHistoryTable)
  {
    std::string text;
    
    mBackEnd->history()->entries_model()->get_field(rowIndex, [[aTableColumn identifier] intValue], text);
    
    return [NSString stringWithUTF8String: text.c_str()];
  }
  else if (aTableView == mHistoryDetailsTable)
  {
    std::string text;
    
    mBackEnd->history()->details_model()->get_field(rowIndex, [[aTableColumn identifier] intValue], text);
    
    return [NSString stringWithUTF8String: text.c_str()];
  }

  return @"foo";
}


- (void)tableViewSelectionDidChange:(NSNotification *)aNotification
{
  NSTableView *sender= [aNotification object];
  
  if (sender == mHistoryTable)
  {
    if ([mHistoryTable selectedRow] >= 0)
      mBackEnd->history()->current_entry([mHistoryTable selectedRow]);
    [mHistoryDetailsTable reloadData];
  }
  else if (sender == mHistoryDetailsTable)
  {
  }
  else if (sender == mMessagesTable)
  {
    std::string text;
    mBackEnd->log()->get_field([mMessagesTable selectedRow], 3, text);
    [mOutputActionText setString: [NSString stringWithCPPString: text]];

    mBackEnd->log()->get_field([mMessagesTable selectedRow], 4, text);
    [mOutputResponseText setString: [NSString stringWithCPPString: text]];
  }
  else if (sender == snippetTable)
  {
    int row = [snippetTable selectedRow];
    if (row < 0)
      [mSnippetText setString: @""];
    else
    {
      std::string text;
      DbSqlEditorSnippets::get_instance()->get_field(row, DbSqlEditorSnippets::Script, text);
      [mSnippetText setString: [NSString stringWithCPPString: text]];
    }
  }  
}


- (CGFloat)tableView:(NSTableView *)tableView heightOfRow:(NSInteger)row
{
  if (tableView == mHistoryDetailsTable)
  {
    std::string text;
    mBackEnd->history()->details_model()->get_field(row, 0, text);
    size_t lines= 0;
    const char *ptr= text.c_str();
    
    do
    {
      ptr= strchr(ptr, '\n');
      lines++;
    } while (ptr++);

    return [tableView rowHeight] * lines;
  }
  else
    return [tableView rowHeight];
}


- (void)loadSelectedHistoryItems:(BOOL)append
{
  std::list<int> sel_indexes;
  NSIndexSet *iset = [mHistoryDetailsTable selectedRowIndexes];
  
  if ([iset count] > 0)
    for (int row = [iset firstIndex]; row <= [iset lastIndex]; row = [iset indexGreaterThanIndex: row])
      sel_indexes.push_back(row);

  if (sel_indexes.empty() || mBackEnd->history()->current_entry() < 0)
    return;
  
  std::string sql= mBackEnd->restore_sql_from_history(mBackEnd->history()->current_entry(), sel_indexes);
  
  if (append)
    [[self activeEditor] setString: [[[self activeEditor] string] stringByAppendingString: [NSString stringWithCPPString: sql]]];
  else
    [[self activeEditor] setString: [NSString stringWithCPPString: sql]];
}


- (bec::ListModel*)listModelForTableView:(NSTableView*)table
{
  if (table == mHistoryTable)
    return mBackEnd->history()->entries_model().get();
  return 0;
}

- (IBAction)activateHistoryEntry:(id)sender
{
  switch ([sender tag])
  {
    case 1:
      [self loadSelectedHistoryItems: YES];
      break;

    case 2:
      [self loadSelectedHistoryItems: NO];
      break;

    default:
      [self loadSelectedHistoryItems: YES];      
      break;
  }
}


- (void)activateHistoryDetailEntry:(id)sender
{
  std::string text, query;
  NSIndexSet *selection= [mHistoryDetailsTable selectedRowIndexes];
  
  for (NSUInteger row= [selection firstIndex]; row != NSNotFound; row= [selection indexGreaterThanIndex:row])
  {
    mBackEnd->history()->details_model()->get_field(row, 1, text);
    query.append("\n").append(text);
  }

  MSQLEditorController *editor = [self activeEditor];
  
  [editor setString: [[editor string] stringByAppendingString: [NSString stringWithCPPString: query]]];
}


- (void)updateResultsets
{
  int n, rs_count = mBackEnd->recordset_count();
  NSMutableArray *leftoverResults = [[[mLowerTabView tabViewItems] mutableCopy] autorelease];
  int non_rs_tabs = 0;
  
  for (NSTabViewItem *item in leftoverResults)
  {
    if ([item isKindOfClass: [ResultTabViewItem class]])
      break;
    non_rs_tabs++;
  }
  
  for (n = 0; n < rs_count; ++n)
  {
    Recordset::Ref rset(mBackEnd->recordset(n));
    
    NSInteger index = [mLowerTabView indexOfTabViewItemWithIdentifier: [NSString stringWithFormat: @"rset%li", rset->key()]];
    if (index == NSNotFound)
    {
      MResultsetViewer *rsview = [[[MResultsetViewer alloc] initWithRecordset: rset] autorelease];

      ResultTabViewItem *tabViewItem= 
        [[[ResultTabViewItem alloc] initWithIdentifier: [NSString stringWithFormat:@"rset%li", rset->key()]] autorelease];

      [tabViewItem setRSViewer: rsview];
      [tabViewItem setLabel: [NSString stringWithCPPString: rset->caption()]];
      [tabViewItem setView: [rsview view]];
      
      [mLowerTabView insertTabViewItem: tabViewItem atIndex: n+non_rs_tabs];
      [rsview refresh];
    }
    else
    {
      [leftoverResults removeObject: [mLowerTabView tabViewItemAtIndex: index]];
    }
  }

  // remove tabs that are not used anymore
  for (id tab in leftoverResults)
  {
    if ([tab isKindOfClass: [ResultTabViewItem class]])
      [mLowerTabView removeTabViewItem: tab];
  }
  
//  [mLowerTabView selectFirstTabViewItem:nil];
}


static int processTaskFinish(WBSQLQueryPanel *self)
{
  [self->mMessagesTable reloadData];
  [self updateResultsets];
  
  if (self->mBackEnd->exec_sql_error_count() > 0)
  {
    [self->mOutputTabView selectTabViewItemWithIdentifier: @"actions"];
    [self->mOutputSegmented selectSegmentWithTag: 0];
  }
  else
  {
    // re-select the top editor to force re-selection of its resultset tab
    [self->mUpperTabView selectTabViewItemWithIdentifier: [[self->mUpperTabView selectedTabViewItem] identifier]];
  }
  
  return 0;
}

static int processTaskProgress(float progress, const std::string &message, WBSQLQueryPanel *self)
{
  [self->mMessagesTable reloadData];
  [self->mMessagesTable scrollRowToVisible: [self->mMessagesTable numberOfRows]-1];
  
  return 0;
}


- (void)refreshTable:(NSTableView*)table
{
  [table reloadData];
  [[table delegate] tableViewSelectionDidChange:[NSNotification notificationWithName:NSTableViewSelectionDidChangeNotification
                                                                              object: table]];
  [table scrollRowToVisible: [table numberOfRows]-1];
}


static int reloadTable(NSTableView *table, WBSQLQueryPanel *self)
{
  if ([NSThread mainThread] == [NSThread currentThread])
  {
    [self refreshTable: table];
  }
  else
  {
    [self performSelectorOnMainThread: @selector(refreshTable:)
                           withObject: table
                        waitUntilDone: NO];
  }
  return 0;
}

- (void)removeRecordsetWithIdentifier:(id)identifier
{
  NSInteger index = [mLowerTabView indexOfTabViewItemWithIdentifier: identifier];
  if (index != NSNotFound)
  {
    id item = [mLowerTabView tabViewItemAtIndex: index];
    if (item)
      [mLowerTabView removeTabViewItem: item];
  }  
}


static void recordsetListChanged(Recordset::Ref rs, bool added, WBSQLQueryPanel *self)
{
  if (!added)
  {
    if (![NSThread isMainThread])
    {
      NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
      [self performSelectorOnMainThread: @selector(removeRecordsetWithIdentifier:) 
                             withObject: [NSString stringWithFormat:@"rset%li", rs->key()]
                          waitUntilDone: NO];
      [pool release];
    }
    else
      [self removeRecordsetWithIdentifier: [NSString stringWithFormat:@"rset%li", rs->key()]];
  }
  else
    [self performSelectorOnMainThread:@selector(updateResultsets) withObject:nil waitUntilDone:NO];
}

static void addTextToOutput(const std::string &text, bool bring_to_front, WBSQLQueryPanel *self)
{
  [self->mTextOutputLock lock];
  self->mTextOutputBuffer.append(text);
  [self->mTextOutputLock unlock];

  if ([NSThread isMainThread])
  {
    [self flushOutputBuffer];
    if (bring_to_front)
    {
      [self->mOutputSegmented selectSegmentWithTag: 1];
      [self->mOutputTabView selectTabViewItemWithIdentifier: @"text"];
      [self->mLowerTabView selectTabViewItemWithIdentifier: @"messages"];
    }
  }
  else
    [self performSelectorOnMainThread:@selector(flushOutputBuffer) withObject:nil waitUntilDone:NO];
}

#pragma mark User actions



#define QUERY_AREA_EXPANDED_MIN_HEIGHT (100)
#define QUERY_AREA_COLLAPSED_MIN_HEIGHT (0)

#define RESULTS_AREA_EXPANDED_MIN_HEIGHT (100)
#define RESULTS_AREA_COLLAPSED_MIN_HEIGHT (28)


- (void)executeQuery:(id)sender currentStatementOnly: (bool) currentStatementOnly
{
  std::string text;
  if (currentStatementOnly)
    text= [[[self activeEditor] currentSqlStatement] UTF8String];
  else
    text= [[[self activeEditor] stringOrSelection] UTF8String] ? : "";
  if (text.empty())
    return;
  
  [mLowerTabView selectTabViewItemWithIdentifier: @"messages"];
  
  mBackEnd->exec_sql(text, [[self activeEditor] backEnd], false, currentStatementOnly);  
}


- (void) tabView: (NSTabView*) tabView 
draggedHandleAtOffset: (NSPoint) offset
{
  if (tabView == mLowerTabView)
  {
    NSPoint pos= [mWorkView convertPointFromBase: [[NSApp currentEvent] locationInWindow]];

    float position= pos.y - offset.y - [mWorkView dividerThickness];
    [mWorkView setPosition: position ofDividerAtIndex: 0];
  }
}


- (void) tabViewDraggerClicked: (NSTabView*) tabView 
{
  if (mLastClick > 0 && [NSDate timeIntervalSinceReferenceDate] - mLastClick < 0.3)
  {  
    if (mQueryAreaOpen && mResultsAreaOpen)
    {
      mResultsAreaOpen= NO;
      [mWorkView setPosition: NSHeight([mWorkView frame]) - RESULTS_AREA_COLLAPSED_MIN_HEIGHT
        ofDividerAtIndex: 0];
    }
    else if (mQueryAreaOpen && !mResultsAreaOpen)
    {
      mResultsAreaOpen= YES;
      mQueryAreaOpen= NO;
      [mWorkView setPosition: 0
        ofDividerAtIndex: 0];
    }
    else
    {
      mResultsAreaOpen= YES;
      mQueryAreaOpen= YES;
      
      [mWorkView setPosition: 200
        ofDividerAtIndex: 0];
    }
    mLastClick= 0;
  }
  else
    mLastClick= [NSDate timeIntervalSinceReferenceDate];
}

- (void)addEditorTabWithBackEndIndex:(int)editor_index
{
  MSQLEditorController *editor = [[[MSQLEditorController alloc] init] autorelease];
  MScintillaView *view = [[[MScintillaView alloc] initWithFrame:NSMakeRect(0, 0, 100, 100)] autorelease];
  
  [editor setView: view];
  [editor setEditorBackEnd: mBackEnd->sql_editor(editor_index)];
  [view setEditorBackEnd: [editor backEnd]];
  
  NSTabViewItem *item = [[NSTabViewItem alloc] initWithIdentifier: editor];
  [item setView: view];
  [item setLabel: [NSString stringWithCPPString: mBackEnd->sql_editor_caption(editor_index)]];
  
  [mEditors addObject: editor];
  
  [mUpperTabView addTabViewItem:item];
  
  // update resultsets before selecting, so that the matching rset gets auto-selected
  [self updateResultsets];
  
  [mUpperTabView selectTabViewItemWithIdentifier: [item identifier]];
  
  [[mView window] makeFirstResponder: [[editor view] content]];
}


- (void)addEditorTab
{
  int editor_index = mBackEnd->add_sql_editor(true);

  [self addEditorTabWithBackEndIndex: editor_index];  
}


- (MSQLEditorController*)activeEditor
{
  return [[mUpperTabView tabViewItemAtIndex: mBackEnd->active_sql_editor_index()] identifier];

  //return [[mUpperTabView selectedTabViewItem] identifier];
}

#pragma mark Overview
- (void)updateOverviewToolbar
{
  // update overview toolbar
  bec::ToolbarItemList items = mBackEnd->live_physical_overview()->get_toolbar_items(bec::NodeId(0));
  for (bec::ToolbarItemList::const_iterator iter= items.begin(); iter != items.end(); ++iter)
  {
    switch (iter->type)
    {
      case bec::ToolbarAction:
      {
        NSButton *button = [mOverviewToolbar addButtonWithIcon: [[GRTIconCache sharedIconCache] imageForIconId: iter->icon]
                                                        target: self
                                                        action: @selector(activateOverviewToolbarItem:)
                                                           tag: 0];
        [[button cell] setRepresentedObject: [NSString stringWithCPPString: iter->command]];
        [button setToolTip: [NSString stringWithCPPString: iter->tooltip]];
        break;
      }
        
      case bec::ToolbarSeparator:
        [mOverviewToolbar addSeparator];
        break;
        
      case bec::ToolbarLabel:
        [mOverviewToolbar addLabelWithTitle: [NSString stringWithCPPString: iter->caption]];
        break;
    }
  }
}


- (void)activateOverviewToolbarItem:(id)sender
{
  std::string action = [[[sender cell] representedObject] UTF8String];
  mBackEnd->live_physical_overview()->activate_toolbar_item(bec::NodeId(0), action);
}

#pragma mark Output

- (IBAction)clearOutput:(id)sender
{
  if ([[[mOutputTabView selectedTabViewItem] identifier] isEqualTo: @"actions"])
  {
    mBackEnd->log()->reset();
    [mMessagesTable reloadData];
  }
  else if ([[[mOutputTabView selectedTabViewItem] identifier] isEqualTo: @"text"])
  {
    [mTextOutput setString: @""];
  }
  else
  {
    mBackEnd->history()->entries_model()->delete_all_entries();
    [mHistoryTable reloadData];
  }
}


- (void)flushOutputBuffer
{
  [mTextOutputLock lock];

  if (!mTextOutputBuffer.empty())
  {
    NSRange range;
    range = NSMakeRange([[mTextOutput string] length], 0);
    [mTextOutput replaceCharactersInRange: range withString: [NSString stringWithUTF8String: mTextOutputBuffer.c_str()]];
  
    range = NSMakeRange([[mTextOutput string] length], 0);
    [mTextOutput scrollRangeToVisible: range];  
  
    mTextOutputBuffer.clear();
  }

  [mTextOutputLock unlock];
}

#pragma mark Snippets


- (void)doubleClickSnippet:(id)sender
{
  if (sender == snippetTable)
    if ([snippetTable clickedRow] >= 0)
      DbSqlEditorSnippets::get_instance()->activate_node([snippetTable clickedRow]);
}


- (void)activateSnippetToolbarItem:(id)sender
{
  std::string action = [[[sender cell] representedObject] UTF8String];
  int row = [snippetTable selectedRow];
  DbSqlEditorSnippets::get_instance()->activate_toolbar_item(row, action);
}


- (void)selectSnippetCollection:(id)sender
{
  DbSqlEditorSnippets::get_instance()->select_category([[sender titleOfSelectedItem] UTF8String]);
  [snippetTable reloadData];
}

#pragma mark Public getters + setters

- (NSView*) topView;
{
  return mView;
}

- (NSString*)title
{
  return [NSString stringWithUTF8String: mBackEnd->caption().c_str()];
}


- (id)identifier
{
  return [NSString stringWithFormat:@"dbquery%p", mBackEnd.get()];
}

- (NSImage*)tabIcon
{
  return [NSImage imageNamed: @"tab.sqlquery.16x16"];
}


- (bec::UIForm*)formBE
{
  return boost::get_pointer(mBackEnd);
}


- (void)setActiveEditorTitle:(NSString*)title
{
  [[mUpperTabView selectedTabViewItem] setLabel: title];
}

- (void)updateActiveRecordsetTitle
{
  MResultsetViewer *rset = [self selectedResultset];
  if (rset)
  {
    [[mLowerTabView selectedTabViewItem] setLabel: [NSString stringWithCPPString: [rset recordset]->caption()]];
  }
}

static void refreshUIPartial(const int what, WBSQLQueryPanel *panel)
{
  switch (what)
  {
    case Db_sql_editor::RefreshSidebar: // possibly deprecated
      [panel->mOverview rebuildAll];
      break;
    case Db_sql_editor::RefreshEditor:
      [[panel activeEditor] setString: [NSString stringWithCPPString: panel->mBackEnd->sql_editor()->sql()]];
      break;
    case Db_sql_editor::RefreshEditorBackend:
      panel->mBackEnd->sql_editor()->sql([[[panel activeEditor] string] UTF8String]);
      break;
    case Db_sql_editor::RefreshEditorTitle:
      [panel setActiveEditorTitle: [NSString stringWithCPPString: panel->mBackEnd->sql_editor_caption()]];
      break;
    case Db_sql_editor::RefreshSnippets:
      [panel->snippetTable reloadData];
      break;
    case Db_sql_editor::RunCurrentScript:
      [panel executeQuery: panel currentStatementOnly: false];
      break;
    case Db_sql_editor::RunCurrentStatement:
      [panel executeQuery: panel currentStatementOnly: true];
      break;
    case Db_sql_editor::RefreshRecordsetTitle:
      [panel updateActiveRecordsetTitle];
      break;
    case Db_sql_editor::RefreshOverview:
    {
      id group = [panel->mOverview itemContainerForNode: bec::NodeId(0)];
      int i = [group indexOfTabViewItem: [group selectedTabViewItem]];
      if (i != NSNotFound)
      {
        bec::NodeId node(bec::NodeId(0).append(i));
        try 
        {
          panel->mBackEnd->live_physical_overview()->focus_node(node); // refocus the node          
        }
        catch (const std::exception &exc) 
        {
          NSLog(@"Ignoring exception focusing overview node: %s", exc.what());
        }
        
        for (int j= 0; j < 3; j++)
        {
          // this might not be the node that needs to be refreshed. this callback needs an extra argument
          // to tell the node to refresh
          [panel->mOverview refreshNodeChildren: bec::NodeId(node).append(j)];
        }
      }
      else
        [panel->mOverview rebuildAll];
      break;
    }
    case Db_sql_editor::RefreshOverviewFully:
      [panel->mOverview rebuildAll];
      break;
  }
}


static int insertText(const std::string &text, WBSQLQueryPanel *panel)
{
  ScintillaView *editor = [[panel activeEditor] view];
  
  [editor insertText: [NSString stringWithCPPString: text]];

  [[editor window] makeFirstResponder: [editor content]];
  
  return 0;
}

static int editorCreated(int editor_index, WBSQLQueryPanel *panel)
{
  [panel addEditorTabWithBackEndIndex: editor_index];
  return 0;
}

#pragma mark Create + destroy

- (id)initWithBE:(const Db_sql_editor::Ref&)be
{
  self= [super init];
  if (self)
  {
    [NSBundle loadNibNamed: @"WBSQLQueryPanel"
                     owner: self];
    
    mBackEnd= be;
    mBackEnd->log()->refresh_ui_cb= boost::bind(reloadTable, mMessagesTable, self);
    mBackEnd->history()->entries_model()->refresh_ui_cb= boost::bind(reloadTable, mHistoryTable, self);
    mBackEnd->history()->details_model()->refresh_ui_cb= boost::bind(reloadTable, mHistoryDetailsTable, self);

    mBackEnd->sql_editor_text_insert_signal.connect(boost::bind(insertText, _1, self));
    
    mBackEnd->set_partial_refresh_ui_slot(boost::bind(refreshUIPartial, _1, self));
    mBackEnd->output_text_slot= boost::bind(addTextToOutput, _1, _2, self);
    
    mBackEnd->exec_sql_task->progress_cb(boost::bind(processTaskProgress, _1, _2, self));
    mBackEnd->exec_sql_task->finish_cb(boost::bind(processTaskFinish, self));
    mBackEnd->recordset_list_changed.connect(boost::bind(recordsetListChanged, _1, _2, self));
    
    mBackEnd->sql_editor_new_ui.connect(boost::bind(editorCreated, _1, self));
    
    mBackEnd->set_frontend_data(self);
    
    // dock the backend provided sidebar
    {
      mforms::View *sidebar = mBackEnd->get_sidebar();
      mSidebarView = nsviewForView(sidebar);
      [mView addSubview: mSidebarView positioned: NSWindowBelow relativeTo: [[mView subviews] lastObject]];
      [mView setPosition: 200 ofDividerAtIndex: 0];
    }
    
    
    mTextOutputLock= [[NSLock alloc] init];
    
    [mTextOutput setFont: [NSFont fontWithName: @"Andale Mono" 
                                          size:[NSFont smallSystemFontSize]]];
    
    [mSplitterDelegate setTopExpandedMinHeight: QUERY_AREA_EXPANDED_MIN_HEIGHT];
    [mSplitterDelegate setTopCollapsedMinHeight: QUERY_AREA_COLLAPSED_MIN_HEIGHT];
    [mSplitterDelegate setBottomExpandedMinHeight: RESULTS_AREA_EXPANDED_MIN_HEIGHT];
    [mSplitterDelegate setBottomCollapsedMinHeight: RESULTS_AREA_COLLAPSED_MIN_HEIGHT];
  
    [mLowerTabView setDelegate: self];
    [(id)mLowerTabView createDragger];
    
    [mWorkView setDividerThickness: 0];
    [mView setDividerThickness: 2];
    
    [(MContainerView*)[mLowerTabView superview] setMinContentSize: NSMakeSize(40, RESULTS_AREA_COLLAPSED_MIN_HEIGHT+20)];
    
    [mHistoryTable setTarget: self];
    [mHistoryTable setDoubleAction: @selector(activateHistoryEntry:)];
    [mHistoryDetailsTable setTarget: self];
    [mHistoryDetailsTable setDoubleAction: @selector(activateHistoryDetailEntry:)];

    for (id item in [mLowerTabView tabViewItems])
    {
      [(id)mLowerTabView setIcon:nil forTabViewItem: [item identifier]];
    }
    
    if (mBackEnd->is_physical_overview_enabled())
    {
      NSTabViewItem *item = [mLowerTabView tabViewItemAtIndex: 0];
      NSView *view;
      NSRect rect;
      
      mOverview = [[WBOverviewPanel alloc] initWithOverviewBE: mBackEnd->live_physical_overview()];
      [mOverview setNoBackground];
      [mOverview setNoHeader];
      
      view = [item view];
      rect = [view frame];
      [view addSubview: [mOverview topView]];
      rect.origin = NSZeroPoint;
      rect.size.height -= NSHeight([mOverviewToolbar frame]) + 2;
      [mOverview setFrame: rect];
      [mOverview setAutoresizingMask: NSViewHeightSizable|NSViewWidthSizable];
      
      [mOverview rebuildAll];
      
      std::string default_schema = mBackEnd->active_schema();
      if (!default_schema.empty())
      {        
        [self performSelector: @selector(activeSchemaChanged:) 
                   withObject: [NSString stringWithCPPString: default_schema]
                   afterDelay: 0.1];
      }
    }
    else
    {
      [mLowerTabView removeTabViewItem: [mLowerTabView tabViewItemAtIndex:0]];
    }

    //XXX[self enableCancelButton:NO];

    mEditors = [[NSMutableArray alloc] init];
    
    mQueryAreaOpen = YES;
    mResultsAreaOpen = YES;
    
    [self updateOverviewToolbar];
    
    // update snippets list
    DbSqlEditorSnippets *snippets = DbSqlEditorSnippets::get_instance();
    [snippetDS setListModel: snippets];
    
    [snippetTable setTarget: self];
    [snippetTable setDoubleAction: @selector(doubleClickSnippet:)];
        
    // update snippets toolbar
    bec::ToolbarItemList items = snippets->get_toolbar_items();
    for (bec::ToolbarItemList::const_iterator iter= items.begin(); iter != items.end(); ++iter)
    {
      switch (iter->type)
      {
        case bec::ToolbarAction:
        {
          NSButton *button = [mSnippetsToolbar addButtonWithIcon: [[GRTIconCache sharedIconCache] imageForIconId: iter->icon]
                                       target: self
                                       action: @selector(activateSnippetToolbarItem:)
                                          tag: 0];
          [[button cell] setRepresentedObject: [NSString stringWithCPPString: iter->command]];
          [button setToolTip: [NSString stringWithCPPString: iter->tooltip]];
          break;
        }

        case bec::ToolbarSeparator:
          [mSnippetsToolbar addSeparator];
          break;

        case bec::ToolbarLabel:
          [mSnippetsToolbar addLabelWithTitle: [NSString stringWithCPPString: iter->caption]];
          break;
      }
    }
    
    [mSnippetsToolbar addLabelWithTitle: @"Snippet Collection:"];
    NSArray *categories = [NSArray arrayWithCPPStringVector: snippets->get_category_list()];
    [mSnippetsToolbar addSelectionPopUpWithItems: categories
                                          target: self
                                          action: @selector(selectSnippetCollection:)
                                    defaultValue: nil];
    
    DbSqlEditorSnippets::get_instance()->select_category([[categories objectAtIndex: 0] UTF8String]);
    [snippetTable reloadData];
    
    // realize pre-existing editors
    for (int i = 0; i < mBackEnd->sql_editor_count(); i++)
    {
      [self addEditorTabWithBackEndIndex: [mUpperTabView numberOfTabViewItems]];
    }
    
    if (mBackEnd->is_physical_overview_enabled())
      [mLowerTabView selectTabViewItemWithIdentifier: @"overview"];
  }
  return self;
}


- (Db_sql_editor::Ref)backEnd
{
  return mBackEnd;
}

- (void)setRightSidebar:(BOOL)flag
{
  sidebarAtRight = flag;

  id view1 = [[mView subviews] objectAtIndex: 0];
  id view2 = [[mView subviews] objectAtIndex: 1];
  
  if (sidebarAtRight)
  {
    if (view2 != mSidebarView)
    {
      [[view1 retain] autorelease];
      [view1 removeFromSuperview];
      [mView addSubview: view1];
    }    
  }
  else
  {
    if (view1 != mSidebarView)
    {
      [[view1 retain] autorelease];
      [view1 removeFromSuperview];
      [mView addSubview: view1];
    }
  }
}

- (MResultsetViewer*)selectedResultset
{
  NSTabViewItem *selectedItem= [mLowerTabView selectedTabViewItem];

  if ([selectedItem isKindOfClass: [ResultTabViewItem class]])
    return [(ResultTabViewItem*)selectedItem RSViewer];

  return nil;
}


- (void)activeSchemaChanged:(NSString*)schemaName
{
  id group = [mOverview itemContainerForNode: bec::NodeId(0)];
  int index = 0;
  for (id tab in [group tabViewItems])
  {
    if ([[tab label] isEqual: schemaName])
    {
      [group selectTabViewItemWithIdentifier: [tab identifier]];
      
      bec::NodeId node(bec::NodeId(0).append(index));
      [mOverview refreshNode: node];
      for (int j= 0; j < 3; j++)
        [mOverview refreshNodeChildren: bec::NodeId(node).append(j)];
      break;
    }
    index++;
  }
}
  
- (void)tabView:(NSTabView *)tabView didSelectTabViewItem:(NSTabViewItem *)tabViewItem
{
  if (tabView == mUpperTabView)
  {
    MSQLEditorController *editor = [tabViewItem identifier];
    Sql_editor::Ref ed([editor backEnd]);

    for (int i= 0; i < mBackEnd->sql_editor_count(); i++)
    {
      if (mBackEnd->sql_editor(i) == ed)
      {
        mBackEnd->active_sql_editor_index(i);
        [mLowerTabView selectTabViewItemWithIdentifier: [NSString stringWithFormat:@"rset%li", mBackEnd->active_recordset_for_sql_editor(i)]];
        break;
      }
    }
  }
  else if (tabView == mLowerTabView)
  {
    MResultsetViewer *rset= [self selectedResultset];
    if (rset)
    {
      mBackEnd->active_recordset([rset recordset]);
      int ed(mBackEnd->active_sql_editor_index());
      if (ed >= 0)
      {
        Sql_editor::Ref editor(mBackEnd->sql_editor(ed));

        for (NSTabViewItem *item in [mUpperTabView tabViewItems])
        {
          if ([[item identifier] isKindOfClass: [MSQLEditorController class]]
              && [(MSQLEditorController*)[item identifier] backEnd] == editor)
          {
            if ([mUpperTabView selectedTabViewItem] != item)
              [mUpperTabView selectTabViewItemWithIdentifier: [item identifier]];
            break;
          }
        }
      }
    }
    else
      mBackEnd->active_recordset(Recordset::Ref());
  }
}

- (void)tabViewDidChangeNumberOfTabViewItems:(NSTabView*)tabView
{
  if (tabView == mUpperTabView)
  {
    if ([tabView numberOfTabViewItems] == 0)
      [self addEditorTab];
  }
}

- (BOOL)tabView:(NSTabView *)tabView
willCloseTabViewItem:(NSTabViewItem*)tabViewItem
{
  if (tabView == mUpperTabView)
  {
    MSQLEditorController *editor = [tabViewItem identifier];
    
    Sql_editor::Ref ed = [editor backEnd];
    int idx = -1;
    
    for (int i= 0; i < mBackEnd->sql_editor_count(); i++)
    {
      if (mBackEnd->sql_editor(i) == ed)
        idx = i;
    }
    if (idx < 0)
      return NO;

    if (!mBackEnd->sql_editor_will_close(idx))
      return NO;
        
    mBackEnd->remove_sql_editor(idx);
    
    return YES;
  }
  else
  {
    id ident = [tabViewItem identifier];
    if ([ident isEqual: @"history"] ||
        [ident isEqual: @"messages"])
      return NO;
    
    if ([tabViewItem view] == mOverview)
      return [mOverview willClose];
    
    if ([tabViewItem isKindOfClass: [ResultTabViewItem class]])
    {
      [[(ResultTabViewItem*)tabViewItem RSViewer] close];
      return NO;
    }
    return YES;
  }
}

- (BOOL)willClose
{
  return mBackEnd->can_close();
}

- (void) dealloc
{ 
  mBackEnd->close();
  
  [mTextOutputLock release];
  
  [mEditors release];
  [mOverview release];
  
  [mSplitterDelegate release];
  [mView release];
  [snippetDS release];
  [super dealloc]; 
}


- (void)searchString:(NSString*)text
{
  [[[self activeEditor] view] findAndHighlightText: text
                                         matchCase: NO
                                         wholeWord: NO
                                          scrollTo: YES
                                              wrap: YES
                                         backwards: NO];
}

//--------------------------------------------------------------------------------------------------

/**
 * Executes commands sent by the main form that should be handled here.
 */
- (void) performCommand: (const std::string) command
{
  if (command == "wb.sidebarHide") // Actually means toggle.
  {
    sidebarHidden = !sidebarHidden;
    if (!sidebarHidden)
    {
      if (sidebarAtRight) 
        [mView setPosition: NSWidth([mView frame])-lastSidebarWidth ofDividerAtIndex: 0];
      else
        [mView setPosition: lastSidebarWidth ofDividerAtIndex: 0];
    }
    else
    {
      lastSidebarWidth = NSWidth([mSidebarView frame]);
      if (sidebarAtRight)
        [mView setPosition: NSWidth([mView frame]) ofDividerAtIndex: 0];
      else
        [mView setPosition: 0 ofDividerAtIndex: 0];
    }
  }
}

@end


