/* 
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#import "WBTabSwitcher.h"
#import "WBTabItem.h"
#import "WBTabView.h"

@interface NSObject(TabDelegateExtras)
- (BOOL)tabView:(NSTabView*)tabView itemIsClosable:(NSTabViewItem*)item;
@end

@implementation WBTabbedWindowTitle

- (id)initWithFrame:(NSRect)frameRect
{
  self = [super initWithFrame: frameRect];
  if (self)
  {
    mActiveGradient = [[NSGradient alloc] initWithStartingColor: [NSColor colorWithCalibratedWhite: 206/255.0
                                                                                             alpha: 1.0]
                                                    endingColor: [NSColor colorWithCalibratedWhite: 167/255.0
                                                                                             alpha: 1.0]];
    
    NSRect rect = [self frame];
    rect.origin = NSZeroPoint;
    mTrack = [[NSTrackingArea alloc] initWithRect: rect
                                          options: NSTrackingMouseEnteredAndExited|NSTrackingMouseMoved|NSTrackingActiveAlways
                                            owner: self
                                         userInfo: nil];
    [self addTrackingArea: mTrack];
    
    mHomeIcon = [[NSImage imageNamed: @"WB_Home_mac.png"] retain];
  }
  return self;
}


- (void)dealloc
{
  [mHomeIcon release];
  [mTrack release];
  [[NSNotificationCenter defaultCenter] removeObserver: self];
  [mActiveGradient release];
  [super dealloc];
}


- (NSTabViewItem*)tabViewItemAtPoint: (NSPoint)p
{
  CGFloat x = 8;
  CGFloat width;
  NSDictionary *attrs = [NSDictionary dictionaryWithObjectsAndKeys: 
                         [NSFont systemFontOfSize: 11.5], NSFontAttributeName,
                         nil];
  for (NSTabViewItem *item in [mOwnerTabView tabViewItems])
  {
    NSString *title = [item label];
    
    width = 35 + [title sizeWithAttributes: attrs].width;
    if (p.x >= x && p.x <= x + width)
      return item;
    x += width;
  }
  return nil;
}


- (void)resizeWithOldSuperviewSize:(NSSize)oldSize
{
  [super resizeWithOldSuperviewSize: oldSize];

  [self setNeedsDisplay: YES];

  [self removeTrackingArea: mTrack];
  NSRect rect = [self frame];
  rect.origin = NSZeroPoint;
  mTrack = [[NSTrackingArea alloc] initWithRect: rect
                                        options: NSTrackingMouseEnteredAndExited|NSTrackingMouseMoved|NSTrackingActiveAlways
                                          owner: self
                                       userInfo: nil];
  [self addTrackingArea: mTrack];
}


- (void)mouseDown:(NSEvent *)theEvent
{
  NSTabViewItem *item = [self tabViewItemAtPoint: [self convertPointFromBase: [theEvent locationInWindow]]];
  mClosePressed = NO;
  if (item)
  {
    [mOwnerTabView selectTabViewItem: item];
    if (mHoverItem && NSPointInRect([self convertPointFromBase: [theEvent locationInWindow]], mCloseButtonRect))
      mClosePressed = YES;
    [self setNeedsDisplay: YES];
  }
}


- (void)mouseUp:(NSEvent *)theEvent
{
  if (mClosePressed && mHoverItem && NSPointInRect([self convertPointFromBase: [theEvent locationInWindow]], mCloseButtonRect))
  {
    [self setNeedsDisplay: YES];
    if ([[mOwnerTabView delegate] tabView: mOwnerTabView willCloseTabViewItem: mHoverItem])
    {
      [mOwnerTabView removeTabViewItem: mHoverItem];
      mHoverItem = nil;
    }
  }
  mClosePressed = NO;
}


- (void)mouseMoved:(NSEvent *)theEvent
{
  NSTabViewItem *item = [self tabViewItemAtPoint: [self convertPointFromBase: [theEvent locationInWindow]]];
  if (![[mOwnerTabView delegate] tabView: mOwnerTabView itemIsClosable: item])
    item = nil;

  mHoverItem = item;
  [self setNeedsDisplay: YES];
}


- (void) mouseEntered:(NSEvent *)theEvent
{
  NSTabViewItem *item = [self tabViewItemAtPoint: [self convertPointFromBase: [theEvent locationInWindow]]];
  if ([[mOwnerTabView delegate] tabView: mOwnerTabView itemIsClosable: item])
  {
    mHoverItem = item;
    [self setNeedsDisplay: YES];
  }
}


- (void) mouseExited:(NSEvent *)theEvent
{
  [self setNeedsDisplay: YES];
  mHoverItem = nil;
}


- (void)mouseDragged:(NSEvent *)theEvent
{
}


- (void)drawRect:(NSRect)rect
{
  [super drawRect: rect];
  
  NSTabViewItem *selectedItem = [mOwnerTabView selectedTabViewItem];
  CGFloat selectedItemMinX, selectedItemMaxX;
  CGFloat x = 8;
  CGFloat width;
  CGFloat height = NSHeight([self frame]);
  BOOL isMain = [[self window] isMainWindow];
  
  [NSBezierPath setDefaultLineWidth: 1.0];
  
  NSDictionary *blackAttrs = [NSDictionary dictionaryWithObjectsAndKeys: 
                              [NSFont systemFontOfSize: 11.5], NSFontAttributeName,
                              [NSColor colorWithDeviceWhite: 53/255.0 alpha: 1.0], NSForegroundColorAttributeName,
                              nil];
  
  NSDictionary *whiteAttrs = [NSDictionary dictionaryWithObjectsAndKeys: 
                              [NSFont systemFontOfSize: 11.5], NSFontAttributeName,
                              [NSColor colorWithDeviceWhite: 245/255.0 alpha: 1.0], NSForegroundColorAttributeName,
                              nil];
  NSDictionary *white2Attrs = [NSDictionary dictionaryWithObjectsAndKeys: 
                              [NSFont systemFontOfSize: 11.5], NSFontAttributeName,
                              [NSColor colorWithDeviceWhite: 200/255.0 alpha: 1.0], NSForegroundColorAttributeName,
                              nil];
  
  for (NSTabViewItem *item in [mOwnerTabView tabViewItems])
  {
    NSString *title = [item label];
    
    if ([title isEqualTo: @"Home"])
      width = 30 + [mHomeIcon size].width;
    else
      width = 35 + [title sizeWithAttributes: blackAttrs].width;
    
    if (selectedItem == item)
    {
      selectedItemMinX = x+0.5;
      selectedItemMaxX = x+width+0.5;
      
      [[NSColor colorWithDeviceWhite: 246/255.0 alpha: 1.0] set];
      
      NSBezierPath *path = [NSBezierPath bezierPath];
    
      [path moveToPoint: NSMakePoint(selectedItemMinX-5, 0)];
      [path appendBezierPathWithArcWithCenter: NSMakePoint(selectedItemMinX-5, 5)
                                       radius: 5
                                   startAngle: 270
                                     endAngle: 360];
      [path lineToPoint: NSMakePoint(selectedItemMinX, height-5)];
      [path appendBezierPathWithArcWithCenter: NSMakePoint(selectedItemMinX+5, height-5.5)
                                       radius: 5
                                   startAngle: 180
                                     endAngle: 90
                                    clockwise: YES];
      [path lineToPoint: NSMakePoint(selectedItemMaxX - 5, height-0.5)];
      [path appendBezierPathWithArcWithCenter: NSMakePoint(selectedItemMaxX-5, height-5.5)
                                       radius: 5
                                   startAngle: 90
                                     endAngle: 0
                                    clockwise: YES];
      [path lineToPoint: NSMakePoint(selectedItemMaxX, 5)];
      [path appendBezierPathWithArcWithCenter: NSMakePoint(selectedItemMaxX+5, 5)
                                       radius: 5
                                   startAngle: 180
                                     endAngle: 270];
      
      if (isMain)
      {
        [[[[NSGradient alloc] initWithStartingColor: [NSColor colorWithDeviceWhite: 206/255.0 alpha:1.0]
                                        endingColor: [NSColor colorWithDeviceWhite: 230/255.0 alpha:1.0]] autorelease]
         drawInBezierPath: path angle: 90];
      }
      else
      {
        [[[[NSGradient alloc] initWithStartingColor: [NSColor colorWithDeviceWhite: 233/255.0 alpha:1.0]
                                        endingColor: [NSColor colorWithDeviceWhite: 240/255.0 alpha:1.0]] autorelease]
         drawInBezierPath: path angle: 90];
      }

      
      [path setLineWidth: 1.0];
      [path stroke];
      

      if (isMain)
        [[NSColor colorWithDeviceWhite: 141/255.0 alpha: 0.8] set];
      else
        [[NSColor colorWithDeviceWhite: 210/255.0 alpha: 0.8] set];

      path = [NSBezierPath bezierPath];
      [path moveToPoint: NSMakePoint(selectedItemMinX-5.5, 1.5)];
      [path appendBezierPathWithArcWithCenter: NSMakePoint(selectedItemMinX-5.5, 6)
                                       radius: 5
                                   startAngle: 270
                                     endAngle: 360];
      [path lineToPoint: NSMakePoint(selectedItemMinX-0.5, height-4.5)];
      [path appendBezierPathWithArcWithCenter: NSMakePoint(selectedItemMinX+5, height-4.5)
                                       radius: 5
                                   startAngle: 180
                                     endAngle: 90
                                    clockwise: YES];
      [path lineToPoint: NSMakePoint(selectedItemMaxX - 5.0, height+0.5)];
      [path appendBezierPathWithArcWithCenter: NSMakePoint(selectedItemMaxX-5.0, height-4.5)
                                       radius: 5
                                   startAngle: 90
                                     endAngle: 0
                                    clockwise: YES];
      [path lineToPoint: NSMakePoint(selectedItemMaxX, 6)];
      [path appendBezierPathWithArcWithCenter: NSMakePoint(selectedItemMaxX+5.0, 6)
                                       radius: 5
                                   startAngle: 180
                                     endAngle: 270];
      
      [path setLineWidth: 1.0];
      [path stroke];
    }
    
    if ([title isEqualTo: @"Home"])
    {
      [mHomeIcon drawInRect: NSMakeRect(x + 15, (int)((height-[mHomeIcon size].height)/2), [mHomeIcon size].width, [mHomeIcon size].height)
                   fromRect: NSMakeRect(0, 0, [mHomeIcon size].width, [mHomeIcon size].height)
                  operation: NSCompositeSourceOver fraction: 1.0];
    }
    else
    {
      [title drawInRect: NSMakeRect(x + 15, 0, width, height-6) withAttributes: selectedItem == item ? whiteAttrs : white2Attrs];
      [title drawInRect: NSMakeRect(x + 14, 1, width, height-6) withAttributes: blackAttrs];
      
      if (mHoverItem == item)
      {
        NSImage *image = [[[NSImage alloc] initWithContentsOfFile: 
                           [[NSBundle bundleForClass: [WBTabItem class]] 
                            pathForImageResource: mClosePressed ? @"TabClose_Pressed.png" : @"TabClose_Unpressed.png"]] autorelease];
        mCloseButtonRect = NSMakeRect(x+width-18, 4, [image size].width, [image size].height);
        [image drawInRect: mCloseButtonRect fromRect: NSMakeRect(0, 0, [image size].width, [image size].height)
                operation: NSCompositeSourceOver fraction: 1.0];
      }
    }
    x += width;
  }
  
  // stroke the baseline
  [[NSColor colorWithDeviceWhite: 227/255.0 alpha: 1.0] set];
  [NSBezierPath strokeLineFromPoint: NSMakePoint(0, 0.5) toPoint: NSMakePoint(selectedItemMinX-3, 0.5)];  
  [NSBezierPath strokeLineFromPoint: NSMakePoint(selectedItemMaxX+4, 0.5) toPoint: NSMakePoint(NSWidth(rect), 0.5)];
  if (isMain)
    [[NSColor colorWithDeviceWhite: 135/255.0 alpha: 1.0] set];
  else
    [[NSColor colorWithDeviceWhite: 210/255.0 alpha: 1.0] set];
  [NSBezierPath strokeLineFromPoint: NSMakePoint(0, 1.5) toPoint: NSMakePoint(selectedItemMinX-5, 1.5)];  
  [NSBezierPath strokeLineFromPoint: NSMakePoint(selectedItemMaxX+3, 1.5) toPoint: NSMakePoint(NSWidth(rect), 1.5)];

}

@end
