/**
 * Copyright (c) 2008, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Threading;
using System.Windows.Forms;

using Aga.Controls.Tree;
using ScintillaNet;

using MySQL.Controls;
using MySQL.Forms;
using MySQL.Grt;
using MySQL.GUI.Mdc;
using MySQL.GUI.Workbench.Plugins;
using MySQL.GUI.Workbench.Properties;
using MySQL.Utilities;
using MySQL.Workbench;
using MySQL.Utilities.SysUtils;

namespace MySQL.GUI.Workbench
{
  public partial class MainForm : Form, IMySQLMainForm
  {
    #region Member Variables

    private String[] resourcePaths = {
      "",
      "images/ui",
      "images/icons",
      "images/grt",
      "images/grt/structs",
      "images/cursors",
      "images/home",
      "images/sql",
    };

    // The Workbench context
    protected WbContext wbContext;
    // the GRT manager
    protected GrtManager grtManager;

    // A timer for helping backend timers.
    private System.Windows.Forms.Timer timer = null;
    
    // Special Forms
    private ModelOverviewForm workbenchPhysicalOverviewForm = null;

    // TODO: this should go once the entire menu/toolbar handling is in the backend.
    private WorkbenchToolbarManager workbenchToolbarManager = null;

    private Form findForm = null;

    private bool shuttingDown = false;
    private ImageList tabImageList = null;

    #endregion

    #region Constructors

    /// <summary>
    /// Standard constructor
    /// </summary>
    private MainForm()
    {
      InitializeComponent();
    }

    /// <summary>
    /// Constructor that takes a WbContext and passes it to the sub-forms that get created
    /// </summary>
    /// <param name="WbContext">The WbContext Backend Wrapper</param>
    public MainForm(WbContext WbContext)
      : this()
    {
      wbContext = WbContext;
      grtManager = wbContext.get_grt_manager();

      workbenchToolbarManager = new WorkbenchToolbarManager(wbContext);

      tabImageList = new ImageList();
      tabImageList.ColorDepth = ColorDepth.Depth32Bit;
      tabImageList.ImageSize = new Size(16, 16);
      ImageListHelper.Add(ApplicationCommand(AppCommand.AppGetResourcePath, null, "WB_Home.png"), tabImageList);
      contentTabControl.ImageList = tabImageList;


      // Create a timer to be triggered when the backend needs
      timer = new System.Windows.Forms.Timer();
      timer.Tick += new EventHandler(timer_Tick);

      // Prepare Statusbar
      PictureBox statusStripImg = new PictureBox();
      statusStripImg.SizeMode = PictureBoxSizeMode.CenterImage;
      statusStripImg.Image = Resources.statusbar_separator;
      statusStripImg.BackColor = Color.Transparent;
      ToolStripControlHost host = new ToolStripControlHost(statusStripImg);
      host.Alignment = ToolStripItemAlignment.Right;
      mainStatusStrip.Items.Add(host);

      // output img
      statusStripImg = new PictureBox();
      statusStripImg.Name = "grtShellStripButton";
      statusStripImg.SizeMode = PictureBoxSizeMode.CenterImage;
      statusStripImg.Image = Resources.statusbar_output;
      statusStripImg.BackColor = Color.Transparent;
      statusStripImg.Click += new System.EventHandler(grtOutputImg_Click);
      mainFormToolTip.SetToolTip(statusStripImg, "Display Output Window");
      host = new ToolStripControlHost(statusStripImg);
      host.Alignment = ToolStripItemAlignment.Right;
      mainStatusStrip.Items.Add(host);

      statusStripImg = new PictureBox();
      statusStripImg.SizeMode = PictureBoxSizeMode.CenterImage;
      statusStripImg.Image = Resources.statusbar_separator;
      statusStripImg.BackColor = Color.Transparent;
      host = new ToolStripControlHost(statusStripImg);
      host.Alignment = ToolStripItemAlignment.Right;
      mainStatusStrip.Items.Add(host);
    }

    #endregion

    #region Implement Interfaces

    delegate void DelegateFunc();

    public string StatusBarText
    {
      get { return statusText.Text; }
      set 
      {
        if (InvokeRequired)
        {
          DelegateFunc f = delegate
          {
            statusText.Text = value.Replace("\r\n", "\n").Replace("\n", "");
          };

          BeginInvoke(f);
        }
        else
          statusText.Text = value.Replace("\r\n", "\n").Replace("\n", "");
      }
    }

    #endregion

    #region Properties

    public ModelOverviewForm WorkbenchPhysicalOverviewForm
    {
      get { return workbenchPhysicalOverviewForm; }
    }

    #endregion

    #region Callbacks

    public void ShowStatusText(String message)
    {
      StatusBarText = message;
    }

    public bool ShowProgress(String title, String detail, float progress)
    {
      if (progress < 0)
      {
        if (title != "" && detail != "")
          StatusBarText = title + ": " + detail;
        else if (title != "")
          StatusBarText = title;
        else
          StatusBarText = "Finished";

        statusProgress.Visible = false;
        return false;
      }
      else if (!statusProgress.Visible)
        statusProgress.Visible = true;

      if (title != "" && detail != "")
        StatusBarText = title + ": " + detail;
      else
        StatusBarText = title;

      statusProgress.Value = (int)(progress * 100);

      return false;
    }

    private bool RequestInputPressedOk;

    public void OnRequestInputButton(Object sender, EventArgs e)
    {
      Button b = sender as Button;
      if (b == null)
        return;
      Form f = b.Parent as Form;
      if (f == null)
        return;
      f.Close();
      RequestInputPressedOk = (b.Text == "OK");
    }

    public bool RequestInput(String title, int flags, out String result)
    {
      this.RequestInputPressedOk = false;
      result = "";

      Form dialog = new Form();
      if (flags != 0)
        dialog.Text = "Input Password";
      else
        dialog.Text = "Input Text";
      dialog.Size = new Size(350, 125);
      dialog.StartPosition = FormStartPosition.CenterParent;
      dialog.FormBorderStyle = FormBorderStyle.FixedDialog;

      Label prompt = new Label();
      prompt.Location = new Point(10, 5);
      prompt.AutoSize = true;
      prompt.Text = title;

      TextBox input = new TextBox();
      input.Location = new Point(10, 30);
      input.Size = new Size(dialog.Size.Width - (input.Location.X*2 + 1), 50);
      if (flags != 0)
        input.PasswordChar = '*';

      Button ok = new Button();
      ok.Text = "OK"; // if you rename this also update OnRequestInputButton()
      ok.Location = new Point(dialog.Size.Width/2 - ok.Size.Width - 10, 
        input.Location.Y + input.Size.Height + 10);
      ok.Click += new EventHandler(OnRequestInputButton);
      
      Button cancel = new Button();
      cancel.Text = "Cancel";
      cancel.Location = new Point(dialog.Size.Width/2 + 10, 
        input.Location.Y + input.Size.Height + 10);
      cancel.Click += new EventHandler(OnRequestInputButton);

      dialog.AcceptButton = ok;
      dialog.CancelButton = cancel;

      dialog.Controls.Add(prompt);
      dialog.Controls.Add(input);
      dialog.Controls.Add(ok);
      dialog.Controls.Add(cancel);
      dialog.ShowDialog();

      if (RequestInputPressedOk)
        result = input.Text;

      return RequestInputPressedOk;
    }

    public String ApplicationCommand(AppCommand command, MySQL.Forms.AppViewImpl view, String str)
    {
      if (IsDisposed || Disposing)
        return "";

      String result = "";
      switch (command) 
      {
        case AppCommand.AppDockView:
          if (str == "maintab")
            DockDocument(view.GetHost(), true, true);
          else
            if (str == "editortab")
              DockDocument(view.GetHost(), false, true);
          break;
        case AppCommand.AppSelectView:
          foreach (ITabDocument document in contentTabControl.Documents)
          {
            if (document is AppViewDockContent)
            {
              AppViewDockContent content = (document as AppViewDockContent);
              if (content.GetAppViewIdentifier() == str)
              {
                content.Activate();
                result = "1";
                break;
              }
            }
          }
          result = "0";
          break;
        case AppCommand.AppUndockView:
          UndockDocument(view.GetHost());
          break;
        case AppCommand.AppSetViewTitle:
          {
            ITabDocument document = view.GetHost();
            if (contentTabControl.IndexFromDocument(document) > 0)
              view.GetHost().TabText = str;
          }
          break;
        case AppCommand.AppQuit: // Not yet used.
          Close();
          break;
        case AppCommand.AppGetResourcePath:
          // Try to find the given file in known locations. For now we only search the image paths.
          // That will get improved later to find any resource in any location (including resource DLLs etc.).
          result = "";
          if (File.Exists(str))
            return str;
          foreach (String path in resourcePaths)
            if (File.Exists(path + "/" + str))
            {
              result = path + "/" + str;
              break;
            }
          break;
        case AppCommand.AppSetStatusText:
          StatusBarText = str;
          break;
        case AppCommand.AppGetBounds:
          // Returns the current bounds of the main window, encoded in a string (we only have one return type: string).
          // This can be used to center messages over the application window, regardless on which screen/monitor
          // the app windows currently is.
          int[] values= { Left, Top, Width, Height };
          result = String.Format("{0} {1} {2} {3} ", values);
          break;
      }
      return result;
    }

    public void RefreshGUI(RefreshType refresh, String str, IntPtr ptr)
    {
      if (IsDisposed || Disposing)
        return;

      SuspendLayout();

      switch (refresh)
      {
        case RefreshType.RefreshCloseDocument:
        {
          if (workbenchPhysicalOverviewForm != null)
          {
            workbenchPhysicalOverviewForm.ResetDocument(true);
            workbenchPhysicalOverviewForm.Close();
            workbenchPhysicalOverviewForm = null;
          }

          wbContext.flush_idle_tasks();
          wbContext.close_document_finish();

          break;
        }
        case RefreshType.RefreshNewDiagram:
        {
          BaseWindowsCanvasView canvas = BaseWindowsCanvasView.GetFromFixedId(ptr);
          if (canvas != null)
          {
            // Open only diagrams which were open when this model was closed last time.
            ModelDiagramForm modelDiagramForm = canvas.GetOwnerForm() as ModelDiagramForm;
            if (modelDiagramForm != null && !((ModelViewForm)modelDiagramForm.BackendClass).is_closed())
              DockDocument(modelDiagramForm, true, true);
          }
          break;
        }
        case RefreshType.RefreshViewName:
        {
          BaseWindowsCanvasView canvas = BaseWindowsCanvasView.GetFromFixedId(ptr);
          if (canvas != null)
          {
            ModelDiagramForm modelDiagramForm = canvas.GetOwnerForm() as ModelDiagramForm;

            if (modelDiagramForm != null)
              modelDiagramForm.TabText = str;
          }
          break;
        }

        case RefreshType.RefreshOverviewNodeChildren:
          if (workbenchPhysicalOverviewForm != null)
            workbenchPhysicalOverviewForm.RefreshNodeChildren(new NodeId(str));
          foreach (ITabDocument content in contentTabControl.Documents)
          {
            if (content is DbSqlEditor)
            {
              DbSqlEditor dbSqlEditor = content as DbSqlEditor;
              if (dbSqlEditor.LivePhysicalOverviewBE.matches_handle(ptr))
              {
                dbSqlEditor.LivePhysicalOverview.RefreshNodeChildren(new NodeId(str));
                break;
              }
            }
          }
          break;

        case RefreshType.RefreshNewModel:
          ShowPhysicalOverviewForm(new NodeId(str)); // Make sure the physical overview exists.

          // Let the backend create what is necessary.
          wbContext.new_model_finish();
          break;

        case RefreshType.RefreshDocument:
        case RefreshType.RefreshCloseEditor:
        case RefreshType.RefreshOverviewNodeInfo:
          ForwardRefreshToAllDocuments(refresh, str, ptr);
          break;

        case RefreshType.RefreshSchema:
        case RefreshType.RefreshSchemaList:
        case RefreshType.RefreshSelection:
        case RefreshType.RefreshZoom:
          ForwardRefreshToActivDocument(refresh, str, ptr);
          break;

        case RefreshType.RefreshTimer:
          UpdateTimer();
          break;

        case RefreshType.RefreshFinishEdits:
          // Force all ongoing edits to be Commited (like in listview cells)
          try
          {
            Control activeControl = ControlUtilities.GetLeafActiveControl(this);
            if (activeControl != null)
            {
              if (activeControl.Parent is Aga.Controls.Tree.TreeViewAdv)
              {
                ActiveControl = null;
              }
            }
          }
          catch (Exception e)
          {
            Program.HandleException(e);
          }
          break;

        default:
          break;
      }

      ResumeLayout();
    }

    /// <summary>
    /// Creates a new diagram form with a given name and id. UI handling (e.g. docking) is done later.
    /// </summary>
    public BaseWindowsCanvasView CreateNewDiagram(string viewId, string name, IntPtr payload)
    {
      ModelDiagramForm modelDiagramForm = new ModelDiagramForm(wbContext, viewId, payload);

      modelDiagramForm.Text = name;
      modelDiagramForm.TabText = name;

      return modelDiagramForm.Canvas;
    }

    /// <summary>
    /// Frees a model diagram form once its backend was freed.
    /// </summary>
    public void DestroyView(BaseWindowsCanvasView canvasView)
    {
      ModelDiagramForm modelDiagramForm = canvasView.GetOwnerForm() as ModelDiagramForm;
      if (modelDiagramForm != null)
        modelDiagramForm.Close();
    }

    public void SwitchedView(BaseWindowsCanvasView canvasView)
    {
      ModelDiagramForm modelDiagramForm = canvasView.GetOwnerForm() as ModelDiagramForm;
      if (modelDiagramForm != null)
        DockDocument(modelDiagramForm, true, true);
    }

    public void ToolChanged(BaseWindowsCanvasView canvasView)
    {
      ModelDiagramForm modelDiagramForm = canvasView.GetOwnerForm() as ModelDiagramForm;
      if (modelDiagramForm != null)
        modelDiagramForm.OnToolChanged();
    }

    public IntPtr OpenPlugin(GrtManager GrtManager, GrtModule GrtModule, string AssemblyName,
      string ClassName, GrtValue GrtList, GUIPluginFlags flags)
    {
      IntPtr ptr = IntPtr.Zero;

      try
      {
        // Load assembly
        Assembly assembly = Assembly.LoadFrom(System.IO.Path.Combine(
          Application.StartupPath, AssemblyName));

        // Find class
        foreach (Type type in assembly.GetTypes())
        {
          if (type.IsClass == true && type.FullName.EndsWith("." + ClassName))
          {
            // use global grtManager
            Object[] args = { grtManager, GrtList };

            if (typeof(DockablePlugin).IsAssignableFrom(type))
            {
              // If ForceNewWindowFlag is not set and this is an object editor
              if ((GUIPluginFlags.ForceNewWindowFlag & flags) == 0 &&
                (GUIPluginFlags.StandaloneWindowFlag & flags) == 0 &&
                typeof(ObjectEditorPlugin).IsAssignableFrom(type))
              {
                // Check if a plugin of this type is already open. If we can find one on the active
                // document page then reuse that. If not try the other pages and close the first
                // editor of that plugin type we find.
                DockablePlugin plugin = null;
                if (contentTabControl.ActiveDocument is IWorkbenchDocument)
                  plugin = (contentTabControl.ActiveDocument as IWorkbenchDocument).FindPluginOfType(type);

                if (plugin == null)
                {
                  foreach (ITabDocument document in contentTabControl.Documents)
                    if (document is IWorkbenchDocument && document != contentTabControl.ActiveDocument)
                    {
                      if ((document as IWorkbenchDocument).ClosePluginOfType(type))
                        break;
                    }
                }
                else
                {
                  // If so, try to change the current GRT Object and exit
                  if ((plugin as ObjectEditorView).EditorPlugin.ChangeGrtList(grtManager, GrtList))
                  {
                    // Release the current editor in the BE.
                    PluginFormClosed(plugin, new FormClosedEventArgs(CloseReason.None));

                    // return old Ptr as Ptr for the new editor
                    return plugin.GetFixedPtr();
                  }
                  else
                    System.Diagnostics.Debug.Print(String.Format("Object editor for {0} does not support reuse", ClassName));
                }
              }

              ObjectEditorPlugin objectEditorPlugin = Activator.CreateInstance(type, args) as ObjectEditorPlugin;
              if (objectEditorPlugin != null)
              {
                ObjectEditorView pluginForm;
                if ((GUIPluginFlags.StandaloneWindowFlag & flags) != 0)
                  pluginForm = new StandaloneWindowPlugin(objectEditorPlugin);
                else
                  pluginForm = new DockedWindowPlugin(objectEditorPlugin);
                //pluginForm.Show();

                // Make sure form is removed from GRT editor list when closed
                pluginForm.EditorPlugin.FormClosed += new FormClosedEventHandler(PluginFormClosed);

                InstallFocusChangeHandlers(pluginForm);

                // Get fixed ptr
                ptr = pluginForm.GetFixedPtr();

                if (ptr == IntPtr.Zero)
                  throw new Exception("Class not found");
              }

              break;
            }
            else if (typeof(Plugin).IsAssignableFrom(type))
            {
              Plugin plugin = Activator.CreateInstance(type, args) as Plugin;

              try
              {
                plugin.Execute();
              }
              catch (Exception e)
              {
                Program.HandleException(e);
              }
            }
          }
        }
      }
      catch (Exception e)
      {
        Program.HandleException(e);
      }

      return ptr;
    }

    public void CreateMainFormView(String ViewName, UIForm viewBE)
    {
      try
      {
        DockablePlugin form = null;
        switch (ViewName)
        {
          case "dbquery":
            DbSqlEditor db_sql_editor = new DbSqlEditor(wbContext, viewBE);
            DockDocument(db_sql_editor, true, true);
            break;
        }
        if (null != form)
        {
          form.FormClosed += new FormClosedEventHandler(PluginFormClosed);
        }
      }
      catch (Exception e)
      {
        Program.HandleException(e);
      }
    }

    public void ShowPlugin(IntPtr ptr)
    {
      if (ptr != IntPtr.Zero)
      {
        TabDocument pluginForm = DockablePlugin.GetFromFixedPtr(ptr);
        if (pluginForm != null)
        {
          if (pluginForm is ObjectEditorPlugin)
          {
            ObjectEditorView objectEditorView = (pluginForm as ObjectEditorPlugin).ContainerForm;
            if (objectEditorView is StandaloneWindowPlugin)
              objectEditorView.Show();
            else if (objectEditorView is DockedWindowPlugin)
              DockDocument(objectEditorView, false, true);
          }
          else if (pluginForm is DockablePlugin && !(pluginForm is WizardPlugin))
            DockDocument(pluginForm, false, true);
          else
            pluginForm.ShowDialog();
        }
      }
    }

    public void HidePlugin(IntPtr ptr)
    {
      if (ptr != IntPtr.Zero)
      {
        TabDocument editorForm = DockablePlugin.GetFromFixedPtr(ptr);

        if (editorForm is ObjectEditorPlugin)
          editorForm = (editorForm as ObjectEditorPlugin).ContainerForm;

        if (editorForm != null)
          editorForm.Hide();
      }
    }

    #region Edit Menu Handlers

    // Note: the EditCan* functions are called to determine the enabled state of the edit menu
    //       items. However this is always interpreted as edit actions for WB items.
    //       Hence, if a normal edit control is focused currently we have to return false.
    //       This will make the menu entries disabled unfortunately, but allows us to do clipboard
    //       operations in the edit controls.
    //       We, nonetheless, handle the actions triggered by the shortcuts (which are processed
    //       by WB).
    // TODO: edit menu update must be reworked to also take normal platform specific actions into account.
    private void EditUndo()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase)
        (activeControl as TextBoxBase).Undo();
      else
        if (activeControl is ComboBox)
          Win32Api.Undo(activeControl as ComboBox);
        else
          if (activeControl is Scintilla)
            (activeControl as Scintilla).UndoRedo.Undo();
          else
            if (wbContext.edit_can_undo())
              wbContext.edit_undo();
    }

    private bool EditCanUndo()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase || activeControl is ComboBox)
        return false;
      else
        if (activeControl is Scintilla)
          return (activeControl as Scintilla).UndoRedo.CanUndo;
        else
          return wbContext.edit_can_undo();
    }

    private void EditRedo()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase || activeControl is ComboBox)
        return;
      else
        if (activeControl is Scintilla)
          (activeControl as Scintilla).UndoRedo.Redo();
        else
          if (wbContext.edit_can_redo())
            wbContext.edit_redo();
    }

    private bool EditCanRedo()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase || activeControl is ComboBox)
        return false;
      else
        if (activeControl is Scintilla)
          return (activeControl as Scintilla).UndoRedo.CanRedo;
        else
          return wbContext.edit_can_redo();
    }


    private void EditCopy()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase)
        (activeControl as TextBoxBase).Copy();
      else
        if (activeControl is ComboBox)
          Win32Api.Copy(activeControl as ComboBox);
        else
          if (activeControl is Scintilla)
            (activeControl as Scintilla).Clipboard.Copy();
          else
            if (activeControl is GridView)
              (activeControl as GridView).Copy();
            else
              if (wbContext.edit_can_copy())
                wbContext.edit_copy();
    }

    private bool EditCanCopy()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase || activeControl is ComboBox)
        return false;
      else
        if (activeControl is Scintilla)
          return (activeControl as Scintilla).Selection.Length > 0;
        else
          if (activeControl is GridView)
            return (activeControl as GridView).CanCopy();
          else
            return wbContext.edit_can_copy();
    }

    private void EditCut()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase)
        (activeControl as TextBoxBase).Cut();
      else
        if (activeControl is ComboBox)
          Win32Api.Cut(activeControl as ComboBox);
        else
          if (activeControl is Scintilla)
              (activeControl as Scintilla).Clipboard.Cut();
            else
              if (wbContext.edit_can_copy())
                wbContext.edit_cut();
    }

    private bool EditCanCut()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase || activeControl is ComboBox)
        return false;
      else
        if (activeControl is Scintilla)
          return (activeControl as Scintilla).Selection.Length > 0;
        else
          if ((activeControl is TreeViewAdv) || (activeControl != null && activeControl.Parent is TreeViewAdv))
            return false; // Returning false tells the backend not to take over the operation.
          else
            return wbContext.edit_can_cut();
    }

    private void EditPaste()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase)
        (activeControl as TextBoxBase).Paste();
      else
        if (activeControl is ComboBox)
          Win32Api.Paste(activeControl as ComboBox);
        else
          if (activeControl is Scintilla)
            (activeControl as Scintilla).Clipboard.Paste();
          else
            if (wbContext.edit_can_paste())
              wbContext.edit_paste();
    }

    private bool EditCanPaste()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase || activeControl is ComboBox || activeControl is GridView)
        return false;
      else
        if (activeControl is Scintilla)
          return (activeControl as Scintilla).Clipboard.CanPaste;
        else
          return wbContext.edit_can_paste();
    }

    private void EditSelectAll()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase)
        (activeControl as TextBoxBase).SelectAll();
      else
        if (activeControl is Scintilla)
          (activeControl as Scintilla).Selection.SelectAll();
        else
          if (wbContext.edit_can_select_all())
            wbContext.edit_select_all();
    }

    private bool EditCanSelectAll()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase || activeControl is ComboBox)
        return false;
      else
        if (activeControl is Scintilla)
          return true;
        else
          return wbContext.edit_can_select_all();
    }

    private void EditDelete()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase)
        (activeControl as TextBoxBase).SelectedText = "";
      else
        if (activeControl is Scintilla)
          (activeControl as Scintilla).Selection.Text = "";
        else
          if (wbContext.edit_can_delete())
            wbContext.edit_delete();
    }

    private bool EditCanDelete()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is TextBoxBase || activeControl is ComboBox)
        return false;
      else
        if (activeControl is Scintilla)
          return (activeControl as Scintilla).Selection.Length > 0;
        else
          return wbContext.edit_can_delete();
    }

    private void EditFind()
    {
      // TODO: Remove once the simple search code moved to the individual pages.
    }

    private bool EditCanFind()
    {
      //return (workbenchToolbarManager.searchBox != null);
      return false; // TODO: The simple search code should now go to the individual pages.
    }

    private void EditFindReplace()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is MySQL.Grt.Db.Sql.SqlEditor)
      {
        MySQL.Grt.Db.Sql.SqlEditor editor = activeControl as MySQL.Grt.Db.Sql.SqlEditor;

        editor.ShowFindReplaceDialog();
      }
    }

    private bool EditCanFindReplace()
    {
      Control activeControl = ControlUtilities.GetLeafActiveControl(this);
      if (activeControl is MySQL.Grt.Db.Sql.SqlEditor)
        return true;
      return false;
    }

    private void SearchActiveTab()
    {
      // TODO: remove this once the code moved to the individual pages.
      /*
      String text= workbenchToolbarManager.lastSearchString;
      if (text != "")
      {
        if (!wbContext.try_searching_diagram(text))
        {
          // if active main tab is Overview, search it
          if (contentTabControl.ActiveDocument == workbenchPhysicalOverviewForm)
            workbenchPhysicalOverviewForm.SearchAndFocusNode(text);
          else
            if (contentTabControl.ActiveDocument is DbSqlEditor)
              (contentTabControl.ActiveDocument as DbSqlEditor).SearchText(text);
        }
      }
      */
    }
#endregion

    private void RegisterCommands()
    {
      List<String> commands= new List<string>();

      commands.Add("overview.mysql_model");
      commands.Add("show_about");
      commands.Add("diagram_size");
      commands.Add("view_model_navigator");
      commands.Add("view_catalog");
      commands.Add("view_layers");
      commands.Add("view_user_datatypes");
      commands.Add("view_object_properties");
      commands.Add("view_object_description");
      commands.Add("view_undo_history");
      commands.Add("reset_layout");
      commands.Add("wb.page_setup");
      commands.Add("closetab");
      commands.Add("help_version_check");
      commands.Add("wb.sidebarHide");

      wbContext.add_frontend_commands(commands);

      // special command in Windows implemented in wrapper to perform 
      // edit menu actions
      wbContext.set_edit_menu_delegates(
        EditUndo, EditCanUndo,
        EditRedo, EditCanRedo,
        EditCopy, EditCanCopy,
        EditCut, EditCanCut, EditPaste, EditCanPaste, 
        EditSelectAll, EditCanSelectAll, EditDelete, EditCanDelete,
        SearchActiveTab,
        EditFind, EditCanFind,
        EditFindReplace, EditCanFindReplace);
    }

    public void PerformCommand(String command)
    {
      if (command == "overview.mysql_model")
        ShowPhysicalOverviewForm();
      else if (command == "show_about")
      {
        new WorkbenchSplashScreen(true).ShowDialog();
      }
      else if (command == "diagram_size")
        ShowDiagramOptionsForm();
      else if (command == "reset_layout")
        ResetWindowLayout();
      else if (command == "show_find_dialog")
      {
        ShowSimpleFindForm();
      }
      else if (command == "wb.page_setup")
        ShowPageSettingsForm();
      else if (command == "closetab")
      {
        CloseActiveTab();
      }
      else if (command.Equals("help_version_check"))
        Program.CheckForNewVersion();
      else if (command.Equals("wb.sidebarHide"))
      {
        ForwardCommandToAllDocuments(command);
      }
      else
        ForwardCommandToActivDocument(command);
    }

    /// <summary>
    /// Closes the active tab page in tab controls of a specific type. That is, all tabs with entities
    /// which should be closed as a whole (e.g. model + diagram documents, object editors).
    /// </summary>
    private void CloseActiveTab()
    {
      // Start at the bottom of the hierarchy and go up. This will include also tab controls
      // from child forms (e.g. model diagram, model overview, SQL IDE etc.), so we don't need to
      // forward this close call and handle it in different places.
      Control control = ControlUtilities.GetLeafActiveControl(this);
      if (control is FlatTabControl)
        control = (control as FlatTabControl).SelectedTab;

      Control host = null;
      while (host == null)
      {
        while ((control != null) && !(control is TabPage))
          control = control.Parent;

        if (control == null)
          break;

        host = control.Parent;
        if (host is FlatTabControl)
        {
          // Reset host if the tab control is not what we are looking for and start over.
          FlatTabControl.TabStyleType style = (host as FlatTabControl).TabStyle;
          if (style == FlatTabControl.TabStyleType.TopNormal || style == FlatTabControl.TabStyleType.TopHanging)
            break;
        }
        control = host;
        host = null;
      }

      if (host != null)
        (host as FlatTabControl).CloseTabPage(control as TabPage);
    }

    private void ShowSimpleFindForm()
    {
      if (findForm == null)
      {
        findForm = new ObjectFindForm(wbContext);
      }
      findForm.ShowDialog();
    }

    private void ShowPageSettingsForm()
    {
      PageSettingsForm form = new PageSettingsForm(wbContext);

      form.ShowDialog();

      form.Dispose();
    }

    
    private void ShowDiagramOptionsForm()
    {
      DiagramOptionsForm form = new DiagramOptionsForm(wbContext);

      form.ShowDialog();

      form.Close();
      form.Dispose();
    }


    public String ShowFileDialog(String type, String title, String extensions)
    {
      FileDialog dialog;

      if (type == "save")
      {
        dialog = new SaveFileDialog();
      }
      else if (type == "open")
      {
        dialog = new OpenFileDialog();
      }
      else
        return "";

      String[] exts = extensions.Split(new char[] { ',' });

      dialog.RestoreDirectory = true;
      dialog.Title = title;
      dialog.DefaultExt = exts[0];
      String filter = "";
      foreach (String ext in exts)
      {
        if (ext.Contains("|"))
          filter = filter + ext + "|";
        else if (ext == "mwb")
          filter = filter + String.Format("{0} (*.mwb)|*.mwb|",
            "MySQL Workbench Models");
        else if (ext == "sql")
          filter = filter + String.Format("{0} (*.sql)|*.sql|",
            "SQL Script Files");
        else
          filter = filter + String.Format("{0} files (*.{0})|*.{0}|", ext);
      }
      filter = filter + String.Format("{0} (*.*)|*.*", "All files");

      dialog.Filter = filter;

      if (dialog.ShowDialog() == DialogResult.OK)
        return dialog.FileName;

      return "";
    }

    public void QuitApplication()
    {
      if (!shuttingDown)
        Close();
    }

    #endregion

    #region Standard Forms Handling

    private void grtOutputImg_Click(object sender, EventArgs e)
    {
      wbContext.show_output();
    }

    public void ShowPhysicalOverviewForm()
    {
      ShowPhysicalOverviewForm(null);
    }

    public void ShowPhysicalOverviewForm(NodeId nodeId)
    {
      if (workbenchPhysicalOverviewForm == null)
        workbenchPhysicalOverviewForm = new ModelOverviewForm(wbContext, wbContext.get_physical_overview());

      SuspendLayout();
      contentTabControl.SuspendLayout();
      try
      {
        if ((null == nodeId) || !nodeId.is_valid())
          workbenchPhysicalOverviewForm.RebuildModelContents();
        else
          workbenchPhysicalOverviewForm.RefreshNodeChildren(nodeId);

        DockDocument(workbenchPhysicalOverviewForm, true, true);
      }
      finally
      {
        // resume suspended dock layout
        contentTabControl.ResumeLayout(true);
        ResumeLayout();
      }

      workbenchPhysicalOverviewForm.Activate();
    }

    public void SetupUI()
    {
      RegisterCommands();
    }

    public void LockGUI(bool value)
    {
      if (value)
        Cursor.Current = Cursors.WaitCursor;
      else
        Cursor.Current = null;
    }

    public void ResetWindowLayout()
    {
      // Ignore.
    }

    #endregion

    #region Document Handling

    private void tabControl_ContentAdded(object sender, ControlEventArgs e)
    {
      if (e.Control is TabPage)
      {
        ITabDocument document = (sender as FlatTabControl).DocumentFromPage(e.Control as TabPage);
        if (document is ModelDiagramForm)
        {
          ModelDiagramForm form = document as ModelDiagramForm;
          ((ModelViewForm)form.BackendClass).set_closed(false);
        }
      }
    }

    private void tabControl_ContentRemoved(object sender, ControlEventArgs e)
    {
      if (e.Control is TabPage)
      {
        ITabDocument document = (sender as FlatTabControl).DocumentFromPage(e.Control as TabPage);
        if (document is ModelDiagramForm)
        {
          ModelDiagramForm form = document as ModelDiagramForm;
          ((ModelViewForm)form.BackendClass).set_closed(true);
        }
      }
    }

    private void tabControl_SelectedIndexChanged(object sender, EventArgs e)
    {
      // Reset status bar to clean up any previously done action message.
      StatusBarText = "Ready";

      // For the glass effect behind the tabs (and the proper content drawing) we need the
      // height of the tabs as well as menu and toolbar of the active tab.
      int topAreaHeight = 0;

      // Suspend dock layout to prevent flicker.
      SuspendLayout();
      contentTabControl.SuspendLayout();
      try
      {
        FlatTabControl tabControl = sender as FlatTabControl;
        if (tabControl.ActiveDocument != null)
        {
          ITabDocument activeDocument = tabControl.ActiveDocument;
          topAreaHeight += activeDocument.ToolbarHeight;

          if (activeDocument is IWorkbenchDocument)
          {
            IWorkbenchDocument wbDoc = activeDocument as IWorkbenchDocument;

            // If the current View is changed, update ModelCatalogForm
            if (activeDocument is ModelDiagramForm)
            {
              ModelDiagramForm form = activeDocument as ModelDiagramForm;

              wbContext.set_active_form(form.BackendClass);

              form.FocusCanvasControl();
            }
            else
            {
              wbContext.set_active_form(wbDoc.BackendClass);  
            }
          }
          else if (activeDocument is Plugins.DockablePlugin)
          {
            Plugins.DockablePlugin form = activeDocument as Plugins.DockablePlugin;
            wbContext.set_active_form(form.BackendClass);
          }
          else if (activeDocument is UIForm)
          {
            wbContext.set_active_form(activeDocument as UIForm);
          }
          else if (activeDocument is MySQL.Forms.AppViewDockContent)
          {
            MySQL.Forms.AppViewDockContent form = activeDocument as MySQL.Forms.AppViewDockContent;

            wbContext.set_active_form_from_appview(form);
          }
          else
            wbContext.set_active_form(null);
        }
        else
          wbContext.set_active_form(null);
      }
      finally
      {
        // Resume suspended dock layout.
        contentTabControl.ResumeLayout(true);
        ResumeLayout();
      }

      // Different pages might have a different height of the menu/toolbar area.
      AdjustGlassFrame(topAreaHeight);
    }

    /// <summary>
    /// Remove plugin form from WbContext editor cache.
    /// </summary>
    /// <param name="sender">The form</param>
    /// <param name="e">The close event arguments</param>
    internal void PluginFormClosed(object sender, FormClosedEventArgs e)
    {
      DockablePlugin plugin = sender as DockablePlugin;

      if (plugin != null)
      {
        wbContext.close_gui_plugin(plugin.GetFixedPtr());
        plugin.ReleaseHandle();
      }

      Program.CollectGarbageOnIdle();
    }

    private void ForwardRefreshToActivDocument(RefreshType refresh, String str, IntPtr ptr)
    {
      if (contentTabControl.ActiveDocument is IWorkbenchDocument)
        (contentTabControl.ActiveDocument as IWorkbenchDocument).RefreshGUI(refresh, str, ptr);
    }

    private void ForwardRefreshToAllDocuments(RefreshType refresh, String str, IntPtr ptr)
    {
      foreach (ITabDocument document in contentTabControl.Documents)
        if (document is IWorkbenchDocument)
          (document as IWorkbenchDocument).RefreshGUI(refresh, str, ptr);
    }

    private void ForwardCommandToActivDocument(String command)
    {
      if (contentTabControl.ActiveDocument is IWorkbenchDocument)
        (contentTabControl.ActiveDocument as IWorkbenchDocument).PerformCommand(command);
    }

    private void ForwardCommandToAllDocuments(String command)
    {
      foreach (ITabDocument document in contentTabControl.Documents)
        if (document is IWorkbenchDocument)
          (document as IWorkbenchDocument).PerformCommand(command);
    }

    /// <summary>
    /// Docks the given document to either the own tab control (as main document) or as
    /// editor window to the currently active main document.
    /// </summary>
    /// <param name="document">The document to dock.</param>
    /// <param name="main">If true then dock that document to our own tab control, otherwise
    /// dock it to the currently active main document.</param>
    private void DockDocument(ITabDocument document, bool main, bool activate)
    {
      if (main)
      {
        int index = -1;
        if (!contentTabControl.HasDocument(document))
        {
          IWorkbenchDocument wbDocument = document as IWorkbenchDocument;
          SetupDockLayout(document, (wbDocument != null) ? wbDocument.BackendClass : null);
          index = contentTabControl.AddDocument(document);

          InstallFocusChangeHandlers(document as Control);
        }

        if (index == 0) // The main tab.
        {
          contentTabControl.TabPages[index].ImageIndex = 0;
          contentTabControl.SetCloseButtonVisibility(0, FlatTabControl.CloseButtonVisiblity.HideButton);
        }
        
        if (activate)
        {
          document.Activate();
          AdjustGlassFrame(document.ToolbarHeight);
        }
      }
      else
      {
        ITabDocument host = contentTabControl.ActiveDocument;
        if (host is ModelDiagramForm)
        {
          ModelDiagramForm form = host as ModelDiagramForm;
          form.DockDocument(document, activate);
        }
        else
          if (host is ModelOverviewForm)
          {
            ModelOverviewForm form = host as ModelOverviewForm;
            form.DockDocument(document, activate);
          }
      }
    }

    /// <summary>
    /// Undocks the given document from our tab control if it is there.
    /// If not forward the call to the currently active document.
    /// </summary>
    /// <param name="document"></param>
    private void UndockDocument(ITabDocument document)
    {
      if (contentTabControl.HasDocument(document))
        contentTabControl.RemoveDocument(document);
      else
      {
        ITabDocument host = contentTabControl.ActiveDocument;
        if (host is ModelDiagramForm)
          (host as ModelDiagramForm).UndockDocument(document);
        else
          if (host is ModelOverviewForm)
            (host as ModelOverviewForm).UndockDocument(document);
      }
    }

    #endregion

    #region UI Event Handling

    private void spacerPanel_Paint(object sender, PaintEventArgs e)
    {
      Graphics g = e.Graphics;

      g.DrawLine(Pens.White, e.ClipRectangle.Left, 1, e.ClipRectangle.Right - 1, 1);
    }

    private void MainForm_KeyDown(object sender, KeyEventArgs e)
    {
      //wbContext.handle_key_event(true, e);
    }

    private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
    {
      shuttingDown = true;
      Hide();

      e.Cancel = !wbContext.request_quit();
      ITabDocument[] documents = contentTabControl.DocumentsToArray();

      /* This code is doing work which is already done in request_quit(), so it can be 
       * removed once we can be sure that works in any circumstance.
       *
      if (!e.Cancel)
      {
        foreach (ITabDocument document in documents)
        {
          if (document is DbSqlEditor)
          {
            DbSqlEditor dbSqlEditor = document as DbSqlEditor;
            e.Cancel = !dbSqlEditor.Backend.can_close();
            if (e.Cancel)
              break;
          }
        }
      }

      if (!e.Cancel && wbContext.has_unsaved_changes())
      {
        DialogResult result = MessageBox.Show("Do you want to save changes to the document before closing?\n\n" +
                        "If you don't save your changes, they will be lost.",
                        "Close Workbench", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Exclamation);
        if (result == DialogResult.Cancel)
          e.Cancel = true;
        else 
          if (result != DialogResult.No && !wbContext.save_changes())
            e.Cancel = true;
      }
      */

      // Reset termination flag, in case it was set.
      if (e.Cancel)
      {
        grtManager.resetTermination();
        shuttingDown = false;
        return;
      }

      RemoveFocusChangeHandlers(this);
      SaveFormState();

      // Go through each workbench document and close it unconditionally.
      contentTabControl.SuspendLayout();
      try
      {
        // Close all documents (editors implement the document interface too, so we get them
        // with that loop as well).
        for (int i = documents.Length - 1; i >= 0; i--)
        {
          // WBA is closed via the appview's DocumentClosing() function not by normal closing.
          if (documents[i] is MySQL.Forms.AppViewDockContent)
          {
            MySQL.Forms.AppViewDockContent content = documents[i] as MySQL.Forms.AppViewDockContent;
            content.GetAppView().DocumentClosing();
          }
          else
            documents[i].Close();
        }
      }
      finally
      {
        contentTabControl.ResumeLayout();
      }
      
      wbContext.perform_quit();
    }

    private void LoadFormState()
    {
      int x = wbContext.read_state("left", "mainform", 100);
      int y = wbContext.read_state("top", "mainform", 100);
      int w = wbContext.read_state("width", "mainform", 1200);
      int h = wbContext.read_state("height", "mainform", 900);

      // Sanity checks to avoid restoring the application into a state where it cannot be reached.
      if (x + w < 100)
        x = 100 - w;
      if (y + h < 100)
        y = 100 - h;
      Rectangle workingArea= Screen.GetWorkingArea(new Point(x, y));
      if (x > workingArea.Right - 100)
        x = workingArea.Right - 100;
      if (y > workingArea.Bottom - 100)
        y = workingArea.Bottom - 100;

      // First set the bounds then the state. If the window is maximized this will properly set
      // RestoreBounds so un-maximizing will later do the right job.
      Bounds = new Rectangle(x, y, w, h);
      WindowState = (FormWindowState)wbContext.read_state("windowstate", "mainform", (int)FormWindowState.Normal);

      // Don't restore into minimized state. Confusing for the user and makes trouble for layouting.
      if (WindowState == FormWindowState.Minimized)
        WindowState = FormWindowState.Normal;

      /* Disabled by order.
      if (wbContext.read_state("outputvisible", "mainform", 0) == 1)
        wbContext.show_output();
      */
    }

    private void SaveFormState()
    {
      wbContext.save_state("windowstate", "mainform", (int)WindowState);

      if (WindowState == FormWindowState.Normal)
      {
        wbContext.save_state("left", "mainform", Left);
        wbContext.save_state("top", "mainform", Top);
        wbContext.save_state("width", "mainform", Width);
        wbContext.save_state("height", "mainform", Height);
      }
      else
      {
        wbContext.save_state("left", "mainform", RestoreBounds.Left);
        wbContext.save_state("top", "mainform", RestoreBounds.Top);
        wbContext.save_state("width", "mainform", RestoreBounds.Width);
        wbContext.save_state("height", "mainform", RestoreBounds.Height);
      }

      bool outputVisible = false;
      foreach (ITabDocument content in contentTabControl.Documents)
      {
        if (content.TabText == "Output")
        {
          outputVisible = true;
          break;
        }
      }
      wbContext.save_state("outputvisible", "mainform", outputVisible ? 1 : 0);
    }

    private void MainForm_Load(object sender, EventArgs e)
    {
      LoadFormState();
      InstallFocusChangeHandlers(this);

      SplashScreen.CloseSplash();

      BringToFront();
      Activate();
    }


    public void UpdateTimer()
    {
      int interval = (int)(1000 * wbContext.delay_for_next_timer());
      if (interval > 0)
      {
        timer.Interval = interval;
        timer.Start();
      }
      else if (interval < 0)
        timer.Stop();
      else
      {
        timer.Interval = 1;
        timer.Start();
      }
    }


    private void timer_Tick(object sender, EventArgs e)
    {
      wbContext.flush_timers();
      UpdateTimer();
    }


    private void FocusChanged(object sender, EventArgs e)
    {
      // force a refresh of the menu so that Edit -> Copy/Paste/Cut/Select All can get updated accordingly
      if (!shuttingDown)
        wbContext.validate_edit_menu();
    }

    private EventHandler FocusChangeHandler= null;


    private void RemoveFocusChangeHandlers(Control control)
    {
      if (FocusChangeHandler == null)
        return;

      // Remove focus change handler we set previously.
      control.GotFocus -= FocusChangeHandler;

      foreach (Control sub in control.Controls)
        RemoveFocusChangeHandlers(sub);
    }

    private void InstallFocusChangeHandlers(Control control)
    {
      if (FocusChangeHandler == null)
        FocusChangeHandler= new EventHandler(FocusChanged);

      // Recursively install event handler for GotFocus and LostFocus.
      control.GotFocus -= FocusChangeHandler;
      control.GotFocus += FocusChangeHandler;

      foreach  (Control sub in control.Controls)
        InstallFocusChangeHandlers(sub);
    }

    #endregion

    private void tabControl_TabClosing(object sender, TabClosingEventArgs e)
    {
      e.canClose = true;

      ITabDocument document = (sender as FlatTabControl).DocumentFromPage(e.page);
      if (document is DbSqlEditor)
        e.canClose = (document as DbSqlEditor).Backend.can_close();
      if (document is Plugins.DockablePlugin)
      {
        Plugins.DockablePlugin plugin = document as Plugins.DockablePlugin;
        plugin.RunOnFormClosing(null);
      }
      else
        if (document is IWorkbenchDocument)
          e.canClose = (document as IWorkbenchDocument).BackendClass.can_close();
        else
          if (document is MySQL.Forms.AppViewDockContent)
          {
            MySQL.Forms.AppViewDockContent content = document as MySQL.Forms.AppViewDockContent;
            e.canClose = content.GetAppView().DocumentClosing();
          }

      if (e.canClose)
        RemoveFocusChangeHandlers(document as Control);

      // Because of the order of processing in the UI we have to anticipate notification handling.
      // It is too late when the regular messages come in (then the page for which they are meant
      // is already gone, leaving so for instance plugins registered, which don't exist anymore).
      if (e.canClose && document is ModelOverviewForm)
        (document as ModelOverviewForm).RefreshGUI(RefreshType.RefreshCloseEditor, "", IntPtr.Zero);
    }

    private void contentTabControl_TabClosed(object sender, TabClosedEventArgs e)
    {
      ITabDocument document = (sender as FlatTabControl).DocumentFromPage(e.page);
      if (document is DbSqlEditor)
        document.Close();
      else if (document is Plugins.DockablePlugin)
        PluginFormClosed(sender, null);
      else
        if (document is IWorkbenchDocument)
          (document as IWorkbenchDocument).BackendClass.close();
        else
          if (document is MySQL.Forms.AppViewDockContent)
          {
            // Remove the appview from the page, otherwise it gets disposed with the page,
            // what we don't want.
            e.page.Controls.Clear();
          }
    }

    /// <summary>
    /// Restores the form to a usable state and activates it.
    /// </summary>
    public void Activate(bool restoreIfMinimized)
    {
      if (restoreIfMinimized && WindowState == FormWindowState.Minimized)
        WindowState = FormWindowState.Normal;
      Activate();
    }

    /// <summary>
    /// Make the upper part of the window extend the glass frame (if we are on Aero).
    /// </summary>
    /// <param name="toolbarHeight"></param>
    private void AdjustGlassFrame(int toolbarHeight)
    {
      if (ControlUtilities.IsCompositionEnabled())
      {
        int topAreaHeight = contentTabControl.ItemSize.Height + contentTabControl.Margin.Top +
          toolbarHeight + Margin.Top;

        Win32.MARGINS margins = new Win32.MARGINS(2, topAreaHeight, 2, 0);
        Win32.DwmExtendFrameIntoClientArea(Handle, margins);
      }
    }

    /// <summary>
    /// Prepares the given document to conform to the overall layout. This requires to draw an
    /// intermediate set of controls (panels) and add menu + toolbar.
    /// </summary>
    /// <param name="document">The document to adjust, which comes with the actual content.</param>
    /// <param name="backend">The backend UI form used to get toolbar and menu.</param>
    private void SetupDockLayout(ITabDocument document, UIForm backend)
    {
      // Setting up the layout requires some additional nesting of controls.
      // A tab document should have only one container as top level control, which we re-host.
      Control root = document.Content;
      Control content = (root.Controls.Count > 0) ? root.Controls[0] : null;

      root.Controls.Clear();

      // Content area.
      DrawablePanel contentPanel = new DrawablePanel();
      contentPanel.BackColor = Color.FromArgb(255, 40, 55, 82);
      contentPanel.CustomBackground = false;
      contentPanel.Dock = DockStyle.Fill;
      contentPanel.Padding = new Padding(7);
      root.Controls.Add(contentPanel);
      if (content != null)
      {
        contentPanel.Controls.Add(content);
        content.Dock = DockStyle.Fill;
      }

      // Menu + toolbar area. We have two types of tab documents:
      // - appview based (and hence backend-managed)
      // - native Form (with interface via WBContext).
      // Appview based documents have own methods to return their menu and toolbar because
      // retrieval via managed code requires a managed UIForm. However appview classes are
      // already managed via mforms and adding another reference management via UIForm
      // will lead to unpredictable results.
      ToolStrip toolbar;
      MenuStrip menu;
      if (backend != null)
      {
        // Retrieval via WBContext will return a default toolbar if no other is defined.
        toolbar = wbContext.toolbar_for_form(backend);
        menu = wbContext.menu_for_form(backend);
      }
      else
      {
        AppViewDockContent dockContent = document as AppViewDockContent;
        toolbar = dockContent.GetAppView().GetToolBar();
        if (toolbar == null)
          toolbar = wbContext.toolbar_for_form(null);
        menu = dockContent.GetAppView().GetMenuBar();
        if (menu == null)
          menu = wbContext.menu_for_form(null);
      }
      DrawablePanel menuPanel = new DrawablePanel();
      if (toolbar != null || menu != null)
      {
        // For menu + toolbar we have a special gradient background, which we implement by
        // placing them on another panel.
        menuPanel.BackColor = Color.Transparent;
        menuPanel.CustomBackground = true;
        menuPanel.Dock = DockStyle.Top;
        menuPanel.Padding = new Padding(5, 0, 5, 0);
        menuPanel.AutoSize = true;
        menuPanel.PaintBackground += new EventHandler<PaintEventArgs>(menuPanel_PaintBackground);

        root.Controls.Add(menuPanel);
        if (toolbar != null)
        {
          menuPanel.Controls.Add(toolbar);
          toolbar.Dock = DockStyle.Top;
          toolbar.AutoSize = true;
          toolbar.Padding = new Padding(0, 0, 0, 3);
        }

        if (menu != null)
        {
          menuPanel.Controls.Add(menu);
          menu.Dock = DockStyle.Top;
          menu.AutoSize = true;
        }
      }
    }
    
    void menuPanel_PaintBackground(object sender, PaintEventArgs e)
    {
      ToolStripHelper.MenuPanelPaint(e.Graphics, (sender as Control).ClientRectangle);
    }

  }
}
