/* 
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "base/log_impl.h"
#include <string>
#include <stdio.h>
#include <stdarg.h>
#include "base/wb_memory.h"
#include <time.h>

static const char* LevelText[] = {"", "Error", "Warning", "Info", "Debug1", "Debug2", "Debug3"};

//=============================================================================
//
//=============================================================================
struct base::Logger::LoggerImpl
{
  LoggerImpl()
  {
    for (int i = 0; i < base::Logger::NumOfLevels + 1; ++i)
      _levels[i] = true;
    _levels[0] = false; // Disable None level
#ifndef DEBUG
    // Disable debug levels for production
    _levels[LogDebug]  = false;
    _levels[LogDebug2] = false;
    _levels[LogDebug3] = false;
#endif
  }

  bool level_is_enabled(const base::Logger::LogLevel level) const
  {
    return _levels[level];
  }

  std::string       _filename;
  bool              _levels[base::Logger::NumOfLevels + 1];
};

base::Logger::LoggerImpl*  base::Logger::_impl = 0;

//-----------------------------------------------------------------------------
base::Logger::Logger(const std::string& dir)
{
  if (!_impl)
    _impl = new base::Logger::LoggerImpl();

  // Currently we hardcode filename
  _impl->_filename = dir + "/wb.log";
  // truncate log file we do not need gigabytes of logs
  base::FILE_scope_ptr fp = fopen(_impl->_filename.c_str(), "w");
}

//-----------------------------------------------------------------------------
void base::Logger::enable_level(const LogLevel level)
{
  if (level <= NumOfLevels)
    _impl->_levels[level] = true;
}

//-----------------------------------------------------------------------------
void base::Logger::disable_level(const LogLevel level)
{
  if (level <= NumOfLevels)
    _impl->_levels[level] = false;
}

//-----------------------------------------------------------------------------
void base::Logger::log(const base::Logger::LogLevel level, const char* const domain, const char* format, ...)
{
  const base::Logger::LogLevel lvl = (level >= 0 && level <= NumOfLevels) ? level : (base::Logger::LogLevel)0;
  if (_impl && _impl->level_is_enabled(lvl))
  {
    base::FILE_scope_ptr fp = fopen(_impl->_filename.c_str(), "a");
    if (fp)
    {
      const time_t t = time(NULL);
      struct tm tm;
      #ifdef _WIN32
      localtime_s(&tm, &t);
      #else
      localtime_r(&t, &tm);
      #endif
      fprintf(fp, "%02u:%02u:%02u [%7s][%15s]: ", tm.tm_hour, tm.tm_min, tm.tm_sec, LevelText[lvl], domain);
      va_list args;
      va_start(args, format);
      vfprintf(fp, format, args);
      va_end(args);
      fputs("\n", fp);
      
#ifdef DEBUG
      fprintf(stderr, "%02u:%02u:%02u [%7s][%15s]: ", tm.tm_hour, tm.tm_min, tm.tm_sec, LevelText[lvl], domain);
      va_start(args, format);
      vfprintf(stderr, format, args);
      va_end(args);
      fprintf(stderr, "\n");
#endif
    }
  }
}

//-----------------------------------------------------------------------------
std::string base::Logger::get_state()
{
  std::string state = "";
  if (_impl)
  {
    for (int i = 0; i < base::Logger::NumOfLevels + 1; ++i)
    {
      state += _impl->level_is_enabled((base::Logger::LogLevel)i) ? "1" : "0";
    }
  }
  return state;
}

//-----------------------------------------------------------------------------
void base::Logger::set_state(const std::string& state)
{
  if (_impl && state.length() >= base::Logger::NumOfLevels)
  {
    for (int i = 0; i < base::Logger::NumOfLevels + 1; ++i)
    {
      const char level = state[i];
      if (level == '1')
      {
        base::Logger::enable_level((base::Logger::LogLevel)i);
      }
      else if (level == '0')
      {
        base::Logger::disable_level((base::Logger::LogLevel)i);
      }
    }
  }
}
