//
//  MySQLWorkbench
//
//  Created by Alfredo Kojima on 5/Feb/09.
//  Copyright 2009 Sun Microsystems Inc. All rights reserved.
//

#import "MFDrawBox.h"
#include "mforms/mforms.h"
#include <cairo/cairo-quartz.h>

@implementation MFDrawBoxImpl

- (id)initWithObject:(::mforms::DrawBox*)aBox
{
  self= [super initWithFrame:NSMakeRect(10,10,10,10)];
  if (self)
  {
    mOwner= aBox;
    mOwner->set_data(self);
    mTag= 0;
  }
  return self;
}

- (void) dealloc
{
  [super dealloc];
}



- (mforms::Object*)mformsObject
{
  return mOwner;
}

- (BOOL)isFlipped
{
  return YES;
}

- (void)setTag:(NSInteger)tag
{
  mTag= tag;
}


- (NSInteger)tag
{
  return mTag;
}

- (void)drawRect:(NSRect)rect
{
  NSRect frame = [self frame];

  CGContextRef cgref = (CGContextRef)[[NSGraphicsContext currentContext] graphicsPort];

  cairo_surface_t *surface= cairo_quartz_surface_create_for_cg_context(cgref, NSWidth(frame), NSHeight(frame));
  
  cairo_t *cr = cairo_create(surface);
  try
  {
    mOwner->repaint(cr, rect.origin.x, rect.origin.y, rect.size.width, rect.size.height);
  }
  catch (...)
  {
    cairo_destroy(cr);
    cairo_surface_destroy(surface);
    throw;
  }
  cairo_destroy(cr);
  cairo_surface_destroy(surface);
}

//--------------------------------------------------------------------------------------------------

/**
 * Helper to avoid code duplication.
 */
- (void) doMouseUp: (NSEvent*) event
{
  NSPoint p = [self convertPointFromBase: [event locationInWindow]];
  NSInteger button= [event buttonNumber];
  switch ([event clickCount])
  {
    case 1:
      mOwner->mouse_click(button, p.x, p.y);
      break;
    case 2:
      mOwner->mouse_double_click(button, p.x, p.y);
      break;
    default:
      mOwner->mouse_up(button, p.x, p.y);
      break;
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * Helper to avoid code duplication.
 */
- (void) doMouseDown: (NSEvent*) event
{
  NSPoint p = [self convertPointFromBase: [event locationInWindow]];
  NSInteger button= [event buttonNumber];
  mOwner->mouse_down(button, p.x, p.y);
}

//--------------------------------------------------------------------------------------------------

- (void)mouseDown:(NSEvent*)event
{
  [self doMouseDown: event];
}

//--------------------------------------------------------------------------------------------------

- (void)mouseUp:(NSEvent*)event
{
  [self doMouseUp: event];
}

//--------------------------------------------------------------------------------------------------

- (void)otherMouseDown:(NSEvent*)event
{
  [self doMouseDown: event];
}

//--------------------------------------------------------------------------------------------------

- (void)otherMouseUp:(NSEvent*)event
{
  [self doMouseUp: event];
}

//--------------------------------------------------------------------------------------------------

- (void)rightMouseDown:(NSEvent*)event
{
  [self doMouseDown: event];
}

//--------------------------------------------------------------------------------------------------

- (void)rightMouseUp:(NSEvent*)event
{
  [self doMouseUp: event];
}

//--------------------------------------------------------------------------------------------------

- (void)mouseMoved:(NSEvent *)event
{
  NSPoint p = [self convertPointFromBase: [event locationInWindow]];
  mOwner->mouse_move(p.x, p.y);
}

//--------------------------------------------------------------------------------------------------

- (void)mouseEntered:(NSEvent *)event
{
  mOwner->mouse_enter();
}

//--------------------------------------------------------------------------------------------------

- (void)mouseExited:(NSEvent *)event
{
  mOwner->mouse_leave();
}

//--------------------------------------------------------------------------------------------------

- (BOOL)acceptsFirstResponder
{
  return YES;
}

//--------------------------------------------------------------------------------------------------

- (void) updateTrackingAreas
{
  [super updateTrackingAreas];

  // Create one tracking area which covers the whole control and make it get mouse events
  // only when the mouse is within the area but regardless of the first responder status.
  [self removeTrackingArea: mTrackingArea];
  [mTrackingArea release];
  
  NSTrackingAreaOptions options= NSTrackingMouseEnteredAndExited | NSTrackingMouseMoved | 
                                 NSTrackingCursorUpdate | NSTrackingActiveInActiveApp |
                                 NSTrackingInVisibleRect;
  mTrackingArea= [[NSTrackingArea alloc] initWithRect: [self bounds] options: options owner: self userInfo: nil];
  [self addTrackingArea: mTrackingArea];
}

//--------------------------------------------------------------------------------------------------

- (void) invalidate
{
  [self setNeedsDisplay: YES];
}

- (NSSize)preferredSize
{
  int w, h;
  mOwner->get_layout_size(&w, &h);
  NSSize size= NSMakeSize(w, h);
  
  // If a fixed size is set honour that but don't go below the
  // minimal required size.
  if ([self widthIsFixed])
    size.width= MAX(size.width, NSWidth([self frame]));
  if ([self heightIsFixed])
    size.height= MAX(size.height, NSHeight([self frame]));
  return size;
}

static bool drawbox_create(::mforms::DrawBox *self)
{
  [[[MFDrawBoxImpl alloc] initWithObject:self] autorelease];
  
  return true;  
}

static void drawbox_set_needs_repaint(::mforms::DrawBox *self)
{
  // Invalidate the draw box in a thread safe manner.
  // See also http://developer.apple.com/mac/library/documentation/Cocoa/Conceptual/Multithreading/ThreadSafetySummary/ThreadSafetySummary.html
  
  [self->get_data() performSelectorOnMainThread: @selector(invalidate)
                                    withObject: nil
                                  waitUntilDone: NO];
}


void cf_drawbox_init()
{
  ::mforms::ControlFactory *f = ::mforms::ControlFactory::get_instance();
  
  f->_drawbox_impl.create= &drawbox_create;
  f->_drawbox_impl.set_needs_repaint= &drawbox_set_needs_repaint;
}

@end


