/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#import "WbModelStoredNoteEditor.h"
#import "MCPPUtilities.h"
#include "base/string_utilities.h"

@implementation StoredNoteEditor

static void call_refresh(StoredNoteEditor *self)
{
  [self refresh];
}


- (id)initWithModule:(grt::Module*)module GRTManager:(bec::GRTManager*)grtm arguments:(const grt::BaseListRef&)args
{
  self= [super initWithModule:module GRTManager:grtm arguments:args];
  if (self != nil)
  {
    // load GUI. Top level view in the nib is the NSTabView that will be docked to the main window
    if (![[NSBundle bundleForClass:[self class]] loadNibFile:@"WbModelStoredNoteEditor"
                                           externalNameTable:[NSDictionary dictionaryWithObject:self forKey:NSNibOwner] 
                                                    withZone:nil])
      NSLog(@"Could not load WbModelStoredNoteEditor.xib");
    
    // take the minimum size of the view from the initial size in the nib.
    // Therefore the nib should be designed as small as possible
    // note: the honouring of the min size is not yet implemented
    [self setMinimumSize: [tabView frame].size];
    
    [self reinitWithArguments: args];
  }
  return self;
}


- (void)reinitWithArguments:(const grt::BaseListRef&)args
{
  [super reinitWithArguments: args];
  delete mBackEnd;
  
    // setup the editor backend with the note object (args[0])
  mBackEnd= new StoredNoteEditorBE(_grtm, GrtStoredNoteRef::cast_from(args[0]));
 
  if (db_ScriptRef::can_wrap(args[0]))
      _editingSQL = YES;
  // register a callback that will make [self refresh] get called
  // whenever the backend thinks its needed to refresh the UI from the backend data (ie, the
  // edited object was changed from somewhere else in the application)
  mBackEnd->set_refresh_ui_slot(boost::bind(call_refresh, self));
  [self setupEditor]; 
      
  // update the UI
  [self refresh];
}


- (void) dealloc
{
  delete mBackEnd;
  [super dealloc];
}


/** Fetches object info from the backend and update the UI
 */
- (void)refresh
{
  if (mBackEnd)
  {
    bool isutf8= true;
    std::string data= mBackEnd->get_text(isutf8);
    
    if (isutf8)
    {
      [noteText setString: [NSString stringWithCPPString:data]];
      [noteText setEditable: YES];
    }
    else
    {
      [noteText setEditable: NO];
      [noteText setString: NSLocalizedString(@"Data is not UTF8 and cannot be displayed.", @"attached file editor")];
    }
    [applyButton setEnabled: NO];
    [revertButton setEnabled: NO];
  }
}


- (id)identifier
{
  // an identifier for this editor (just take the object id)
  return [NSString stringWithCPPString:mBackEnd->get_object().id()];
}


- (NSString*)title
{
  // the title for the editor
  return [NSString stringWithCPPString: base::strfmt("%s - Attached File", mBackEnd->get_name().c_str())];
}


- (NSView*)dockableView
{
  // the view to be docked to the main window
  return tabView;
}


- (BOOL)matchesIdentifierForClosingEditor:(NSString*)identifier
{
  return mBackEnd->should_close_on_delete_of([identifier UTF8String]);
}


- (IBAction)loadText:(id)sender
{
  NSOpenPanel *panel= [NSOpenPanel openPanel];
  
  [panel setTitle: NSLocalizedString(@"Load from File", @"loadText from file dialog")];
  if ([panel runModal] == NSOKButton)
  {
    mBackEnd->load_file([[panel filename] UTF8String]);
    [self refresh];
  }
}


- (IBAction)saveText:(id)sender
{
  NSSavePanel *panel= [NSSavePanel savePanel];
  
  [panel setTitle: NSLocalizedString(@"Save to File", @"saveText to file dialog")];
  if ([panel runModal] == NSOKButton)
  {
    mBackEnd->save_file([[panel filename] UTF8String]);
  }
}


- (IBAction)applyChanges:(id)sender
{
  NSString *text = [noteText string];

  mBackEnd->set_text(text ? [text UTF8String] : "");
}


- (IBAction)revertChanges:(id)sender
{
  [revertButton setEnabled: NO];
  [self refresh];
}

- (void)textDidChange:(NSNotification *)aNotification
{
  // Stop the timer in case it is about to trigger.
  [mSyntaxCheckTimer invalidate];
  
  // Set up a new timer.
  mSyntaxCheckTimer= [NSTimer scheduledTimerWithTimeInterval: 0.5
                                                      target: self
                                                    selector: @selector(checkSyntax:)
                                                    userInfo: nil
                                                     repeats: NO];
  if ([aNotification object] == noteText)
  {
    [applyButton setEnabled: YES];
    [revertButton setEnabled: YES];
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * Callback from the backend telling us about syntax errors it founds.
 */
static int process_syntax_error(const int line, const int err_tok_line_pos, const int err_tok_len,
                                const std::string& err_msg, StoredNoteEditor* self)
{
  // If this is the first new error then reset all error markers.
  // Don't do this in advance in checkSyntax, otherwise we get some ugly flickering.
  if (self->mErrorCount == 0)
    [self resetSyntaxErrors];
  
  Scintilla::ScintillaCocoa *backend= self->noteText.backend;
  int line_start_pos= backend->WndProc(SCI_POSITIONFROMLINE, line - 1, 0);
  
  backend->WndProc(SCI_SETINDICATORCURRENT, 0, 0);
  backend->WndProc(SCI_INDICATORFILLRANGE, line_start_pos + err_tok_line_pos, err_tok_len);
  
  backend->WndProc(SCI_MARKERADD, line - 1, 0);
  
  ++self->mErrorCount;
  
  return 0;
}

//--------------------------------------------------------------------------------------------------

/**
 * Prepares the code editor and the backend to have proper syntax highlighting, error parsing etc.
 */
- (void) setupEditor
{
  [[NSNotificationCenter defaultCenter] removeObserver: self
                                                  name: NSTextDidChangeNotification
                                                object: noteText];
  
  [[NSNotificationCenter defaultCenter] addObserver: self
                                           selector: @selector(textDidChange:)
                                               name: NSTextDidChangeNotification
                                             object: noteText];
  
  if (_editingSQL)
  {
    [WBPluginEditorBase setupCodeEditor: noteText backend: mBackEnd->get_sql_editor() withStatus: YES];
    [noteText setStatusText: @""/*"No errors found"*/];
  

    // Connect the parser with our callback to get notifications about syntax errors.
    mBackEnd->get_sql_editor()->sql_parser_err_cb(boost::bind(process_syntax_error, _1, _2, _3, _4, self));
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * Remove all markers we set for previous syntax errors.
 */
- (void) resetSyntaxErrors
{
  [noteText setStatusText: @""/*"No errors found"*/];
  int length= noteText.backend->WndProc(SCI_GETLENGTH, 0, 0);
  
  [noteText setGeneralProperty: SCI_SETINDICATORCURRENT parameter: 0 value: 0];
  [noteText setGeneralProperty: SCI_INDICATORCLEARRANGE parameter: 0 value: length];
  
  [noteText setGeneralProperty: SCI_MARKERDELETEALL parameter: -1 value: 0];
}

//--------------------------------------------------------------------------------------------------

/**
 * Triggered once a change in the editor content happend and there was a pause of at least 500 ms.
 */
- (void) checkSyntax:(NSTimer*) timer
{
  mSyntaxCheckTimer = nil;
  NSString *text;
  
  text= [noteText string];
  if (text)
  {
    [noteText setStatusText: @"Parsing SQL ..."];
    mBackEnd->get_sql_editor()->sql([text UTF8String]);
    
    mErrorCount = 0;
    mBackEnd->get_sql_editor()->check_sql(true);
    
    // If there was no syntax error then reset any remaining error markers now.
    if (mErrorCount == 0)
      [self resetSyntaxErrors];
    else
      [noteText setStatusText: [NSString stringWithFormat: @"%d error(s) found.", mErrorCount]];
  }
  else
    [self resetSyntaxErrors];
}

//--------------------------------------------------------------------------------------------------

@end
