/* 
 * Copyright (c) 2009, 2012, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using System.Text;

using MySQL.Controls;
using MySQL.Grt;
using MySQL.Grt.Db;
using MySQL.Grt.Db.Sql;
using MySQL.Utilities;
using MySQL.Workbench;
using MySQL.Forms;

using ScintillaNet;

namespace MySQL.GUI.Workbench
{
  public partial class DbSqlEditor : Plugins.DockablePlugin, IWorkbenchObserver
  {
    static int InstanceCount = 0;

    private List<RecordsetView> pendingRelayouts = new List<RecordsetView>();
    private SqlEditor currentlyRunning = null; // The editor which is currently executing a query.

    private SqlEditorFormWrapper dbSqlEditorBE;
    public SqlEditorFormWrapper Backend { get { return dbSqlEditorBE; } }

    protected WbContext wbContext;

    private bool canTrackChanges = false; // True when initialization is done and UI changes can be tracked.

    #region Initialization

    public DbSqlEditor(WbContext wbContext, UIForm uiForm)
    {
      InstanceCount++;

      this.wbContext = wbContext;
      dbSqlEditorBE = uiForm as SqlEditorFormWrapper;
      Initialize();
    }

    protected void Initialize()
    {
      Logger.LogInfo("WQE.net", "Launching SQL IDE\n");

      InitializeComponent();

      if (InstanceCount == 1)
        RegisterCommands();

      grtManager = dbSqlEditorBE.grt_manager();

      outputSelector.SelectedIndex = 0;

      TabText = dbSqlEditorBE.caption();

      sqlEditors = new List<SqlEditor>();
      for (int i = 0; i < dbSqlEditorBE.sql_editor_count(); i++)
        AddSqlEditor(i);
      dbSqlEditorBE.sql_editor_new_ui_cb(OnSqlEditorNew);

      logView = new GridView(dbSqlEditorBE.log());
      logView.AutoScroll = true;
      logView.RowHeadersVisible = false;
      logView.Parent = actionPanel;
      logView.AllowAutoResizeColumns = false;

      // Some visual setup of the header.
      logView.ColumnHeadersBorderStyle = DataGridViewHeaderBorderStyle.Single;
      logView.ColumnHeadersDefaultCellStyle.Font = ControlUtilities.GetFont("Segoe UI", 7.5f);
      logView.ColumnHeadersHeightSizeMode = DataGridViewColumnHeadersHeightSizeMode.AutoSize;

      dbSqlEditorBE.log().refresh_ui_cb(logView.ProcessModelRowsChange);
      logView.ProcessModelChange();
      logView.Columns[0].DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleCenter;
      logView.Columns[0].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      logView.Columns[1].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      logView.Columns[2].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      logView.Columns[3].AutoSizeMode = DataGridViewAutoSizeColumnMode.Fill;
      logView.Columns[4].AutoSizeMode = DataGridViewAutoSizeColumnMode.Fill;
      logView.Columns[5].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      logView.SelectionMode = DataGridViewSelectionMode.FullRowSelect;
      logView.ForeColor = Color.Black;
      logView.ShowBusyAnimation = true;
      logView.CellContextMenuStripNeeded += new DataGridViewCellContextMenuStripNeededEventHandler(logView_CellContextMenuStripNeeded);

      historyEntriesView = new GridView(dbSqlEditorBE.history().entries_model());
      historyEntriesView.AutoScroll = true;
      historyEntriesView.ForeColor = Color.Black;
      historyEntriesView.MultiSelect = false;
      historyEntriesView.RowHeadersVisible = false;
      historyEntriesView.Parent = historySplitContainer.Panel1;
      dbSqlEditorBE.history().entries_model().refresh_ui_cb(ProcessModelHistoryEntryRowsChange);
      historyEntriesView.ProcessModelChange();
      historyEntriesView.AutoSizeColumnsMode = DataGridViewAutoSizeColumnsMode.Fill;
      historyEntriesView.SelectionMode = DataGridViewSelectionMode.FullRowSelect;
      historyEntriesView.RowEnter += new DataGridViewCellEventHandler(historyEntriesView_RowEnter);
      historyEntriesView.CellContextMenuStripNeeded += historyEntriesView_CellContextMenuStripNeeded;
      {
        SqlIdeMenuManager.MenuContext popupMenuContext = new SqlIdeMenuManager.MenuContext();
        popupMenuContext.GetNodesMenuItems = historyEntriesView.Model.get_popup_items_for_nodes;
        popupMenuContext.GetSelectedNodes = historyEntriesView.SelectedNodes;
        popupMenuContext.TriggerNodesAction = historyEntriesView.Model.activate_popup_item_for_nodes;
        SqlIdeMenuManager.InitMenu(historyListMenuStrip, popupMenuContext);
      }

      historyDetailsView = new GridView(dbSqlEditorBE.history().details_model());
      historyDetailsView.AutoScroll = true;
      historyDetailsView.ForeColor = Color.Black;
      historyDetailsView.RowHeadersVisible = false;
      historyDetailsView.Parent = historySplitContainer.Panel2;
      historyDetailsView.CellContextMenuStripNeeded += new DataGridViewCellContextMenuStripNeededEventHandler(historyDetailsView_CellContextMenuStripNeeded);
      dbSqlEditorBE.history().details_model().refresh_ui_cb(historyDetailsView.ProcessModelRowsChange);
      historyDetailsView.ProcessModelChange();
      historyDetailsView.AutoSizeColumnsMode = DataGridViewAutoSizeColumnsMode.Fill;
      historyDetailsView.Columns[0].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      historyDetailsView.SelectionMode = DataGridViewSelectionMode.FullRowSelect;
      historyDetailsView.CellDoubleClick += new DataGridViewCellEventHandler(historyDetailsView_CellDoubleClick);

      dbSqlEditorBE.exec_sql_task.finish_cb(AfterExecSql);
      dbSqlEditorBE.exec_sql_task.progress_cb(OnExecSqlProgress);
      dbSqlEditorBE.recordset_list_changed_cb(RecordsetListChanged);
      dbSqlEditorBE.output_text_ui_cb(RecordsetTextOutput);
      resultSetTextBox.Font = grtManager.get_font_option("workbench.general.Editor:Font");

      dbSqlEditorBE.sql_editor_text_insert_cb(OnSqlEditorTextInsert);

      dbSqlEditorBE.refresh_ui().set_partial_refresh_slot(OnPartialRefresh);

      Control sidebar = dbSqlEditorBE.get_sidebar_control();
      sideArea.Controls.Add(sidebar);
      sidebar.Dock = DockStyle.Fill;

      // Help/snippets palette.
      Control palette = dbSqlEditorBE.get_palette_control();
      rightSplitContainer.Panel1.Controls.Add(palette);
      palette.Dock = DockStyle.Fill;
      palette.Show();

      outputPageToolStrip.Renderer = new FlatSubToolStripRenderer();

      ManagedNotificationCenter.AddObserver(this, "GNFormTitleDidChange");
    }

    #endregion

    #region Finalization

    private void Destroy()
    {
      Logger.LogInfo("WQE.net", "Shutting down SQL editor (" + TabText + ")\n");

      if (InstanceCount == 1)
        UnregisterCommands();

      // Make a copy of the keys first as the dictionary is modified while closing the recordsets.
      long[] keys = new long[recordset2placeholder.Count];
      recordset2placeholder.Keys.CopyTo(keys, 0);
      foreach (long key in keys)
        CloseRecordset(key);

      foreach (SqlEditor sqlEditor in sqlEditors)
        sqlEditor.Dispose();
      sqlEditors.Clear();

      historyDetailsView.Dispose();
      historyEntriesView.Dispose();

      dbSqlEditorBE.Dispose();

      InstanceCount--;
    }

    delegate void CollectGarbageDelegate();

    static void CollectGarbage()
    {
      GC.Collect();
    }

    #endregion

    #region Refresh Messages

    protected void OnPartialRefresh(int what)
    {
      SqlEditorFormWrapper.PartialRefreshType refresh_type = (SqlEditorFormWrapper.PartialRefreshType)what;
      switch (refresh_type)
      {
        case SqlEditorFormWrapper.PartialRefreshType.RefreshEditor:
          SqlEditor sqlEditor = ActiveEditor;
          sqlEditor.SqlText = dbSqlEditorBE.sql_editor().sql();
          sqlEditor.Focus();
          break;
       
        case SqlEditorFormWrapper.PartialRefreshType.RunCurrentScript:
          ExecuteSqlScript();
          break;
        
        case SqlEditorFormWrapper.PartialRefreshType.RefreshEditorTitle:
          OnSqlEditorTitleChange();
          break;
        
        case SqlEditorFormWrapper.PartialRefreshType.RefreshRecordsetTitle:
          OnRecordsetCaptionChange();
          break;

        case SqlEditorFormWrapper.PartialRefreshType.RefreshEditorBackend:
          // No action here. The backend copy of the text is always kept up-to-date
          // (for error checking etc.).
          break;

        case SqlEditorFormWrapper.PartialRefreshType.RunCurrentStatement:
          ExecuteCurrentSqlStatement();
          break;

        case SqlEditorFormWrapper.PartialRefreshType.ShowFindPanel:
          {
            SqlEditor editor = ActiveEditor;
            if (editor != null)
              editor.Commands.Execute(BindableCommand.ShowFind);
          }
          break;

        case SqlEditorFormWrapper.PartialRefreshType.ShowSpecialCharacters:
          {
            SqlEditor editor = ActiveEditor;
            if (editor != null)
            {
              editor.EndOfLine.IsVisible = true;
              editor.WhiteSpace.Mode = WhiteSpaceMode.VisibleAlways;
            }
          }
          break;

        case SqlEditorFormWrapper.PartialRefreshType.HideSpecialCharacters:
          {
            SqlEditor editor = ActiveEditor;
            if (editor != null)
            {
              editor.EndOfLine.IsVisible = false;
              editor.WhiteSpace.Mode = WhiteSpaceMode.Invisible;
            }
          }
          break;

        default:
          Logger.LogDebug("WQE.net", 1, "Unhandled partial refresh message\n");
          break;
      }
    }

    /// <summary>
    /// Notifications coming in from the notification center.
    /// </summary>
    /// <param name="name">The name of the notification. We only get called for notifications we want.</param>
    /// <param name="sender">The object that caused the notification to be sent.</param>
    /// <param name="info">Name/Value string pairs for data to be passed.</param>
    public void HandleNotification(string name, IntPtr sender, Dictionary<string, string> info)
    {
      if (name == "GNFormTitleDidChange" && dbSqlEditorBE.form_id() == info["form"])
        TabText = dbSqlEditorBE.caption();
    }

    public override void PerformCommand(string command)
    {
      switch (command)
      {
        case "wb.toggleSchemataBar":
          mainSplitContainer.Panel1Collapsed = !mainSplitContainer.Panel1Collapsed;
          wbContext.save_state("sidebar_visible", "query_editor", !mainSplitContainer.Panel1Collapsed);
          break;
        case "wb.toggleOutputArea":
          contentSplitContainer.Panel2Collapsed = !contentSplitContainer.Panel2Collapsed;
          wbContext.save_state("output_visible", "query_editor", !contentSplitContainer.Panel2Collapsed);
          break;
        case "wb.toggleSideBar":
          mainContentSplitContainer.Panel2Collapsed = !mainContentSplitContainer.Panel2Collapsed;
          wbContext.save_state("support_sidebar_visible", "query_editor", !mainContentSplitContainer.Panel2Collapsed);
          break;
        case "close_editor":
          editorTabControl.CloseTabPage(editorTabControl.SelectedTab);
          break;
      }
    }

    private List<String> commands = new List<string>();

    private void RegisterCommands()
    {
      commands.Add("wb.toggleSchemataBar");
      commands.Add("wb.toggleOutputArea");
      commands.Add("wb.toggleSideBar");
      commands.Add("close_editor");

      wbContext.add_frontend_commands(commands);
    }

    private void UnregisterCommands()
    {
      wbContext.remove_frontend_commands(commands);
    }

    #endregion

    #region SQL Script Execution

    private void ExecuteSqlScript()
    {
      DoExecuteSqlScript(false);
    }

    private void ExecuteCurrentSqlStatement()
    {
      DoExecuteSqlScript(true);
    }

    private int DoExecuteSqlScript(bool currentStatementOnly)
    {
      Logger.LogDebug("WQE.net", 1, "Executing SQL script (" + TabText + ")\n");

      currentlyRunning = ActiveEditor;
      if (currentlyRunning == null)
      {
        // Error, probably in inner logic. Execution cannot be run for no active sql editor.
        Logger.LogWarning("WQE.net", "Script execution failed as there was no currently active SQL editor.\n");
        return 0;
      }

      // Check for pending changes.
      FlatTabControl resultTabControl = GetResultTabControlForEditor(currentlyRunning);
      if (resultTabControl != null)
      {
        bool doForAll = false;
        DialogResult result = DialogResult.OK;
        foreach (TabPage page in resultTabControl.TabPages)
        {
          RecordsetView view = page.Tag as RecordsetView;
          MySQL.Grt.Db.RecordsetWrapper rs = page2recordset[page];
          if (view.Dirty)
          {
            if (!doForAll)
            {
              result = CustomMessageBox.Show(MessageType.MessageWarning, "Unsaved changes pending",
                "There are still pending changes in " + rs.caption() + ". Do you want to save " +
                "first before running the query?", "Save changes", "Cancel", "Ignore changes",
                "Use my decision for all result sets.", out doForAll);
            }

            switch (result)
            {
              case DialogResult.Cancel:
              {
                currentlyRunning = null; 
                return 1;
              }

              case DialogResult.OK:
                view.SaveChanges();
                break;

              case DialogResult.Ignore:
                view.DiscardChanges();
                break;
            }
          }
        }

      }

      String sql;
      if (currentStatementOnly)
      {
        currentlyRunning.RunBackgroundAction(true);
        sql = currentlyRunning.CurrentSqlStatement;
      }
      else
      {
        sql = (currentlyRunning.Selection.Length > 0) ? currentlyRunning.Selection.Text : currentlyRunning.Text;
      }

      if (sql.Length == 0)
      {
        currentlyRunning = null;
        return 1;
      }

      dbSqlEditorBE.exec_sql(sql, true, currentStatementOnly);
      editorTabControl.SetBusy(TabIndexFromEditor(currentlyRunning), true);

      return 0;
    }

    private int OnExecSqlProgress(float progress, String msg)
    {
      logView.ProcessModelRowsChange();
      return 0;
    }

    private void UpdateRecordsets(int editorIndex)
    {
      FlatTabControl resultTabControl = GetResultTabControlForEditor(sqlEditors[editorIndex]);
      if (resultTabControl == null)
        return;

      // Get a list of existing record set pages for this editor, to avoid recreation
      // of pages which still are in the editors result list.
      List<long> leftoverResults = new List<long>();
      foreach (TabPage page in resultTabControl.TabPages)
        leftoverResults.Add((page.Tag as RecordsetView).Model.key());

      // Go through list of recordsets and add new ones that aren't there yet.
      int lastIndex = -1;

      for (int n = 0, rsCount = dbSqlEditorBE.recordset_count(editorIndex); n < rsCount; ++n)
      {
        MySQL.Grt.Db.RecordsetWrapper rs = dbSqlEditorBE.recordset(editorIndex, n);
        if (recordset2page.ContainsKey(rs.key()))
        {
          leftoverResults.Remove(rs.key());
          continue;
        }

        lastIndex = n;
        TabPage page = new TabPage(rs.caption());
        page.BackColor = SystemColors.ButtonFace;

        RecordsetView recordsetView = new RecordsetView();
        page.Tag = recordsetView;
        //components.Add(recordsetView);

        Form mainForm = ParentForm;
        if (!mainForm.Visible)
          pendingRelayouts.Add(recordsetView);

        recordsetView.Embed(page, rs);
        resultTabControl.TabPages.Add(page);
        resultTabControl.SetCloseButtonVisibility(n, FlatTabControl.CloseButtonVisiblity.ShowButton);
        recordset2page.Add(rs.key(), page);
        recordset2placeholder.Add(rs.key(), recordsetView);
        page2recordset.Add(page, rs);
        recordsetView.ProcessModelChange();
      }

      if (lastIndex > -1)
        resultTabControl.SelectedIndex = lastIndex;

      // close recordsets that have gone
      foreach (long key in leftoverResults)
        CloseRecordset(key);

      SplitContainer editorSplitContainer = resultTabControl.Parent.Parent as SplitContainer;

      // Hide SQL text area if we started collapsed (means: in edit mode for a table).
      // If there is no result set, however, show the editor.
      editorSplitContainer.Panel2Collapsed = resultTabControl.TabCount == 0;
      if (resultTabControl.TabCount > 0)
        editorSplitContainer.Panel1Collapsed = dbSqlEditorBE.sql_editor_start_collapsed(editorIndex);

      UpdateActiveRecordsetInBackend();
    }

    private int AfterExecSql()
    {
      // For normal sql execution we have an editor (which is not necessarily the active editor).
      // If there is no running editor then we probably have an EXPLAIN call that just finished
      // which is always for the active editor.
      SqlEditor editor = currentlyRunning;
      if (editor == null)
        editor = ActiveEditor;
      if (editor == null)
        return 0;

      int tabIndex = TabIndexFromEditor(editor);
      int editorIndex = EditorIndexFromTabIndex(tabIndex);
      UpdateRecordsets(editorIndex);

      if (dbSqlEditorBE.exec_sql_error_count() == 0)
        editorTabControl_SelectedIndexChanged(null, null);

      logView.ProcessModelRowsChange();
      logView.AutoResizeColumn(1);
      logView.AutoResizeColumn(2);
      logView.AutoResizeColumn(5);

      if (logView.Rows.Count > 0)
        logView.SetRowSelected(logView.Rows.Count - 1);

      editorTabControl.SelectedIndex = tabIndex;
      sqlEditors[editorIndex].Focus();
      editorTabControl.SetBusy(tabIndex, false);
      currentlyRunning = null;

      return 0;
    }

    void onSizeChanged(object sender, EventArgs e)
    {
      foreach (RecordsetView view in pendingRelayouts)
      {
        // Workaround for data grids added while the form was minimized.
        // They refuse to compute their own layout until a new resize event occurs.
        view.GridView.Dock = DockStyle.None;
        view.GridView.Dock = DockStyle.Fill;
      }
      pendingRelayouts.Clear();
    }
    #endregion

    #region SQL Editors

    private List<SqlEditor> sqlEditors;
    private Dictionary<SqlEditor, TabPage> sqlEditor2page = new Dictionary<SqlEditor, TabPage>();
    private Dictionary<TabPage, SqlEditor> page2SqlEditor = new Dictionary<TabPage, SqlEditor>();

    private SqlEditor ActiveEditor
    {
      get
      {
        int index = dbSqlEditorBE.active_sql_editor_index();
        if (index < 0 || index > sqlEditors.Count - 1)
          return null;
        return sqlEditors[index];
      }
    }

    /// <summary>
    /// Returns the index of the tab page that hosts the given editor.
    /// There is no 1:1 relationship between the sql editor collections and the tab collection,
    /// because we can have other types of forms docked (e.g. object editors).
    /// </summary>
    private int TabIndexFromEditor(SqlEditor editor)
    {
      TabPage page = sqlEditor2page[editor];
      if (page == null)
        return -1;
      return editorTabControl.TabPages.IndexOf(page);
    }

    /// <summary>
    /// The reverse operation to the previous function, without traveling the layout hierarchy.
    /// This way we could change our UI without affecting the mapping.
    /// </summary>
    private int EditorIndexFromTabIndex(int tabIndex)
    {
      if (tabIndex < 0 || tabIndex > editorTabControl.TabCount - 1)
        return -1;

      return EditorIndexFromTabPage(editorTabControl.TabPages[tabIndex]);
    }

    private int EditorIndexFromTabPage(TabPage page)
    {
      if (!page2SqlEditor.ContainsKey(page))
        return -1;

      return dbSqlEditorBE.sql_editor_index(page2SqlEditor[page].BE);
    }

    private int OnSqlEditorNew(int index)
    {
      AddSqlEditor(index);
      return 0;
    }

    private void OnSqlEditorTitleChange()
    {
      int editorIndex = dbSqlEditorBE.active_sql_editor_index();
      if (editorIndex >= 0 && editorIndex < sqlEditors.Count)
      {
        SqlEditor editor = sqlEditors[editorIndex];
        sqlEditor2page[editor].Text = dbSqlEditorBE.sql_editor_caption(editorIndex);

        // Also update the tooltip text on the way.
        if (!dbSqlEditorBE.sql_editor_is_scratch(editorIndex))
          sqlEditor2page[editor].ToolTipText = dbSqlEditorBE.sql_editor_path(editorIndex);
      }
    }

    private SqlEditor AddSqlEditor(int index)
    {
      Logger.LogDebug("WQE.net", 1, "Adding new SQL editor (" + TabText + ")\n");

      SqlEditor sqlEditor = new SqlEditor(grtManager);
      sqlEditor.BE = dbSqlEditorBE.sql_editor(index);
      sqlEditor.Text = sqlEditor.BE.sql();
      String caption = dbSqlEditorBE.sql_editor_caption(index);

      TabPage page = new TabPage(caption);
      page.Tag = sqlEditor;

      /*
      if (dbSqlEditorBE.sql_editor_is_scratch(index))
        page.ImageIndex = 1;
      else
        page.ImageIndex = 0;
      */

      // Create control hierarchy.
      SplitContainer editorSplitContainer = new SplitContainer();
      editorSplitContainer.Orientation = Orientation.Horizontal;
      editorSplitContainer.Dock = DockStyle.Fill;

      // Upper part (editor + toolbar).
      editorSplitContainer.Panel1.Controls.Add(sqlEditor);
      ToolStrip toolStrip = dbSqlEditorBE.get_editor_toolbar(index);
      editorSplitContainer.Panel1.Controls.Add(toolStrip);
      toolStrip.Dock = DockStyle.Top;

      // Lower part (record set views).
      FlatTabControl resultTabControl = new FlatTabControl();
      resultTabControl.BackgroundColor = System.Drawing.Color.FromArgb(40, 55, 82);
      resultTabControl.CanCloseLastTab = true;
      resultTabControl.ContentPadding = new System.Windows.Forms.Padding(0);
      editorSplitContainer.Panel2.Controls.Add(resultTabControl);
      resultTabControl.Dock = System.Windows.Forms.DockStyle.Fill;
      resultTabControl.HideWhenEmpty = false;
      resultTabControl.ItemPadding = new System.Windows.Forms.Padding(6, 0, 6, 0);
      resultTabControl.MaxTabSize = 200;
      resultTabControl.Name = "resultTabControl";
      resultTabControl.ShowCloseButton = true;
      resultTabControl.ShowFocusState = true;
      resultTabControl.ShowToolTips = true;
      resultTabControl.CanReorderTabs = true;
      resultTabControl.AllowDrop = true;
      resultTabControl.TabStyle = MySQL.Controls.FlatTabControl.TabStyleType.BottomNormal;
      resultTabControl.TabClosing += new System.EventHandler<MySQL.Controls.TabClosingEventArgs>(ResultTabClosing);
      resultTabControl.TabClosed += new System.EventHandler<MySQL.Controls.TabClosedEventArgs>(ResultTabClosed);
      resultTabControl.MouseClick += new MouseEventHandler(recordsetTabControlMouseClick);
      resultTabControl.SelectedIndexChanged += new EventHandler(resultTabControl_SelectedIndexChanged);

      page.Controls.Add(editorSplitContainer);
      editorTabControl.TabPages.Add(page);

      sqlEditor.Selection.Start = sqlEditor.Text.Length;
      sqlEditor.Selection.End = sqlEditor.Text.Length;
      sqlEditor.KeyDown += new KeyEventHandler(sqlEditor_KeyDown);

      sqlEditors.Add(sqlEditor);
      sqlEditor2page[sqlEditor] = page;
      page2SqlEditor[page] = sqlEditor;

      // Set an initial height for the result set panel depending on what the user last had.
      int splitterDistance = wbContext.read_state("recordset_height", "query_editor", editorSplitContainer.Height / 2);
      if (editorSplitContainer.Height > splitterDistance + editorSplitContainer.SplitterWidth)
        editorSplitContainer.SplitterDistance = editorSplitContainer.Height - editorSplitContainer.SplitterWidth - splitterDistance;

      editorSplitContainer.SplitterMoved += new SplitterEventHandler(editorSplitContainer_SplitterMoved);
      UpdateRecordsets(index);

      // Select tab last so that callbacks will work on attached recordset.
      editorTabControl.SelectedIndex = editorTabControl.TabCount - 1;

      return sqlEditor;
    }

    private void RemoveSqlEditor(int index)
    {
      if (index == -1)
        index = dbSqlEditorBE.active_sql_editor_index();

      // This call implicitly deletes all recordsets too.
      dbSqlEditorBE.remove_sql_editor(index);
      SqlEditor sqlEditor = sqlEditors[index];
      sqlEditor.Tag = null;
      sqlEditors.RemoveAt(index);
      TabPage page = sqlEditor2page[sqlEditor];
      sqlEditor2page.Remove(sqlEditor);
      page2SqlEditor.Remove(page);
      sqlEditor.Dispose();
    }

    private void sqlEditor_KeyDown(object sender, KeyEventArgs e)
    {
      switch (e.KeyCode)
      {
        case Keys.Enter:
          if (e.Control && !e.Shift && !e.Alt)
            ExecuteSqlScript();
          break;
      }
    }

    private int OnSqlEditorTextInsert(String text)
    {
      SqlEditor sqlEditor = ActiveEditor;
      if (sqlEditor == null)
        return 0;

      sqlEditor.InsertText(sqlEditor.Caret.Position, text);
      sqlEditor.Selection.Start = sqlEditor.Caret.Position + text.Length;
      sqlEditor.Selection.End = sqlEditor.Selection.Start;
      if (sqlEditor.CanFocus)
        sqlEditor.Focus();
      return 0;
    }

    #region Event handling

    void editorTabControl_SelectedIndexChanged(object sender, EventArgs e)
    {
      // This could set -1 as active editor index in the backend if a page was selected
      // that does not hold an SQL editor (e.g. an object editor page).
      dbSqlEditorBE.active_sql_editor_index(EditorIndexFromTabIndex(editorTabControl.SelectedIndex));
      UpdateActiveRecordsetInBackend();

      SqlEditor sqlEditor = ActiveEditor;
      if (sqlEditor != null && sqlEditor.CanFocus)
        sqlEditor.Focus();
      UpdateApplyCancel();
    }

    void editorTabControl_TabClosing(object sender, MySQL.Controls.TabClosingEventArgs e)
    {
      int editorIndex = EditorIndexFromTabPage(e.page);
      if (editorIndex < 0)
        return; // Not an SQL editor.
      e.canClose = dbSqlEditorBE.sql_editor_will_close(editorIndex);
    }

    void editorTabControl_TabClosed(object sender, MySQL.Controls.TabClosedEventArgs e)
    {
      int editorIndex = EditorIndexFromTabPage(e.page);
      if (editorIndex > -1)
        RemoveSqlEditor(editorIndex);

      // If the last editor was closed create a new, empty one.
      if (sqlEditors.Count > 0)
      {
        editorIndex = EditorIndexFromTabIndex(editorTabControl.SelectedIndex);
        if (editorIndex > -1)
          dbSqlEditorBE.active_sql_editor_index(editorIndex);
      }
      else
      {
        dbSqlEditorBE.new_sql_script_file(); // This will call our callback for setting up the editor.
        dbSqlEditorBE.active_sql_editor_index(0);
        ActiveControl = ActiveEditor;
      }
    }

    void resultTabControl_SelectedIndexChanged(object sender, EventArgs e)
    {
      UpdateActiveRecordsetInBackend();
      UpdateApplyCancel();
    }

    private void editorTabControl_TabMoving(object sender, TabMovingEventArgs e)
    {
      // When moving tab pages we have to consider the sql editor indexing, which
      // might be different from that of the tab control (e.g. because of object editors).
      int from = EditorIndexFromTabIndex(e.FromIndex);
      if (from > -1)
      {
        // We are indeed moving a tab with an sql editor. Determine the target sql editor
        // index by iterating from the target index to the source index until another sql editor
        // is found. This is then the new sql editor target index.
        int offset = (e.FromIndex < e.ToIndex) ? -1 : 1;
        for (int index = e.ToIndex; index != e.FromIndex; index += offset)
        {
          int to = EditorIndexFromTabIndex(index);
          if (to > -1)
          {
            dbSqlEditorBE.sql_editor_reorder(from, to);

            // Also update our sqlEditor cache.
            SqlEditor editor = sqlEditors[from];
            sqlEditors[from] = sqlEditors[to];
            sqlEditors[to] = editor;
            break;
          }
        }
      }
    }

    private void applyButton_Click(object sender, EventArgs e)
    {
      RecordsetView view = ActiveRecordsetView;
      if (view != null)
        view.SaveChanges();
      UpdateApplyCancel();
    }

    private void cancelButton_Click(object sender, EventArgs e)
    {
      RecordsetView view = ActiveRecordsetView;
      if (view != null)
        view.DiscardChanges();
      UpdateApplyCancel();
    }

    #endregion

    public void SearchText(String text)
    {
      SqlEditor editor = ActiveEditor;
      if (editor == null)
        return;

      Range textPosition = editor.FindReplace.FindNext(text);
      if (textPosition != null)
        textPosition.Select();
    }

    // IWorkbenchDocument override.
    public override UIForm BackendForm
    {
      get { return dbSqlEditorBE; }
    }

    #endregion

    #region Recordsets

    // Simple mapper from a record set to its tab page/view or vice versa.
    // No information is stored to which editor they belong.
    private Dictionary<long, TabPage> recordset2page = new Dictionary<long, TabPage>();
    private Dictionary<long, RecordsetView> recordset2placeholder = new Dictionary<long, RecordsetView>();
    private Dictionary<TabPage, MySQL.Grt.Db.RecordsetWrapper> page2recordset = new Dictionary<TabPage, MySQL.Grt.Db.RecordsetWrapper>();

    public MySQL.Grt.Db.RecordsetWrapper ActiveRecordset
    {
      get
      {
        if (editorTabControl.SelectedTab == null)
          return null;

        FlatTabControl resultTabControl = GetResultTabControlForEditor(ActiveEditor);
        if (resultTabControl == null || resultTabControl.SelectedTab == null)
          return null;
        
        return page2recordset.ContainsKey(resultTabControl.SelectedTab) ?
          page2recordset[resultTabControl.SelectedTab] : null;
      }
    }

    public RecordsetView ActiveRecordsetView
    {
      get
      {
        MySQL.Grt.Db.RecordsetWrapper recordset = ActiveRecordset;
        if (recordset == null)
          return null;

        return recordset2placeholder[recordset.key()];
      }
    }

    private void UpdateActiveRecordsetInBackend()
    {
      if (editorTabControl.SelectedTab == null)
        return;

      int index = dbSqlEditorBE.active_sql_editor_index();
      MySQL.Grt.Db.RecordsetWrapper activeRecordset = ActiveRecordset;
      if (activeRecordset == null)
        dbSqlEditorBE.active_recordset(index, null);
      else
        dbSqlEditorBE.active_recordset(index, activeRecordset);
    }
    
    private void OnRecordsetCaptionChange()
    {
      if (editorTabControl.SelectedTab == null)
        return;

      FlatTabControl resultTabControl = GetResultTabControlForEditor(ActiveEditor);
      if (resultTabControl == null || resultTabControl.SelectedTab == null)
        return;

      MySQL.Grt.Db.RecordsetWrapper rs = page2recordset[resultTabControl.SelectedTab];
      resultTabControl.SelectedTab.Text = rs.caption();

      UpdateApplyCancel();
    
      // Trigger also a menu validation as the dirty state change might also change menu items.
      wbContext.validate_menu_for_form(dbSqlEditorBE);
    }

    delegate void DelegateFunc();
    private void RecordsetListChanged(int editor_index, long key, bool added)
    {
      if (!added)
      {
        if (InvokeRequired)
        {
          DelegateFunc deleg = delegate 
          {
            CloseRecordset(key);
          };
          Invoke(deleg);
        }
        else
          CloseRecordset(key);
      }
    }

    private void CloseRecordset(long key)
    {
      if (recordset2page.ContainsKey(key))
      {
        TabPage page = recordset2page[key];
        recordset2page.Remove(key);

        RecordsetView recordsetView = recordset2placeholder[key];
        page.Tag = null;
        recordset2placeholder.Remove(key);
        //components.Remove(recordsetView);
        recordsetView.Dispose();

        page2recordset.Remove(page);
        page.Dispose();
      }
    }

    private delegate void RecordsetTextOutputDelegate(String text, bool bringToFront);

    private void RecordsetTextOutput(String text, bool bringToFront)
    {
      if (InvokeRequired)
      {
        RecordsetTextOutputDelegate f = new RecordsetTextOutputDelegate(RecordsetTextOutput);
        Invoke(f, new Object[] {text, bringToFront} );
      }
      else
      {
        resultSetTextBox.AppendText(text);
        if (bringToFront)
          outputSelector.SelectedIndex = 1;
      }
    }

    private void ResultTabClosing(object sender, TabClosingEventArgs e)
    {
      e.canClose = false;
      MySQL.Grt.Db.RecordsetWrapper rs = page2recordset.ContainsKey(e.page) ? page2recordset[e.page] : null;
      if (rs != null)
        e.canClose = rs.can_close();

      if (e.canClose)
      {
        // Collapse the result set pane if this is the last record set we close.
        FlatTabControl resultTabControl = e.page.Parent as FlatTabControl;
        (resultTabControl.Parent.Parent as SplitContainer).Panel2Collapsed = resultTabControl.TabCount == 1;
      }
    }

    private void ResultTabClosed(object sender, TabClosedEventArgs e)
    {
      MySQL.Grt.Db.RecordsetWrapper rs = page2recordset.ContainsKey(e.page) ? page2recordset[e.page] : null;
      if (rs != null)
        CloseRecordset(rs.key());
    }

    private void recordsetTabContextMenuClick(object sender, EventArgs e)
    {
      FlatTabControl resultTabControl = GetResultTabControlForEditor(ActiveEditor);
      if (resultTabControl == null)
        return;

      TabPage page = resultTabControl.SelectedTab;
      if (page == null)
        return;
      if (!page2recordset.ContainsKey(page))
        return;

      MySQL.Grt.Db.RecordsetWrapper rs = page2recordset[page];

      ToolStripMenuItem item = sender as ToolStripMenuItem;
      int clickedTab = (int)item.Owner.Tag;
      switch (item.Tag as string)
      {
        case "0": // Rename editor and tab. (currently not active)
          String newName = page.Text;
          if (StringInputForm.ShowModal("Rename a page", "Enter a new name for the page", "Name", ref newName) == DialogResult.OK)
          {
            rs.caption(newName);
            page.Text = rs.caption();
          }
          break;
        case "1": // Close recordset.
          resultTabControl.CloseTabPage(page);
          break;
        case "2": // Close all recordsets.
          while (resultTabControl.TabCount > 0)
            resultTabControl.CloseTabPage(resultTabControl.TabPages[0]);
          break;
        case "3": // Close all editors but the clicked one.
          while (resultTabControl.TabCount > 1)
          {
            if (resultTabControl.TabPages[0] == page)
              resultTabControl.CloseTabPage(resultTabControl.TabPages[1]);
            else
              resultTabControl.CloseTabPage(resultTabControl.TabPages[0]);
          }
          break;
      }
    }

    private void editorTabControlMouseClick(object sender, MouseEventArgs e)
    {
      switch (e.Button)
      {
        case MouseButtons.Right:
          {
            int clickedIndex = editorTabControl.TabIndexFromPosition(e.Location);
            if (clickedIndex < 0)
              return;

            // Keep the found index for later handling in the item click handler.
            editorTabsContextMenuStrip.Tag = clickedIndex;

            string path = "";
            int editorIndex = EditorIndexFromTabIndex(clickedIndex);
            if (editorIndex > -1)
              path = dbSqlEditorBE.sql_editor_path(editorIndex);
            copyFullPathToClipboardToolStripMenuItem.Enabled = path.Length > 0;
            Point p = editorTabControl.PointToScreen(e.Location);
            editorTabsContextMenuStrip.Show(p);
          }
          break;
      }
    }

    private void recordsetTabControlMouseClick(object sender, MouseEventArgs e)
    {
      switch (e.Button)
      {
        case MouseButtons.Right:
          {
            FlatTabControl resultTabControl = GetResultTabControlForEditor(ActiveEditor);
            if (resultTabControl == null)
              return;

            int clickedIndex = resultTabControl.TabIndexFromPosition(e.Location);
            if (clickedIndex < 0)
              return;

            // Keep the found index for later handling in the item click handler.
            recordsetTabsContextMenuStrip.Tag = clickedIndex;

            TabPage page = resultTabControl.TabPages[clickedIndex];
            renamePage_ToolStripMenuItem.Enabled = page2recordset.ContainsKey(page);
            Point p = resultTabControl.PointToScreen((Point)e.Location);
            recordsetTabsContextMenuStrip.Show(p);
          }
          break;
      }
    }

    private void editorTabContextMenuItemClick(object sender, EventArgs e)
    {
      ToolStripMenuItem item = sender as ToolStripMenuItem;
      int clickedTab = (int)item.Owner.Tag;
      TabPage page = editorTabControl.TabPages[clickedTab];
      switch (item.Tag as string)
      {
        case "0": // Rename editor and tab. (currently not active)
          String newName = page.Text;
          if (StringInputForm.ShowModal("Rename a page", "Enter a new name for the page", "Name", ref newName) == DialogResult.OK)
          {
            dbSqlEditorBE.sql_editor_caption(clickedTab, newName);
            page.Text = newName;
          }
          break;
        case "1": // Close editor.
          editorTabControl.CloseTabPage(page);
          break;
        case "2": // Close all editors.
          // Close them backwards and only exactly the number of editors that are there when we started.
          // Because on close of the last one a new empty one is created which we do not want to close again.
          // Additionally, some tabs might not be closable due to changed content and the user refusing
          // to close them.
          for (int i = editorTabControl.TabCount - 1; i >= 0; i--)
            editorTabControl.CloseTabPage(editorTabControl.TabPages[i]);
          break;
        case "3": // Close all editors but the clicked one.
          // Similar to case 2, but we don't need to care for a newly created editor, as we always
          // keep one open.
          List<TabPage> open_tabs = new List<TabPage>();
          
          for(int index = 0; index < editorTabControl.TabCount; index++)
          {
            if(editorTabControl.TabPages[index] != page)
              open_tabs.Add(editorTabControl.TabPages[index]);
          }

          foreach(TabPage open_page in open_tabs)
            editorTabControl.CloseTabPage(open_page);

          break;

        case "4": // Copy full path to clipboard (only valid for SQL editors).
          System.Windows.Forms.Clipboard.SetText(dbSqlEditorBE.sql_editor_path(clickedTab));
          break;

        case "5": // Close all of the same type.
          ITabDocument document = editorTabControl.DocumentFromIndex(clickedTab);
          bool closeDocument = document != null; // Only two types of pages: documents (table editors) or sql editors.
          for (int i = editorTabControl.TabCount - 1; i >= 0; i--)
          {
            if (closeDocument ^ editorTabControl.DocumentFromIndex(i) == null)
              editorTabControl.CloseTabPage(editorTabControl.TabPages[i]);
          }
          break;
      }
    }

    #endregion

    #region Log Panel

    private GridView logView;

    /* TODO: do we want to show these details somewhere?
    void logView_SelectionChanged(object sender, EventArgs e)
    {
      String logEventAction = "";
      String logEventMessage = "";

      if (logView.SelectedRows.Count == 0)
      {
        logEventDetailHeaderLabel.Text = "";
      }
      else
      {
        int logEventNo = logView.SelectedRows[0].Index;
        int logEventTypeCode = 0;
        String logEventTime = "";
        dbSqlEditorBE.get_log_event_details(logEventNo, out logEventTypeCode, out logEventTime, out logEventAction, out logEventMessage);
        String logEventType;
        switch (logEventTypeCode)
        {
          case (int)Msg_type.MT_error:
            logEventType = "Error";
            break;
          case (int)Msg_type.MT_warning:
            logEventType = "Warning";
            break;
          case (int)Msg_type.MT_info:
            logEventType = "Info";
            break;
          default:
            logEventType = "Unknown";
            break;
        }
        logEventDetailHeaderLabel.Text = String.Format("Event {0} of type {1} at {2}", logEventNo + 1, logEventType, logEventTime);
      }
      logEventDetailActionTextBox.Text = logEventAction;
      logEventDetailMessageTextBox.Text = logEventMessage;
    }
    */

    #endregion

    #region History Panel

    private GridView historyEntriesView;
    private GridView historyDetailsView;

    ContextMenuStrip logViewMenuStrip = null;
    ContextMenuStrip historyListMenuStrip = new ContextMenuStrip();
    ContextMenuStrip historyDetailsMenuStrip = null;

    private void historyEntriesView_CellContextMenuStripNeeded(object sender, DataGridViewCellContextMenuStripNeededEventArgs e)
    {
      e.ContextMenuStrip = historyListMenuStrip;
    }

    void historyDetailsView_CellContextMenuStripNeeded(object sender, DataGridViewCellContextMenuStripNeededEventArgs e)
    {
      if (historyDetailsMenuStrip == null)
      {
        // Prepare context menu strip on first use. Since this is actually an mforms managed
        // menu we don't get item clicks from there. So we hook into the click events here too.
        // TODO: this entire handling needs a general review.
        historyDetailsMenuStrip = dbSqlEditorBE.history().get_details_context_menu();
        foreach (ToolStripItem item in historyDetailsMenuStrip.Items)
          item.Click += new EventHandler(historyDetailsMenuItemClick);
      }

      if (historyDetailsView.SelectedRows.Count > 0)
        e.ContextMenuStrip = historyDetailsMenuStrip;
    }

    private void logView_CellContextMenuStripNeeded(object sender, DataGridViewCellContextMenuStripNeededEventArgs e)
    {
      if (logViewMenuStrip == null)
      {
        // Prepare context menu strip on first use. Since this is actually an mforms managed
        // menu we don't get item clicks from there. So we hook into the click events here too.
        // TODO: this entire handling needs a general review.
        logViewMenuStrip = dbSqlEditorBE.get_log_context_menu();
        foreach (ToolStripItem item in logViewMenuStrip.Items)
          item.Click += new EventHandler(logMenuItemClick);
      }

      if (logView.SelectedRows.Count > 0)
        e.ContextMenuStrip = logViewMenuStrip;
    }

    enum HistoryAction { HistoryAppendToEditor, HistoryReplaceEditor, HistoryCopyToClipboard };

    void historyDetailsMenuItemClick(object sender, EventArgs e)
    {
      ToolStripItem item = sender as ToolStripItem;
      switch (item.Tag.ToString())
      {
        case "copy_row":
          loadSelectedHistoryItems(HistoryAction.HistoryCopyToClipboard);
          break;
        case "append_selected_items":
          loadSelectedHistoryItems(HistoryAction.HistoryAppendToEditor);
          break;
        case "replace_sql_script":
          loadSelectedHistoryItems(HistoryAction.HistoryReplaceEditor);
          break;
      }
    }

    void logMenuItemClick(object sender, EventArgs e)
    {
      ToolStripItem item = sender as ToolStripItem;
      StringBuilder sql = new StringBuilder();
      foreach (NodeId node in logView.SelectedNodes())
      {
        if (sql.Length > 0)
          sql.Append('\n');

        string part;
        logView.Model.get_field_repr(node, 3, out part);
        sql.Append(part);
      }

      SqlEditor sqlEditor = ActiveEditor;
      switch (item.Tag.ToString())
      {
        case "copy_row":
            System.Windows.Forms.Clipboard.SetText(dbSqlEditorBE.get_text_for_actions(logView.SelectedNodes(), false));
          break;
        case "append_selected_items":
          if (sqlEditor != null)
          {
            sqlEditor.IsRefreshEnabled = true;
            sqlEditor.SqlText += sql.ToString();
          }
          break;
        case "replace_sql_script":
          if (sqlEditor != null)
          {
            sqlEditor.IsRefreshEnabled = true;
            sqlEditor.SqlText = sql.ToString();
          }
          break;
        case "clear":
          dbSqlEditorBE.log().reset();
          break;
      }
    }

    void historyEntriesView_RowEnter(Object sender, DataGridViewCellEventArgs e)
    {
      dbSqlEditorBE.history().current_entry(e.RowIndex);
    }

    void loadSelectedHistoryItems(HistoryAction action)
    {
      if (null == historyEntriesView.CurrentRow)
        return;

      List<int> sel_indexes = new List<int>();
      foreach (DataGridViewRow row in historyDetailsView.SelectedRows)
        sel_indexes.Add(row.Index);
      sel_indexes.Sort();
      String sql = dbSqlEditorBE.restore_sql_from_history(historyEntriesView.CurrentRow.Index, sel_indexes);

      SqlEditor sqlEditor = ActiveEditor;
      switch (action)
      {
        case HistoryAction.HistoryAppendToEditor:
          if (sqlEditor != null)
          {
            sqlEditor.IsRefreshEnabled = true;
            sqlEditor.SqlText += sql;
          }
          break;
        case HistoryAction.HistoryReplaceEditor:
          if (sqlEditor != null)
          {
            sqlEditor.IsRefreshEnabled = true;
            sqlEditor.SqlText = sql;
          }
          break;
        case HistoryAction.HistoryCopyToClipboard:
          System.Windows.Forms.Clipboard.SetText(sql);
          break;
      }
    }

    void historyDetailsView_CellDoubleClick(Object sender, DataGridViewCellEventArgs e)
    {
      loadSelectedHistoryItems(HistoryAction.HistoryAppendToEditor);
    }

    private int ProcessModelHistoryEntryRowsChange()
    {
      int ret_val = historyEntriesView.ProcessModelRowsChange();
      int selected_entry = dbSqlEditorBE.history().current_entry();

      if (selected_entry != -1)
        historyEntriesView.SetRowSelected(selected_entry);

      return ret_val;
    }

    #endregion

    #region Event Handling

    private void DbSqlEditor_Shown(object sender, EventArgs e)
    {
      LoadFormState();
      canTrackChanges = true;
    }

    private void mainSplitContainer_SplitterMoved(object sender, SplitterEventArgs e)
    {
      if (!canTrackChanges || IsDisposed || Disposing)
        return;

      int sidebarWidth;
      bool leftAligned = wbContext.read_option_value("", "Sidebar:RightAligned", "0") == "0";
      if (leftAligned)
        sidebarWidth = mainSplitContainer.SplitterDistance;
      else
        sidebarWidth = mainSplitContainer.Width - mainSplitContainer.SplitterWidth - mainSplitContainer.SplitterDistance;
      wbContext.save_state("sidebar_width", "query_editor", sidebarWidth);
    }

    private void contentSplitContainer_SplitterMoved(object sender, SplitterEventArgs e)
    {
      if (!canTrackChanges || IsDisposed || Disposing)
        return;

      int splitterDistance = contentSplitContainer.Height - contentSplitContainer.SplitterWidth - contentSplitContainer.SplitterDistance;
      wbContext.save_state("output_height", "query_editor", splitterDistance);
    }

    private void mainContentSplitContainer_SplitterMoved(object sender, SplitterEventArgs e)
    {
      if (!canTrackChanges || IsDisposed || Disposing)
        return;

      int splitterDistance = mainContentSplitContainer.Width - mainContentSplitContainer.SplitterWidth - mainContentSplitContainer.SplitterDistance;
      wbContext.save_state("support_sidebar_width", "query_editor", splitterDistance);
    }

    private void editorSplitContainer_SplitterMoved(object sender, SplitterEventArgs e)
    {
      if (!canTrackChanges || IsDisposed || Disposing)
        return;

      SplitContainer container = sender as SplitContainer;
      int splitterDistance = container.Height - container.SplitterWidth - container.SplitterDistance;
      wbContext.save_state("recordset_height", "query_editor", splitterDistance);
    }

    void outputPaneIndexChanged(object sender, System.EventArgs e)
    {
      ToolStripComboBox selector = sender as ToolStripComboBox;
      if (outputPageContent.Controls.Count > 0)
        outputPageContent.Controls.RemoveAt(0);
      switch (selector.SelectedIndex)
      {
        case 0:
          outputPageContent.Controls.Add(actionPanel);
          actionPanel.Dock = DockStyle.Fill;
          break;
        case 1:
          outputPageContent.Controls.Add(textOutputPage);
          textOutputPage.Dock = DockStyle.Fill;
          break;
        case 2:
          outputPageContent.Controls.Add(historyPage);
          historyPage.Dock = DockStyle.Fill;
          break;
      }
    }

    #endregion

    #region Form Layout

    private void LoadFormState()
    {
      // Object side bar.
      int splitterDistance = wbContext.read_state("sidebar_width", "query_editor", 200);
      bool leftAligned = wbContext.read_option_value("", "Sidebar:RightAligned", "0") == "0";

      if (leftAligned)
      {
        if (mainSplitContainer.Width > splitterDistance)
          mainSplitContainer.SplitterDistance = splitterDistance;
      }
      else
      {
        mainSplitContainer.FixedPanel = FixedPanel.Panel2;
        contentSplitContainer.Parent = mainSplitContainer.Panel1;
        sideArea.Parent = mainSplitContainer.Panel2;
        if (mainSplitContainer.Width > splitterDistance + mainSplitContainer.SplitterWidth)
          mainSplitContainer.SplitterDistance = mainSplitContainer.Width - mainSplitContainer.SplitterWidth - splitterDistance;
      }

      // Output tab. Distance measured from bottom (for easy default value).
      splitterDistance = wbContext.read_state("output_height", "query_editor", 200);
      if (contentSplitContainer.Height > splitterDistance + contentSplitContainer.SplitterWidth)
        contentSplitContainer.SplitterDistance = contentSplitContainer.Height - contentSplitContainer.SplitterWidth - splitterDistance;

      // Support side bar. Distance measured from right (for easy default value).
      splitterDistance = wbContext.read_state("support_sidebar_width", "query_editor", 200);
      if (mainContentSplitContainer.Height > splitterDistance + mainContentSplitContainer.SplitterWidth)
        mainContentSplitContainer.SplitterDistance = mainContentSplitContainer.Width - mainContentSplitContainer.SplitterWidth - splitterDistance;

      // Visibility of the sidebar/output areas.
      bool visible = wbContext.read_state("sidebar_visible", "query_editor", true);
      dbSqlEditorBE.set_tool_item_checked("wb.toggleSchemataBar", visible);
      mainSplitContainer.Panel1Collapsed = !visible;

      visible = wbContext.read_state("output_visible", "query_editor", true);
      dbSqlEditorBE.set_tool_item_checked("wb.toggleOutputArea", visible);
      contentSplitContainer.Panel2Collapsed = !visible;

      visible = wbContext.read_state("support_sidebar_visible", "query_editor", true);
      dbSqlEditorBE.set_tool_item_checked("wb.toggleSideBar", visible);
      mainContentSplitContainer.Panel2Collapsed = !visible;
    }
    
    /// <summary>
    /// Docks the given document to the editor tab control (usually object editors).
    /// </summary>
    public void DockDocument(ITabDocument document, bool activate)
    {
      if (!editorTabControl.HasDocument(document))
        editorTabControl.AddDocument(document);
      if (activate)
        document.Activate();

      if (contentSplitContainer.Panel2Collapsed)
      {
        contentSplitContainer.Panel2Collapsed = false;

        // Set a splitter distance or we end up at almost full display. Use a relatively small
        // value for panel2. The document's min height will kick in and does the right job.
        contentSplitContainer.SplitterDistance = contentSplitContainer.Height - 100;
      }
    }

    public void UndockDocument(ITabDocument document)
    {
      editorTabControl.RemoveDocument(document);
    }

    #endregion

    #region Miscellaneous

    /// <summary>
    /// Helper method to ease access to the result set tab control, stored in our
    /// nested tab hierarchy.
    /// </summary>
    private FlatTabControl GetResultTabControlForEditor(SqlEditor editor)
    {
      SplitContainer editorSplitContainer = GetSplitContainerForEditor(editor);
      if (editorSplitContainer == null || editorSplitContainer.Panel2.Controls.Count == 0)
        return null;

      return editorSplitContainer.Panel2.Controls[0] as FlatTabControl;
    }

    /// <summary>
    /// Helper method to ease access to the result set tab control, stored in our
    /// nested tab hierarchy.
    /// </summary>
    private SplitContainer GetSplitContainerForEditor(SqlEditor editor)
    {
      if (editor == null)
        return null;

      int index = TabIndexFromEditor(editor);
      if (index < 0 || index > editorTabControl.TabCount - 1)
        return null;

      TabPage editorTabPage = editorTabControl.TabPages[index];
      if (editorTabPage.Controls.Count == 0)
        return null;

      return editorTabPage.Controls[0] as SplitContainer;
    }

    /// <summary>
    /// Update visibility and enabled state of apply and cancel button.
    /// </summary>
    private void UpdateApplyCancel()
    {
      RecordsetView view = ActiveRecordsetView;
      if (view == null)
      {
        applyButton.Visible = false;
        cancelButton.Visible = false;
        readOnlyLabel.Visible = false;
        readOnlyPictureBox.Visible = false;
        labelsTooltip.RemoveAll();
      }
      else
      {
        if (view.GridView.ReadOnly)
        {
          readOnlyLabel.Visible = true;
          readOnlyPictureBox.Visible = true;
          labelsTooltip.SetToolTip(readOnlyLabel, view.GridView.Model.readonly_reason());
          labelsTooltip.SetToolTip(readOnlyPictureBox, view.GridView.Model.readonly_reason());
          applyButton.Visible = false;
          cancelButton.Visible = false;
        }
        else
        {
          readOnlyLabel.Visible = false;
          readOnlyPictureBox.Visible = false;
          labelsTooltip.RemoveAll();
          applyButton.Visible = true;
          cancelButton.Visible = true;
          applyButton.Enabled = view.Dirty;
          cancelButton.Enabled = view.Dirty;
        }
      }
    }

    #endregion

  }
}
