/* 
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#ifndef __APPLE__
  #include <algorithm>
#endif

using namespace mforms;

MenuBase::MenuBase()
: _parent(0)
{
  _impl = &mforms::ControlFactory::get_instance()->_menu_item_impl;
}

MenuBase::~MenuBase()
{
  for (std::vector<MenuItem*>::iterator iter = _items.begin(); iter != _items.end(); ++iter)
    (*iter)->release();
  _items.clear();  
}

void MenuBase::add_item(MenuItem *item)
{
  insert_item(-1, item);
}

void MenuBase::insert_item(int index, MenuItem *item)
{
  if (index < 0 || index > (int) _items.size())
    index = _items.size();

  item->_parent = this;
  
  _impl->insert_item(this, index, item);
  _items.insert(_items.begin()+index, item);
  
 // item->retain();
}

void MenuBase::remove_all()
{
  _impl->remove_item(this, NULL); // null means remove all
  std::vector<MenuItem*>::iterator iter;
  for (iter = _items.begin(); iter != _items.end(); ++iter)
  {
    (*iter)->_parent = 0;
    (*iter)->release();
  }
  _items.clear();
}

void MenuBase::remove_item(MenuItem *item)
{
  std::vector<MenuItem*>::iterator iter = std::find(_items.begin(), _items.end(), item);
  if (iter != _items.end())
  {
    (*iter)->_parent = 0;
    _impl->remove_item(this, item);
    item->release();
    _items.erase(iter);
  }
}

void MenuBase::set_enabled(bool flag)
{
  _impl->set_enabled(this, flag);
}

bool MenuBase::get_enabled()
{
  return _impl->get_enabled(this);
}

MenuItem *MenuBase::find_item(const std::string &name)
{
  for (std::vector<MenuItem*>::const_iterator iter = _items.begin(); iter != _items.end(); ++iter)
  {
    if ((*iter)->get_name() == name)
      return *iter;
    MenuItem *item;
    if ((item = (*iter)->find_item(name)))
      return item;
  }
  return 0;
}


void MenuBase::validate()
{
  for (std::vector<MenuItem*>::const_iterator iter = _items.begin(); iter != _items.end(); ++iter)
    (*iter)->validate();
}


MenuBar *MenuBase::get_menubar()
{
  if (dynamic_cast<MenuBar*>(this) != NULL)
    return dynamic_cast<MenuBar*>(this);

  MenuBase *p = _parent;
  while (p && p->get_parent())
    p = p->get_parent();
  
  if (p)
    return dynamic_cast<MenuBar*>(p);
  return 0;
}


MenuItem::MenuItem(const std::string &title, const MenuItemType type)
: MenuBase()
{
  _impl->create_menu_item(this, title, type); // ! Warning there will be no checked menu with this!
}
  
void MenuItem::set_title(const std::string &title)
{
  _impl->set_title(this, title);
}

std::string MenuItem::get_title() 
{
  return _impl->get_title(this);
}
  
void MenuItem::set_shortcut(const std::string &shortcut)
{
  _impl->set_shortcut(this, shortcut);
}

  
void MenuItem::set_checked(bool flag)
{
  _impl->set_checked(this, flag);
}

bool MenuItem::get_checked() 
{
  return _impl->get_checked(this);
}

void MenuItem::callback()
{
  _clicked_signal();
}

void MenuItem::validate()
{
  if (_validate)
    set_enabled(_validate());
  
  if (!_items.empty())
    MenuBase::validate();
}

void MenuItem::set_validator(const boost::function<bool ()> &slot)
{
  _validate = slot;
}

void MenuItem::set_name(const std::string &name)
{
  _name= name;
}


MenuBar::MenuBar()
: MenuBase()
{
  _impl->create_menu_bar(this);
}


void MenuBar::will_show_submenu_from(MenuItem *item)
{
  _signal_will_show(item);
}


void MenuBar::set_item_enabled(const std::string &item_name, bool flag)
{
  MenuItem *item = find_item(item_name);
  if (item)
    item->set_enabled(flag);
}


void MenuBar::set_item_checked(const std::string &item_name, bool flag)
{
  MenuItem *item = find_item(item_name);
  if (item)
    item->set_checked(flag);
}
