/* 
 * Copyright (c) 2007, 2012, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System;
using System.ComponentModel;
using System.Windows.Forms;

using ScintillaNet;

using MySQL.Grt;

namespace MySQL.GUI.Workbench.Plugins
{
  public partial class StoredNoteEditor : ObjectEditorPlugin
  {
    #region Member Variables

    private MySQL.Grt.StoredNoteEditorBE storedNoteEditorBE { get { return Backend as MySQL.Grt.StoredNoteEditorBE; } }

    #endregion

    #region Constructors

    protected StoredNoteEditor()
    {
      InitializeComponent();
    }

    public StoredNoteEditor(GrtManager GrtManager, GrtValue GrtObject)
      : base(new MySQL.Grt.StoredNoteEditorBE(GrtManager, GrtObject))
    {
      InitializeComponent();

      // Init BE
      InitializingControls = true;
      try
      {
        sqlEd = new SqlEditorHelper(GrtManager, content, null);
        sqlEd.sqlEditor.DocumentChange += new System.EventHandler<ScintillaNet.NativeScintillaEventArgs>(textBox_DocumentChange);
        RefreshFormData();
      }
      finally
      {
        InitializingControls = false;
      }

      storedNoteEditorBE.add_refresh_ui_handler(RefreshFormDataInvoke);
    }

    #endregion

    #region Form implemenation

    protected override void RefreshFormData()
    {
      TabText = storedNoteEditorBE.get_name();

      if (sqlEd.EditorBE != storedNoteEditorBE)
        sqlEd.SetEditorBackend(storedNoteEditorBE, null);

      // Disable syntax highlighting and error checking if we have a note here (not a script).
      if (storedNoteEditorBE.is_sql_script())
        sqlEd.sqlEditor.BackgroundAction = sqlEd.sqlEditor.CheckSql;
      else
      {
        sqlEd.sqlEditor.BackgroundAction = null;
        sqlEd.sqlEditor.Folding.IsEnabled = false;
        sqlEd.sqlEditor.Styles.ClearAll();
        sqlEd.sqlEditor.Margins.Margin1.Width = 0; // markers
        sqlEd.sqlEditor.Margins.Margin2.Width = 0; // indicators
      }

      bool isutf8;
      sqlEd.sqlEditor.Text = storedNoteEditorBE.get_text(out isutf8);

      if (isutf8)
        sqlEd.sqlEditor.Enabled = true;
      else
      {
        sqlEd.sqlEditor.Enabled = false;
        sqlEd.sqlEditor.Text = "The contents of this file cannot be displayed because it is in an unknown encoding.\r\n" +
        "You may save the file and view it with an external program.";
      }

      applyButton.Enabled = false;
      discardButton.Enabled = false;
    }

    private void applyButton_Click(object sender, EventArgs e)
    {
      storedNoteEditorBE.set_text(sqlEd.sqlEditor.Text);
      RefreshFormData();
    }

    private void discardButton_Click(object sender, EventArgs e)
    {
      RefreshFormData();
    }

    private void saveButton_Click(object sender, EventArgs e)
    {
      if (saveFileDialog1.ShowDialog() == DialogResult.OK)
      {
        storedNoteEditorBE.save_file(saveFileDialog1.FileName);
      }
    }

    private void loadButton_Click(object sender, EventArgs e)
    {
      if (openFileDialog1.ShowDialog() == DialogResult.OK)
      {
        storedNoteEditorBE.load_file(openFileDialog1.FileName);
        RefreshFormData();
        applyButton.Enabled = true;
        discardButton.Enabled = true;
      }
    }

    private void cutItem_Click(object sender, EventArgs e)
    {
      sqlEd.sqlEditor.Clipboard.Cut();
    }

    private void copyItem_Click(object sender, EventArgs e)
    {
      sqlEd.sqlEditor.Clipboard.Copy();
    }

    private void pasteItem_Click(object sender, EventArgs e)
    {
      sqlEd.sqlEditor.Clipboard.Paste();
    }

    private void selectAllItem_Click(object sender, EventArgs e)
    {
      sqlEd.sqlEditor.Selection.SelectAll();
    }

    private void undoItem_Click(object sender, EventArgs e)
    {
      sqlEd.sqlEditor.UndoRedo.Undo();
    }

    private void redoItem_Click(object sender, EventArgs e)
    {
      sqlEd.sqlEditor.UndoRedo.Redo();
    }

    private void editorContextMenu_Opening(object sender, CancelEventArgs e)
    {
      // Enable/Disable menu items depending on the edit content.
      undoItem.Enabled = sqlEd.sqlEditor.UndoRedo.IsUndoEnabled;
      cutItem.Enabled = sqlEd.sqlEditor.Selection.Length > 0;
      copyItem.Enabled = sqlEd.sqlEditor.Selection.Length > 0;
      pasteItem.Enabled = sqlEd.sqlEditor.Clipboard.CanPaste;
      selectAllItem.Enabled = sqlEd.sqlEditor.TextLength > 0;
    }

    private const uint TEXT_MODIFIED_FLAGS = Constants.SC_MOD_BEFOREDELETE | Constants.SC_MOD_BEFOREINSERT |
      Constants.SC_MOD_DELETETEXT | Constants.SC_MOD_INSERTTEXT;

    private void textBox_DocumentChange(object sender, ScintillaNet.NativeScintillaEventArgs e)
    {
      SCNotification scn = e.SCNotification;
      int modType = scn.modificationType;

      if ((modType & TEXT_MODIFIED_FLAGS) > 0)
      {
        applyButton.Enabled = true;
        discardButton.Enabled = true;
      }
    }

    #endregion

  }
}