/* 
 * Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"
#include "db_helpers.h"
#include "base/string_utilities.h"



std::string bec::get_host_identifier_for_connection(const db_mgmt_ConnectionRef &connection)
{
  grt::DictRef params(connection->parameterValues());
  std::string host_id;
  std::string driver;
  
  if (connection->driver().is_valid())
    driver = connection->driver()->name();
  if (g_str_has_suffix(driver.c_str(), "Socket"))
  {
    host_id = "local";
  }
  else if (g_str_has_suffix(driver.c_str(), "SSH"))
  {
    host_id = base::strfmt("%s:%i@%s",
                           params.get_string("hostName").c_str(),
                           (int) params.get_int("port"),
                           params.get_string("sshHost").c_str());
  }
  else
  {
    host_id = base::strfmt("%s:%i", 
                           params.get_string("hostName").c_str(),
                           (int) params.get_int("port"));
  }
  if (connection->driver().is_valid())
    host_id = base::strfmt("%s@%s", 
                           connection->driver()->owner()->name().c_str(),
                           host_id.c_str());
  
  return host_id;
}


std::string bec::get_description_for_connection(const db_mgmt_ConnectionRef &connection)
{
  std::string conn_type;
  std::string driver, server;
  grt::DictRef params(connection->parameterValues());
  
  if (connection->driver().is_valid())
  {
    driver = connection->driver()->name();
    server = db_mgmt_RdbmsRef::cast_from(connection->driver()->owner())->caption();
  }
  else
    return "Invalid Connection Description";

  std::string user = params.get_string("userName");
  
  if (g_str_has_suffix(driver.c_str(), "Socket"))
  {
    std::string path = base::trim(params.get_string("socket"));
    conn_type = base::strfmt("%s using local socket/pipe at \"%s\" with user %s", 
                             server.c_str(),
                             path.empty()?"default path":path.c_str(),
                             user.c_str());
  }
  else if (g_str_has_suffix(driver.c_str(), "SSH"))
  {    
    conn_type = base::strfmt("%s at %s:%i through SSH tunnel at %s with user %s",
                             server.c_str(),
                             params.get_string("hostName").c_str(),
                             (int) params.get_int("port"),
                             params.get_string("sshHost").c_str(),
                             user.c_str());
  }
  else // TCP
  {
    conn_type = base::strfmt("%s at %s:%i with user %s",
                             server.c_str(),
                             params.get_string("hostName").c_str(),
                             (int) params.get_int("port"),
                             user.c_str());
  }

  return conn_type;  
}
