/* 
 * Copyright (c) 2010, 2011 Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#import "MScintillaView.h"
#include "MSQLEditorController.h"
#import "MCPPUtilities.h"
#import "NSMenu_extras.h"

@implementation MScintillaView


- (void)setEditorBackEnd:(const Sql_editor::Ref&)be
{
  mBackEnd= be;
}


- (NSMenu *)menuForEvent:(NSEvent *)theEvent
{
  if (mBackEnd)
  {      
    bec::MenuItemList items= mBackEnd->get_context_menu();
    
    if (!items.empty())
    {      
      for (bec::MenuItemList::iterator iter= items.begin(); iter != items.end(); ++iter)
      {        
        bool enabled= iter->enabled;
        if (iter->name == "undo")
          enabled= [[self content] canUndo];
        else if (iter->name == "redo")
          enabled= [[self content] canRedo];
        else if (iter->name == "cut")
          enabled= [[self content] selectedRange].length > 0;
        else if (iter->name == "copy")
          enabled= [[self content] selectedRange].length > 0;
        else if (iter->name == "paste")
        {
          NSArray* supportedTypes = [NSArray arrayWithObjects: NSStringPboardType, nil];
          NSString *bestType = [[NSPasteboard generalPasteboard] availableTypeFromArray: supportedTypes];
          if (bestType)
            enabled= true;
          else
            enabled= false;
        }
        else
          if (iter->name == "toggle_wrap_lines")
          {
            int wrap_mode = [self getGeneralProperty: SCI_GETWRAPMODE]; // None=0, Word=1, Char=2
            iter->checked = (wrap_mode == 1);
          }
        iter->enabled= enabled;
      }

      return [NSMenu menuFromMenuItems:items
                                action:@selector(activateMenuItem:)
                                target:self];
    }
  }
  return [super menuForEvent: theEvent];
}



- (void)activateMenuItem:(id)sender
{
  if (mBackEnd && [sender representedObject])
  {
    std::string action([[sender representedObject] UTF8String]);
    
    if (action == "undo")
    {
      [[self content] performSelector:@selector(undo:) withObject:nil];
    }
    else if (action == "redo")
    {
      [[self content] performSelector:@selector(redo:) withObject:nil];
    }
    else if (action == "cut")
    {
      [[self content] performSelector:@selector(cut:) withObject:nil];
    }
    else if (action == "copy")
    {
      [[self content] performSelector:@selector(copy:) withObject:nil];
    }
    else if (action == "paste")
    {
      [[self content] performSelector:@selector(paste:) withObject:nil];
    }
    else if (action == "delete")
    {
      [[self content] deleteBackward:nil];
    }
    else if (action == "select_all")
    {
      [[self content] selectAll:nil];
    }
    else
      if (action == "toggle_wrap_lines")
      {
        int wrap_mode = [self getGeneralProperty: SCI_GETWRAPMODE]; // None=0, Word=1, Char=2
        if (wrap_mode == 1)
          [self setGeneralProperty: SCI_SETWRAPMODE value: 0];
        else
          [self setGeneralProperty: SCI_SETWRAPMODE value: 1];
      }
      else
      {
        try
        {
          mBackEnd->activate_context_menu_item(action); 
        }
        catch (const std::exception &exc)
        {
          NSRunAlertPanel(@"Error", @"Could not execute plugin: %s", @"OK", nil, nil,
                          exc.what());
        }
      }
  }
}

- (void) dealloc
{
  [mFindPanel release];
  [super dealloc];
}


- (id)delegate
{
  return delegate;
}


- (BOOL)expandsOnLayoutVertically:(BOOL)vertically
{
  return YES;
}

#pragma mark Find & Replace panel support
- (void)enableReplaceInFindPanel: (BOOL)flag
{
  if (!flag)
  {
    [mFindTypeSegmented setSelectedSegment: 0];
    for (id view in [mFindPanel subviews])
      if ([view tag] >= 10 && [view tag] <= 13)
        [view setHidden: YES];
    [mFindPanel setFrameSize: NSMakeSize(NSWidth([mFindPanel frame]), 23)];
  }
  else
  {
    [mFindTypeSegmented setSelectedSegment: 1];
    for (id view in [mFindPanel subviews])
      if ([view tag] >= 10 && [view tag] <= 13)
        [view setHidden: NO];    
    [mFindPanel setFrameSize: NSMakeSize(NSWidth([mFindPanel frame]), 46)];
  }
  [mFindLabel setStringValue: @""];
}

- (NSView*)embedableFindPanel
{
  if (!mFindPanel)
  {
    if (![[NSBundle bundleForClass: [MScintillaView class]] loadNibFile: @"EmbedableFindPane"
                                                      externalNameTable: [NSDictionary dictionaryWithObject: self forKey: NSNibOwner]
                                                               withZone: NSDefaultMallocZone()])
      NSLog(@"Could not load EmbedableFindPane nib file for Find Panel");
    
    mIgnoreCase = YES;
    mWrapAround = YES;
  }
  [self enableReplaceInFindPanel: NO];
  return mFindPanel;
}


- (void)focusFindPanel
{
  [mFindLabel setStringValue: @""];
  [[mFindPanel window] makeFirstResponder: mFindText];
  [mFindText selectText: nil];
}


- (IBAction)findActionClicked:(id)sender
{
  int count;
  switch ([sender tag])
  {
    case 5: // find&replace button
      [self enableReplaceInFindPanel: [mFindTypeSegmented selectedSegment] == 1];
      break;

    case 1: // find text
    case 6: // find back/next segmented
      if ([[mFindText stringValue] length] == 0)
        [mFindLabel setStringValue: @""];
      else if (![sender isKindOfClass: [NSSegmentedControl class]] || [sender selectedSegment] == 1)
      {
        if ([self findAndHighlightText: [mFindText stringValue]
                             matchCase: !mIgnoreCase
                             wholeWord: mMatchWhole
                              scrollTo: YES
                                  wrap: mWrapAround
                             backwards: NO])
          [mFindLabel setStringValue: @"Found match"];
        else
          [mFindLabel setStringValue: @"Not found"];
      }
      else
      {
        if ([self findAndHighlightText: [mFindText stringValue]
                             matchCase: !mIgnoreCase
                             wholeWord: mMatchWhole
                              scrollTo: YES
                                  wrap: mWrapAround
                             backwards: YES])
          [mFindLabel setStringValue: @"Found match"];
        else
          [mFindLabel setStringValue: @"Not found"];
      }
      break;

    case 7: // done
      if (mFindPanel)
        [delegate scintillaViewFindPanelClose: self];
      break;
      
    case 10: // replace all
      if ([[mFindText stringValue] length] > 0)
      {
        if ((count = [self findAndReplaceText: [mFindText stringValue]
                                       byText: [mReplaceText stringValue]
                                    matchCase: !mIgnoreCase
                                    wholeWord: mMatchWhole
                                        doAll: YES]) > 0)
          [mFindLabel setStringValue: [NSString stringWithFormat: @"Replaced %i matches", count]];
        else
          [mFindLabel setStringValue: @"No matches found"];
        [self setNeedsDisplay: YES];
      }
      break;
      
    case 11: // replace selection (without find)
      if ([[self content] selectedRange].length > 0)
      {
        NSString *text = [mReplaceText stringValue];
        [[self content] insertText: text];
        // For a single replace we set the new selection to the replaced text.
        [self setGeneralProperty: SCI_SETSELECTIONEND value: [self getGeneralProperty: SCI_SETSELECTIONSTART] + [text length]];
        [mFindLabel setStringValue: @""];
      }
      break;

    case 12: // replace and find
    case 13: // replace textfield
      if ([[mFindText stringValue] length] > 0)
      {
        if ([self findAndReplaceText: [mFindText stringValue]
                              byText: [mReplaceText stringValue]
                           matchCase: !mIgnoreCase
                           wholeWord: mMatchWhole
                               doAll: NO] > 0)
          [mFindLabel setStringValue: @"Replaced 1 match"];
        else
          [mFindLabel setStringValue: @"Not found"];
      }
      break;

      // Menu
      /*
      // regex is not supported natively in Scintilla. In windows, it fetches the whole text
      // from backend to perform the search
    case 20: // plain text
      [[findMenu itemWithTag: 20] setState: NSOnState];
      [[findMenu itemWithTag: 21] setState: NSOffState];      
      break;
    case 21: // regex
      [[findMenu itemWithTag: 20] setState: NSOffState];
      [[findMenu itemWithTag: 21] setState: NSOnState];
      break;
       */
    case 30: // ignore case
      mIgnoreCase = [sender state] != NSOnState;
      [sender setState: mIgnoreCase ? NSOnState : NSOffState];
      break;
    case 31: // match whole words
      mMatchWhole = [sender state] != NSOnState;
      [sender setState: mMatchWhole ? NSOnState : NSOffState];
      break;
    case 32: // wrap around
      mWrapAround = [sender state] != NSOnState;
      [sender setState: mWrapAround ? NSOnState : NSOffState];
      break;
  }
}

- (void)setShowsSpecialCharacters:(BOOL)flag
{
  [self setGeneralProperty: SCI_SETVIEWWS value: flag ? SCWS_VISIBLEALWAYS : SCWS_INVISIBLE];
  [self setGeneralProperty: SCI_SETVIEWEOL value: flag];
}

- (void)menuWillOpen:(NSMenu *)menu
{
  [[menu itemWithTag: 30] setState: mIgnoreCase ? NSOnState : NSOffState];
  [[menu itemWithTag: 31] setState: mMatchWhole ? NSOnState : NSOffState];
  [[menu itemWithTag: 32] setState: mWrapAround ? NSOnState : NSOffState];
}

@end
