/* 
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#import "WBQueryTab.h"
#import "WBSQLQueryPanel.h"
#import "MSQLEditorController.h"
#import "MTabSwitcher.h"
#import "MResultsetViewer.h"
#import "WBSplitViewUnbrokenizerDelegate.h"
#import "MVerticalLayoutView.h"
#import "MScintillaView.h"

#include <mforms/toolbar.h>
#include <mforms/../cocoa/MFView.h>

#include "sqlide/wb_sql_editor_form.h"

@interface ResultTabViewItem : NSTabViewItem
{
  MResultsetViewer *viewer;
}

- (void)setRSViewer: (MResultsetViewer*)viewer;
- (MResultsetViewer*)RSViewer;

@end


@implementation ResultTabViewItem

- (void)setRSViewer: (MResultsetViewer*)aViewer
{
  [viewer autorelease];
  viewer = [aViewer retain];
}

- (MResultsetViewer*)RSViewer
{
  return viewer;
}

- (void) dealloc
{
  [viewer release];
  [super dealloc];
}

@end





@implementation WBQueryTab


- (NSView*)topView
{
  return mPanel;
}

- (NSString*)title
{
  SqlEditorForm::Ref be(mOwner.backEnd);
  return [NSString stringWithCPPString: be->sql_editor_caption(be->sql_editor_index([mEditorController backEnd]))];
}

- (NSImage*)tabIcon
{
  return [NSImage imageNamed: @"tab_icon_db.query.QueryBuffer.png"];
}

- (MScintillaView*)editorView
{
  return mEditor;
}

- (NSString*)string
{
  return [mEditorController string];
}

- (void)setString:(NSString*)string
{
}

- (void)appendString:(NSString*)string
{
}


- (MSQLEditorController*)editorController
{
  return mEditorController;
}

- (void)updateResultsetTabs
{
  SqlEditorForm::Ref backEnd = mOwner.backEnd;
  int editor_index = backEnd->sql_editor_index([mEditorController backEnd]);
  int n, rs_count = backEnd->recordset_count(editor_index);
  NSMutableArray *leftoverResults = [[[mResultsTabView tabViewItems] mutableCopy] autorelease];
  int non_rs_tabs = 0;
  
  for (NSTabViewItem *item in leftoverResults)
  {
    if ([item isKindOfClass: [ResultTabViewItem class]])
      break;
    non_rs_tabs++;
  }
  
  int last_added_tab = -1;
  for (n = 0; n < rs_count; ++n)
  {
    Recordset::Ref rset(backEnd->recordset(editor_index, n));
    
    NSInteger index = [mResultsTabView indexOfTabViewItemWithIdentifier: [NSString stringWithFormat: @"rset%li", rset->key()]];
    if (index == NSNotFound)
    {
      MResultsetViewer *rsview = [[[MResultsetViewer alloc] initWithRecordset: rset] autorelease];
      [[[rsview gridView] enclosingScrollView] setBorderType: NSNoBorder];
      
      ResultTabViewItem *tabViewItem= 
      [[[ResultTabViewItem alloc] initWithIdentifier: [NSString stringWithFormat:@"rset%li", rset->key()]] autorelease];
      
      [tabViewItem setRSViewer: rsview];
      [tabViewItem setLabel: [NSString stringWithCPPString: rset->caption()]];
      [tabViewItem setView: [rsview view]];
      
      [mResultsTabView insertTabViewItem: tabViewItem atIndex: n+non_rs_tabs];
      last_added_tab = n + non_rs_tabs;
      [rsview refresh];
    }
    else
    {
      [leftoverResults removeObject: [mResultsTabView tabViewItemAtIndex: index]];
    }
  }
  
  if (last_added_tab >= 0)
    [mResultsTabView selectTabViewItemAtIndex: last_added_tab];
  
  // remove tabs that are not used anymore
  for (id tab in leftoverResults)
  {
    if ([tab isKindOfClass: [ResultTabViewItem class]])
      [mResultsTabView removeTabViewItem: tab];
  }
}


- (MResultsetViewer*)selectedResultset
{
  NSTabViewItem *selectedItem= [mResultsTabView selectedTabViewItem];
  
  if ([selectedItem isKindOfClass: [ResultTabViewItem class]])
    return [(ResultTabViewItem*)selectedItem RSViewer];
  
  return nil;
}


- (void)updateSplitterPosition
{
  int tabCount = [mResultsTabView numberOfTabViewItems];
  if (!mQueryCollapsed)
  {
    if (tabCount == 0)
    {
      mSplitterPosition = NSHeight([[mResultsTabView superview] frame]) + [mSplitView dividerThickness];
      [mSplitView setPosition: NSHeight([mSplitView frame]) - [mSplitView dividerThickness]
             ofDividerAtIndex: 0];
    }
    else
    {
      if (tabCount > 0 && mLastResultTabViewCount == 0)
        [mSplitView setPosition: NSHeight([mSplitView frame]) - mSplitterPosition
               ofDividerAtIndex: 0];
    }
  }
  mLastResultTabViewCount = tabCount;
  mSplitterUpdatePending = NO;
}


- (void)updateActiveRecordsetTitle
{
  MResultsetViewer *rset = [self selectedResultset];
  if (rset)
  {
    [[mResultsTabView selectedTabViewItem] setLabel: [NSString stringWithCPPString: [rset recordset]->caption()]];
    [mResultsTabSwitcher setNeedsDisplay: YES];
    
    bool flag = [rset recordset]->has_pending_changes();
    [mApplyButton setEnabled: flag];
    [mCancelButton setEnabled: flag];    
  }
}

- (void)removeRecordsetWithIdentifier:(id)identifier
{
  NSInteger index = [mResultsTabView indexOfTabViewItemWithIdentifier: identifier];
  if (index != NSNotFound)
  {
    id item = [mResultsTabView tabViewItemAtIndex: index];
    if (item)
      [mResultsTabView removeTabViewItem: item];
  }  
}


- (IBAction)actionButtonClicked:(id)sender
{
  if ([sender tag] == 1) // apply
  {
    if ([[self selectedResultset] hasPendingChanges])
      [[self selectedResultset] saveEdits: nil];
  }
  else
  {
    if ([[self selectedResultset] hasPendingChanges])
      [[self selectedResultset] discardEdits: nil];
  }
}


- (void)setQueryCollapsed: (BOOL)flag
{
  mSplitterPosition = NSHeight([[mResultsTabView superview] frame]) + [mSplitView dividerThickness];
  [mSplitView setPosition: 0
         ofDividerAtIndex: 0];
  mQueryCollapsed = YES;
}


- (IBAction)activateQueryArea: (id)sender
{
  [[mPanel window] makeFirstResponder: [mEditor content]];
}


////

- (void)showFindPanel:(MScintillaView*)editor
{
  NSView *panel = [mEditor embedableFindPanel];
  if (![panel superview])
  {
    [mEditorContainer addSubview: panel positioned: NSWindowBelow relativeTo: mEditor];
    [mEditorContainer tile];
  }
  [mEditor focusFindPanel];
}


- (void)scintillaViewFindPanelClose:(MScintillaView *)editor
{
  NSView *panel = [mEditor embedableFindPanel];
  [panel retain];
  [panel removeFromSuperview];
  [mEditorContainer tile];
  
  [[mPanel window] makeFirstResponder: [mEditor content]];
}


#pragma mark Resultset TabView delegates

- (BOOL)tabView:(NSTabView *)tabView
willCloseTabViewItem:(NSTabViewItem*)tabViewItem
{
  if ([tabViewItem isKindOfClass: [ResultTabViewItem class]])
  {
    if ([[(ResultTabViewItem*)tabViewItem RSViewer] recordset]->can_close(true))
      [[(ResultTabViewItem*)tabViewItem RSViewer] close];
    return NO;
  }
  return YES;
}

- (BOOL)tabView:(NSTabView*)tabView itemHasCloseButton:(NSTabViewItem*)item
{
  return YES;
}

- (void)tabView:(NSTabView *)tabView didSelectTabViewItem:(NSTabViewItem *)tabViewItem
{
  MResultsetViewer *rset= [self selectedResultset];
  SqlEditorForm::Ref backEnd = mOwner.backEnd;
  int index = backEnd->sql_editor_index([mEditorController backEnd]);
  if (index >= 0)
  {
    if (rset)
    {
      Recordset::Ref recordset([rset recordset]);
      backEnd->active_recordset(index, recordset);
      if (!recordset->is_readonly())
      {
        /*
        [mApplyButton sizeToFit];
        [mCancelButton sizeToFit];
        [mApplyButton setFrameSize: NSMakeSize(70, NSHeight([mApplyButton frame]))];
        [mCancelButton setFrameSize: NSMakeSize(70, NSHeight([mCancelButton frame]))];
         */

        [mApplyButton setHidden: NO];
        [mCancelButton setHidden: NO];
        [mInfoText setHidden: YES];
        [mInfoIcon setHidden: YES];
        
        bool flag = recordset->has_pending_changes();
        [mApplyButton setEnabled: flag];
        [mCancelButton setEnabled: flag];
      }
      else
      {
        [mApplyButton setHidden: YES];
        [mCancelButton setHidden: YES];
        [mInfoText setToolTip: [NSString stringWithCPPString: recordset->readonly_reason()]];
        [mInfoText setHidden: NO];
        [mInfoIcon setToolTip: [NSString stringWithCPPString: recordset->readonly_reason()]];
        [mInfoIcon setHidden: NO];
      }
    }
    else
    {
      backEnd->active_recordset(index, Recordset::Ref());
      [mApplyButton setHidden: YES];
      [mCancelButton setHidden: YES];
      [mInfoText setHidden: YES];
      [mInfoIcon setHidden: YES];
    }
    [mResultsTabSwitcher tile];
  }
}


//- (void)tabView:(NSTabView *)tabView willDisplayMenu:(NSMenu *)menu forTabViewItem:(NSTabViewItem *)item
//{
//}


- (void)tabViewDidChangeNumberOfTabViewItems:(NSTabView *)aTabView
{
  if (!mSplitterUpdatePending)
  {
    mSplitterUpdatePending = YES;
    [self performSelector: @selector(updateSplitterPosition) withObject:nil afterDelay:0.1];
  }
}

- (BOOL)tabView:(NSTabView *)tabView willReorderTabViewItem:(NSTabViewItem *)item
{
  if (tabView == mResultsTabView)
  {
    int index = 0;
    // find the new relative index of the recordset
    for (NSTabViewItem *ti in [tabView tabViewItems])
    {
      if (ti == item)
        break;
      if ([ti isKindOfClass: [ResultTabViewItem class]])
        index++;
    }
    if ([item isKindOfClass: [ResultTabViewItem class]])
    {
      MResultsetViewer *rsviewer = [(ResultTabViewItem*)item RSViewer];
      if (rsviewer && [rsviewer recordset])
      {
        SqlEditorForm::Ref backEnd = mOwner.backEnd;
        if (!backEnd->recordset_reorder(backEnd->sql_editor_index([mEditorController backEnd]), [rsviewer recordset], index))
          return NO;
      }
    }
  }
  return YES;
}

#pragma mark init/dealloc

- (id)initWithOwner: (WBSQLQueryPanel*)owner
            backEnd: (Sql_editor::Ref)backend
{
  self = [super init];
  if (self)
  {
    mOwner = owner;
    
    [NSBundle loadNibNamed: @"WBQueryTab" owner: self];
    
    [mEditorContainer setBackgroundColor: [NSColor colorWithDeviceWhite: 230/255.0 alpha: 1.0]];
    //[mEditorContainer setPaddingLeft: 0 right: 0 top: 3 bottom: 0];
    
    SqlEditorForm::Ref editorBE = owner.backEnd;
    
    boost::shared_ptr<mforms::ToolBar> toolbar(editorBE->sql_editor_toolbar(editorBE->sql_editor_index(backend)));
    [mEditorContainer addSubview: nsviewForView(toolbar.get()) 
                      positioned: NSWindowBelow
                      relativeTo: nil];

    mEditorController = [[MSQLEditorController alloc] init];
    
    [mEditorController setView: mEditor];
    [mEditorController setEditorBackEnd: backend];
    [mEditor setEditorBackEnd: [mEditorController backEnd]];
    [mResultsTabSwitcher setTabStyle: MEditorBottomTabSwitcher];
    [mResultsTabSwitcher setAllowTabReordering: YES];
    
    [mSplitViewDelegate setTopCollapsedMinHeight: 30];
    [mSplitViewDelegate setTopExpandedMinHeight: 30];
    [mSplitViewDelegate setBottomCollapsedMinHeight: 60];
    [mSplitViewDelegate setBottomExpandedMinHeight: 60];

    [mInfoText setStringValue: @"Read Only"];
    [mInfoText sizeToFit];

    mLastResultTabViewCount = -1;
    [self updateSplitterPosition];
    
    [mEditorContainer setExpandSubviewsByDefault: NO];
    [(MVerticalLayoutView*)[mResultsTabSwitcher superview] setExpandSubviewsByDefault: YES];    
  }
  return self;
}


- (void)dealloc
{
  [mSplitViewDelegate release];
  [mEditorController release];
  [mPanel release];
  [super dealloc];
}
@end
