/**
 * Copyright (c) 2008, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using Aga.Controls.Tree;

using MySQL.Grt;
using MySQL.Workbench;
using MySQL.Controls;
using MySQL.Utilities;

namespace MySQL.GUI.Workbench
{
  public partial class ModelLayerForm : TabDocument
  {
    private ModelDiagramForm model;
    private MySQL.Grt.TreeModel layerTreeBE;
    private SimpleGrtTreeModel layerTreeModel;
    private GrtManager manager;
    private bool doubleClickedItem = false;

    private ModelLayerForm()
    {
      InitializeComponent();
    }

    public ModelLayerForm(ModelDiagramForm model, GrtManager manager)
      : this()
    {
      this.model = model;
      this.manager = manager;

      layerTreeBE = model.DiagramWrapper.get_layer_tree();
      layerTreeModel = new SimpleGrtTreeModel(layerTreeView, layerTreeBE, nodeStateIcon, false);
      layerTreeModel.AddColumn(nameNodeControl, 0, false);
      layerTreeModel.AddColumn(selectedNodeControl, 11, false);
      layerTreeModel.EnableRowColor(10);
      layerTreeView.Model = layerTreeModel;

      layerTreeBE.add_tree_refresh_handler(invokeRefresh);

      layerTreeView.Expanded+=new EventHandler<TreeViewAdvEventArgs>(layerTreeView_Expanded);
      layerTreeView.Collapsed+=new EventHandler<TreeViewAdvEventArgs>(layerTreeView_Collapsed);

      ExpandImportantNodes();
    }

    public void UpdateLayerTree()
    {
      if (manager.try_soft_lock_globals_tree())
      {
        AgaTreeStateKeeper stateKeeper = new AgaTreeStateKeeper(layerTreeView);
        try
        {
          layerTreeModel.RefreshModel();
        }
        finally
        {
          stateKeeper.RestoreStates();
          manager.soft_unlock_globals_tree();
        }
      }
    }

    private void invokeRefresh(NodeId node, int oldCount)
    {
      if (InvokeRequired)
        BeginInvoke((Action) (() => UpdateLayerTree()));
      else
        UpdateLayerTree();
    }

    private void ExpandImportantNodes()
    {
      layerTreeView.Root.Expand();
			foreach (TreeNodeAdv node in layerTreeView.Root.Children)
      {
        node.Expand();

        foreach (TreeNodeAdv node2 in node.Children)
          node2.Expand();
      }
    }


    void layerTreeView_Expanded(object sender, TreeViewAdvEventArgs e)
    {
      if (doubleClickedItem)
      {
        doubleClickedItem = false;
        e.Node.Collapse();
      }
    }

    void layerTreeView_Collapsed(object sender, TreeViewAdvEventArgs e)
    {
      if (doubleClickedItem)
      {
        doubleClickedItem = false;
        e.Node.Expand();
      }
    }

    private void layerTreeView_DoubleClick(object sender, EventArgs e)
    {
			if (layerTreeView.SelectedNode != null)
			{
				GrtTreeNode node = layerTreeView.SelectedNode.Tag as GrtTreeNode;
				if (node != null)
					layerTreeBE.activate_node(node.NodeId);
			}
		}

    public void updateContextMenu(List<NodeId> nodes)
    {
      layerMenuStrip.Items.Clear();

      List<MySQL.Grt.MenuItem> items = layerTreeModel.GrtTree.get_popup_items_for_nodes(nodes);
      foreach (MySQL.Grt.MenuItem grtMenuItem in items)
      {
        ToolStripMenuItem item = new ToolStripMenuItem(grtMenuItem.get_caption());
        item.Enabled = grtMenuItem.get_enabled();
        item.Tag = grtMenuItem.get_name();
        item.Click += new EventHandler(contextMenuClick);
        layerMenuStrip.Items.Add(item);
      }
    }

    private void layerMenuStrip_Opening(object sender, CancelEventArgs e)
    {
      List<NodeId> nodes = new List<NodeId>();
      foreach (TreeNodeAdv node in layerTreeView.SelectedNodes)
      {
        GrtTreeNode treeNode = node.Tag as GrtTreeNode;
        nodes.Add(treeNode.NodeId);
      }
      updateContextMenu(nodes);
    }

    void contextMenuClick(object sender, EventArgs e)
    {
      ToolStripMenuItem menuItem = sender as ToolStripMenuItem;

      List<NodeId> nodes = new List<NodeId>();
      GrtTreeNode treeNode = layerTreeView.SelectedNode.Tag as GrtTreeNode;
      nodes.Add(treeNode.NodeId);
      string command = menuItem.Tag as string;

      if (!layerTreeModel.GrtTree.activate_popup_item_for_nodes(command, nodes))
      {
        // The backend could not handle the menu command. See if we can make sense of it here.
        // Currently there is no command defined that could not be handled by the backend.
      }
    }
  }
}